package cn.kinyun.scrm.weixin.sdk.api.applet;

import cn.kinyun.scrm.weixin.sdk.entity.ErrorCode;
import cn.kinyun.scrm.weixin.sdk.entity.miniprogram.req.account.*;
import cn.kinyun.scrm.weixin.sdk.entity.miniprogram.resp.account.*;
import cn.kinyun.scrm.weixin.sdk.exception.WeixinException;
import com.google.common.base.Preconditions;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;

/**
 * @author Created by chenjiaqiang on 2022/01/17
 **/
@Slf4j
@Component
public class AppletAccountAPI {

    @Autowired
    private RestTemplate restTemplate;

    @Value("${wx.applet.account.basicInfo}")
    private String getBasicInfoUrl;

    @Value("${wx.applet.account.modifyDomain}")
    private String modifyDomainUrl;

    @Value("${wx.applet.account.getEffectiveServerDomain}")
    private String getEffectiveServerDomainUrl;

    @Value("${wx.applet.account.setWebViewDomain}")
    private String setWebViewDomainUrl;

    @Value("${wx.applet.account.checkWxVerifyNickname}")
    private String checkWxVerifyNicknameUrl;

    @Value("${wx.applet.account.setNickname}")
    private String setNicknameUrl;

    @Value("${wx.applet.account.modifyHeadImage}")
    private String modifyHeadImageUrl;

    @Value("${wx.applet.account.modifySignature}")
    private String modifySignatureUrl;

    @Value("${wx.applet.account.changeWxaSearchStatus}")
    private String changeWxaSearchStatusUrl;

    @Value("${wx.applet.account.getEffectiveJumpDomain}")
    private String getEffectiveJumpDomainUrl;

    /**
     * 获取小程序的基本信息
     *
     * @param accessToken
     * @return
     */
    public GetBasicInfoResp getBasicInfo(@NonNull String accessToken) {
        log.info("getBasicInfo with accessToken={}", accessToken);
        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        // 发送请求
        String url = MessageFormat.format(getBasicInfoUrl, accessToken);
        ResponseEntity<GetBasicInfoResp> resp = restTemplate.getForEntity(url, GetBasicInfoResp.class);
        GetBasicInfoResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 设置服务器域名
     *
     * @param accessToken
     * @param req
     * @return
     */
    public ModifyDomainResp modifyDomain(@NonNull String accessToken, @NonNull ModifyDomainReq req) {
        log.info("modifyDomain with req={}", req);
        req.validate();
        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        ResponseEntity<ModifyDomainResp> resp = null;
        // 构造请求体
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<ModifyDomainReq> httpEntity = new HttpEntity<>(req, headers);
        // 发送请求
        String url = MessageFormat.format(modifyDomainUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, ModifyDomainResp.class);

        ModifyDomainResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取发布后生效服务器域名列表
     * @param accessToken
     * @return
     */
    public GetEffectiveServerDomainResp getEffectiveServerDomain(@NonNull String accessToken) {
        log.info("getEffectiveServerDomain with accessToken={}", accessToken);
        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        // 构造请求体
        HttpHeaders headers = new HttpHeaders();
        HttpEntity<String> httpEntity = new HttpEntity<>("{}", headers);
        // 发送请求
        String url = MessageFormat.format(getEffectiveServerDomainUrl, accessToken);
        ResponseEntity<GetEffectiveServerDomainResp> resp = restTemplate.postForEntity(url, httpEntity, GetEffectiveServerDomainResp.class);
        GetEffectiveServerDomainResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 设置业务域名
     *
     * @param req
     * @param accessToken
     * @return
     */
    public ErrorCode setWebViewDomain(@NonNull SetWebViewDomainReq req, @NonNull String accessToken) {
        log.info("setWebViewDomain with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        ResponseEntity<ErrorCode> resp = null;
        // 构造请求体
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<SetWebViewDomainReq> httpEntity = new HttpEntity<>(req, headers);
        // 发送请求
        String url = MessageFormat.format(setWebViewDomainUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 微信认证名称检测
     *
     * @param accessToken
     * @param req
     * @return
     */
    public CheckWxVerifyNicknameResp checkWxVerifyNickname(@NonNull String accessToken, @NonNull CheckWxVerifyNicknameReq req) {
        log.info("checkWxVerifyNickname with req={}", req);
        req.validate();
        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        ResponseEntity<CheckWxVerifyNicknameResp> resp = null;
        // 构造请求体
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<CheckWxVerifyNicknameReq> httpEntity = new HttpEntity<>(req, headers);
        // 发送请求
        String url = MessageFormat.format(checkWxVerifyNicknameUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, CheckWxVerifyNicknameResp.class);

        CheckWxVerifyNicknameResp result = resp.getBody();
        log.info("checkWxVerifyNickname with result={}", resp.getBody());
        WeixinException.isSuccess(result);
        return result;

    }

    /**
     * 设置名称
     *
     * @param req
     * @param accessToken
     * @return
     */
    public SetNicknameResp setNickname(@NonNull SetNicknameReq req, @NonNull String accessToken) {
        log.info("setNickname with req={}", req);
        req.validate();
        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        ResponseEntity<SetNicknameResp> resp = null;
        // 构造请求体
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<SetNicknameReq> httpEntity = new HttpEntity<>(req, headers);
        // 发送请求
        String url = MessageFormat.format(setNicknameUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, SetNicknameResp.class);

        SetNicknameResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 修改头像
     *
     * @param accessToken
     * @param req
     * @return
     */
    public ErrorCode modifyHeadImage(@NonNull String accessToken, @NonNull ModifyHeadImageReq req) {
        log.info("modifyHeadImage with req={}", req);
        req.validate();
        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        ResponseEntity<ErrorCode> resp = null;
        // 构造请求体
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<ModifyHeadImageReq> httpEntity = new HttpEntity<>(req, headers);
        // 发送请求
        String url = MessageFormat.format(modifyHeadImageUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 修改简介
     *
     * @param accessToken
     * @param req
     * @return
     */
    public ErrorCode modifySignature(@NonNull String accessToken, @NonNull ModifySignatureReq req) {
        log.info("modifySignature with req={}", req);
        req.validate();
        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");

        ResponseEntity<ErrorCode> resp = null;
        // 构造请求体
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<ModifySignatureReq> httpEntity = new HttpEntity<>(req, headers);
        // 发送请求
        String url = MessageFormat.format(modifySignatureUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;

    }

    /**
     * 修改搜索设置
     *
     * @param accessToken
     * @param req
     * @return
     */
    public ErrorCode changeWxaSearchStatus(@NonNull String accessToken, @NonNull ChangeWxaSearchStatusReq req) {
        log.info("changeWxaSearchStatus with req={}", req);
        req.validate();
        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");

        ResponseEntity<ErrorCode> resp = null;
        // 构造请求体
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<ChangeWxaSearchStatusReq> httpEntity = new HttpEntity<>(req, headers);
        // 发送请求
        String url = MessageFormat.format(changeWxaSearchStatusUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;

    }

    /**
     * 获取发布后生效业务域名列表
     * @param accessToken
     * @return
     */
    public GetEffectiveJumpDomainResp getEffectiveJumpDomain(@NonNull String accessToken) {
        log.info("getEffectiveJumpDomain, token:{}", accessToken);
        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");

        ResponseEntity<GetEffectiveJumpDomainResp> resp = null;
        // 构造请求体
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<String> httpEntity = new HttpEntity<>("{}", headers);

        // 发送请求
        String url = MessageFormat.format(getEffectiveJumpDomainUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, GetEffectiveJumpDomainResp.class);

        GetEffectiveJumpDomainResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }
}
