/**
 * kuaike.com Inc. Copyright (c) 2014-2019 All Rights Reserved.
 */
package cn.kinyun.scrm.weixin.sdk.api.account;

import cn.kinyun.scrm.weixin.sdk.entity.account.CreateQrcodeResult;
import cn.kinyun.scrm.weixin.sdk.exception.WeixinException;
import com.google.common.collect.Maps;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.text.MessageFormat;
import java.util.Map;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

/**
 * 公众号创建二维码的接口.
 * 
 * <p>
 * 为了满足用户渠道推广分析和用户帐号绑定等场景的需要，公众平台提供了生成带参数二维码的接口。使用该接口可以获得多个带不同场景值的二维码，用户扫描后，公众号可以接收到事件推送。
 * </p>
 * 
 * @title WxQrcodeAPI
 * @author yanmaoyuan
 * @date 2019年7月25日
 * @version 1.0
 * @see <a href="https://mp.weixin.qq.com/wiki?t=resource/res_main&id=mp1443433542">生成带参数的二维码</a>
 */
@Slf4j
@Component
public class WxQrcodeAPI {

    @Autowired
    private RestTemplate restTemplate;

    @Value("${wx.qrcode.create}")
    private String wxQrcodeCreate;

    @Value("${wx.qrcode.get}")
    private String wxQrcodeGet;

    /**
     * 创建临时二维码
     * 
     * @throws WeixinException
     */
    public CreateQrcodeResult createQrcode(@NonNull String accessToken, @NonNull Integer sceneId, Integer expireSeconds)
        throws WeixinException {
        log.info("create temp qrcode with sceneId={}, expireSeconds={}", sceneId, expireSeconds);

        /**
         * 构造请求头
         */
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        /**
         * 构造请求体
         */
        // 请求参数
        Map<String, Object> params = Maps.newHashMap();
        if (expireSeconds != null && expireSeconds > 0) {
            params.put("expire_seconds", expireSeconds);
        }
        params.put("action_name", "QR_SCENE");

        Map<String, Object> scene = Maps.newHashMap();
        scene.put("scene_id", sceneId);

        Map<String, Object> actionInfo = Maps.newHashMap();
        actionInfo.put("scene", scene);

        params.put("action_info", actionInfo);

        // 设置群发消息体
        HttpEntity<Map<String, Object>> request = new HttpEntity<Map<String, Object>>(params, headers);

        /**
         * 发送请求
         */
        String url = MessageFormat.format(wxQrcodeCreate, accessToken);
        ResponseEntity<CreateQrcodeResult> response =
            restTemplate.postForEntity(url, request, CreateQrcodeResult.class);

        /**
         * 处理错误码
         */
        CreateQrcodeResult result = response.getBody();
        WeixinException.isSuccess(result);

        return result;
    }

    /**
     * 创建临时二维码
     * 
     * @throws WeixinException
     */
    public CreateQrcodeResult createQrcode(@NonNull String accessToken, @NonNull String sceneStr, Integer expireSeconds)
        throws WeixinException {
        log.info("create temp qrcode with sceneStr={}, expireSeconds={}", sceneStr, expireSeconds);
        /**
         * 构造请求头
         */
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        /**
         * 构造请求体
         */
        // 请求参数
        Map<String, Object> params = Maps.newHashMap();
        if (expireSeconds != null && expireSeconds > 0) {
            params.put("expire_seconds", expireSeconds);
        }

        params.put("action_name", "QR_STR_SCENE");

        Map<String, Object> scene = Maps.newHashMap();
        scene.put("scene_str", sceneStr);

        Map<String, Object> actionInfo = Maps.newHashMap();
        actionInfo.put("scene", scene);

        params.put("action_info", actionInfo);

        // 设置群发消息体
        HttpEntity<Map<String, Object>> request = new HttpEntity<Map<String, Object>>(params, headers);

        /**
         * 发送请求
         */
        String url = MessageFormat.format(wxQrcodeCreate, accessToken);
        ResponseEntity<CreateQrcodeResult> response =
            restTemplate.postForEntity(url, request, CreateQrcodeResult.class);

        /**
         * 处理错误码
         */
        CreateQrcodeResult result = response.getBody();
        WeixinException.isSuccess(result);

        return result;
    }

    /**
     * 创建永久二维码
     * 
     * @throws WeixinException
     */
    public CreateQrcodeResult createQrcode(@NonNull String accessToken, @NonNull Integer sceneId) throws WeixinException {
        log.info("create qrcode with sceneId={}", sceneId);
        /**
         * 构造请求头
         */
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        /**
         * 构造请求体
         */
        // 请求参数
        Map<String, Object> params = Maps.newHashMap();
        params.put("action_name", "QR_LIMIT_SCENE");

        Map<String, Object> scene = Maps.newHashMap();
        scene.put("scene_id", sceneId);

        Map<String, Object> actionInfo = Maps.newHashMap();
        actionInfo.put("scene", scene);

        params.put("action_info", actionInfo);

        // 设置群发消息体
        HttpEntity<Map<String, Object>> request = new HttpEntity<Map<String, Object>>(params, headers);

        /**
         * 发送请求
         */
        String url = MessageFormat.format(wxQrcodeCreate, accessToken);
        ResponseEntity<CreateQrcodeResult> response =
            restTemplate.postForEntity(url, request, CreateQrcodeResult.class);

        /**
         * 处理错误码
         */
        CreateQrcodeResult result = response.getBody();
        WeixinException.isSuccess(result);

        return result;
    }

    /**
     * 创建永久二维码
     * 
     * @throws WeixinException
     */
    public CreateQrcodeResult createQrcode(@NonNull String accessToken, @NonNull String sceneStr) throws WeixinException {
        log.info("create qrcode with sceneStr={}", sceneStr);
        /**
         * 构造请求头
         */
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        /**
         * 构造请求体
         */
        // 请求参数
        Map<String, Object> params = Maps.newHashMap();
        params.put("action_name", "QR_LIMIT_STR_SCENE");

        Map<String, Object> scene = Maps.newHashMap();
        scene.put("scene_str", sceneStr);

        Map<String, Object> actionInfo = Maps.newHashMap();
        actionInfo.put("scene", scene);

        params.put("action_info", actionInfo);

        // 设置群发消息体
        HttpEntity<Map<String, Object>> request = new HttpEntity<Map<String, Object>>(params, headers);

        /**
         * 发送请求
         */
        String url = MessageFormat.format(wxQrcodeCreate, accessToken);
        ResponseEntity<CreateQrcodeResult> response =
            restTemplate.postForEntity(url, request, CreateQrcodeResult.class);

        /**
         * 处理错误码
         */
        CreateQrcodeResult result = response.getBody();
        WeixinException.isSuccess(result);

        return result;
    }

    /**
     * 通过ticket换取二维码
     * 
     * @param ticket
     * @return
     */
    public String getQrcodeUrl(@NonNull String ticket) {
        try {
            ticket = URLEncoder.encode(ticket, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            
        }
        return MessageFormat.format(wxQrcodeGet, ticket);
    }
}