package cn.kinyun.scrm.weixin.sdk.api;

import cn.kinyun.scrm.weixin.sdk.entity.user.req.TagModReq;
import cn.kinyun.scrm.weixin.sdk.entity.user.resp.TagListResp;
import cn.kinyun.scrm.weixin.sdk.entity.user.resp.TagModResp;
import cn.kinyun.scrm.weixin.sdk.exception.WeixinException;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

/**
 * 标签管理
 * 
 * @author : shanyu
 * @version V1.0
 * @Project: weixin-platform
 * @Package com.kuaike.weixin.api
 * @Description: 标签管理
 * @date Date : 2019年04月28日 14:39
 */
@Slf4j
@Component
public class WxTagAPI {

    @Autowired
    private RestTemplate restTemplate;

    @Value("${wx.tag.create}")
    private String wxTagCreate;

    @Value("${wx.tag.get}")
    private String wxTagGet;

    @Value("${wx.tag.update}")
    private String wxTagUpdate;

    @Value("${wx.tag.delete}")
    private String wxTagDelete;

    @Value("${wx.user.tage.get")
    private String wxUserTageGet;

    /**
     * 创建标签
     *
     * @param accessToken 授权方accessToken（第三方平台调用）||公众号accessToken（公众号调用）
     * @param tagModReq 新增标签（只需要name，不需要id）
     * @return
     */
    public TagModResp createTag(@NonNull String accessToken, @NonNull TagModReq tagModReq) throws WeixinException {
        log.info("create tag with params={}", tagModReq);
        String url = MessageFormat.format(wxTagCreate, accessToken);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<TagModReq> request = new HttpEntity<TagModReq>(tagModReq, headers);
        ResponseEntity<TagModResp> response = restTemplate.postForEntity(url, request, TagModResp.class);
        TagModResp result = response.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 编辑标签
     *
     * @param accessToken 授权方accessToken（第三方平台调用）||公众号accessToken（公众号调用）
     * @param tagModReq 新增标签（id和name都需要）
     * @return
     */
    public TagModResp updateTag(@NonNull String accessToken, @NonNull TagModReq tagModReq) throws WeixinException {
        log.info("update tag with params={}", tagModReq);
        String url = MessageFormat.format(wxTagUpdate, accessToken);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<TagModReq> request = new HttpEntity<TagModReq>(tagModReq, headers);
        ResponseEntity<TagModResp> response = restTemplate.postForEntity(url, request, TagModResp.class);
        TagModResp result = response.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 删除标签
     *
     * @param accessToken 授权方accessToken（第三方平台调用）||公众号accessToken（公众号调用）
     * @param tagModReq 新增标签（只需要id）
     * @return
     */
    public TagModResp deleteTag(@NonNull String accessToken, @NonNull TagModReq tagModReq) throws WeixinException {
        log.info("delete tag with params={}", tagModReq);
        String url = MessageFormat.format(wxTagDelete, accessToken);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<TagModReq> request = new HttpEntity<TagModReq>(tagModReq, headers);
        ResponseEntity<TagModResp> response = restTemplate.postForEntity(url, request, TagModResp.class);
        TagModResp result = response.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     *  获取标签列表
     *
     * @param accessToken 授权方accessToken（第三方平台调用）||公众号accessToken（公众号调用）
     * @return
     */
    public TagListResp getTagList(@NonNull String accessToken) throws WeixinException {
        log.info("get tag list.");
        String url = MessageFormat.format(wxUserTageGet, accessToken);
        ResponseEntity<TagListResp> response = restTemplate.postForEntity(url, null, TagListResp.class);
        TagListResp result = response.getBody();
        WeixinException.isSuccess(result);
        return result;
    }
}
