package com.kuaike.scrm.common.perm.interceptor;

import com.kuaike.common.errorcode.CommonErrorCode;
import com.kuaike.common.utils.*;
import com.kuaike.scrm.common.dto.CurrentUserInfo;
import com.kuaike.scrm.common.perm.service.AccessPermitService;
import com.kuaike.scrm.common.perm.dto.AccessBizInfo;
import com.kuaike.scrm.common.perm.dto.AccessUserInfo;
import com.kuaike.scrm.common.utils.LoginUtils;
import com.kuaike.scrm.common.perm.utils.PermUtils;
import com.kuaike.scrm.common.utils.ServletUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.apache.http.HttpStatus;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.handler.HandlerInterceptorAdapter;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Collections;
import java.util.Map;
import java.util.Objects;

/**
 * @author sunwen
 */
@Slf4j
@Component
public class AccessInterceptor extends HandlerInterceptorAdapter {

    @Resource
    private RedisTemplate<String, Object> redisTemplate;

    @Value("${spring.redis.key.prefix}")
    private String redisKeyPrefix;

    @Value("${permission.prefix:}")
    private String prefix;

    @Autowired
    private AccessPermitService accessPermitService;

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) throws Exception {
        LoginUtils.clear();
        LoginUtils.setSession(request.getSession());

        if (!(handler instanceof HandlerMethod)) {
            // 访问URL不是后端接口(如图片网页资源)，无需权限验证
            return true;
        }

        // 初始化客户信息
        // 即使无需登录，也初始化客户信息。
        LoginUtils.initCurrentUser(redisTemplate, redisKeyPrefix);

        // 无需登录的注解
        boolean exist = LoginUtils.hasLoginNeedless(handler);
        if (exist) {
            return true;
        }

        String requestUri = request.getRequestURI();
        String actualUri = PermUtils.getPathWithPrefix(prefix, requestUri);
        log.debug("requestUri:{}", actualUri);
        if (StringUtils.isNotBlank(requestUri) && requestUri.endsWith("np")) {
            return true;
        }

        CurrentUserInfo currentUserInfo  = LoginUtils.getCurrentUser();
        log.debug("currentUserInfo:{}", currentUserInfo);
        if (null == currentUserInfo) {
            returnAjaxNotLoginResponse(request,response);
            return false;
        }

        Long userId = currentUserInfo.getId();
        if (Objects.isNull(userId)) {
            log.warn("当前操作用户userId为空");
            returnAjaxNotLoginResponse(request,response);
            return false;
        }

        // 查询当前登录的用户信息
        AccessUserInfo user = accessPermitService.getAccessUser();

        if (Objects.isNull(user)) {
            log.warn("根据userId:{}未查询到用户信息", userId);
            returnAjaxNotLoginResponse(request,response);
        }

        if (user != null && NumberUtils.INTEGER_ZERO.equals(user.getEnabled())) {
            log.warn("用户:{}已被禁用", user.getId());
            returnAjaxNotLoginResponse(request,response);
            return false;
        }

        if (user != null && StringUtils.isNotBlank(user.getWeworkUserId())
                && NumberUtils.INTEGER_ZERO.equals(user.getIsAgentAuth())) {
            log.warn("weworkUser:{}", user.getWeworkUserId());
            returnAjaxBusinessCustomerForbid(response, "当前用户未授权,暂不能登录");
            return false;
        }

        // 商户被禁用后，在线用户踢出
        AccessBizInfo businessCustomer = accessPermitService.getAccessBiz();
        if (Objects.isNull(businessCustomer)) {
            returnAjaxBusinessCustomerForbid(response, "商户不存在");
            return false;
        }

        // 校验接口权限
        boolean permit = accessPermitService.checkPermission(actualUri);
        log.debug("uri:{}, permit:{}", actualUri, permit);
        if (!permit) {
            returnNoPermissionResponse(response);
            return false;
        }

        return super.preHandle(request, response, handler);
    }

    @Override
    public void afterCompletion(HttpServletRequest request, HttpServletResponse response, Object handler, Exception ex) throws Exception {
        LoginUtils.clear();
    }

    private void returnAjaxNotLoginResponse(HttpServletRequest request, HttpServletResponse response) {
		long code = ErrorCodeUtil.getUniverseErrorCode(CommonErrorCode.NO_LOGIN);
		Map<String, String> map = Collections.singletonMap("redirectUrl", ServletUtil.getBasePath(request) + "/login?service=");
        response.setStatus(HttpStatus.SC_UNAUTHORIZED);
		WebCommonUtil.returnMessage(response, JacksonUtil.obj2Str(new ApiResult(code, "用户未登录", map)));
	}

    private void returnNoPermissionResponse(HttpServletResponse response) {
        response.setStatus(HttpStatus.SC_FORBIDDEN);
        WebCommonUtil.returnMessage(response,
                JacksonUtil.obj2Str(ApiResultUtils.error(CommonErrorCode.PERMISSION_DENY)));
    }


    // 用户所在商户被禁用或过期
    private void returnAjaxBusinessCustomerForbid(HttpServletResponse response, String msg) {
        response.setStatus(HttpStatus.SC_FORBIDDEN);
        WebCommonUtil.returnMessage(response,
                JacksonUtil.obj2Str(ApiResultUtils.error(CommonErrorCode.PERMISSION_DENY, msg)));
    }

}
