package com.baijiayun.openapi.sdk.utils;

import com.baijiayun.openapi.sdk.base.BjyPartnerInfo;
import com.baijiayun.openapi.sdk.base.BjyResult;
import com.baijiayun.openapi.sdk.base.Request;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.NameValuePair;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.*;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.HttpMultipartMode;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.FileBody;
import org.apache.http.entity.mime.content.StringBody;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;

import java.io.*;
import java.lang.reflect.Field;
import java.lang.reflect.Type;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.Map.Entry;

@Slf4j
public class BjyServiceUtil {

    private static final String SIGN_SPLIT = "&";

    private static final ObjectMapper MAPPER;

    private BjyServiceUtil() {}
    static {
        MAPPER = new ObjectMapper();
        MAPPER.configure(SerializationFeature.FAIL_ON_EMPTY_BEANS, false);
        MAPPER.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);

        // 注册自定义的File序列化器，阻止序列化 File 对象
        SimpleModule module = new SimpleModule();
        module.addSerializer(File.class, new IgnoreFileSerializer());
        MAPPER.registerModule(module);
    }

    /**
     * 生成签名字符串
     *
     * @param params
     * @return
     */
    public static String createSignData(Map<String, String> params) {
        StringBuilder signData = new StringBuilder();

        if (params != null && !params.isEmpty()) {
            int size = params.size();
            String[] paraNames = new String[size];
            Arrays.sort(params.keySet().toArray(paraNames), (o1, o2) -> {
                int result = o1.compareTo(o2);
                if (result == 0) {
                    return 0;
                }
                if (o1.equals("partner_key")) {
                    result = 1;
                }
                if (o2.equals("partner_key")) {
                    result = -1;
                }
                return result;
            });
            for (int i = 0; i < size; i++) {
                String paraName = paraNames[i];
                String value = params.get(paraName);
                if (i != 0) {
                    signData.append(SIGN_SPLIT);
                }
                signData.append(paraName).append("=").append(value);
            }
        }
        return signData.toString();
    }

    /**
     * 生成签名数据
     *
     * @param signData 签名字符串
     * @return 生成MD5字符串
     */
    public static String getSignature(String signData) {
        return DigestUtils.md5Hex(signData);
    }

    ///////////// 百家云服务端对接方法 //////////////

    /**
     * 组装请求url，在 uri 前面拼接 privateDomain
     *
     * @see <a href="https://dev.baijiayun.com/wiki/detail/54">专属域名说明</a>
     * @param privateDomain 专属域名
     * @param uri openapi uri
     * @return 请求url
     */
    public static String getUrl(String privateDomain, String uri) {
        if (StringUtils.isBlank(privateDomain)) {
            return uri;
        }
        if (StringUtils.isBlank(uri)) {
            return privateDomain;
        }
        if (!privateDomain.endsWith("/") && !uri.trim().startsWith("/")) {
            privateDomain += "/";
        } else if (privateDomain.endsWith("/") && uri.startsWith("/")){
            uri = uri.substring(1);
        }
        return privateDomain + uri;
    }

    /**
     * 参数服务端签名
     *
     * @see <a href="https://dev.baijiayun.com/wiki/detail/97">接口对接标准流程说明</a>
     * @param paramMap 参数
     * @param partnerInfo 百家云合作者信息
     */
    public static void sign(Map<String, String> paramMap, BjyPartnerInfo partnerInfo) {
        // 放入固定参数
        paramMap.put("partner_id", partnerInfo.getPartnerId());
        paramMap.put("timestamp", String.valueOf(System.currentTimeMillis()));

        // 计算签名
        List<String> list = new ArrayList<>();
        TreeMap<String, String> sortedMap = new TreeMap<>(paramMap);
        sortedMap.forEach((key, values) -> {
            if (StringUtils.isEmpty(values)) {
                list.add(key + "=");
            } else {
                list.add(key + "=" + values);
            }
        });
        // partner_key总是拼在字符串最后面，并不参与key的排序。
        // partner_key只是计算签名时需要，在发送请求时不需要发partner_key
        String signData = StringUtils.join(list, "&") + "&partner_key=" + partnerInfo.getPartnerKey();
        String sign = DigestUtils.md5Hex(signData);
        paramMap.put("sign", sign);
    }

    //////////////// get&post /////////////////

    public static <T> BjyResult<T> get(BjyPartnerInfo base,
                                       String uri,
                                       Object params,
                                       Class<T> dataType) throws IOException {
        // 发送请求
        String resultJson = invoke(Request.Method.GET, base, uri, params);

        // 解析结果
        try {
            return str2BjcloudResult(resultJson, dataType);
        } catch (IOException e) {
            log.error("解析json失败", e);
            throw new IllegalArgumentException("解析json失败", e);
        }
    }

    public static <T> BjyResult<T> get(BjyPartnerInfo base,
                                       String uri,
                                       Object params,
                                       TypeReference<BjyResult<T>> typeReference) throws IOException {
        String resultJson = invoke(Request.Method.GET, base, uri, params);

        // 解析结果
        try {
            return str2Obj(resultJson, typeReference);
        } catch (IOException e) {
            log.error("解析json失败", e);
            throw new IllegalArgumentException("解析json失败", e);
        }
    }

    public static BjyResult<Void> post(BjyPartnerInfo base,
                                       String uri,
                                       Object params) throws IOException {
        // 重载方法，无需返回值。
        return post(base, uri, params, Void.class);
    }

    public static <T> BjyResult<T> post(BjyPartnerInfo base,
                                        String uri,
                                        Object params,
                                        Class<T> dataType) throws IOException {
        String resultJson = invoke(Request.Method.POST, base, uri, params);

        // 解析结果
        try {
            return str2BjcloudResult(resultJson, dataType);
        } catch (IOException e) {
            log.error("解析json失败", e);
            throw new IllegalArgumentException("解析json失败", e);
        }
    }

    // jackson 对深度嵌套的范型类型反序列化，需要使用 TypeReference 将类型传入。例如 BjcloudResult<List<Map<String, RoomInfo>>>
    public static <T> T post(BjyPartnerInfo base,
                             String uri,
                             Object params,
                             TypeReference<T> typeReference) throws IOException {
        String resultJson = invoke(Request.Method.POST, base, uri, params);

        // 解析结果
        try {
            return str2Obj(resultJson, typeReference);
        } catch (IOException e) {
            log.error("解析json失败", e);
            throw new IllegalArgumentException("解析json失败", e);
        }
    }

    //////////// Http请求 //////////////
    public static CloseableHttpClient getHttpClient() {
        RequestConfig config = RequestConfig.custom().setConnectTimeout(60000).setSocketTimeout(35000).build();
        return HttpClientBuilder.create().setDefaultRequestConfig(config).build();
    }

    public static String invoke(Request.Method method, BjyPartnerInfo base, String uri, Object params) throws IOException {
        String url = getUrl(base.getPrivateDomain(), uri);

        Map<String, String> paramMap = beanToMap(params);
        sign(paramMap, base);// 生成签名

        // 发送请求
        String resultJson;
        if (method == Request.Method.GET) {
            resultJson = doGet(url, paramMap, null);
        } else {
            // 是否有文件？
            Map<String, File> fileMap = beanToFileMap(params);
            if (fileMap.isEmpty()) {
                resultJson = doPost(url, paramMap, null, null);
            } else {
                resultJson = doPost(url, paramMap, fileMap, null);
            }
        }
        log.info("resultJson＝{}", resultJson);
        return resultJson;
    }

    public static String doGet(String url, Map<String, String> params, Map<String, String> headers) throws IOException {
        Preconditions.checkArgument(StringUtils.isNotBlank(url), "url is blank");

        try (CloseableHttpClient httpClient = getHttpClient()) {
            if (params != null && !params.isEmpty()) {
                url = url + "?" + EntityUtils.toString(getUrlEncodedFormEntity(params));
            }
            HttpGet httpGet = new HttpGet(url);
            setHeaders(httpGet, headers);
            return execute(httpClient, httpGet);
        } catch (IOException e) {
            log.error("doGet exception - ", e);
            throw e;
        }
    }

    public static String doPost(String url, Map<String, String> params, Map<String, File> files, Map<String, String> headers) throws IOException {
        Preconditions.checkArgument(StringUtils.isNotBlank(url), "url is blank");
        try (CloseableHttpClient httpClient = getHttpClient()) {
            HttpPost post = new HttpPost(url);
            if (files == null || files.isEmpty()) {
                post.setEntity(getUrlEncodedFormEntity(params));// application/x-www-form-urlencoded
            } else {
                post.setEntity(getMultipartFormEntity(params, files));// multipart/form-data
            }
            setHeaders(post, headers);
            return execute(httpClient, post);
        } catch (IOException e) {
            log.error("doPost exception - ", e);
            throw e;
        }
    }

    private static HttpEntity getUrlEncodedFormEntity(Map<String, String> params) {
        List<NameValuePair> pairs = Lists.newArrayListWithCapacity(params.size());
        for (Entry<String, String> entry : params.entrySet()) {
            String value = entry.getValue();
            if (value != null) {
                pairs.add(new BasicNameValuePair(entry.getKey(), value));
            }
        }
        return new UrlEncodedFormEntity(pairs, StandardCharsets.UTF_8);
    }

    private static HttpEntity getMultipartFormEntity(Map<String, String> params, Map<String, File> files) {
        MultipartEntityBuilder builder = MultipartEntityBuilder.create();
        builder.setMode(HttpMultipartMode.RFC6532);// 中文文件名兼容
        if (params != null && !params.isEmpty()) {
            for (Entry<String, String> param : params.entrySet()) {
                builder.addPart(param.getKey(), new StringBody(param.getValue(), ContentType.TEXT_PLAIN.withCharset(StandardCharsets.UTF_8)));
            }
        }
        if (files != null && !files.isEmpty()) {
            for (Entry<String, File> entry : files.entrySet()) {
                File file = entry.getValue();
                builder.addPart(entry.getKey(), new FileBody(file, ContentType.DEFAULT_BINARY, new String(file.getName().getBytes(StandardCharsets.UTF_8))));
            }
        }
        return builder.build();
    }

    private static void setHeaders(HttpRequestBase req, Map<String, String> headers) {
        if (headers != null && !headers.isEmpty()) {
            headers.forEach(req::setHeader);
        }
    }

    private static String execute(CloseableHttpClient client, HttpUriRequest request) throws IOException {
        CloseableHttpResponse response = client.execute(request);
        int statusCode = response.getStatusLine().getStatusCode();
        if (statusCode != 200) {
            request.abort();
            throw new IllegalStateException("HttpClient,error status code :" + statusCode);
        } else {
            HttpEntity entity = response.getEntity();
            String result = null;
            if (entity != null) {
                result = EntityUtils.toString(entity, StandardCharsets.UTF_8);
            }

            EntityUtils.consume(entity);
            response.close();
            return result;
        }
    }

    //////////// 序列化&反序列化 //////////////

    /**
     * Jackson方式的bean转换成Map
     *
     * @param bean 请求对象 Java Bean
     * @return 可以供HttpClient发送的Map
     */
    public static Map<String, String> beanToMap(Object bean) throws IOException {
        HashMap<String, String> params = Maps.newHashMap();
        Map<String, Object> paramMap = MAPPER.convertValue(bean, new TypeReference<HashMap<String, Object>>() {});
        if (MapUtils.isNotEmpty(paramMap)) {
            for (Entry<String, Object> entry : paramMap.entrySet()) {
                String key = entry.getKey();
                Object value = entry.getValue();
                if (null == value) {
                    continue;// ignore null value
                }
                if (value instanceof File || value instanceof InputStream || value instanceof OutputStream) {
                    // 有文件，不要转换为map中的元素。
                    log.info("ignore file field: {}", key);
                } else if (value instanceof String) {
                    params.put(key, (String) value);
                } else if (value instanceof Number || value instanceof Boolean || value instanceof Enum || value instanceof Character || value instanceof CharSequence) {
                    params.put(key, String.valueOf(value));
                } else {
                    // 如果是 List、Set、Map或者其它复杂的对象类型，则转换为json字符串。
                    params.put(key, MAPPER.writeValueAsString(value));
                }
            }
        }
        return params;
    }

    @SuppressWarnings("unchecked")
    public static Map<String, File> beanToFileMap(Object obj) {
        Map<String, File> params = Maps.newHashMap();

        // 如果是Map类型，尝试遍历Map类型
        if (obj instanceof Map) {
            Map<String, Object> paramMap = (Map<String, Object>) obj;
            if (MapUtils.isNotEmpty(paramMap)) {
                for (Entry<String, Object> entry : paramMap.entrySet()) {
                    String key = entry.getKey();
                    Object value = entry.getValue();
                    if (null == value) {
                        continue;// ignore null value
                    }
                    if (value instanceof File) {
                        params.put(key, (File) value);
                    }
                }
            }
            return params;
        }

        // 忽略掉不可能的类型
        if (obj instanceof Iterable || obj instanceof Enum || obj instanceof Number || obj instanceof Boolean ||
                obj instanceof Character || obj instanceof CharSequence) {
            return params;
        }

        // 通过反射获取File类型的属性
        Class<?> clazz = obj.getClass();
        Field[] fields = clazz.getDeclaredFields();

        for (Field field : fields) {
            if (field.getType().equals(File.class)) {
                field.setAccessible(true);
                try {
                    File file = (File) field.get(obj);
                    if (file == null) {// 忽略空字段
                        continue;
                    }

                    String fieldName = field.getName();
                    // 判断是否有注解的序列化名称
                    JsonProperty jsonProperty = field.getAnnotation(JsonProperty.class);
                    if (jsonProperty != null) {
                        fieldName = jsonProperty.value();
                    }
                    params.put(fieldName, file);
                } catch (IllegalAccessException e) {
                    log.error("get file field error: {}", field, e);
                }
            }
        }
        return params;
    }

    public static <T> T str2Obj(String s, TypeReference<T> valueType) throws IOException {
        return MAPPER.readValue(s, valueType);
    }

    public static <T> T str2Obj(String s, Class<T> valueType) throws IOException {
        return MAPPER.readValue(s, valueType);
    }

    public static <T> T str2Obj(String s, JavaType valueType) throws IOException {
        return MAPPER.readValue(s, valueType);
    }

    public static JavaType constructType(Type type) {
        return MAPPER.getTypeFactory().constructType(type);
    }

    /**
     * 泛型转换
     * @param json json数据
     * @param valueType 值类型
     * @return 转换好的范型类型对象
     * @param <T> 值类型
     * @throws IOException JSON反序列化失败后抛出IO异常
     */
    public static <T> BjyResult<T> str2BjcloudResult(String json, Class<T> valueType) throws IOException {
        JavaType javaType = MAPPER.getTypeFactory().constructParametricType(BjyResult.class, valueType);
        return MAPPER.readValue(json, javaType);
    }
}
