
package com.baijia.tianxiao.filter;

import com.baijia.tianxiao.dto.mobile.AppAuthToken;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.util.ErrorCodeUtil;
import com.baijia.tianxiao.util.encrypt.EncryptUtils;
import com.baijia.tianxiao.util.json.JacksonUtil;
import com.baijia.tianxiao.util.properties.PropertiesReader;
import com.baijia.tianxiao.validation.ValidateTokenUtil;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.*;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import java.io.IOException;

/**
 * @author caoliang
 * @version 1.0
 * @title WrapTokenFromCookieFilter
 * @desc 从cookie中取出org_auth_token，解析后初始化TianXiaoContext
 * @date 2015年12月1日
 */
public class WrapOrgTokenFilter implements Filter {

    private static final String DEFAULTCOOKIENAME = "ORG_AUTH_TOKEN";
    private static final String DEFAULTPARAMNAME = "auth_token";
    private static final String DEFAULTPROPNAME = "passport-client.properties";
    private static final String DEFAULTVALIDTOKENURL = "valid.token.url";
    private static final String TX_APP_ENVIRONMENT = PropertiesReader.getValue("rest", "webapp.deploy.environment");


    private final Logger logger = LoggerFactory.getLogger(this.getClass());

    private String cookieName = DEFAULTCOOKIENAME;

    private String paramName = DEFAULTPARAMNAME;

    @Override
    public void init(FilterConfig filterConfig) throws ServletException {
        // nothing to do
    }

    @Override
    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain)
            throws IOException, ServletException {
        HttpServletRequest servletRequest = (HttpServletRequest) request;
        String encryptedToken = retrieveCookieValue(servletRequest);
        // 先从cookie找，没有找到从参数找，还没有就是没传。pc版做的时候，需要兼容pc的初始化。
        if (StringUtils.isNotBlank(encryptedToken)) {
            initTianxiaoContext(encryptedToken);
            if (logger.isDebugEnabled()) {
                logger.debug("initTianxiaoContext from cookie");
            }
        } else {
            if (logger.isDebugEnabled()) {
                logger.debug("can't find token in cookie. try find by request param");
            }
            encryptedToken = servletRequest.getParameter(paramName);
            if (StringUtils.isNotBlank(encryptedToken)) {
                initTianxiaoContext(encryptedToken);
                if (logger.isDebugEnabled()) {
                    logger.debug("initTianxiaoContext from request");
                }
            } else {
                // if (logger.isDebugEnabled()) {
                // logger.debug("can't find token in cookie and request.");
                // JacksonUtil.writeObj(response.getOutputStream(),
                // new AppBaseResponse("token不能为空", CommonErrorCode.NO_LOGIN.getSubsystemErrorCode(), null));
                // return;
                // }
            }
        }
        try {
            if (StringUtils.isNotBlank(encryptedToken)) {
                AppAuthToken token = JacksonUtil.str2Obj(EncryptUtils.strDecode(encryptedToken), AppAuthToken.class);
                if (!ValidateTokenUtil.isTokenValid(token)) {
                    logger.debug("InnerSync token fail");
                    clearTianxiaoContext();
                    JacksonUtil.writeObj(response.getOutputStream(),
                            new AppBaseResponse("身份信息已失效，请重新登录", ErrorCodeUtil.getUniverseErrorCode(CommonErrorCode.NEED_RE_LOGIN), null));
                    return;
                }
                if (null != token && StringUtils.isNotEmpty(token.getAppEnv()) && !token.getAppEnv().equals(TX_APP_ENVIRONMENT)) {
                    logger.warn("different environment between app and webapp--- webappEnv:[{}],appEnv:[{}],AppAuthToken:[{}]", TX_APP_ENVIRONMENT, token.getAppEnv(), token);
                    clearTianxiaoContext();
                    JacksonUtil.writeObj(response.getOutputStream(),
                            new AppBaseResponse("APP环境已切换，请重新登录", ErrorCodeUtil.getUniverseErrorCode(CommonErrorCode.NEED_RE_LOGIN), null));
                    return;
                }
            }
        } catch (Exception e) {
            logger.warn("valid token error!", e);
        }
        logger.debug("InnerSync token success");

        try {
            chain.doFilter(request, response);
        } finally {
            clearTianxiaoContext();
        }

    }

    @Override
    public void destroy() {

    }

    protected String getCookieName() {
        return cookieName;
    }

    public void setCookieName(String cookieName) {
        this.cookieName = cookieName;
    }

    public String getParamName() {

        return paramName;
    }

    public void setParamName(String paramName) {

        this.paramName = paramName;
    }

    private void initTianxiaoContext(String encryptedToken) {
        try {
            AppAuthToken token = JacksonUtil.str2Obj(EncryptUtils.strDecode(encryptedToken), AppAuthToken.class);
            TianxiaoMContext.setOrgId(token.getUser_id().intValue());
            TianxiaoMContext.setTXCascadeId(token.getTx_cascade_user_id());
            TianxiaoMContext.setTXLoginAccountId(token.getTx_account_id());
            TianxiaoMContext.setAppAuthToken(token);
        } catch (Exception e) {
            logger.warn("decode token error! encryptedToken is :" + encryptedToken, e);
        }
    }

    private String retrieveCookieValue(final HttpServletRequest request) {
        final Cookie cookie = org.springframework.web.util.WebUtils.getCookie(request, getCookieName());
        if (logger.isDebugEnabled()) {
            logger.info("get token from cookie | cookieName:{}, cookie value:{}", getCookieName(),
                    cookie == null ? null : cookie.getValue());

        }
        return cookie == null ? null : cookie.getValue();
    }

    private void clearTianxiaoContext() {
        TianxiaoMContext.clear();
    }

}