package com.baijia.tianxiao.util.excel;

import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.poi.hssf.usermodel.HSSFClientAnchor;
import org.apache.poi.hssf.usermodel.HSSFRichTextString;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.*;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.text.SimpleDateFormat;
import java.util.*;

public class ExcelHelper<T> {
    /**
     * @param request
     * @param response
     * @param data       要导出的数据
     * @param fields     要导出的字段
     * @param fieldValue 表头
     */
    public void exportToExcel(HttpServletRequest request, HttpServletResponse response, List<T> data, List<String> fields,
                              List<String> fieldValue) {
        exportToExcel(request, response, data, fields, fieldValue, null);
    }

    /**
     * @param request
     * @param response
     * @param data       要导出的数据
     * @param fields     要导出的字段
     * @param fieldValue 表头
     * @param fileName   导出文件名称
     */
    public void exportToExcel(HttpServletRequest request, HttpServletResponse response, List<T> data, List<String> fields,
                              List<String> fieldValue, String fileName) {
        if (fields == null || fieldValue == null) {
            throw new RuntimeException("fields or header can't be null.");
        }

        Workbook wb = new HSSFWorkbook();
        Sheet sheet = wb.createSheet("ExportDataList");

        // Create a row and put some cells in it. Rows are 0 based.
        Row headerTitle = sheet.createRow(0);
        int index = 0;
        for (int i = 0; i < fields.size(); i++) {
            Cell cell = headerTitle.createCell(index++ % fields.size());
            cell.setCellValue(fieldValue.get(i));
        }

        for (int i = 0; i < data.size(); i++) {
            Row row = sheet.createRow(i + 1);
            T o = data.get(i);

            for (int j = 0; j < fields.size(); j++) {
                Cell cell = row.createCell(index++ % fields.size());

                try {
                    cell.setCellValue(BeanUtils.getProperty(o, fields.get(j)));
                } catch (Exception e) {
                    throw new RuntimeException("export data error", e);
                }
            }
        }

        if (fileName == null || "".equals(fileName)) {
            fileName = System.currentTimeMillis() + ".xls";
        }

        response.setContentType("application/vnd.ms-excel");

        try {
            response.setHeader("Content-Disposition",
                    "attachment; filename=" + URLEncoder.encode(fileName, "utf-8"));
        } catch (UnsupportedEncodingException e1) {
            response.setHeader("Content-Disposition", "attachment; filename=" + fileName);
        }
        try {
            wb.write(response.getOutputStream());
        } catch (IOException e) {
            throw new RuntimeException("export data error", e);
        }
    }

    public void exportToExcel(HttpServletRequest request, HttpServletResponse response, String fileName, SheetWrapper... sheetWrappers) {

        Workbook wb = new HSSFWorkbook();

        if (null != sheetWrappers) {
            int sheetIndex = 1;
            for (SheetWrapper sheetWrapper : sheetWrappers) {

                String[] fields = sheetWrapper.fields;
                String[] fieldValue = sheetWrapper.fieldValue;
                List<T> data = sheetWrapper.data;

                if (fields == null || fieldValue == null) {
                    throw new RuntimeException("fields or header can't be null.");
                }

                Sheet sheet = wb.createSheet(StringUtils.isNotBlank(sheetWrapper.sheetName) ? sheetWrapper.sheetName : "ExportDataList" + sheetIndex++);

                // Create a row and put some cells in it. Rows are 0 based.
                Row headerTitle = sheet.createRow(0);
                int index = 0;
                for (int i = 0; i < fields.length; i++) {
                    Cell cell = headerTitle.createCell(index++ % fields.length);
                    cell.setCellValue(fieldValue[i]);
                }

                for (int i = 0; i < data.size(); i++) {
                    Row row = sheet.createRow(i + 1);
                    T o = data.get(i);

                    for (int j = 0; j < fields.length; j++) {
                        Cell cell = row.createCell(index++ % fields.length);

                        try {
                            cell.setCellValue(BeanUtils.getProperty(o, fields[j]));
                        } catch (Exception e) {
                            throw new RuntimeException("export data error", e);
                        }
                    }
                }
            }
        }


        if (fileName == null || "".equals(fileName)) {
            fileName = System.currentTimeMillis() + ".xls";
        }

        response.setContentType("application/vnd.ms-excel");
        response.setHeader("Content-Disposition", "attachment; filename=" + fileName);
        try {
            wb.write(response.getOutputStream());
        } catch (IOException e) {
            throw new RuntimeException("export data error", e);
        }
    }


    public <A> void exportToExcel(HttpServletResponse response, String fileName, String sheetName, List<A> datas,
                                  Excelable<A> exporter) {
        Workbook wb = new HSSFWorkbook();
        if (StringUtils.isBlank(sheetName)) {
            sheetName = "ExportDataList";
        }
        createSheet(wb, sheetName, datas, exporter);
        if (StringUtils.isBlank(fileName)) {
            SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMdd_HHmmss");
            fileName = sdf.format(new Date()) + ".xls";
        }
        response.setContentType("application/vnd.ms-excel");
        response.setHeader("Content-Disposition", "attachment; filename=" + fileName);
        try {
            wb.write(response.getOutputStream());
        } catch (IOException e) {
            throw new RuntimeException("export data error", e);
        }
    }

    public <A> Sheet createSheet(Workbook wb, String sheetName, List<A> datas, Excelable<A> exporter) {
        Sheet sheet;
        if (sheetName != null) {
            sheet = wb.createSheet(sheetName);
        } else {
            sheet = wb.createSheet();
        }
        if (datas != null && datas.isEmpty() == false) {
            List<ExcelCell[]> result = preprocessData(datas, exporter);
            int size = result.size();
            Map<ExcelCellStyle, CellStyle> styleMap = new HashMap<ExcelCellStyle, CellStyle>();
            for (int i = 0; i < size; i++) {
                createRow(wb, sheet, i, result.get(i), styleMap);
            }
        }
        return sheet;
    }

    public <A> List<ExcelCell[]> preprocessData(List<A> datas, Excelable<A> exporter) {
        List<ExcelCell[]> result = new ArrayList<ExcelCell[]>(datas.size() + 1);
        result.add(exporter.exportRowName());
        for (A data : datas) {
            result.add(exporter.exportRowValue(data));
        }
        return result;
    }

    public Row createRow(Workbook wb, Sheet sheet, int rowNum, ExcelCell[] values,
                         Map<ExcelCellStyle, CellStyle> styleMap) {
        Row row = sheet.createRow(rowNum);
        ExcelCell value = null;
        int size = values.length;
        Drawing p = sheet.createDrawingPatriarch();
        for (int i = 0; i < size; i++) {
            Cell cell = row.createCell(i);
            value = values[i];
            if (value == null) {
                continue;
            }
            cell.setCellValue(value.getValue().toString());
            if (StringUtils.isBlank(value.getComment()) == false) {
                cell.setCellComment(createCellComment(p, value, (short) i, rowNum));
            }
            if (value.getCellType() != null) {
                cell.setCellType(value.getCellType());
            }
            ExcelCellStyle excelCellStyle = value.getCellStyle();
            if (excelCellStyle != null) {
                CellStyle cellStyle = styleMap.get(excelCellStyle);
                if (cellStyle == null) {
                    cellStyle = wb.createCellStyle();
                    styleMap.put(excelCellStyle, cellStyle);
                    Font font = null;
                    if (excelCellStyle.getAlignment() != null) {
                        cellStyle.setAlignment(excelCellStyle.getAlignment());
                    }
                    if (excelCellStyle.getVerticalAlignment() != null) {
                        cellStyle.setVerticalAlignment(excelCellStyle.getVerticalAlignment());
                    }
                    if (excelCellStyle.getFontColor() != null) {
                        if (font == null) {
                            font = wb.createFont();
                        }
                        font.setColor(excelCellStyle.getFontColor());
                    }
                    if (excelCellStyle.getBoldWeight() != null) {
                        if (font == null) {
                            font = wb.createFont();
                        }
                        font.setBoldweight(excelCellStyle.getBoldWeight());
                    }
                    if (font != null) {
                        cellStyle.setFont(font);
                    }
                }
                cell.setCellStyle(cellStyle);
            }
        }
        return row;
    }

    public Comment createCellComment(Drawing draw, ExcelCell value, short col, int row) {
        short col2 = col;
        int row2 = row;
        col2++;
        row2++;
        int len = 0;
        try {
            len = value.getComment().getBytes("GBK").length;
        } catch (UnsupportedEncodingException e1) {
            // TODO Auto-generated catch block
            e1.printStackTrace();
        }
        if (len > 24) {
            col2 = (short) (col + 3);
            row2 = row + (len - 1) / 24 + 1;
        } else {
            col2 = (short) (col + (len + 2) / 9 + 1);
        }
        Comment comment = draw.createCellComment(new HSSFClientAnchor(0, 0, 0, 127, col, row, col2, row2));
        comment.setString(new HSSFRichTextString(value.getComment()));
        comment.setVisible(value.isCommentVisible());
        return comment;
    }

    /**
     * 中文名用此方法处理一下
     *
     * @param s
     * @return
     */
    public static String toUtf8String(String s) {
        StringBuffer sb = new StringBuffer();
        for (int i = 0; i < s.length(); i++) {
            char c = s.charAt(i);
            if (c >= 0 && c <= 255) {
                sb.append(c);
            } else {
                byte[] b;
                try {
                    b = Character.toString(c).getBytes("utf-8");
                } catch (Exception ex) {
                    System.out.println(ex);
                    b = new byte[0];
                }
                for (int j = 0; j < b.length; j++) {
                    int k = b[j];
                    if (k < 0)
                        k += 256;
                    sb.append("%" + Integer.toHexString(k).toUpperCase());
                }
            }
        }
        return sb.toString();
    }

    public static void main(String[] args) {
        Workbook wb = new HSSFWorkbook();
        Sheet sheet = wb.createSheet("ExportDataList");

        // Create a row and put some cells in it. Rows are 0 based.
        Row headerTitle = sheet.createRow(0);
        Cell cell = headerTitle.createCell(0);
        int dx1 = 0;
        int dy1 = 0;
        int dx2 = 0;
        int dy2 = 127;
        short col = 0;
        int row = 0;
        short col2 = 1;
        int row2 = 1;
        String value = "123456789012345";
        int len = 0;
        try {
            len = value.getBytes("GBK").length;
        } catch (UnsupportedEncodingException e1) {
            // TODO Auto-generated catch block
            e1.printStackTrace();
        }
        System.out.println(len);
        if (len > 24) {
            col2 = (short) (col + 3);
            row2 = row + (len - 1) / 24 + 1;
        } else {
            col2 = (short) (col + (len + 2) / 9 + 1);
            dx2 = ((len + 3) % 9) * 113;
        }
        cell.setCellValue(value);
        Drawing draw = sheet.createDrawingPatriarch();

        Comment comment = draw.createCellComment(new HSSFClientAnchor(dx1, dy1, dx2, dy2, col, row, col2, row2));
        comment.setString(new HSSFRichTextString(value));
        comment.setVisible(false);
        cell.setCellComment(comment);
        OutputStream os = null;
        try {
            os = new FileOutputStream("/info/" + System.currentTimeMillis() + ".xls");
            wb.write(os);
            os.flush();
        } catch (IOException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } finally {
            if (os != null) {
                try {
                    os.close();
                } catch (IOException e) {
                    // TODO Auto-generated catch block
                    e.printStackTrace();
                }
                os = null;
            }
        }

    }

    public static class SheetWrapper<T> {
        public String sheetName;
        public List<T> data;
        public String[] fields;
        public String[] fieldValue;

        public static <T> SheetWrapper<T> create() {
            return new SheetWrapper<T>();
        }
    }
}
