package com.baijia.tianxiao.redis.lock;

import com.google.common.base.Preconditions;

import org.apache.commons.lang3.StringUtils;
import org.springframework.data.redis.core.RedisOperations;

import java.util.concurrent.TimeUnit;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

/**
 * @author weihongyan
 * @apiNote <(▰˘◡˘▰)> redis分布式锁简易实现.
 * @since 22/05/2017 8:01 PM
 * @see RedisDistributeLock
 */
// FIXME 注意跨系统时使用的serializer. 不同的话, 估计就GG了. 但考虑到序列化方式不同时,缓存也不可以跨系统使用,故锁在此不统一序列化方式.
@Slf4j
public class RedisDistributeLockImpl implements RedisDistributeLock {

    private final RedisOperations<String, String> redisTemplate;

    public RedisDistributeLockImpl(RedisOperations<String, String> redisTemplate) {
        this.redisTemplate = redisTemplate;
    }

    // 默认持有锁时间为2分钟
    private static final Long defaultHoldLockExpMills = 2 * 60 * 1000L;
    // 默认自旋sleep时间为200毫秒
    private static final Long defaultSpinMills = 200L;

    @Override
    public LockObject lockFastFail(String key) {
        return this.lockFastFail(key, this.defaultHoldLockExpMills);
    }

    @Override
    public LockObject lockFastFail(@NonNull String key, long holdLockExpMills) {
        Preconditions.checkArgument(holdLockExpMills > 0, "预计锁持有时间必须大于0!");
        LockObject lockObject = new LockObject(holdLockExpMills);

        if (this.redisTemplate.opsForValue().setIfAbsent(key, lockObject.toJson())
            && this.redisTemplate.expire(key, lockObject.getLockExpMills(), TimeUnit.MILLISECONDS)) {
            return lockObject;
        } else {
            Long expire = this.redisTemplate.getExpire(key, TimeUnit.MILLISECONDS);
            if (expire < 0) {
                String json = this.redisTemplate.opsForValue().get(key);
                if (StringUtils.isNotBlank(json)) {
                    LockObject noExpireLock = LockObject.formJson(json);
                    this.redisTemplate.expire(key, noExpireLock.getLockExpMills(), TimeUnit.MILLISECONDS);
                    log.warn(
                        "[RedisDestributeLockImpl] lockFastFail find no expire lock! add expire done! "
                            + "lockKey:{}, oldExpire:{}ms, newExpire:{}ms, lock:{}",
                        key, expire, noExpireLock.getLockExpMills(), json);
                }
            }
        }
        // log.debug("[RedisDestributeLockImpl] lockFastFail failed! will return or spin. key:{}", keyEnums);
        return null;
    }

    @Override
    public LockObject lockOrWait(String key, long waitMills) {
        return this.lockOrWait(key, waitMills, this.defaultHoldLockExpMills);
    }

    @Override
    public LockObject lockOrWait(String key, long waitMills, long holdLockExpMills) {
        return this.lockOrWait(key, waitMills, holdLockExpMills, this.defaultSpinMills);
    }

    @Override
    public LockObject lockOrWait(@NonNull String key, long waitMills, long holdLockExpMills, long spinMills) {
        Preconditions.checkArgument(holdLockExpMills > 0, "预计锁持有时间必须大于0!");
        Preconditions.checkArgument(spinMills > 0, "预计锁自旋等待时间必须大于0!");
        long start = System.currentTimeMillis();
        LockObject result = null;
        while (waitMills < 0 || System.currentTimeMillis() - start < waitMills) {
            result = this.lockFastFail(key, holdLockExpMills);
            if (null == result) {
                try {
                    Thread.sleep(spinMills);
                } catch (InterruptedException e) {
                    e.printStackTrace();
                }
            } else {
                log.info("[RedisDestributeLockImpl] get lock! key:{}, lock:{}", key, result.toJson());
                return result;
            }
        }
        log.warn(
            "[RedisDestributeLockImpl] lockOrWait failed! lockKey:{}, waitMills:{}, holdLockExpMills:{}, spinMils:{}",
            key, waitMills, holdLockExpMills, spinMills);
        return result;
    }

    @Override
    // FIXME 这里应该使用watch命令, 但是codis并不支持 ㄟ( ▔, ▔ )ㄏ
    public boolean unlock(@NonNull String key, @NonNull LockObject lockObject) {
        String json = this.redisTemplate.opsForValue().get(key);
        if (StringUtils.isNotBlank(json)) {
            if (lockObject.equals(LockObject.formJson(json))) {
                this.redisTemplate.delete(key);
                log.debug("[RedisDestributeLockImpl] unlock success! lockKey:{}", key);
                return true;
            }
        }
        log.error("[RedisDestributeLockImpl] unlock failed! lockKey:{}, unlockObject:{}, lockObject:{}", key,
            lockObject.toJson(), json);
        return false;
    }
    // private String getRedisKey(RedisKeyEnums keyEnums, String...keySuffix) {
    // String redisKey = keyEnums.getRedisKey();
    // if (null != keySuffix && keySuffix.length > 0) {
    // for (String suffix : keySuffix) {
    // if (StringUtils.isNotEmpty(suffix)) {
    // redisKey += "_" + suffix;
    // } else {
    // redisKey += "_" + "null";
    // }
    // }
    // }
    // return redisKey;
    // }
}
