
/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2016 All Rights Reserved.
 */

package com.baijia.tianxiao.dto;

import com.baijia.tianxiao.annotation.Show;
import com.baijia.tianxiao.constants.FieldType;
import com.baijia.tianxiao.util.collection.CollectorUtil;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import org.apache.commons.lang3.StringUtils;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.Date;
import java.util.List;
import java.util.Map;

import lombok.Data;

/**
 * @title Field
 * @desc TODO
 * @author shanyu
 * @date 2016年3月11日
 * @version 1.0
 */
@Data
public class FieldShow {

    /**
     * 列名
     */
    private String name;

    /**
     * 对应的po列名
     */
    private String poName;

    /**
     * 前端展示名
     */
    private String showName;

    /**
     * 字段是否锁定
     */
    private Integer lock;

    /**
     * 字段类型
     */
    private String type;

    /**
     * 操作类型 0 不可操作；1，可操作
     */
    private Integer operate;

    /**
     * 是否可见
     */
    private Integer viewType;

    /**
     * 获取列展示信息
     * 
     * @param cls
     * @return
     */
    public static List<FieldShow> getColumnShows(Class<?> cls) {
        Preconditions.checkArgument(cls != null, "cls is null!");
        List<FieldShow> result = Lists.newArrayList();
        Field[] fields = cls.getDeclaredFields();
        for (Field field : fields) {
            Show show = field.getAnnotation(Show.class);
            if (show != null) {
                FieldShow fieldShow = new FieldShow();
                fieldShow.setName(field.getName());
                fieldShow.setShowName(show.name());
                fieldShow.setLock(show.lock());
                fieldShow.setType(getType(field));
                fieldShow.setViewType(show.viewType());
                fieldShow.setPoName(getPoName(field.getName(), show));
                fieldShow.setOperate(show.operate());
                result.add(fieldShow);
            }
        }
        Method[] methods = cls.getDeclaredMethods();
        for (Method method : methods) {
            Show show = method.getAnnotation(Show.class);
            String name = getField(method.getName());
            if (StringUtils.isNoneBlank(name) && show != null) {
                FieldShow fieldShow = new FieldShow();
                fieldShow.setName(name);
                fieldShow.setShowName(show.name());
                fieldShow.setLock(show.lock());
                fieldShow.setType(getType(method));
                fieldShow.setViewType(show.viewType());
                fieldShow.setPoName(getPoName(name, show));
                fieldShow.setOperate(show.operate());
                result.add(fieldShow);
            }
        }
        return result;
    }

    /**
     * 获取字段类型
     * 
     * @param method
     * @return
     */
    private static String getType(Method method) {
        if (method.getReturnType() == String.class) {
            return FieldType.string.getMessage();
        } else if (method.getReturnType() == Date.class) {
            return FieldType.date.getMessage();
        } else {
            return FieldType.number.getMessage();
        }
    }

    /**
     * 获取字段类型
     * 
     * @param field
     * @return
     */
    private static String getType(Field field) {
        if (field.getType() == String.class) {
            return FieldType.string.getMessage();
        } else if (field.getType() == Date.class) {
            return FieldType.date.getMessage();
        } else {
            return FieldType.number.getMessage();
        }
    }

    /**
     * 获取对应的po名
     * 
     * @param name
     * @param show
     * @return
     */
    private static String getPoName(String name, Show show) {
        if (StringUtils.isBlank(show.poName())) {
            return name;
        }
        return show.poName();
    }

    /**
     * 获取列展示信息
     * 
     * @param cls
     * @return
     */
    public static Map<String, FieldShow> getColumnShowMap(Class<?> cls) {
        Preconditions.checkArgument(cls != null, "cls is null!");
        Map<String, FieldShow> map = Maps.newHashMap();
        Field[] fields = cls.getDeclaredFields();
        for (Field field : fields) {
            Show show = field.getAnnotation(Show.class);
            if (show != null) {
                FieldShow fieldShow = new FieldShow();
                fieldShow.setName(field.getName());
                fieldShow.setShowName(show.name());
                fieldShow.setLock(show.lock());
                fieldShow.setType(getType(field));
                fieldShow.setViewType(show.viewType());
                fieldShow.setPoName(getPoName(field.getName(), show));
                fieldShow.setOperate(show.operate());
                map.put(field.getName(), fieldShow);
            }
        }
        Method[] methods = cls.getDeclaredMethods();
        for (Method method : methods) {
            Show show = method.getAnnotation(Show.class);
            String name = getField(method.getName());
            if (StringUtils.isNoneBlank(name) && show != null) {
                FieldShow fieldShow = new FieldShow();
                fieldShow.setName(name);
                fieldShow.setShowName(show.name());
                fieldShow.setLock(show.lock());
                fieldShow.setType(getType(method));
                fieldShow.setViewType(show.viewType());
                fieldShow.setPoName(getPoName(name, show));
                fieldShow.setOperate(show.operate());
                map.put(name, fieldShow);
            }
        }
        return map;
    }

    private static String getField(String methodName) {
        if (methodName.startsWith("get")) {
            char start = methodName.charAt(3);
            StringBuilder sb = new StringBuilder();
            sb.append(String.valueOf(start).toLowerCase());
            if (methodName.length() > 4) {
                sb.append(methodName.substring(4));
            }
            return sb.toString();
        }
        return null;
    }

    public static boolean isFieldsUsable(Class<?> cls, List<FieldShow> fields) {
        Map<String, FieldShow> fieldMap = getColumnShowMap(cls);
        for (FieldShow show : fields) {
            if (!fieldMap.containsKey(show.getName())) {
                return false;
            }
            FieldShow field = fieldMap.get(show.getName());
            show.setLock(field.getLock());
            show.setType(field.getType());
            show.setOperate(field.getOperate());
            show.setPoName(field.getPoName());
            show.setViewType(field.viewType);
        }
        Map<String, FieldShow> map = CollectorUtil.collectMap(fields, new Function<FieldShow, String>() {
            @Override
            public String apply(FieldShow arg0) {
                return arg0.getName();
            }
        });
        for (FieldShow fieldShow : fieldMap.values()) {
            if (fieldShow.lock == 1) {
                if (!map.containsKey(fieldShow.getName())) {
                    return false;
                }
            }
        }
        return true;
    }
}
