package com.baijia.tianxiao.util.httpclient;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.NameValuePair;
import org.apache.http.StatusLine;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.HttpClient;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.entity.mime.MultipartEntity;
import org.apache.http.entity.mime.content.FileBody;
import org.apache.http.entity.mime.content.StringBody;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.baijia.commons.lang.utils.file.FileUtils;
import com.baijia.tianxiao.util.json.JacksonUtil;

import lombok.NonNull;

/**
 * 
 * @title HttpClientUtils
 * @desc TODO
 * @author wangzhigang
 * @date 2014年10月17日
 * @version 1.0
 */
public class HttpClientUtils {

    private static final Logger logger = LoggerFactory.getLogger(HttpClientUtils.class);

    public static final String CHARSET = "UTF-8";

    public static CloseableHttpClient getHttpClient() {
        RequestConfig config = RequestConfig.custom().setConnectTimeout(60000).setSocketTimeout(35000).build();
        return HttpClientBuilder.create().setDefaultRequestConfig(config).build();
    }

    public static CloseableHttpClient getHttpClient(int timeOut) {
        RequestConfig config = RequestConfig.custom().setConnectTimeout(timeOut).setSocketTimeout(timeOut).build();
        return HttpClientBuilder.create().setDefaultRequestConfig(config).build();
    }

    public static String doGet(String url, Map<String, String> params) {
        return doGet(url, params, CHARSET);
    }

    public static String doPost(String url, Map<String, String> params) {
        return doPost(url, params, CHARSET);
    }

    /**
     * HTTP Get 获取内容
     * 
     * @param url 请求的url地址 ?之前的地址
     * @param params 请求的参数
     * @param charset 编码格式
     * @return 页面内容
     */
    public static String doGet(String url, Map<String, String> params, String charset) {
        return doGet(url, params, charset, null);
    }

    public static String doGet(String url, Map<String, String> params, String charset, Collection<Header> headers) {
        if (StringUtils.isBlank(url)) {
            return null;
        }
        try {
            if (params != null && !params.isEmpty()) {
                List<NameValuePair> pairs = new ArrayList<NameValuePair>(params.size());
                for (Map.Entry<String, String> entry : params.entrySet()) {
                    String value = entry.getValue();
                    if (value != null) {
                        pairs.add(new BasicNameValuePair(entry.getKey(), value));
                    }
                }
                url += "?" + EntityUtils.toString(new UrlEncodedFormEntity(pairs, charset));
            }
            HttpGet httpGet = new HttpGet(url);
            if (CollectionUtils.isNotEmpty(headers)) {
                for (Header header : headers) {
                    httpGet.addHeader(header);
                }
            }
            return execute(null, httpGet, charset, null);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * HTTP Post 获取内容
     * 
     * @param url 请求的url地址 ?之前的地址
     * @param params 请求的参数
     * @param charset 编码格式
     * @return 页面内容
     */
    public static String doPost(String url, Map<String, String> params, String charset) {
        return doPost(url, params, charset, null);
    }

    public static String doPost(String url, Map<String, String> params, String charset, Collection<Header> headers) {
        if (StringUtils.isBlank(url)) {
            return null;
        }
        try {
            List<NameValuePair> pairs = null;
            if (params != null && !params.isEmpty()) {
                pairs = new ArrayList<NameValuePair>(params.size());
                for (Map.Entry<String, String> entry : params.entrySet()) {
                    String value = entry.getValue();
                    if (value != null) {
                        pairs.add(new BasicNameValuePair(entry.getKey(), value));
                    }
                }
            }
            HttpPost httpPost = new HttpPost(url);
            if (pairs != null && pairs.size() > 0) {
                httpPost.setEntity(new UrlEncodedFormEntity(pairs, CHARSET));
            }
            if (CollectionUtils.isNotEmpty(headers)) {
                for (Header header : headers) {
                    httpPost.addHeader(header);
                }
            }
            return execute(null, httpPost, null, null);
        } catch (Exception e) {
            logger.error("post error",e);
        }
        return null;
    }

    /**
     * 上传文件
     * 
     * @param url
     * @param params
     * @param files
     * @return
     */
    public static String doPostFiles(String url, Map<String, String> params, Map<String, File> files) {
        RequestConfig config = RequestConfig.custom().setConnectTimeout(10000).setSocketTimeout(60000).build();
        CloseableHttpClient httpClient = HttpClientBuilder.create().setDefaultRequestConfig(config).build();
        try {
            logger.info("doPost - start, url:{}", url);

            long start = System.currentTimeMillis();
            long end = System.currentTimeMillis();
            logger.info("HttpClientUtils.doPost: get httpClinet, cost:{}", (end - start));

            HttpPost httpPost = new HttpPost(url);

            MultipartEntity multipartEntity = new MultipartEntity();

            long getPostTime = System.currentTimeMillis();
            logger.info("HttpClientUtils.doPost: getHttppOST cost:{}", (getPostTime - end));
            // add files
            for (Map.Entry<String, File> file : files.entrySet()) {
                multipartEntity.addPart(file.getKey(), new FileBody(file.getValue()));
            }

            // add params
            for (Map.Entry<String, String> param : params.entrySet()) {
                multipartEntity.addPart(param.getKey(), new StringBody(param.getValue()));
            }

            httpPost.setEntity(multipartEntity);
            logger.info("doPost - execute, url:{}", url);
            return execute(httpClient, httpPost, null, null);
        } catch (Exception e) {
            throw new RuntimeException("dopost exception - ", e);
        } finally {
            try {
                httpClient.close();
            } catch (final IOException ignore) {
            }
        }
    }

    public static String doJsonPost(String url, Map<String, String> params, String charset) throws Exception {
        return doJsonPost(url, params, charset, null);
    }
    
    public static String doJsonPostNew(String url, Map<String, Object> params, String charset) throws Exception {
        return doJsonPostNew(url, params, charset, null);
    }

    public static String doJsonPost(String url, Map<String, String> params, String charset, Integer timeOut)
        throws Exception {
        String jsonPost = JacksonUtil.obj2Str(params);

        HttpPost httpPost = new HttpPost(url);
        HttpEntity requestEntity = new StringEntity(jsonPost, ContentType.APPLICATION_JSON);
        httpPost.setEntity(requestEntity);

        return execute(null, httpPost, charset, timeOut);
    }
    
    public static String doJsonPostNew(String url, Map<String, Object> params, String charset, Integer timeOut)
        throws Exception {
        String jsonPost = JacksonUtil.obj2Str(params);

        HttpPost httpPost = new HttpPost(url);
        HttpEntity requestEntity = new StringEntity(jsonPost, ContentType.APPLICATION_JSON);
        httpPost.setEntity(requestEntity);

        return execute(null, httpPost, charset, timeOut);
    }

    private static String execute(HttpClient httpClient, @NonNull HttpUriRequest request, String charset,
        Integer timeOut) {
        boolean closeHttpClient = false;
        if (httpClient == null) {
            closeHttpClient = true;
            if (timeOut != null) {
                httpClient = getHttpClient(timeOut);
            } else {
                httpClient = getHttpClient();
            }
        }
        if (StringUtils.isBlank(charset)) {
            charset = CHARSET;
        }

        try {
            long current = System.currentTimeMillis();
            if (logger.isDebugEnabled()) {
                logger.debug("send request:{}", request);
            }
            String result = httpClient.execute(request, new StringResponseHandler(request, charset));
            logger.info("cost:{} ms to execute http method:{},url:{}", System.currentTimeMillis() - current,
                request.getMethod(), request.getURI());
            return result;
        } catch (IOException e) {
            throw new RuntimeException("do request exception - ", e);
        } finally {
            if (closeHttpClient) {
                if (httpClient != null) {
                    if (httpClient instanceof Closeable) {
                        try {
                            ((Closeable) httpClient).close();
                        } catch (final IOException ignore) {
                        }
                    }
                }
            }
        }

    }

    static class StringResponseHandler implements ResponseHandler<String> {

        private HttpUriRequest request;

        private String charset;

        public StringResponseHandler(HttpUriRequest request, String charset) {
            this.request = request;
            this.charset = charset;
        }

        @Override
        public String handleResponse(HttpResponse response) throws ClientProtocolException, IOException {
            try {
                StatusLine status = response.getStatusLine();
                if (logger.isDebugEnabled()) {
                    logger.debug("doPost - execute end, url:{}, status:{}", request.getURI(), status);
                }
                if (status.getStatusCode() != HttpStatus.SC_OK) {
                    request.abort();
                    throw new RuntimeException("HttpClient,error status code :" + status);
                }
                HttpEntity responseEntity = response.getEntity();
                String result = null;
                if (responseEntity != null) {
                    result = EntityUtils.toString(responseEntity, charset);
                }
                if (logger.isDebugEnabled()) {
                    logger.debug("http response - url:{}, status:{}, response:{}", request.getURI(), status, result);
                }
                return result;
            } finally {
                if (response != null) {
                    final HttpEntity entity = response.getEntity();
                    if (entity != null) {
                        try {
                            EntityUtils.consume(entity);
                        } catch (final IOException ex) {
                        }
                    }
                }
            }

        }

    }
    
    
    public static byte[] download(String url) throws Exception {
        try {
            HttpGet httpGet = new HttpGet(url);
            CloseableHttpClient httpClient = getHttpClient();
            CloseableHttpResponse response = httpClient.execute(httpGet);
            int statusCode = response.getStatusLine().getStatusCode();
            if (statusCode != 200) {
                httpGet.abort();
                throw new IllegalStateException("download - error status code :" + statusCode);
            }

            HttpEntity entity = response.getEntity();
            byte[] result = FileUtils.readBytes(entity.getContent());
            EntityUtils.consume(entity);
            response.close();
            return result;
        } catch (Exception e) {
            logger.warn("download exception - ", e);
            return null;
        }
    }
    
    
    
    public static byte[] downloadByPost(String url, Map<String, String> params) throws Exception {
        try {
        	
        	String jsonPost = JacksonUtil.obj2Str(params);

            HttpPost httpPost = new HttpPost(url);
            HttpEntity requestEntity = new StringEntity(jsonPost, ContentType.APPLICATION_JSON);
            httpPost.setEntity(requestEntity);
            logger.info("download... url:{}, params:{} ", url, jsonPost);
            CloseableHttpClient httpClient = getHttpClient();
            CloseableHttpResponse response = httpClient.execute(httpPost);
            int statusCode = response.getStatusLine().getStatusCode();
            if (statusCode != 200) {
            	httpPost.abort();
                throw new IllegalStateException("download - error status code :" + statusCode);
            }

            HttpEntity entity = response.getEntity();
            byte[] result = FileUtils.readBytes(entity.getContent());
            EntityUtils.consume(entity);
            response.close();
            return result;
        } catch (Exception e) {
            logger.warn("download exception - ", e);
            return null;
        }
    }
}
