/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */
package com.baijia.tianxiao.util.request;

import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.Map;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;

import org.apache.commons.lang.StringUtils;

import com.baijia.tianxiao.util.GenericsUtils;

import lombok.extern.slf4j.Slf4j;

/**
 * @author wangzhigang
 * @version 1.0
 * @title RequestUtils
 * @desc TODO
 * @date 2015年7月17日
 */
@Slf4j
public class RequestUtils {

    public static String getParam(HttpServletRequest request, String name) {
        return getStringParam(request, name);
    }

    public static String getParam(HttpServletRequest request, String name, String defaultValue) {
        return getStringParam(request, name, defaultValue);
    }

    public static String getStringParam(HttpServletRequest request, String name) {
        return getStringParam(request, name, "");
    }

    public static String getStringParam(HttpServletRequest request, String name, String defaultValue) {
        String value = request.getParameter(name);
        if (StringUtils.isBlank(value)) {
            return defaultValue;
        } else {
            return value.trim();
        }
    }

    public static String[] getStringParams(HttpServletRequest request, String name, String[] defaultValues) {
        String[] values = request.getParameterValues(name);
        if (values == null) {
            return defaultValues;
        } else {
            return values;
        }
    }

    public static int getIntParam(HttpServletRequest request, String name) {
        return getIntParam(request, name, 0);
    }

    public static int getIntParam(HttpServletRequest request, String name, int defaultValue) {
        String value = request.getParameter(name);
        if (StringUtils.isBlank(value)) {
            return defaultValue;
        } else {
            value = value.trim();
            if (!StringUtils.isNumeric(value)) {
                return defaultValue;
            }
            return Integer.parseInt(value);
        }
    }

    public static byte getByteParam(HttpServletRequest request, String name) {
        return getByteParam(request, name, (byte) 0);
    }

    public static byte getByteParam(HttpServletRequest request, String name, byte defaultValue) {
        String value = request.getParameter(name);

        if (StringUtils.isBlank(value)) {
            return defaultValue;
        } else {
            value = value.trim();
            if (!StringUtils.isNumeric(value)) {
                return defaultValue;
            }
            return Byte.parseByte(value);
        }
    }

    public static long getLongParam(HttpServletRequest request, String name) {
        return getLongParam(request, name, 0L);
    }

    public static long getLongParam(HttpServletRequest request, String name, long defaultValue) {
        String value = request.getParameter(name);

        if (StringUtils.isBlank(value)) {
            return defaultValue;
        } else {
            value = value.trim();
            if (!StringUtils.isNumeric(value)) {
                return defaultValue;
            }
            return Long.parseLong(value);
        }
    }

    public static String getPostParamTK(HttpServletRequest request, String key) {
        String params = getPostString(request);
        if (params != null && !params.equals("")) {
            if (params.indexOf("\"tk\":\"") > 0) {
                params = params.substring(params.indexOf("\"tk\":\"") + 6);
                int len = 0;
                for (int i = 0; i < params.length(); i++) {
                    if (params.charAt(i) == '"') {
                        len = i;
                        break;
                    }
                }
                params = params.substring(0, len);
                return params;
            }
        }
        return null;
    }

    public static String getPostString(HttpServletRequest request) {
        byte[] data = getPostData(request);
        return new String(data);
    }

    public static byte[] getPostData(HttpServletRequest request) {
        BufferedInputStream bis = null;
        ByteArrayOutputStream baos = null;
        final byte[] EMPTY_BYTES = new byte[0];

        try {
            bis = new BufferedInputStream(request.getInputStream());
            baos = new ByteArrayOutputStream();

            byte[] buf = new byte[1024];
            int len = 0;
            while ((len = bis.read(buf)) > 0) {
                baos.write(buf, 0, len);
            }
            return baos.toByteArray();
        } catch (Exception e) {
            return EMPTY_BYTES;
        } finally {
            if (bis != null) {
                try {
                    bis.close();
                } catch (IOException e) {
                    return EMPTY_BYTES;
                }
            }
            if (baos != null) {
                try {
                    baos.close();
                } catch (IOException e) {
                    return EMPTY_BYTES;
                }
            }
        }
    }

    public static String getServerHttpPath(HttpServletRequest request) {
        String fmt = "%s://%s/";
        return String.format(fmt, "http", request.getServerName());
    }

    public static String getHttpPath(HttpServletRequest request, String relativePath) {
        if (relativePath.startsWith("/")) {
            relativePath = relativePath.substring(1, relativePath.length());
        }
        String fmt = "%s://%s/%s";
        return String.format(fmt, "http", request.getServerName(), relativePath);
    }

    /**
     * @param request
     * @param requestKey
     * @return
     */
    public static String getParamFromCookie(HttpServletRequest request, String requestKey) {
        Cookie[] cookies = request.getCookies();
        if (GenericsUtils.notNullAndEmpty(cookies)) {
            for (Cookie cookie : cookies) {
                String cName = cookie.getName();
                if (cName.equals(requestKey)) {
                    return cookie.getValue();
                }
            }
        }
        return null;
    }

    /**
     * @param request
     * @return name=xxxx&age=12
     * 
     *         just call request.getQueryString()
     */
    @Deprecated
    public static String getParamsString(HttpServletRequest request) {
        String characterEncoding = request.getCharacterEncoding();
        if (GenericsUtils.isNullOrEmpty(characterEncoding)) {
            characterEncoding = "utf-8";
        }
        StringBuilder sb = new StringBuilder();
        @SuppressWarnings("unchecked")
        Map<String, String[]> parameterMap = request.getParameterMap();
        if (GenericsUtils.notNullAndEmpty(parameterMap)) {
            for (Map.Entry<String, String[]> entry : parameterMap.entrySet()) {
                String paramName = entry.getKey();
                String[] paramValues = entry.getValue();
                for (String value : paramValues) {
                    try {
                        sb.append(URLEncoder.encode(paramName, characterEncoding)).append("=")
                            .append(URLEncoder.encode(value, characterEncoding));
                    } catch (UnsupportedEncodingException e) {
                        GenericsUtils.logErrorAndInfo(log, e,
                            "can not encode queryParam:{} or paramValue:{} with charset:{} ", paramName, value,
                            characterEncoding);
                    }
                }
                sb.append("&");
            }
        }
        return GenericsUtils.deleteLastCharToString(sb);
    }
}
