package com.baijia.tianxiao.util;

import java.beans.PropertyDescriptor;
import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.baijia.tianxiao.beanCopy.BeanInvokeUtils;
import com.baijia.tianxiao.beanCopy.BeanMethodInvoke;
import com.baijia.tianxiao.beanCopy.Invoker;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import lombok.extern.slf4j.Slf4j;

/**
 * 
 * @title GenericsUtils
 * @desc 一些通用的工具方法
 * @author Rezar
 * @date 2015年9月13日
 * @version 1.0
 */
@Slf4j
public class GenericsUtils {

    /**
     * 判断某个对象是否为空或者如果是某些特殊对象的话，判断这些特殊对象的长度属性，是否为Empty
     * 
     * @param obj
     * @return
     */
    public static boolean notNullAndEmpty(Object obj) {

        if (obj == null) {
            return false;
        }

        if (obj instanceof String) {
            return ((String) obj).length() == 0 ? false : true;
        }

        if (obj instanceof Collection<?>) {
            return ((Collection<?>) obj).size() == 0 ? false : true;
        }

        if (obj instanceof Map<?, ?>) {
            return ((Map<?, ?>) obj).size() == 0 ? false : true;
        }

        Class<?> cla = obj.getClass();
        if (cla.isArray()) {
            return Array.getLength(obj) == 0 ? false : true;
        }

        return true;

    }

    public static boolean isNullOrEmpty(Object obj) {
        return !notNullAndEmpty(obj);
    }

    /**
     * 判断一个对象中的某个属性是否为普通的属性，且判断该属性是否为默认值
     * 
     * @param field
     * @param qm
     * @return
     * @throws IllegalArgumentException
     * @throws IllegalAccessException
     */
    public static boolean isNotNull(Field field, Object qm) throws IllegalArgumentException, IllegalAccessException {
        boolean notNull = true;
        Class<?> type = field.getType();
        if (type == int.class) {
            notNull = ((Integer) field.get(qm) == 0 ? false : true);
        }
        if (type == double.class) {
            notNull = ((Double) field.get(qm) == 0.0 ? false : true);
        }
        if (type == float.class) {
            notNull = ((Float) field.get(qm) == 0.0 ? false : true);
        }
        if (type == boolean.class) {
            notNull = ((Boolean) field.get(qm) == false ? false : true);
        }
        if (type == char.class) {
            notNull = ((Character) field.get(qm) == '\u0000' ? false : true);
        }
        if (type == byte.class) {
            notNull = ((Byte) field.get(qm) == 0 ? false : true);
        }
        if (type == short.class) {
            notNull = ((Short) field.get(qm) == 0 ? false : true);
        }
        return notNull;
    }

    public static boolean notEqualsIn(String value, boolean ignoreCase, String...strs) {
        boolean isIn = false;
        for (int i = 0; i < strs.length; i++) {
            if (ignoreCase) {
                isIn = strs[i].equals(value);
            } else {
                isIn = strs[i].equalsIgnoreCase(value);
            }
        }
        return isIn;
    }

    /**
     * 从一个字符串中找到某个字符串出现的位置，从指定开始的位置开始查找
     * 
     * @param str
     * @return
     */
    public static int indexOfIgnoreCase(String str, int fromIndex) {

        str = str.toLowerCase();
        return str.indexOf(str, fromIndex);

    }

    public static StringBuilder deleteLastChar(StringBuilder sb) {
        if (sb != null && sb.length() > 0) {
            return sb.deleteCharAt(sb.length() - 1);
        } else {
            return sb;
        }
    }

    public static String deleteLastCharToString(StringBuilder sb) {
        return deleteLastChar(sb).toString();
    }

    public static <T extends Collection<K>, K> List<K> toList(T cols) {
        return new ArrayList<K>(cols);
    }

    // 测试类
    public static class User {
        private String name;
        private String age;

        public User(String name, String age) {
            this.name = name;
            this.age = age;
        }

        public String toString() {
            return this.name + " : " + this.age;
        }

        public String getName() {

            return name;
        }

        public void setName(String name) {
            this.name = name;
        }

        public String getAge() {
            return age;
        }

        public void setAge(String age) {
            this.age = age;
        }

    }

    public static void main(String[] args) {
        List<User> users = new ArrayList<>();
        for (int i = 0; i < 10; i++) {
            User user = new User("name " + i, "age " + i);
            users.add(user);
        }
        Map<String, String> toKeyMap = toFieldMap(users, "name", "age");
        System.out.println(toKeyMap.get("name 8"));
        List<String> keyListWithInvoker = toFieldList(users, "age");
        System.out.println(keyListWithInvoker);
    }

    /**
     * 根据对象集合,获取对象中某个字段的属性值的集合<br/>
     * 
     * 从一个集合里面取出某个字段的属性作为key,返回所有对象里面这个属性的值的集合
     * 
     * @param cols
     * @param fieldName
     * @return
     */
    @SuppressWarnings("unchecked")
    public static <K, V> List<K> toFieldList(Collection<V> cols, String fieldName) {
        if (isNullOrEmpty(cols) || isNullOrEmpty(fieldName)) {
            return emptyList();
        }
        Iterator<? extends V> iter = cols.iterator();
        List<K> retList = new ArrayList<>(cols.size());
        Invoker<V> fieldReader = null;
        while (iter.hasNext()) {
            V next = iter.next();
            if (fieldReader == null) {
                BeanMethodInvoke<V> findBeanMethodInovker =
                    (BeanMethodInvoke<V>) BeanInvokeUtils.findBeanMethodInovker(next.getClass());
                fieldReader = findBeanMethodInovker.getFieldReader(fieldName);
            }
            Object invoke = fieldReader.invoke(next);
            retList.add((K) invoke);
        }
        return retList;
    }

    /**
     * 根据List ---> keyValue<br/>
     * 从一个集合里面取出某个字段的属性作为key,如果指定了两个以上的keyName,则第一个始终为key值的属性名.第二个为value值的属性名.
     * 
     * @param cols
     * @param keyName
     * @return
     */
    @SuppressWarnings("unchecked")
    public static <T extends Collection<V>, V, K, R> Map<K, R> toFieldMap(T cols, String...keyNames) {
        if (isNullOrEmpty(cols) || isNullOrEmpty(keyNames)) {
            log.info("cols is null:{} or field is null :{} ", isNullOrEmpty(cols), isNullOrEmpty(keyNames));
            return emptyMap();
        }
        String keyOfKey = keyNames[0];
        String keyOfValue = null;
        if (keyNames.length >= 2) {
            keyOfValue = keyNames[1];
        }
        Iterator<V> iter = cols.iterator();
        Map<K, R> retMap = new HashMap<>(cols.size());
        Invoker<V> fieldReaderForKey = null;
        Invoker<V> fieldReaderForValue = null;
        while (iter.hasNext()) {
            V next = iter.next();
            fieldReaderForKey = getFieldReaderInvoker(keyOfKey, fieldReaderForKey, next);
            Object key = fieldReaderForKey.invoke(next);
            Object value = null;
            if (keyOfValue != null) {
                value = getFieldReaderInvoker(keyOfValue, fieldReaderForValue, next).invoke(next);
            } else {
                value = next;
            }
            retMap.put((K) key, (R) value);
        }
        return retMap;
    }

    /**
     * @param keyOfKey
     * @param fieldReader
     * @param next
     * @return
     */
    @SuppressWarnings("unchecked")
    private static <V> Invoker<V> getFieldReaderInvoker(String keyOfKey, Invoker<V> fieldReader, V next) {
        if (fieldReader == null) {
            BeanMethodInvoke<V> findBeanMethodInovker =
                (BeanMethodInvoke<V>) BeanInvokeUtils.findBeanMethodInovker(next.getClass());
            fieldReader = findBeanMethodInovker.getFieldReader(keyOfKey);
        }
        return fieldReader;
    }

    /**
     * 
     * @param cols
     * @param field
     * @return
     */
    @SuppressWarnings("unchecked")
    public static <K, V> List<K> toKeyList(Collection<? extends V> cols, String field) {
        if (isNullOrEmpty(cols) || isNullOrEmpty(field)) {
            log.info("cols is null:{} or field is null :{} ", isNullOrEmpty(cols), isNullOrEmpty(field));
            return emptyList();
        }
        List<K> retList = new ArrayList<K>();
        try {
            Class<?> beanClass = cols.toArray()[0].getClass();
            final PropertyDescriptor pd = new PropertyDescriptor(field, beanClass);
            for (V value : cols) {
                try {
                    retList.add((K) pd.getReadMethod().invoke(value, new Object[] {}));
                } catch (Exception e) {
                    log.debug("encount a error cause by : {} ", e);
                }
            }
        } catch (Exception ex) {
            log.error("exception : {} ", ex);
        }
        return retList;
    }

    public static void setAccessable(String fieldName, Class<?> beanClass) {
        try {
            Field field = beanClass.getDeclaredField(fieldName);
            if (!field.isAccessible()) {
                field.setAccessible(true);
            }
        } catch (Exception e) {
            log.debug("can not set accessible for field : {} in class : {} ", fieldName, beanClass);
        }
    }

    /**
     * 添加这个方法的原因:<br/>
     * 因为之前都是调用 , Collections.emptyList(); ,<br/>
     * 这个方法会返回一个EmptyList对象，该对象继承AbstractList父类 其add方法未实现.<br/>
     * 所以在使用底层方法返回的Collections.emptyList() 对象再添加新数据的时候，会报 java.lang.UnsupportedOperationException
     * 
     * @return
     */
    public static <T> List<T> emptyList() {
        return Lists.newArrayListWithCapacity(0);
    }

    /**
     * @see GenericsUtils.emptyList()
     * @return
     */
    public static <K, V> Map<K, V> emptyMap() {
        return Maps.newHashMapWithExpectedSize(0);
    }

    /**
     * @see GenericsUtils.emptyList()
     * @return
     */
    public static <T> Set<T> emptySet() {
        return Sets.newHashSetWithExpectedSize(0);
    }

    /**
     * @param emptyMap
     * @param key
     * @param listValue
     */
    public static <K, V> void addListIfNotExists(Map<K, List<V>> map, K key, V listValue) {
        List<V> list = map.get(key);
        if (isNullOrEmpty(list)) {
            list = Lists.newArrayList();
            map.put(key, list);
        }
        list.add(listValue);
    }

    private static final String COMMON_SPLIT_RETEX = "[,|:&#]+";

    public static <T> Set<T> stringToNumber(String numStr, Class<T> numType) {
        return stringToNumber(numStr, COMMON_SPLIT_RETEX, numType);
    }

    @SuppressWarnings("unchecked")
    public static <T> Set<T> stringToNumber(String numStr, String splitRegex, Class<T> numType) {
        if (GenericsUtils.isNullOrEmpty(numStr)) {
            return emptySet();
        }

        String[] allValues = numStr.split(splitRegex);
        Set<T> retSet = Sets.newHashSet();
        for (String value : allValues) {
            Object numValue = getNumValueFromStr(numType, value);
            retSet.add((T) numValue);
        }
        return retSet;
    }

    /**
     * @param numType
     * @param value
     * @return
     */
    @SuppressWarnings("unchecked")
    public static <T> T getNumValueFromStr(Class<T> numType, String value) {
        Object numValue = null;
        if (numType == Integer.class || numType == int.class) {
            numValue = Integer.parseInt(value);
        } else if (numType == Float.class || numType == float.class) {
            numValue = Float.parseFloat(value);
        } else if (numType == Short.class || numType == short.class) {
            numValue = Short.parseShort(value);
        } else if (numType == Long.class || numType == long.class) {
            numValue = Long.parseLong(value);
        } else if (numType == Double.class || numType == double.class) {
            numValue = Double.parseDouble(value);
        } else if (numType == Byte.class || numType == byte.class) {
            numValue = Byte.parseByte(value);
        } else if (numType == Character.class || numType == char.class) {
            numValue = new Character(value.charAt(0));
        }
        return (T) numValue;
    }

    /**
     * 打印异常,logger 对象在error输出方法里面输出常规信息的时候,异常栈信息会缺失,需要分开打印
     * 
     * @param logger :异常对象
     * @param ex :异常对象
     * @param format :输出格式,同 can not find any orgInfo with orgId:{} ,使用{}占位
     * @param infos :需要输出的占位信息
     */
    public static void logErrorAndInfo(Logger logger, Exception ex, String format, Object...infos) {
        if (GenericsUtils.notNullAndEmpty(format)) {
            format = format.replace("{}", "%s");
            logger.info("[Pring Error Info ] {} ", String.format(format, infos));
        }
        logger.error("[exception] {} ", ex);// the really excepiton not in GenericsUtils ,please find out the
                                            // exception
                                            // position from StackTrace
    }

    public static void logErrorAndInfo(Class<?> clazz, Exception ex, String format, Object...infos) {
        Logger log = LoggerFactory.getLogger(clazz);
        logErrorAndInfo(log, ex, format, infos);
    }

}
