package com.baijia.tianxiao.util;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.math.BigDecimal;
import java.sql.Blob;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

import javax.servlet.http.HttpServletResponse;
import javax.sql.rowset.serial.SerialBlob;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.baijia.tianxiao.annotation.Snippet;
import com.baijia.tianxiao.dto.FieldShow;
import com.baijia.tianxiao.util.properties.PropertiesReader;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

/**
 * 通用方法
 *
 * @author zhangbing
 * @version 1.0
 * @title BaseUtils
 * @desc TODO
 * @date 2014年10月27日
 */
public class BaseUtils {

    private static final Logger logger = LoggerFactory.getLogger(BaseUtils.class);

    /**
     * 将字符串按照指定的分隔符切分为List
     *
     * @param source 源字符串
     * @param symbol 分隔符
     * @return
     */
    public static List<String> strToList(String source, String symbol) {
        List<String> retList = null;
        if (source != null && !source.isEmpty()) {
            if (source.contains(symbol)) {
                String arr[] = source.split(symbol);
                retList = Arrays.asList(arr);
            } else {
                retList = Arrays.asList(source);
            }
        }

        return retList;
    }

    @SuppressWarnings("unchecked")
    public static <T, V> List<V> getIds(List<T> objs, String property) {
        List<V> ids = Collections.emptyList();
        for (T obj : objs) {
            ids.add((V) getValue(obj, property));
        }
        return ids;
    }

    public static String listToStr(List<String> list, String symbol) {
        String str = "";
        for (String s : list) {
            str += s;
            str += symbol;
        }
        if (str.length() > 0) {
            return str.substring(0, str.length() - symbol.length());
        }
        return str;
    }

    /**
     * format格式的时间字符串转换成毫秒数
     *
     * @param time
     * @param format
     * @return
     */
    public static long strTimeToMills(String time, String format) {
        SimpleDateFormat sdf = new SimpleDateFormat(format);
        long millSeconds = 0l;
        try {
            millSeconds = sdf.parse(time).getTime();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return millSeconds;
    }

    /**
     * 格式化date
     *
     * @param date
     * @param format
     * @return
     */
    public static String getFormatTime(Date date, String format) {
        SimpleDateFormat sdf = new SimpleDateFormat(format);
        return sdf.format(date);
    }

    /**
     * unix时间戳转换成yyyy-MM-dd HH:mm:ss类型时间
     *
     * @param unixTime
     * @return
     */
    public static String unixToFormatTime(String unixTime) {
        long time = Long.parseLong(unixTime);
        String ts = new Timestamp(time * 1000).toString();
        return ts.substring(0, ts.length() - 2);
    }

    /**
     * 获取指定格式的日期字符串
     *
     * @param format 日期格式, yyyy-MM-dd HH:mm:ss etc.
     * @param offset 偏移量：0：获取当前格式时间，负数：获取offset前的时间
     * @param field Calendar.DAY_OF_MONTH, Calendar.HOUR_OF_DAY etc
     * @return
     */
    public static String getFormatDate(String format, int offset, int field) {
        SimpleDateFormat sdf = new SimpleDateFormat(format);
        Calendar cal = Calendar.getInstance();
        cal.add(field, offset);

        return sdf.format(cal.getTime());
    }

    public static Date getDate(String time, String format) {
        Date ret = null;

        if (StringUtils.isNotBlank(time)) {
            SimpleDateFormat sdf = new SimpleDateFormat(format);
            try {
                ret = sdf.parse(time);
            } catch (ParseException e) {
                e.printStackTrace();
            }
        }
        return ret;
    }

    /**
     * 获取当前月的第一天
     *
     * @return
     */
    public static String getFirstDay(String format, int year, int month, int field) {
        SimpleDateFormat sdf = new SimpleDateFormat(format);
        Calendar cal = Calendar.getInstance();
        cal.set(Calendar.YEAR, year);
        cal.set(Calendar.MONTH, month);
        int firstDay = cal.getActualMinimum(field);
        cal.set(field, firstDay);
        return sdf.format(cal.getTime());
    }
    
    /**
     * 获取当前周周一的时间
     * @return
     */
    public static Date getStartDayOfWeek() {
        Calendar cal = Calendar.getInstance();
        cal.set(Calendar.DAY_OF_WEEK, Calendar.MONDAY);
        cal.set(Calendar.HOUR_OF_DAY, 0);
        cal.set(Calendar.MINUTE, 0);
        cal.set(Calendar.SECOND, 0);
        
        return cal.getTime();
    }
    
    /**
     * 获取当前周周日的时间
     * @return
     */
    public static Date getEndDayOfWeek() {
        Calendar cal = Calendar.getInstance();
        cal.set(Calendar.DAY_OF_WEEK, Calendar.SUNDAY);
        cal.add(Calendar.WEEK_OF_YEAR, 1);
        cal.set(Calendar.HOUR_OF_DAY, 23);
        cal.set(Calendar.MINUTE, 59);
        cal.set(Calendar.SECOND, 59);
        
        return cal.getTime();
    }

    /**
     * 获取每周第一天时间戳
     *
     * @return
     */
    public static String getFirstDayOfWeek() {
        Calendar currentTime = Calendar.getInstance();
        int dayofWeek = currentTime.get(Calendar.DAY_OF_WEEK) - 1;
        int mondayPlus = 0;
        if (dayofWeek == 0) {
            mondayPlus = -6;
        } else if (dayofWeek != 1) {
            mondayPlus = 1 - dayofWeek;
        }
        currentTime.add(Calendar.DATE, mondayPlus);
        SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd");
        String startTimestr = df.format(currentTime.getTime());
        startTimestr += " 00:00:00";
        return startTimestr;
    }
    
    /**
     * 获取当前月的最后一天
     *
     * @param format ：时间格式，yyyy-MM-dd/yyyyMMddHHmmss etc.
     * @param field : Calendar.DAY_OF_MONTH
     * @return
     */
    public static String getLastDay(String format, int year, int month, int field) {
        SimpleDateFormat sdf = new SimpleDateFormat(format);
        Calendar cal = Calendar.getInstance();
        cal.set(Calendar.YEAR, year);
        cal.set(Calendar.MONTH, month);
        int lastDay = cal.getActualMaximum(field);
        cal.set(field, lastDay);
        return sdf.format(cal.getTime());
    }

    /**
     * 构造用户主页地址
     *
     * @param domain
     * @param surfix 老师：t, 机构:i
     * @return
     */
    public static String constructUserUrl(String domain, String surfix) {
        String img_server = PropertiesReader.getValue("jigou", "jigou.host");
        StringBuilder sb = new StringBuilder(img_server);
        return sb.append("/").append(surfix).append("/").append(domain).toString();
    }

    public static int inputCount(String input) {
        int count = 0;
        if (StringUtils.isBlank(input) || input == null) {
            return count;
        } else {
            for (int i = 0; i < input.length(); i++) {
                if (input.substring(i, i + 1).getBytes().length == 3) {
                    count = count + 2;
                } else {
                    count++;
                }
            }
            return count;
        }
    }

    /**
     * 采用gzip压缩字符串
     *
     * @param source
     * @return
     */
    public static String gzipCompress(String source) {
        String target = null;
        if (source != null && !source.isEmpty()) {
            ByteArrayOutputStream out = null;
            GZIPOutputStream gzip = null;

            try {
                out = new ByteArrayOutputStream();
                gzip = new GZIPOutputStream(out);
                gzip.write(source.getBytes());
                gzip.flush();
                gzip.finish();

                target = out.toString("ISO-8859-1");
            } catch (IOException e) {
                logger.error("BaseUtil.zipCompress: compress source String failed, e" + e);
            } finally {
                if (gzip != null) {
                    try {
                        gzip.close();
                    } catch (IOException e) {
                        logger.error("BaseUtil.zipCompress: close gzipoutputstream failed, e" + e);
                    }
                }
                if (out != null) {
                    try {
                        out.close();
                    } catch (IOException e) {
                        logger.error("BaseUtil.zipCompress: close bytearrayoutputstream failed, e" + e);
                    }
                }
            }
        }

        return target;
    }

    /**
     * 解压
     *
     * @param source
     * @return
     */
    public static String gzipUncompress(String source) {
        String target = null;
        if (source != null && !source.isEmpty()) {
            ByteArrayOutputStream out = null;
            ByteArrayInputStream in = null;
            GZIPInputStream gunzip = null;
            try {
                out = new ByteArrayOutputStream();
                in = new ByteArrayInputStream(source.getBytes("ISO-8859-1"));

                gunzip = new GZIPInputStream(in);
                byte[] buffer = new byte[256];
                int n;
                while ((n = gunzip.read(buffer)) >= 0) {
                    out.write(buffer, 0, n);
                }

                target = out.toString();
            } catch (IOException e) {
                logger.error("BaseUtil.gzipUncompress: uncompress source String failed, e" + e);
            } finally {
                if (out != null) {
                    try {
                        out.close();
                    } catch (IOException e) {
                        logger.error("BaseUtil.gzipUncompress: close bytearrayoutputstream failed, e" + e);
                    }
                }
                if (gunzip != null) {
                    try {
                        gunzip.close();
                    } catch (IOException e) {
                        logger.error("BaseUtil.gzipUncompress: close GZIPInputStream failed, e" + e);
                    }
                }
                if (in != null) {
                    try {
                        in.close();
                    } catch (IOException e) {
                        logger.error("BaseUtil.gzipUncompress: close ByteArrayInputStream failed, e" + e);
                    }
                }
            }
        }

        return target;
    }

    /**
     * 将字符串转换成blob
     *
     * @param source
     * @param charset 字符集, UTF-8, GBK, ISO-8859-1 etc.
     * @return
     */
    public static Blob strToBlob(String source, String charset) {
        Blob ret = null;
        if (source != null && !source.isEmpty()) {
            try {
                ret = new SerialBlob(source.getBytes(charset));
            } catch (UnsupportedEncodingException | SQLException e) {
                logger.error("BaseUtils.strToBlob: string convert to blob failed," + e);
            }
        }
        return ret;
    }

    /**
     * blob转换成String
     *
     * @param source blob
     * @param charset 字符集, UTF-8, GBK, ISO-8859-1 etc.
     * @return
     */
    public static String blobToString(Blob source, String charset) {
        String ret = null;

        if (source != null) {
            try {
                ret = new String(source.getBytes(1, (int) source.length()), charset);
            } catch (UnsupportedEncodingException | SQLException e) {
                logger.error("BaseUtils.blobToString: blob convert to String failed," + e);
            }
        }

        return ret;
    }

    /**
     * InputStream 流转换成string
     *
     * @param sream
     * @param charset
     * @return
     */
    public static String streamToString(InputStream sream, String charset) {
        String ret = null;
        ByteArrayOutputStream outStream = null;
        byte[] data = null;
        int count = -1;
        try {
            data = new byte[1024];
            outStream = new ByteArrayOutputStream();
            while ((count = sream.read(data, 0, 1024)) != -1) {
                outStream.write(data, 0, count);
            }

            ret = new String(outStream.toByteArray(), charset);
        } catch (IOException e) {
            logger.error("BaseUtils.streamToString: stream convert to String failed," + e);
        }
        return ret;
    }

    /**
     * 手机号码合法性验证 手机号码：11位 移动：134、135、136、137、138、139、150、151、157(TD)、158、159、187、188 联通：130、131、132、152、155、156、185、186
     * 电信：133、153、180、189、（1349卫通
     *
     * @param number 手机号码
     * @return
     */
    public static boolean verifyMobileNum(String number) {
        String regExp = "^[1]([3-9][0-9]{1})[0-9]{8}$";
        Pattern p = Pattern.compile(regExp);
        Matcher matcher = p.matcher(number);

        return matcher.find();
    }

    /**
     * @param objs
     * @param param 变量名称
     * @return
     * @throws NoSuchMethodException
     * @throws SecurityException
     * @throws IllegalAccessException
     * @throws IllegalArgumentException
     * @throws InvocationTargetException
     * @throws NoSuchFieldException
     */
    public static <T> String getStringArray(List<T> objs, String param) {
        StringBuffer sb = new StringBuffer("(");
        try {
            for (T object : objs) {
                Field field = object.getClass().getDeclaredField(param);
                field.setAccessible(true);
                Object id = field.get(object);
                sb.append(id).append(",");
            }
        } catch (Exception e) {
            e.printStackTrace();
        }

        String ret = "";
        if (sb.toString().endsWith(",")) {
            ret = sb.substring(0, sb.length() - 1);
        }
        ret += ")";
        return ret;
    }

    @SuppressWarnings("rawtypes")
    public static <T> List getListFilter(List<T> objs, String param) {
        List<Object> list = new ArrayList<Object>(objs.size());
        try {
            for (T obj : objs) {
                Field field = obj.getClass().getDeclaredField(param);
                field.setAccessible(true);
                Object id = field.get(obj);
                if (id != null) {
                    list.add(id);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return list;
    }

    /**
     * @param objs
     * @param param
     * @return
     */
    public static <T, V> Set<V> getPropertiesList(List<T> objs, String param) {
        Set<V> sets = Sets.newHashSet();
        try {
            for (T object : objs) {
                Field field = object.getClass().getDeclaredField(param);
                field.setAccessible(true);
                @SuppressWarnings("unchecked")
                V id = (V) field.get(object);
                sets.add(id);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }

        return sets;
    }

    /**
     * @param target
     * @param digit 位数
     * @return
     */
    public static Double doubleFormat(Double target, Integer digit) {
        BigDecimal bg = new BigDecimal(target);
        return bg.setScale(digit, BigDecimal.ROUND_HALF_UP).doubleValue();
    }

    /**
     * 获取offset偏移的field时间值
     *
     * @param offset
     * @param field
     * @return
     */
    public static int getCalendarField(int offset, int field) {
        Calendar cal = Calendar.getInstance();
        cal.add(field, offset);
        return cal.get(field);
    }

    /**
     * 获取两个时间之间的时间间隔
     *
     * @param current 当前时间
     * @param source 原始时间
     * @param format 时间格式
     * @return interval
     */
    public static long getInterval(String current, String source, String format) {
        SimpleDateFormat sdf = new SimpleDateFormat(format);

        Date currTime = null;
        Date srcTime = null;
        try {
            currTime = sdf.parse(current);
            srcTime = sdf.parse(source);
        } catch (ParseException e) {
            logger.error("BaseUtils.getInterval: current:{}, source:{}, format:{}", current, source, format);
        }
        return currTime.getTime() - srcTime.getTime();
    }

    /**
     * 将list对象转换成以list中对象属性作为key的MAP
     *
     * @param list
     * @param param
     * @return
     */
    public static <K, V> Map<K, V> listToMap(List<V> list, String param) {
        Map<K, V> map = new HashMap<K, V>();
        for (V v : list) {
            K key = getValue(v, param);
            map.put(key, v);
        }
        return map;
    }

    public static String gbEncoding(final String gbString) {
        char[] utfBytes = gbString.toCharArray();
        String unicodeBytes = "";
        for (int byteIndex = 0; byteIndex < utfBytes.length; byteIndex++) {
            String hexB = Integer.toHexString(utfBytes[byteIndex]);
            if (hexB.length() <= 2) {
                hexB = "00" + hexB;
            }
            unicodeBytes = unicodeBytes + "\\u" + hexB;
        }
        return unicodeBytes;
    }

    public static String decodeUnicode(final String dataStr) {
        int start = 0;
        int end = 0;
        final StringBuffer buffer = new StringBuffer();
        while (start > -1) {
            end = dataStr.indexOf("\\u", start + 2);
            String charStr = "";
            if (end == -1) {
                charStr = dataStr.substring(start + 2, dataStr.length());
            } else {
                charStr = dataStr.substring(start + 2, end);
            }
            char letter = (char) Integer.parseInt(charStr, 16); // 16进制parse整形字符串。
            buffer.append(new Character(letter).toString());
            start = end;
        }
        return buffer.toString();
    }
    
    /**
     * 将T中属性按照Snippet过滤展示
     * @param result
     * @param fields
     * @return
     */
    public static <T, V> List<Map<String, V>> resultWrapper(List<T> result, List<Snippet> snippets) {
        List<Map<String, V>> data = Lists.newArrayList();
        Map<String, V> subMap = null;
        for (T t : result) {
            subMap = Maps.newHashMap();
            for (Snippet snippet : snippets) {
                V v = getValue(t, snippet.getName());
                subMap.put(snippet.getName(), v);
            }
            data.add(subMap);
        }
        return data;
    }

    /**
     * 根据对象属性，获取属性值
     *
     * @param obj object
     * @param param object属性
     * @return
     */
    @SuppressWarnings("unchecked")
    public static <T, V> V getValue(T obj, String param) {
        V ret = null;
        try {
            Field field = obj.getClass().getDeclaredField(param);
            field.setAccessible(true);
            ret = (V) field.get(obj);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return ret;
    }

    public static Long getLong(Number num) {
        if (num == null) {
            return null;
        }
        return num.longValue();
    }

    public static Integer getInt(Number num) {
        if (num == null) {
            return null;
        }
        return num.intValue();
    }

    public static Long getLong(Map<String, Object> map, String key) {
        if (map == null) {
            return null;
        }
        return getLong((Number) map.get(key));
    }

    public static Integer getInt(Map<String, Object> map, String key) {
        if (map == null) {
            return null;
        }
        return getInt((Number) map.get(key));
    }

    public static void disableCache(HttpServletResponse response) {
        response.setHeader("Pragma", "no-cache");
        response.setDateHeader("Expires", 0);
        response.addHeader("Cache-Control", "no-cache");
        response.addHeader("Cache-Control", "no-store");
    }

    /**
     * 字符串转换成ascii码
     *
     * @param source
     * @return
     */
    public static int str2Ascii(String source) {
        char[] chars = source.toCharArray();
        return chars[0];
    }

    /**
     * 字符串转换成ascii码
     *
     * @param source
     * @return
     */
    public static String ascii2Str(int source) {
        return (char) source + "";
    }

    /**
     * 获取当天开始时间
     *
     * @return
     */
    public static String getStartOfDay(Date date) {
        SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd");
        String startTimestr = df.format(date.getTime());
        startTimestr += " 00:00:00";
        return startTimestr;
    }

    /**
     * 获取当天结束时间
     *
     * @return
     */
    public static String getEndOfDay(Date date) {
        SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd");
        String endTimestr = df.format(date.getTime());
        endTimestr += " 23:59:59";
        return endTimestr;
    }
}
