 /**
 * Baijiahulian.com Inc.
 * Copyright (c) 2014-2016 All Rights Reserved.
 */

package com.baijia.tianxiao.annotation;

import java.io.IOException;
import java.lang.reflect.Field;
import java.util.List;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;

import lombok.Data;
import lombok.extern.slf4j.Slf4j;

import com.baijia.tianxiao.enums.CrmErrorCode;
import com.baijia.tianxiao.enums.PropertiesType;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.util.json.JacksonUtil;
import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

/**
 * 用户自定义设置的列表表头
 * @title Snippet
 * @desc TODO 
 * @author zhangbing
 * @date 2016年3月15日
 * @version 1.0
 */
@Data
@Slf4j
public class Snippet {

    //属性名称
    private String name;
    
    //属性对应中文名称
    private String showName;
    
    //属性类型
    private  PropertiesType type;
    
    //是否锁定0:false,1:true
    private boolean lock;
    
    //是否隐藏属性0:false,1:true
    private boolean hidden;
    
    //如果机构没有设置显示哪些属性，取defaultProp=true的属性显示
    private boolean defaultProp;
    
    //查询属性名称
    private String queryProp;
    
    //是否可排序,0:false,1:true
    private int sort;
    
    /**
     * 获取所有表头
     * @param cls
     * @return
     */
    public static List<Snippet> getAllFields(Class<?> clazz) {
        List<Snippet> result = Lists.newArrayList();
        Field[] fields = clazz.getDeclaredFields();
        for (Field field : fields) {
            Option option = field.getAnnotation(Option.class);
            if (option != null) {
                Snippet snippet = new Snippet();
                boolean defaultProp = option.defaultProp();
                
                snippet.setName(option.name());
                snippet.setShowName(option.showName());
                snippet.setDefaultProp(defaultProp);
                snippet.setHidden(option.hidden());
                snippet.setLock(option.lock());
                snippet.setType(option.type());
                snippet.setSort(option.sort());
                
                String queryProp = option.queryProp();
                if (StringUtils.isNotBlank(queryProp)) {
                    snippet.setQueryProp(queryProp);
                } else {
                    snippet.setQueryProp(option.name());
                }
                result.add(snippet);
            }
        }
        return result;
    }
    
    /**
     * 获取默认表头
     * @param cls
     * @return
     */
    public static List<Snippet> getDefaultFields(Class<?> clazz) {
        List<Snippet> result = Lists.newArrayList();
        Field[] fields = clazz.getDeclaredFields();
        for (Field field : fields) {
            Option option = field.getAnnotation(Option.class);
            if (option != null) {
                Snippet snippet = new Snippet();
                boolean defaultProp = option.defaultProp();
                
                if (defaultProp) {
                    snippet.setName(option.name());
                    snippet.setShowName(option.showName());
                    snippet.setDefaultProp(defaultProp);
                    snippet.setHidden(option.hidden());
                    snippet.setLock(option.lock());
                    snippet.setType(option.type());
                    snippet.setSort(option.sort());
                    
                    String queryProp = option.queryProp();
                    if (StringUtils.isNotBlank(queryProp)) {
                        snippet.setQueryProp(queryProp);
                    } else {
                        snippet.setQueryProp(option.name());
                    }
                    
                    result.add(snippet);
                }
            }
        }
        return result;
    }
    
    /**
     * 安全性校验:用户所填写属性是否是clazz中已有的
     * @param clazz
     * @param headers
     * @return
     */
    public static void securityCheck(Class<?> clazz, List<Snippet> headers) throws BussinessException{
        Set<String> properties = getFields(clazz);
        for (Snippet snippet : headers) {
            String property = snippet.getName();
            if (!properties.contains(property)) {
                log.info("invalid propertie name:{}", property);
                throw new BussinessException(CrmErrorCode.ILLEGAL_REQUEST);
            }
        }
    }
    
    /**
     * 提取clazz中属性集合
     */
    public static Set<String> getFields(Class<?> clazz) {
        Field[] fields = clazz.getDeclaredFields();
        Set<String> properties = Sets.newHashSet();
        for (Field field : fields) {
            Option option = field.getAnnotation(Option.class);
            if (option != null) {
                properties.add(option.name());
            }
        }
        return properties;
    }
    
    public static Option getAnnotation(Class<?> clazz, String propName) {
        if (StringUtils.isNotBlank(propName)) {
            Field[] fields = clazz.getDeclaredFields();
            for (Field field : fields) {
                String name = field.getName();
                if (name.equals(propName)) {
                    Option option = field.getAnnotation(Option.class);
                    if (option != null) {
                        return option;
                    }
                }
            }
        }
        return null;
    }
    
    public static void main(String args[]) throws JsonParseException, JsonMappingException, IOException {
        String content = "[{\"name\":\"consulterId\",\"showName\":\"id\",\"type\":\"DIGITAL\",\"lock\":false,\"hidden\":true,\"defaultProp\":true},{\"name\":\"name\",\"showName\":\"学员姓名\",\"type\":\"STRING\",\"lock\":true,\"hidden\":false,\"defaultProp\":true},{\"name\":\"mobile\",\"showName\":\"学员手机\",\"type\":\"STRING\",\"lock\":true,\"hidden\":false,\"defaultProp\":true},{\"name\":\"parentName\",\"showName\":\"家长姓名\",\"type\":\"STRING\",\"lock\":false,\"hidden\":false,\"defaultProp\":true},{\"name\":\"parentMobile\",\"showName\":\"家长手机\",\"type\":\"STRING\",\"lock\":false,\"hidden\":false,\"defaultProp\":true},{\"name\":\"nextRemindTime\",\"showName\":\"下次跟进时间\",\"type\":\"DIGITAL\",\"lock\":false,\"hidden\":false,\"defaultProp\":true},{\"name\":\"createTime\",\"showName\":\"创建时间\",\"type\":\"DIGITAL\",\"lock\":false,\"hidden\":false,\"defaultProp\":false}]";
        List<Snippet> snippets = JacksonUtil.str2List(content, Snippet.class);
        for (Snippet snippet : snippets) {
            System.out.println(snippet.getName() +"--"+ snippet.getShowName() +"--"+ snippet.getType() +"--"+ snippet.isDefaultProp() +"--"+ snippet.isHidden() +"--"+ snippet.isLock());
        }
    }
}

    