/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */
package com.baijia.tianxiao.sal.wechat.impl;

import java.io.File;
import java.util.Date;
import java.util.List;

import lombok.extern.slf4j.Slf4j;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.baijia.tianxiao.constants.MIMEType;
import com.baijia.tianxiao.dal.org.constant.DeleteStatus;
import com.baijia.tianxiao.dal.org.dao.OrgStorageDao;
import com.baijia.tianxiao.dal.org.po.OrgStorage;
import com.baijia.tianxiao.dal.wechat.constant.WechatDataSyncStatus;
import com.baijia.tianxiao.dal.wechat.dao.AuthorizerInfoDao;
import com.baijia.tianxiao.dal.wechat.dao.ComponentAccessTokenDao;
import com.baijia.tianxiao.dal.wechat.po.AuthorizerInfo;
import com.baijia.tianxiao.dto.upload.UploadResult;
import com.baijia.tianxiao.dto.upload.UploadResult.UploadFile;
import com.baijia.tianxiao.sal.wechat.api.AuthorizationInfoService;
import com.baijia.tianxiao.sal.wechat.api.AuthorizerInfoService;
import com.baijia.tianxiao.sal.wechat.dto.wechatapi.OpenPlatformInfoDto;
import com.baijia.tianxiao.sal.wechat.helper.WechatProperties;
import com.baijia.tianxiao.sal.wechat.helper.openplat.OpenPlatApiHelper;
import com.baijia.tianxiao.sal.wechat.util.FileUploadUtils;
import com.baijia.tianxiao.sal.wechat.util.LocalFileHelper;
import com.baijia.tianxiao.util.httpclient.HttpClientUtils;

@Slf4j
@Service
public class AuthorizerInfoServiceImpl implements AuthorizerInfoService {

    private static final Logger logger = LoggerFactory.getLogger(AuthorizerInfoServiceImpl.class);

    @Autowired
    private AuthorizerInfoDao authorizerInfoDao;
    @Autowired
    private ComponentAccessTokenDao componentAccessTokenDao;
    @Autowired
    private OrgStorageDao orgStorageDao;
    @Autowired
    private AuthorizationInfoService authorizationInfoService;


    @Transactional(rollbackFor = Exception.class)
    @Override
    public void update(AuthorizerInfo authorizerInfo) {
        authorizerInfoDao.update(authorizerInfo, true);
        log.info("wechat - AuthorizerInfoServiceImpl - update authorizerInfo:{}", authorizerInfo);
    }

    
    
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void resetOngoing() {
        log.info("wechat - AuthorizerInfoServiceImpl - resetOngoing");
        authorizerInfoDao.resetOngoingStatus();
    }

    
    
    @Transactional(readOnly = true)
    @Override
    public AuthorizerInfo getByOrgId(Integer orgId) {
        return authorizerInfoDao.getByOrgId(orgId);
    }


    
    @Transactional(readOnly = true)
    @Override
    public AuthorizerInfo getByUserName(String userName) {
        return authorizerInfoDao.getByUserName(userName);
    }


    
    @Transactional(readOnly = true)
    @Override
    public AuthorizerInfo getByAuthorizerAppId(String authorizerAppId) {
        return authorizerInfoDao.getByAuthorizerAppId(authorizerAppId);
    }


    
    @Transactional(rollbackFor = Exception.class)
    @Override
    public AuthorizerInfo syncAuthorizerInfo(Integer orgId, String authorizerAppId) {
        logger.info("wechat - sync authorizer info - orgId:{}, authorizerAppId:{}", orgId, authorizerAppId);
        //参数
        OpenPlatformInfoDto openPlatformInfo = WechatProperties.getOpenPlatformInfo();
        String componentAccessToken = componentAccessTokenDao.getByAppId(openPlatformInfo.getAppId()).getComponentAccessToken();

        //请求
        AuthorizerInfo newAuthorizerInfo = OpenPlatApiHelper.requestAuthorizerInfo(openPlatformInfo.getAppId(), componentAccessToken, authorizerAppId);

        //结果处理
        newAuthorizerInfo.setAuthorizerAppId(authorizerAppId);
        newAuthorizerInfo.setOrgId(orgId);

        Date now = new Date();
        AuthorizerInfo info = authorizerInfoDao.getByOrgId(orgId);
        if (info == null) {
            info = newAuthorizerInfo;
            info.setQrcodeStorageId(uploadQrcode(newAuthorizerInfo.getOrgId(), newAuthorizerInfo.getQrcodeUrl()));
            info.setUpdateTime(now);
            info.setCreateTime(now);
            info.setFansSyncStatus(WechatDataSyncStatus.NOTSTART.getValue());
            info.setMediaNewsSyncStatus(WechatDataSyncStatus.NOTSTART.getValue());
            authorizerInfoDao.save(info, true);
            logger.info("wechat - add new authorizer info - info:{}", info);
            return info;
        } else {
            info.setAuthorizerAppId(newAuthorizerInfo.getAuthorizerAppId());
            info.setOrgId(newAuthorizerInfo.getOrgId());
            info.setAlias(newAuthorizerInfo.getAlias());
            info.setNickName(newAuthorizerInfo.getNickName());
            info.setUserName(newAuthorizerInfo.getUserName());
            info.setHeadImg(newAuthorizerInfo.getHeadImg());
            info.setQrcodeUrl(newAuthorizerInfo.getQrcodeUrl());
            info.setQrcodeStorageId(uploadQrcode(orgId, info.getQrcodeUrl()));
            info.setServiceType(newAuthorizerInfo.getServiceType());
            info.setVerifyType(newAuthorizerInfo.getVerifyType());
            info.setUpdateTime(now);
            authorizerInfoDao.update(info, true);
            logger.info("wechat - update authorizer info - info:{}", info);
            return info;
        }
    }


    private Integer uploadQrcode(Integer orgId, String url) {
        if (StringUtils.isBlank(url)) {
            logger.warn("wechat - upload qrcode - url is null - orgId:{}", orgId);
            return 0;
        }

        long start = System.currentTimeMillis();
        File localFile = null;
        Long uid = Long.parseLong(orgId + "");
        try {
            byte[] data = HttpClientUtils.download(url);
            if (data == null) {
                logger.warn("wechat - upload qrcode - download failed - orgId:{}, url:{}", orgId, url);
                return 0;
            }
            localFile = LocalFileHelper.saveToLocal(data, MIMEType.JPEG);

            UploadResult result = FileUploadUtils.uploadToRemote(uid, localFile, false);
            long end = System.currentTimeMillis();
            logger.info("UploadService.upload: uid:{}, file.len:{}, cost:{}", uid, data.length, (end - start));

            return saveToDb(result);
        } catch (Exception e) {
            logger.error("wechat - upload qrcode exception - ", e);
            return 0;
        } finally {
            if (localFile != null) {
                LocalFileHelper.deleteFile(localFile);
            }
        }
    }

    private Integer saveToDb(UploadResult uploadResult) {
        List<UploadFile> uploadFiles = uploadResult.getFiles();
        if (uploadFiles.size() == 0) {
            return 0;
        }
        UploadFile uploadFile = uploadFiles.get(0);
        OrgStorage orgStorage = new OrgStorage();
        orgStorage.setFid(uploadFile.getFid());
        orgStorage.setSn(uploadFile.getSn());
        orgStorage.setSize(uploadFile.getSize());
        orgStorage.setMimeType(MIMEType.JPEG.getCode());
        orgStorage.setSource((short) 0);
        Date now = new Date();
        orgStorage.setCreateTime(now);
        orgStorage.setUpdateTime(now);
        orgStorage.setIsDel(DeleteStatus.NORMAL.getValue());
        orgStorageDao.save(orgStorage, true);
        return orgStorage.getId();
    }
}
