/**
 * Baijiahulian.com Inc. Copyright (c) 2015-2015 All Rights Reserved.
 */
package com.baijia.tianxiao.sal.wechat.impl;


import java.io.File;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

import com.baijia.tianxiao.constant.Flag;
import com.baijia.tianxiao.constants.MIMEType;
import com.baijia.tianxiao.dal.org.constant.DeleteStatus;
import com.baijia.tianxiao.dal.org.dao.OrgStorageDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentDao;
import com.baijia.tianxiao.dal.org.po.OrgStorage;
import com.baijia.tianxiao.dal.org.po.OrgStudent;
import com.baijia.tianxiao.dal.push.constant.MsgUserRole;
import com.baijia.tianxiao.dal.roster.po.TxConsultUser;

import com.baijia.tianxiao.enums.RedisKeyEnums;
import lombok.extern.slf4j.Slf4j;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.core.RedisConnectionUtils;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import com.baijia.tianxiao.dal.roster.dao.TxConsultUserDao;
import com.baijia.tianxiao.dal.wechat.constant.WechatDataSyncStatus;
import com.baijia.tianxiao.dal.wechat.constant.WechatDataSyncType;
import com.baijia.tianxiao.dal.wechat.dao.AuthorizationInfoDao;
import com.baijia.tianxiao.dal.wechat.dao.AuthorizerInfoDao;
import com.baijia.tianxiao.dal.wechat.dao.FansDao;
import com.baijia.tianxiao.dal.wechat.po.AuthorizationInfo;
import com.baijia.tianxiao.dal.wechat.po.AuthorizerInfo;
import com.baijia.tianxiao.dal.wechat.po.Fans;
import com.baijia.tianxiao.dal.wechat.po.OrgWechatDataSyncLog;
import com.baijia.tianxiao.dto.upload.UploadResult;
import com.baijia.tianxiao.dto.upload.UploadResult.UploadFile;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.exception.CustomException;
import com.baijia.tianxiao.exception.WebServiceException;
import com.baijia.tianxiao.exception.WechatException;
import com.baijia.tianxiao.sal.wechat.api.AuthorizationInfoService;
import com.baijia.tianxiao.sal.wechat.api.AuthorizerInfoService;
import com.baijia.tianxiao.sal.wechat.api.FansService;
import com.baijia.tianxiao.sal.wechat.api.WechatDataSyncLogService;
import com.baijia.tianxiao.enums.SalWechatErrorCode;
import com.baijia.tianxiao.sal.wechat.constant.WechatApi;
import com.baijia.tianxiao.sal.wechat.dto.fans.FansDetailDto;
import com.baijia.tianxiao.sal.wechat.dto.fans.FansListDto;
import com.baijia.tianxiao.sal.wechat.dto.fans.FansListRecordDto;
import com.baijia.tianxiao.sal.wechat.helper.WechatProperties;
import com.baijia.tianxiao.sal.wechat.helper.user.FansServiceApiHelper;
import com.baijia.tianxiao.sal.wechat.task.FansSyncJob;
import com.baijia.tianxiao.sal.wechat.task.TaskHelper;
import com.baijia.tianxiao.sal.wechat.util.FileUploadUtils;
import com.baijia.tianxiao.sal.wechat.util.LocalFileHelper;
import com.baijia.tianxiao.sal.wechat.util.StorageUtil;
import com.baijia.tianxiao.sal.wechat.validator.WechatApiValidator;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.util.httpclient.HttpClientUtils;

/**
 * @author : zhenyujian
 * @title : FansServiceImpl
 * @description : 微信粉丝相关
 * @date : 2015年12月8日 下午8:19:13
 */
/**   
 * @title       : FansServiceImpl 
 * @description : 
 * @author      : zhenyujian
 * @date        : 2016年10月29日 下午5:12:08 
 */
@Slf4j
@Service
public class FansServiceImpl implements FansService {

    @Autowired
    private FansDao fansDao;
    @Autowired
    private AuthorizationInfoDao authorizationInfoDao;
    @Autowired
    private AuthorizerInfoDao authorizerInfoDao;
    @Autowired
    private AuthorizationInfoService authorizationInfoService;
    @Autowired
    private TxConsultUserDao txConsultUserDao;
    @Autowired
    private OrgStudentDao orgStudentDao;
    @Autowired
    private OrgStorageDao orgStorageDao;
    
    @Autowired
    private AuthorizerInfoService authorizerInfoService;
    
    @Autowired(required = false)
	private RedisTemplate<String, String> redisTemplate;
    
    @Autowired
	private WechatDataSyncLogService wechatDataSyncLogService;
    
    
    public boolean redisFansSyncInterval(String appId) {
    	//(5分钟内只允许触发一次粉丝更新)
    	String key = String.format(RedisKeyEnums.CRM.TX_WECHAT_FANS_INTERVAL_PRE.getRedisKey(), appId);
    	Long intervalSeconds = 5L;
    	
		RedisConnection connection = redisTemplate.getConnectionFactory().getConnection();
		try {
			connection.select(WechatProperties.getRedisDB());

			if (connection.exists(key.getBytes())) {
				return false;
			} else {
				connection.setEx(key.getBytes(), intervalSeconds, "".getBytes());
				return true;
			}
		} catch (Exception e) {
			log.error("Wechat - redisFansSyncInterval - exception - e:{}", e);
			return true;
		} finally {
			RedisConnectionUtils.releaseConnection(connection, redisTemplate.getConnectionFactory());
		}
	}

    /* 
     * 本地数据库粉丝数与微信端粉丝数不相等 则同步
     */
    @Override
    public boolean isNeedToSync(int orgId) {

        try {
        	AuthorizerInfo authorizerInfo = authorizerInfoDao.getByOrgId(orgId);
        	
        	//判断是否已有同步任务在进行中
        	if( wechatDataSyncLogService.hasTaskOngoing(authorizerInfo.getAuthorizerAppId(), WechatDataSyncType.FANS) ) {
    			log.info("wechat fans - isNeedToSync - RETURN FALSE - orgId:{}",orgId);
    			return false;
    		}
        	
        	//判断两次同步间隔时间是否合法
        	if( !redisFansSyncInterval(authorizerInfo.getAuthorizerAppId()) ){
        		log.info("wechat fans - isNeedToSync - RETURN FALSE - orgId:{}",orgId);
        		return false;
        	}
        	
            AuthorizationInfo authorizationInfo = authorizationInfoService.refreshAccessToken(orgId);
            
            WechatApiValidator._4CallApi(authorizationInfo, authorizerInfo, WechatApi.USER_LIST, true);

            int fansCount = fansDao.countSubscribe(authorizationInfo.getAuthorizerAppId());

            int wechatFansCount = FansServiceApiHelper.getFansCount(authorizationInfo.getAuthorizerAccessToken());

            if(fansCount!=wechatFansCount){
            	log.info("wechat fans - isNeedToSync - RETURN TURE - orgId:{}, fansCount:{},wechatFansCount:{}", orgId, fansCount,wechatFansCount);
            	return true;
            }

        } catch (BussinessException | WechatException e) {
            log.warn("wechat fans - isNeedToSync - orgId:{}, CustomException:{}", orgId, e);
        } catch (Exception e) {
        	log.error("wechat fans - isNeedToSync - Exception - orgId:{}", orgId);
            log.error("wechat fans - isNeedToSync - Exception", e);
        }
        
        log.info("wechat fans - isNeedToSync - RETURN FALSE - orgId:{}",orgId);
        return false;
    }


    @Override
    public FansListDto listFans(Integer orgId, String searchText, PageDto pageDto) throws CustomException {
        AuthorizationInfo authorizationInfo = authorizationInfoDao.getByOrgId(orgId);
        AuthorizerInfo authorizerInfo = authorizerInfoDao.getByOrgId(orgId);

        //权限判断
        WechatApiValidator._4CallApi(authorizationInfo, authorizerInfo, WechatApi.USER_LIST);

        //判断是否需要从微信同步粉丝数据
        if (pageDto.getPageNum() == 1 && isNeedToSync(orgId)) {
            try {
                log.info("wechat - listFans - handle - add fansSync task - appId:{}", authorizerInfo.getAuthorizerAppId());
                TaskHelper.getInstance().addTask(new FansSyncJob(this, authorizerInfo.getAuthorizerAppId()));
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        String authorizerAppId = authorizerInfo.getAuthorizerAppId();
        
        //分页查询
        Integer count = fansDao.count(authorizerAppId);
        List<Fans> fansList = null;
        if (count > 0) {
            fansList = fansDao.search(authorizerAppId, searchText, pageDto);
        } else {
            fansList = new ArrayList<Fans>();
        }
        pageDto.setCurPageCount(fansList.size());
        pageDto.setCount(count);

        //结果填充
        List<FansListRecordDto> recordList = FansListRecordDto.buildList(fansList);
        if (recordList != null && !recordList.isEmpty()) {
            List<String> openIdList = new ArrayList<String>();
            for (FansListRecordDto dto : recordList) {
                openIdList.add(dto.getOpenId());
            }
            Map<String, TxConsultUser> keyOpenIdValueConsultUserIdMap = txConsultUserDao.mapKeyOpenIdValueId(orgId, openIdList);
            for (FansListRecordDto dto : recordList) {
                TxConsultUser user = keyOpenIdValueConsultUserIdMap.get(dto.getOpenId());
                if (user != null) {
                    dto.setConsultUserId(user.getId());
                    dto.setUserRole(getUserRole(user));
                } else {
                    dto.setUserRole(-1);
                    dto.setConsultUserId(-1L);
                }
            }
        }

        FansListDto dto = new FansListDto(recordList);

        return dto;
    }

    public int getUserRole(TxConsultUser user) {
        MsgUserRole userType = null;
        if (user.getStudentId() != null && user.getStudentId() > 0) {
            userType = MsgUserRole.STUDENT;
        } else {
            if (user.getUserId() != null && user.getUserId() > 0) {
                OrgStudent orgStudent = orgStudentDao.getStudent(user.getOrgId(), user.getUserId(), DeleteStatus.NORMAL.getValue());
                if (orgStudent != null) {
                    userType = MsgUserRole.STUDENT;
                }
            }
        }

        if (userType == null) {
            if ( Flag.getBoolean(user.getIsConsulter()) ) {
                userType = MsgUserRole.CLUE;
            } else {
                userType = MsgUserRole.CONSULT;
            }
        }

        return userType.getValue();
    }

    
    
    @Transactional(readOnly = true)
    @Override
    public FansDetailDto getFansDetail(String openId) {
        Fans fans = fansDao.getByOpenId(openId);
        return FansDetailDto.build(fans);
    }


    
    @Transactional(rollbackFor = Exception.class)
    @Override
    @Deprecated
    public void syncFansFullAmount(String authorizerAppId) throws CustomException {
        log.info("wechat - FansService - syncFansFullAmount - start - authorizerAppId:{}", authorizerAppId);
        int fansNum = 0;
        int failNum = 0;
        AuthorizationInfo authorizationInfo = authorizationInfoService.getByAuthorizerAppId(authorizerAppId);
        authorizationInfo = authorizationInfoService.refreshAccessToken(authorizationInfo.getOrgId());
        String accessToken = authorizationInfo.getAuthorizerAccessToken();

        //从微信拉取所有的粉丝id
        List<String> openIds = FansServiceApiHelper.getAllFansOpenIdList(accessToken);
        int openIdSize = (openIds == null ? 0 : openIds.size());
        log.info("wechat - FansService - syncFans - start - openIds size:{}", openIdSize);

        if (openIds != null && !openIds.isEmpty()) {
            fansNum = openIds.size();

            List<String> tempOpendIds = new ArrayList<String>();
            for (int i = 0; i < openIds.size(); i++) {
                tempOpendIds.add(openIds.get(i));
                //批量插入 每次100条
                if ((i != 0 && i % 100 == 0) || i == openIds.size() - 1) {
                    Map<String, Fans> openIdMap = fansDao.mapKeyOpenIdValueFans(tempOpendIds);
                    List<Fans> fansList = new ArrayList<Fans>();
                    Fans fans = null;

                    for (String tempOpenId : tempOpendIds) {
                        try {
                            fans = FansServiceApiHelper.getFansInfo(accessToken, tempOpenId);
                        } catch (WechatException e) {
                            if (e.getErrorCode() == SalWechatErrorCode.WECHAT_ACCESS_TOKEN_INVALID) {
                                //数据同步过程中accesstoken失效 重新获取accesstoken
                                accessToken = authorizationInfoService.forceRefreshAccessToken(authorizationInfo.getOrgId()).getAuthorizerAccessToken();
                                fans = FansServiceApiHelper.getFansInfo(accessToken, tempOpenId);
                            } else {
                                log.error("wechat - FansServiceImpl - syncFansFullAmount - getFansInfo WechatException(all fail) - authorizerAppId:{},e:{}", authorizerAppId, e);
                                throw e;
                            }
                        } catch (WebServiceException e) {
                            log.warn("wechat - FansServiceImpl - syncFansFullAmount - getFansInfo Exception - authorizerAppId:{},openId:{},e:{}", authorizerAppId, tempOpenId, e);
                            failNum++;
                            continue;
                        } catch (Exception e) {
                            log.error("wechat - FansServiceImpl - syncFansFullAmount - getFansInfo Exception - authorizerAppId:{},openId:{},e:{}", authorizerAppId, tempOpenId, e);
                            failNum++;
                            continue;
                        }

                        if (fans != null) {
                            Fans dbFans = openIdMap.get(tempOpenId);
                            if (dbFans == null) {
                                fans.initBasicAttribute();
                                fans.setAuthorizerAppId(authorizerAppId);

                                //临时处理2期 与粉丝聊天必须是粉丝曾主动发起过聊天的情况 //FIXME
                                fans.setLastCommunicationTime(new Date(0));
                                fansList.add(fans);
                            } else {
                                if (!dbFans.getNick().equals(fans.getNick())
                                        || !dbFans.getHeadImgUrl().equals(fans.getHeadImgUrl())
                                        || !dbFans.getSubscribe().equals(fans.getSubscribe())) {
                                    fans.setId(dbFans.getId());
                                    //临时处理2期 与粉丝聊天必须是粉丝曾主动发起过聊天的情况 //FIXME
                                    fans.setLastCommunicationTime(dbFans.getLastCommunicationTime());
                                    fans.setSubscribe(1);
                                    fans.setUpdateTime(new Date());
                                    fansDao.update(fans);
                                } else {
                                    Fans oldFans = fansDao.getByOpenId(tempOpenId);
                                    if (oldFans.getSubscribe() == 0) {
                                        oldFans.setSubscribe(1);
                                        oldFans.setUpdateTime(new Date());
                                        fansDao.update(oldFans);

                                    }
                                }
                            }
                        }
                    }
                    try {
                        batchSaveFans(fansList);
                    } catch (Exception e) {
                        log.warn("wechat - FansServiceImpl - fans batch insert fail", e);
                    }
                    tempOpendIds = new ArrayList<String>();
                }
            }
        }
        log.info("wechat - FansService - syncFansFullAmount - end - authorizerAppId:{}, fansNum:{}, failNum:{}", authorizerAppId, fansNum, failNum);
    }
    
    
    
    @Override
    public Integer syncFansIncrement(String authorizerAppId) throws CustomException {
    	
    	//启动粉丝同步
    	log.info("wechat - FansService - syncFansIncrement - start - authorizerAppId:{}", authorizerAppId);
    	Date now = new Date();
    	Date startTime = now;
    	Integer amountExpected = 0;
    	Integer amountCompleted = 0;
    	
    	AuthorizationInfo authorizationInfo = authorizationInfoService.refreshAccessToken(authorizerAppId);
    	String accessToken = authorizationInfo.getAuthorizerAccessToken();
    	
    	//从微信拉取   所有的粉丝id
        HashSet<String> openIds = FansServiceApiHelper.getAllFansOpenIdHashSet(accessToken);

        //更新所有取消关注的粉丝状态
        this.batchUpdateFansToUnSubscribe(authorizerAppId, openIds, now);
        
        //从数据库查询 所有的粉丝id
        List<String> openIdInDB = fansDao.listOpenIds(authorizerAppId);
        if( CollectionUtils.isNotEmpty(openIdInDB) ){
	        for(String inDB:openIdInDB){
	        	openIds.remove(inDB);
	        }
        }
        
        //说明没有需要新增的粉丝
        if(openIds.size() == 0){
        	log.info("wechat - FansService - syncFansIncrement - return - authorizerAppId:{}", authorizerAppId);
    		return 0;
        }
        
        //检查任务log
    	if(wechatDataSyncLogService.hasTaskOngoing(authorizerAppId, WechatDataSyncType.FANS)){
    		log.info("wechat - FansService - syncFansIncrement - return - authorizerAppId:{}", authorizerAppId);
    		return 0;
    	}
    	
    	//创建任务Log
    	now = new Date();
    	amountExpected = openIds.size();
    	Date syncLogExpireTime = new Date( now.getTime()+amountExpected*500 ); //按每条记录执行时间0.5秒 计算任务超时时间
    	WechatDataSyncStatus syncLogEndStatus = WechatDataSyncStatus.END_SUCCESS;
    	OrgWechatDataSyncLog syncLog = wechatDataSyncLogService.buildDefaultLog(authorizerAppId, WechatDataSyncType.FANS, startTime, amountExpected, syncLogExpireTime);
        
    	try{
	        Fans fans = null;
	        List<Fans> fansList = new ArrayList<Fans>();
	        
	        int i=0;
	        for (String tempOpenId : openIds) {
	            try {
	                fans = FansServiceApiHelper.getFansInfo(accessToken, tempOpenId);
	            } catch (WechatException e) {
	                if (e.getErrorCode() == SalWechatErrorCode.WECHAT_ACCESS_TOKEN_INVALID) {
	                    //数据同步过程中accesstoken失效 重新获取accesstoken
	                    accessToken = authorizationInfoService.forceRefreshAccessToken(authorizationInfo.getOrgId()).getAuthorizerAccessToken();
	                    fans = FansServiceApiHelper.getFansInfo(accessToken, tempOpenId);
	                } else {
	                    log.error("wechat - FansServiceImpl - syncFansIncrement - getFansInfo WechatException(all fail) - authorizerAppId:{},e:{}", authorizerAppId, e);
	                    throw e;
	                }
	            } catch (WebServiceException e) {
	                log.warn("wechat - FansServiceImpl - syncFansIncrement - getFansInfo Exception - authorizerAppId:{},openId:{},e:{}", authorizerAppId, tempOpenId, e);
	                continue;
	            } catch (Exception e) {
	                log.error("wechat - FansServiceImpl - syncFansIncrement - getFansInfo Exception - authorizerAppId:{},openId:{},e:{}", authorizerAppId, tempOpenId, e);
	                continue;
	            }
	
	            if (fans != null) {                
	                fans.initBasicAttribute();
	                fans.setAuthorizerAppId(authorizerAppId);
	                fans.setLastCommunicationTime(new Date(0));//临时处理2期 与粉丝聊天必须是粉丝曾主动发起过聊天的情况 //FIXME
	                fansList.add(fans);
	            }
	            
	            //每100条批量插入
	            if( (i != 0 && i % 100 == 0) || i == openIds.size() - 1 ){
	            	if(CollectionUtils.isNotEmpty(fansList)){
		            	try {
		            		//插入粉丝
		                    batchSaveFans(fansList);
		                    amountCompleted += fansList.size();
		                    
		                    //更新日志进度
		                    now = new Date();
		                    syncLog.setAmountCompleted(amountCompleted);
		                    syncLog.setUpdateTime(now);
		                    wechatDataSyncLogService.saveOrUpdateInNewTransaction(syncLog);
		                    
		                } catch (Exception e) {
		                    log.warn("wechat - syncFansIncrement - fans batch insert fail", e);
		                }
		            	fansList = new ArrayList<Fans>();
	            	}
	            }
	            i++;
	        }
    	}catch(Exception e){
    		syncLogEndStatus = WechatDataSyncStatus.END_FAIL;
    		throw e;
    	}finally{
    		//跟新日志进度
            now = new Date();
            syncLog.setStatus(syncLogEndStatus.getValue());
            syncLog.setAmountCompleted(amountCompleted);
    		syncLog.setStartTime(startTime);				
    		syncLog.setEndTime(now);
    		syncLog.setUpdateTime(now);
    		wechatDataSyncLogService.saveOrUpdateInNewTransaction(syncLog);
    	}
    
    	return amountCompleted; 
    }


    
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void remark(Integer orgId, String openId, String remark) {
        if (remark == null) {
            remark = "";
        }

        AuthorizationInfo authorizationInfo = authorizationInfoService.refreshAccessToken(orgId);
        if (authorizationInfo == null) {
            throw new BussinessException(SalWechatErrorCode.ORG_NUBIND_WECHATAPP);
        }

        FansServiceApiHelper.remark(authorizationInfo.getAuthorizerAccessToken(), openId, remark);
        Fans fans = fansDao.getByOpenId(openId);
        fans.setRemark(remark);
        fansDao.update(fans, true);
        log.info("wechat - FansService - remark - orgId:{}, openId:{}, remark:{}", orgId, openId, remark);
    }


    
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void saveOrUpdateFans(Fans fans) {
        Fans oldFans = fansDao.getByOpenId(fans.getOpenId());
        if (oldFans == null) {
            //插入
            fans.initBasicAttribute();
            fansDao.save(fans, true);
            try{
            	convertFansHeadImg(null, fans);
            }catch(Exception e){
            	log.warn("saveOrUpdateFans - convertFansHeadImg - exception:{}",e);
            }
        } else {
            //更新
            oldFans.refreshBy(fans);
            fansDao.update(oldFans);
            try{
            	convertFansHeadImg(null, oldFans);
            }catch(Exception e){
            	log.warn("saveOrUpdateFans - convertFansHeadImg - exception:{}",e);
            }
        }
    }


    
    @Transactional(readOnly = true)
    @Override
    public Fans getFans(int orgId, String openId) {
        Fans fans = fansDao.getByOpenId(openId);
        if (fans == null) {
            return null;
        }

        AuthorizerInfo authorizerInfo = authorizerInfoDao.getByOrgId(orgId);
        if (authorizerInfo == null || !authorizerInfo.getAuthorizerAppId().equals(fans.getAuthorizerAppId())) {
            return null;
        }

        return fans;
    }


    
    @Transactional(readOnly = true)
    @Override
    public Fans getFans(String openId) {
        return fansDao.getByOpenId(openId);
    }


    
    @Transactional(propagation = Propagation.REQUIRES_NEW, rollbackFor = Exception.class)
    @Override
    public void batchSaveFans(List<Fans> fansList) {
        fansDao.saveAll(fansList,false);
    }

    
    
    @Transactional(propagation = Propagation.REQUIRES_NEW, rollbackFor = Exception.class)
	@Override
	public String convertFansHeadImg(Long orgId, Fans fans) {
    	//判断开关
    	if(!WechatProperties.getSwitchFansImgConvert()){
    		return "";
    	}
    	
    	if(orgId==null){
    		orgId = 0L;
    	}
		if( fans == null || StringUtils.isBlank(fans.getHeadImgUrl()) ){
			return null;
		}
		if( StringUtils.isNotBlank(fans.getHeadImgUrlGsx()) ){
			return fans.getHeadImgUrlGsx();
		}
		
		String wxHeadImgUrl = fans.getHeadImgUrl();
		String headImgUrlGsx = null;
		File localFile = null;
		Date now = new Date();
		MIMEType mime = MIMEType.JPG;
		
		try {
			//文件下载、上传
			byte[] data = HttpClientUtils.download( wxHeadImgUrl );
			if(data!=null){
				localFile = LocalFileHelper.saveToLocal(data, mime);
				UploadResult result = FileUploadUtils.uploadToRemote(orgId, localFile, false);
				UploadFile uploadFile = result.getFiles().get(0);
				saveToDb(uploadFile, mime);
				
				headImgUrlGsx = StorageUtil.constructUrl(uploadFile.getFid(), uploadFile.getSn(), mime.getCode());
				
				//设置url
				fans.setHeadImgUrlGsx(headImgUrlGsx);
				fans.setUpdateTime(now);
				fansDao.update(fans);
				
				//你懂的
				txConsultUserDao.replacePortrait(wxHeadImgUrl, headImgUrlGsx);
				//orgStudentDao.replacePortrait(wxHeadImgUrl, headImgUrlGsx);
			}else{
				log.warn("wechat - fans convertFansHeadImg - download wechat img error");
			}
		} catch (Exception e) {
			log.error("wechat - fans convertFansHeadImg - exception",e);
		} finally {
			if (localFile != null) {
				LocalFileHelper.deleteFile(localFile);
			}
			log.info("convertFansHeadImg fansId:{}, headImg:{}",fans.getId(), headImgUrlGsx);
		}
		return headImgUrlGsx;
	}
	
	
	private Integer saveToDb(UploadFile uploadFile, MIMEType mimeType) {
		OrgStorage orgStorage = new OrgStorage();
		orgStorage.setFid(uploadFile.getFid());
		orgStorage.setSn(uploadFile.getSn());
		orgStorage.setSize(uploadFile.getSize());
		orgStorage.setMimeType(mimeType.getCode());
		Date now = new Date();
		orgStorage.setCreateTime(now);
		orgStorage.setUpdateTime(now);
		orgStorage.setIsDel(DeleteStatus.NORMAL.getValue());
		orgStorage.setSource((short) 0);
		orgStorageDao.save(orgStorage, true);
		return orgStorage.getId();
	}

	
	
	@Transactional(rollbackFor = Exception.class)
	@Override
	public void batchConvertFansHeadImg(String authorizerAppId) {
		Integer pageNum = 1;
		PageDto dto = new PageDto();
		dto.setPageSize(500);
		dto.setPageNum(pageNum);
		List<Fans> fansList = fansDao.listFansWhichGxsImgIsNull(authorizerAppId,dto);
		
		while(CollectionUtils.isNotEmpty(fansList)){
		
			Long orgId = 0L;
			if( StringUtils.isNotBlank(authorizerAppId) ){
				AuthorizationInfo authorizationInfo = authorizationInfoService.getByAuthorizerAppId(authorizerAppId);
				if(authorizationInfo!=null){
					orgId = authorizationInfo.getOrgId().longValue();
				}
			}
			log.info("batchConvertFansHeadImg - start - fans size:{}",fansList.size() );
			for(Fans fans:fansList){
				try{
					String url = convertFansHeadImg(orgId, fans);
				}catch(Exception e){
					log.warn("batchConvertFansHeadImg - exception",e);
				}
			}
			pageNum =pageNum+1;
			dto.setPageNum(pageNum);
			fansList = fansDao.listFansWhichGxsImgIsNull(authorizerAppId,dto);
		}
		log.info("batchConvertFansHeadImg - end - fans size:{}",fansList.size() );
	}

	

	@Override
	public void updateFansInfo(String openId) {
		Fans fansInDB = fansDao.getByOpenId(openId);
		if(fansInDB!=null){
			try {
				AuthorizationInfo authorizationInfo = authorizationInfoService.refreshAccessToken(fansInDB.getAuthorizerAppId());
		    	String accessToken = authorizationInfo.getAuthorizerAccessToken();
                Fans fansInWechat = FansServiceApiHelper.getFansInfo(accessToken, fansInDB.getOpenId());
                if (!fansInDB.getNick().equals(fansInWechat.getNick())
                        || !fansInDB.getHeadImgUrl().equals(fansInWechat.getHeadImgUrl())
                        || !fansInDB.getSubscribe().equals(fansInWechat.getSubscribe())) {
                	fansInWechat.setId(fansInDB.getId());
                    //临时处理2期 与粉丝聊天必须是粉丝曾主动发起过聊天的情况 //FIXME
                	fansInWechat.setLastCommunicationTime(fansInDB.getLastCommunicationTime());
                	fansInWechat.setUpdateTime(new Date());
                    fansDao.update(fansInWechat);
                }
            } catch (Exception e) {
            	log.warn("wechat - updateFansInfo - fail - openId:{}, e:{}", openId, e);
            }
		}
		
	}

	
	@Transactional
	@Override
	public void batchUpdateFansToUnSubscribe(String authorizerAppId, HashSet<String> openIds, Date now) {
		fansDao.batchUpdateFansToUnSubscribe(authorizerAppId, openIds, now);
	}
}

