/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */
package com.baijia.tianxiao.sal.wechat.helper.common;

import java.util.Date;

import lombok.extern.slf4j.Slf4j;

import org.dom4j.Document;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;

import com.baijia.tianxiao.exception.WebServiceException;
import com.baijia.tianxiao.sal.wechat.constant.wechatmsg.WechatMsgEvent;
import com.baijia.tianxiao.sal.wechat.constant.wechatmsg.WechatMsgType;
import com.baijia.tianxiao.sal.wechat.dto.wechatmsg.WechatAbstractMessage;
import com.baijia.tianxiao.sal.wechat.dto.wechatmsg.event.WechatEventMessage;
import com.baijia.tianxiao.sal.wechat.dto.wechatmsg.event.WechatLocationEventMessage;
import com.baijia.tianxiao.sal.wechat.dto.wechatmsg.event.WechatSCANEventMessage;
import com.baijia.tianxiao.sal.wechat.dto.wechatmsg.event.WechatSubscribeEventMessage;
import com.baijia.tianxiao.sal.wechat.dto.wechatmsg.event.WechatUnsubscribeEventMessage;
import com.baijia.tianxiao.sal.wechat.dto.wechatmsg.ordinary.WechatImageMessage;
import com.baijia.tianxiao.sal.wechat.dto.wechatmsg.ordinary.WechatOrdinaryMessage;
import com.baijia.tianxiao.sal.wechat.dto.wechatmsg.ordinary.WechatTextMessage;
import com.baijia.tianxiao.sal.wechat.dto.wechatmsg.ordinary.WechatVoiceMessage;


/**   
 * @title       : MessageBuilder 
 * @description : 
 * @author      : zhenyujian
 * @date        : 2016年1月13日 上午10:26:30 
 */
@Slf4j
public class MessageBuilder {

    public static WechatAbstractMessage build(String xml) {
        try {
            Document document = DocumentHelper.parseText(xml);
        	Element root = document.getRootElement();

        	String msgType = root.elementTextTrim("MsgType");
            
            if (msgType.equals("event")) {
                return buildEventMessage(root);
            } else {
                return buildOrdinaryMessage(root);
            }
        } catch (Exception e) {
        	log.error("",e);
            throw new WebServiceException("build message exception", e);
        }
    }

    private static WechatOrdinaryMessage buildOrdinaryMessage(Element root) {
        String msgType = root.elementTextTrim("MsgType");
        if (msgType.equals("text")) {
            WechatTextMessage message = new WechatTextMessage();
            setTextMessage(root, message);
            return message;
        }else if(msgType.equals("image")){
        	WechatImageMessage message = new WechatImageMessage();
        	setImageMessage(root, message);
        	return message;
        }else if(msgType.equals("voice")){
        	WechatVoiceMessage message = new WechatVoiceMessage();
        	setVoiceMessage(root, message);
        	return message;
        }

        WechatOrdinaryMessage defaultMessage = new WechatOrdinaryMessage();
        setOrdinaryMessage(root, defaultMessage);
        return defaultMessage;
    }

    private static void setOrdinaryMessage(Element root, WechatOrdinaryMessage message) {
        setAbstractMessage(root, message);

        String msgId = root.elementTextTrim("MsgId");
        message.setMsgId(Long.parseLong(msgId));
    }

    private static void setTextMessage(Element root, WechatTextMessage message) {
        setOrdinaryMessage(root, message);

        String content = root.elementTextTrim("Content");
        message.setContent(content);
    }
    
    private static void setImageMessage(Element root, WechatImageMessage message) {
        setOrdinaryMessage(root, message);

        String mediaId = root.elementTextTrim("MediaId");
        String picUrl = root.elementTextTrim("PicUrl");
        message.setMediaId(mediaId);
        message.setPicUrl(picUrl);;
    }
    
    private static void setVoiceMessage(Element root, WechatVoiceMessage message) {
        setOrdinaryMessage(root, message);

        String mediaId = root.elementTextTrim("MediaId");
        String format = root.elementTextTrim("Format");
        String recognition = root.elementTextTrim("Recognition");
        message.setMediaId(mediaId);
        message.setFormat(format);
        message.setRecognition(recognition);
    }

    private static WechatEventMessage buildEventMessage(Element root) {
        String event = root.elementTextTrim("Event");
        if (event.equals("subscribe")) {
            WechatSubscribeEventMessage message = new WechatSubscribeEventMessage();
            setSubscribeEventMessage(root, message);
            return message;
        }

        if (event.equals("unsubscribe")) {
            WechatUnsubscribeEventMessage message = new WechatUnsubscribeEventMessage();
            setUnsubscribeEventMessage(root, message);
            return message;
        }

        if (event.equals("LOCATION")) {
            WechatLocationEventMessage message = new WechatLocationEventMessage();
            setLocationEventMessage(root, message);
            return message;
        }
        
        if (event.equals("SCAN")) {
        	WechatSCANEventMessage message = new WechatSCANEventMessage();
        	setSCANEventMessage(root, message);
            return message;
        }

        WechatEventMessage defaultMessage = new WechatEventMessage();
        setEventMessage(root, defaultMessage);
        return defaultMessage;
    }

    private static void setSubscribeEventMessage(Element root, WechatSubscribeEventMessage message) {
    	if(root.element("Ticket")!=null){
    		message.setTicket( root.elementTextTrim("Ticket") );
    	}
        setEventMessage(root, message);
    }
    
    private static void setSCANEventMessage(Element root, WechatSCANEventMessage message) {
    	if(root.element("Ticket")!=null){
    		message.setTicket( root.elementTextTrim("Ticket") );
    	}
        setEventMessage(root, message);
    }

    private static void setUnsubscribeEventMessage(Element root, WechatUnsubscribeEventMessage message) {
        setEventMessage(root, message);
    }

    private static void setLocationEventMessage(Element root, WechatLocationEventMessage message) {
        setEventMessage(root, message);

        String latitude = root.elementTextTrim("Latitude");
        String longitude = root.elementTextTrim("Longitude");
        String precision = root.elementTextTrim("Precision");

        message.setLatitude(Double.parseDouble(latitude));
        message.setLongitude(Double.parseDouble(longitude));
        message.setPrecision(Double.parseDouble(precision));
    }

    private static void setEventMessage(Element root, WechatEventMessage message) {
        setAbstractMessage(root, message);

        String event = root.elementTextTrim("Event");
        String eventKey = null;
        if(root.element("EventKey")!=null){
        	eventKey = root.elementTextTrim("EventKey");
        }
        
        message.setEvent( WechatMsgEvent.getByVaule(event) );
        message.setEventKey(eventKey);
    }

    private static void setAbstractMessage(Element root, WechatAbstractMessage message) {
    	String msgType = root.elementTextTrim("MsgType");
        String fromUserName = root.elementTextTrim("FromUserName");
        String toUserName = root.elementTextTrim("ToUserName");
        String createTime = root.elementTextTrim("CreateTime");
        
        message.setMsgType( WechatMsgType.getByVaule(msgType) );
        message.setFromUserName(fromUserName);
        message.setToUserName(toUserName);
        message.setCreateTime(new Date(Long.parseLong(createTime) * 1000));
    }
}
