/**
 * Baijiahulian.com Inc. Copyright (c) 2015-2015 All Rights Reserved.
 */
package com.baijia.tianxiao.sal.wechat.impl;

import com.baijia.tianxiao.common.service.WechatMsgRenderService;
import com.baijia.tianxiao.dal.wechat.constant.WechateTemplateMsgType;
import com.baijia.tianxiao.dal.wechat.dao.AuthorizerInfoDao;
import com.baijia.tianxiao.dal.wechat.dao.FansDao;
import com.baijia.tianxiao.dal.wechat.dao.OrgWechatTemplateDao;
import com.baijia.tianxiao.dal.wechat.po.AuthorizationInfo;
import com.baijia.tianxiao.dal.wechat.po.AuthorizerInfo;
import com.baijia.tianxiao.dal.wechat.po.Fans;
import com.baijia.tianxiao.dal.wechat.po.OrgWechatTemplate;
import com.baijia.tianxiao.dal.wechat.po.UnifiedWechatAccount;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.exception.CustomException;
import com.baijia.tianxiao.exception.WechatException;
import com.baijia.tianxiao.sal.wechat.api.AuthorizationInfoService;
import com.baijia.tianxiao.sal.wechat.api.UnifiedWechatAccountService;
import com.baijia.tianxiao.sal.wechat.constant.SalWechatErrorCode;
import com.baijia.tianxiao.sal.wechat.constant.WechatApi;
import com.baijia.tianxiao.sal.wechat.dto.templatemsg.Color;
import com.baijia.tianxiao.sal.wechat.helper.template.TemplateMsgHelper;
import com.baijia.tianxiao.sal.wechat.validator.WechatApiValidator;
import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.Map;

/**
 * @title : WechatMsgRenderServiceImpl
 * @description : 微信模板消息服务 (TemplateMsgService也支持模板消息发送) WechatMsgRenderServiceImpl的产生是因为TemplateMsgService开发完成后
 *              应要求重新设计了接口 所以重新开发了模板消息服务类 原TemplateMsgService依旧保留
 * @author : zhenyujian
 * @date : 2016年3月23日 下午5:02:50
 */
@Slf4j
@Service
public class WechatMsgRenderServiceImpl implements WechatMsgRenderService {

    public static String KEY_TOUSER = "touser";
    public static String KEY_TEMPLATE_ID = "template_id";
    public static String KEY_URL = "url";
    public static String KEY_FIRST = "first";
    public static String KEY_REMARK = "remark";
    public static String KEY_KEYWORD_PREFIX = "keyword";

    @Autowired
    private AuthorizationInfoService authorizationInfoService;
    @Autowired
    private AuthorizerInfoDao authorizerInfoDao;
    @Autowired
    private OrgWechatTemplateDao orgWechatTemplateDao;
    @Autowired
    private FansDao fansDao;
    @Autowired
    private UnifiedWechatAccountService unifiedWechatAccountService;

    @Override
    public Object render(Integer msgType, Map<String, Object> params) {
        WechateTemplateMsgType type = WechateTemplateMsgType.getByVaule(msgType);

        JSONObject data = new JSONObject();
        int i = 1;
        String keyword = WechatMsgRenderServiceImpl.KEY_KEYWORD_PREFIX + i++;
        String value = (String) params.get(keyword);
        while (value != null) {
            data.put(keyword, getTemplateMsgNode(value));
            keyword = WechatMsgRenderServiceImpl.KEY_KEYWORD_PREFIX + i++;
            value = (String) params.get(keyword);
        }

        if (params.get(WechatMsgRenderServiceImpl.KEY_FIRST) != null) {
            data.put("first", getTemplateMsgNode(params.get(WechatMsgRenderServiceImpl.KEY_FIRST) + ""));
        } else {
            data.put("first", "");
        }

        if (params.get(WechatMsgRenderServiceImpl.KEY_REMARK) != null) {
            data.put("remark", getTemplateMsgNode(params.get(WechatMsgRenderServiceImpl.KEY_REMARK) + ""));
        } else {
            data.put("remark", "");
        }

        JSONObject json = new JSONObject();
        json.put("touser", params.get(WechatMsgRenderServiceImpl.KEY_TOUSER));
        json.put("template_id", type.getTemplateId());
        json.put("url", params.get(WechatMsgRenderServiceImpl.KEY_URL));
        json.put("topcolor", Color.BLACK_LIGHT);
        json.put("data", data);

        return json;
    }

    protected static JSONObject getTemplateMsgNode(String value, String color) {
        JSONObject obj = new JSONObject();
        obj.put("value", value);
        obj.put("color", color);
        return obj;
    }

    protected static JSONObject getTemplateMsgNode(String value) {
        return getTemplateMsgNode(value, Color.BLACK_LIGHT);
    }

    @Override
    public boolean sendMsg(Long orgId, Object wechatMsg) {
        try {
            JSONObject json = (JSONObject) wechatMsg;
            String openId = json.getString(WechatMsgRenderServiceImpl.KEY_TOUSER);
            String originalTemplateId = json.getString(WechatMsgRenderServiceImpl.KEY_TEMPLATE_ID);
            log.debug("[send message to]={}",openId);
            Fans fans = fansDao.getByOpenId(openId);
            if (fans!=null && !fans.isSubscribed()) {
                return false;
            }

            AuthorizerInfo authorizerInfo = authorizerInfoDao.getByAuthorizerAppId(fans.getAuthorizerAppId());
            if (authorizerInfo == null) {
                log.info("authorizerInfo is null - orgId:{}, wechatMsg:{}", orgId, wechatMsg);
                return false;
            } else if (orgId != null && authorizerInfo.getOrgId().intValue() != orgId.intValue()) {
                log.info("wrong orgId - orgId:{}, wechatMsg:{}", orgId, wechatMsg);
                UnifiedWechatAccount findMasterAccountWithAnyCampusOrgId =
                    this.unifiedWechatAccountService.findMasterAccountWithAnyCampusOrgId(orgId.intValue(), true);
                if (findMasterAccountWithAnyCampusOrgId == null) {
                    return false;
                }
                log.info("can send with master account :{} ", findMasterAccountWithAnyCampusOrgId.getOrgId());
            }

            AuthorizationInfo authorizationInfo =
                authorizationInfoService.refreshAccessToken(authorizerInfo.getOrgId());

            // 权限判断
            WechatApiValidator._4CallApi(authorizationInfo, authorizerInfo, WechatApi.MSG_TEMPLATE_SEND);

            // 查询|绑定 消息模板
            OrgWechatTemplate template = bindOrgNoticeTemplate(authorizationInfo.getAuthorizerAppId(),
                authorizationInfo.getAuthorizerAccessToken(), originalTemplateId);

            if (template == null) {
                throw new BussinessException(SalWechatErrorCode.WECHAT_UNBIND_TEMPLATE);
            }

            json.put(WechatMsgRenderServiceImpl.KEY_TEMPLATE_ID, template.getAuthorizerTemplateId());

            try {
                // 发送模板消息
                TemplateMsgHelper.sendTemplateMsg(authorizationInfo.getAuthorizerAccessToken(), json.toString());
            } catch (WechatException e) {
                // 数据库模板id与微信端不一致时 删除数据库中的错误模板数据
                if (e.getErrorCode().equals(SalWechatErrorCode.WECHAT_WRONG_TEMPLATE_ID)) {
                    orgWechatTemplateDao.delByAppIdAndOriginalTemplateId(authorizationInfo.getAuthorizerAppId(),
                        originalTemplateId);
                    // 尝试重发
                    template = bindOrgNoticeTemplate(authorizationInfo.getAuthorizerAppId(),
                        authorizationInfo.getAuthorizerAccessToken(), originalTemplateId);
                    json.put(WechatMsgRenderServiceImpl.KEY_TEMPLATE_ID, template.getAuthorizerTemplateId());
                    TemplateMsgHelper.sendTemplateMsg(authorizationInfo.getAuthorizerAccessToken(), json.toString());
                } else {
                    throw e;
                }
            }
        } catch (Exception e) {
            log.warn("wechat - sendMsg - exception ", e);
            return false;
        }
        return true;
    }

    // 绑定模板(每次发送消息前判断公众号是否已绑定了指定消息模板)
    private OrgWechatTemplate bindOrgNoticeTemplate(String authorizerAppId, String accessToken,
        String originalTemplateId) {
        OrgWechatTemplate template = null;
        try {
            Date now = new Date();
            template = orgWechatTemplateDao.getByAppIdAndOriginalTemplateId(authorizerAppId, originalTemplateId);

            if (template == null) {
                String templateId = TemplateMsgHelper.bindTemplate(accessToken, originalTemplateId);

                template = new OrgWechatTemplate();
                template.setAuthorizerAppId(authorizerAppId);
                template.setAuthorizerTemplateId(templateId);
                template.setOriginalTemplateId(originalTemplateId);
                template.setCreateTime(now);
                orgWechatTemplateDao.save(template, true);
            } else {
                // FIXME 更新 待定
            }

        } catch (CustomException e) {
            throw e;
        } catch (Exception e) {
            log.warn("wechat - bindOrgNoticeTemplate fail - e:{}", e);
        }

        return template;
    }

}
