/**
 * Baijiahulian.com Inc. Copyright (c) 2015-2015 All Rights Reserved.
 */
package com.baijia.tianxiao.sal.wechat.impl;


import java.util.Date;

import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONObject;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.baijia.tianxiao.dal.wechat.constant.WechateTemplateMsgType;
import com.baijia.tianxiao.dal.wechat.dao.AuthorizerInfoDao;
import com.baijia.tianxiao.dal.wechat.dao.OrgWechatTemplateDao;
import com.baijia.tianxiao.dal.wechat.po.AuthorizationInfo;
import com.baijia.tianxiao.dal.wechat.po.AuthorizerInfo;
import com.baijia.tianxiao.dal.wechat.po.OrgWechatTemplate;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.exception.CustomException;
import com.baijia.tianxiao.exception.WechatException;
import com.baijia.tianxiao.sal.wechat.api.AuthorizationInfoService;
import com.baijia.tianxiao.sal.wechat.api.TemplateMsgService;
import com.baijia.tianxiao.sal.wechat.constant.SalWechatErrorCode;
import com.baijia.tianxiao.sal.wechat.constant.WechatApi;
import com.baijia.tianxiao.sal.wechat.dto.templatemsg.WechatTemplateMsg;
import com.baijia.tianxiao.sal.wechat.dto.templatemsg.batch.BatchMsg;
import com.baijia.tianxiao.sal.wechat.dto.templatemsg.courseplan.CoursePlanMsgToStu;
import com.baijia.tianxiao.sal.wechat.dto.templatemsg.coursesignin.CourseSigninSuccMsgToStu;
import com.baijia.tianxiao.sal.wechat.dto.templatemsg.coursestart.CourseStartMsgToStu;
import com.baijia.tianxiao.sal.wechat.dto.templatemsg.courseupdate.CourseUpdateMsgToStu;
import com.baijia.tianxiao.sal.wechat.dto.templatemsg.evaluation.CourseEvaluationMsgToStu;
import com.baijia.tianxiao.sal.wechat.dto.templatemsg.evaluation.TeacherEvaluationMsgToStu;
import com.baijia.tianxiao.sal.wechat.helper.template.TemplateMsgHelper;
import com.baijia.tianxiao.sal.wechat.validator.WechatApiValidator;



/**   
 * @title       : TemplateMsgServiceImpl 
 * @description : 模板消息服务类  
 * @author      : zhenyujian
 * @date        : 2016年3月23日 下午5:05:19 
 */
@Slf4j
@Service
public class TemplateMsgServiceImpl implements TemplateMsgService {

    @Autowired
    private AuthorizationInfoService authorizationInfoService;
    @Autowired
    private AuthorizerInfoDao authorizerInfoDao;
    @Autowired
    private OrgWechatTemplateDao orgWechatTemplateDao;

    
    
    
    private OrgWechatTemplate bindOrgNoticeTemplate(String authorizerAppId, String accessToken, String originalTemplateId) {
        OrgWechatTemplate template = null;
        try {
            Date now = new Date();
            template = orgWechatTemplateDao.getByAppIdAndOriginalTemplateId(authorizerAppId, originalTemplateId);

            if (template == null) {
                String templateId = TemplateMsgHelper.bindTemplate(accessToken, originalTemplateId);

                template = new OrgWechatTemplate();
                template.setAuthorizerAppId(authorizerAppId);
                template.setAuthorizerTemplateId(templateId);
                template.setOriginalTemplateId(originalTemplateId);
                template.setCreateTime(now);
                orgWechatTemplateDao.save(template, true);
            } else {
                // FIXME 更新 待定
            }

        } catch (CustomException e) {
            throw e;
        } catch (Exception e) {
            log.error("wechat - bindOrgNoticeTemplate fail - e:{}", e);
        }

        return template;
    }

    

    OrgWechatTemplate templateCheck(Integer orgId, String originalTemplateId, AuthorizationInfo authorizationInfo) {

        AuthorizerInfo authorizerInfo = authorizerInfoDao.getByOrgId(orgId);

        // 权限判断
        WechatApiValidator._4CallApi(authorizationInfo, authorizerInfo, WechatApi.MSG_TEMPLATE_SEND);

        // 查询消息模板
        OrgWechatTemplate template = bindOrgNoticeTemplate(authorizationInfo.getAuthorizerAppId(), authorizationInfo.getAuthorizerAccessToken(), originalTemplateId);
        if (template == null) {
            throw new BussinessException(SalWechatErrorCode.WECHAT_UNBIND_TEMPLATE);
        }

        return template;
    }

    
    @Override
    public void sendTemplateMsg(String msgJson) throws CustomException {
        JSONObject json = JSONObject.fromObject(msgJson);
        int msgTypeValue = json.getInt(WechatTemplateMsg.JSON_KEY_MSG_TYPE);
        WechateTemplateMsgType msgType = WechateTemplateMsgType.getByVaule(msgTypeValue);

        OrgWechatTemplate template = null;
        AuthorizationInfo authorizationInfo = null;
        String originalTemplateId = null;
        try{
	        switch (msgType) {
	            case COURSE_PLAN_TO_STU:
	            	originalTemplateId = WechateTemplateMsgType.COURSE_PLAN_TO_STU.getTemplateId();
	                CoursePlanMsgToStu cpMsg2stu = CoursePlanMsgToStu.fromJsonStr(msgJson);
	                authorizationInfo = authorizationInfoService.refreshAccessToken(cpMsg2stu.getOrgId());
	                template = templateCheck(cpMsg2stu.getOrgId(), originalTemplateId, authorizationInfo);
	                TemplateMsgHelper.sendTemplateMsg(authorizationInfo.getAuthorizerAccessToken(),
	                    cpMsg2stu.buildTemplateMsg(template.getAuthorizerTemplateId()));
	                break;
	
	            case COURSE_START_TO_STU:
	            	originalTemplateId = WechateTemplateMsgType.COURSE_START_TO_STU.getTemplateId();
	                CourseStartMsgToStu snMsg2stu = CourseStartMsgToStu.fromJsonStr(msgJson);
	                authorizationInfo = authorizationInfoService.refreshAccessToken(snMsg2stu.getOrgId());
	                template = templateCheck(snMsg2stu.getOrgId(), originalTemplateId, authorizationInfo);
	                TemplateMsgHelper.sendTemplateMsg(authorizationInfo.getAuthorizerAccessToken(),
	                    snMsg2stu.buildTemplateMsg(template.getAuthorizerTemplateId()));
	                break;
	
	            case COURSE_SIGNIN_TO_STU:
	            	originalTemplateId = WechateTemplateMsgType.COURSE_SIGNIN_TO_STU.getTemplateId();
	                CourseSigninSuccMsgToStu csMsg2stu = CourseSigninSuccMsgToStu.fromJsonStr(msgJson);
	                authorizationInfo = authorizationInfoService.refreshAccessToken(csMsg2stu.getOrgId());
	                template = templateCheck(csMsg2stu.getOrgId(), originalTemplateId, authorizationInfo);
	                TemplateMsgHelper.sendTemplateMsg(authorizationInfo.getAuthorizerAccessToken(),
	                    csMsg2stu.buildTemplateMsg(template.getAuthorizerTemplateId()));
	                break;
	
	            case COURSE_EVALUATION_TO_STU:
	            	originalTemplateId = WechateTemplateMsgType.COURSE_EVALUATION_TO_STU.getTemplateId();
	                CourseEvaluationMsgToStu ceMsg2stu = CourseEvaluationMsgToStu.fromJsonStr(msgJson);
	                authorizationInfo = authorizationInfoService.refreshAccessToken(ceMsg2stu.getOrgId());
	                template = templateCheck(ceMsg2stu.getOrgId(), originalTemplateId, authorizationInfo);
	                TemplateMsgHelper.sendTemplateMsg(authorizationInfo.getAuthorizerAccessToken(),
	                    ceMsg2stu.buildTemplateMsg(template.getAuthorizerTemplateId()));
	                break;
	
	            case TEACHER_EVALUATION_TO_STU:
	            	originalTemplateId = WechateTemplateMsgType.TEACHER_EVALUATION_TO_STU.getTemplateId();
	                TeacherEvaluationMsgToStu jsMsg2stu = TeacherEvaluationMsgToStu.fromJsonStr(msgJson);
	                authorizationInfo = authorizationInfoService.refreshAccessToken(jsMsg2stu.getOrgId());
	                template = templateCheck(jsMsg2stu.getOrgId(), originalTemplateId, authorizationInfo);
	                TemplateMsgHelper.sendTemplateMsg(authorizationInfo.getAuthorizerAccessToken(),
	                    jsMsg2stu.buildTemplateMsg(template.getAuthorizerTemplateId()));
	                break;
	
	            case COURSE_UPDATE_TO_STU:
	            	originalTemplateId = WechateTemplateMsgType.COURSE_UPDATE_TO_STU.getTemplateId();
	                CourseUpdateMsgToStu cuMsg2stu = CourseUpdateMsgToStu.fromJsonStr(msgJson);
	                authorizationInfo = authorizationInfoService.refreshAccessToken(cuMsg2stu.getOrgId());
	                template = templateCheck(cuMsg2stu.getOrgId(), originalTemplateId, authorizationInfo);
	                TemplateMsgHelper.sendTemplateMsg(authorizationInfo.getAuthorizerAccessToken(),
	                    cuMsg2stu.buildTemplateMsg(template.getAuthorizerTemplateId()));
	                break;
	
	            case BATCH:
	            	originalTemplateId = WechateTemplateMsgType.BATCH.getTemplateId();
	                BatchMsg batchMsg = BatchMsg.fromJsonStr(msgJson);
	                log.debug("-----batchMsg={}", batchMsg);
	                authorizationInfo = authorizationInfoService.refreshAccessToken(batchMsg.getOrgId());
	                template =
	                    templateCheck(batchMsg.getOrgId(), originalTemplateId, authorizationInfo);
	                TemplateMsgHelper.sendTemplateMsg(authorizationInfo.getAuthorizerAccessToken(),
	                    batchMsg.buildTemplateMsg(template.getAuthorizerTemplateId()));
	                break;
	
	            default:
	                return;
	        }
        }catch(WechatException e){
        	//数据库模板id与微信端不一致时 删除数据库中的错误模板数据
        	if(e.getErrorCode().equals(SalWechatErrorCode.WECHAT_WRONG_TEMPLATE_ID)){
        		orgWechatTemplateDao.delByAppIdAndOriginalTemplateId(authorizationInfo.getAuthorizerAppId(), originalTemplateId);
        	}else{
        		throw e;
        	}
        }
    }



    
}
