/**
 * Baijiahulian.com Inc. Copyright (c) 2015-2015 All Rights Reserved.
 */
package com.baijia.tianxiao.sal.wechat.impl;

import java.io.File;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import net.sf.json.JSONArray;
import net.sf.json.JSONObject;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.core.RedisConnectionUtils;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import lombok.extern.slf4j.Slf4j;

import com.baijia.tianxiao.constants.MIMEType;
import com.baijia.tianxiao.dal.org.dao.OrgInfoDao;
import com.baijia.tianxiao.dal.org.po.OrgInfo;
import com.baijia.tianxiao.dal.wechat.constant.WechatDataSyncStatus;
import com.baijia.tianxiao.dal.wechat.constant.WechatDataSyncType;
import com.baijia.tianxiao.dal.wechat.constant.WechatMediaNewsType;
import com.baijia.tianxiao.dal.wechat.dao.AuthorizationInfoDao;
import com.baijia.tianxiao.dal.wechat.dao.AuthorizerInfoDao;
import com.baijia.tianxiao.dal.wechat.dao.OrgWechatMediaNewsDao;
import com.baijia.tianxiao.dal.wechat.po.AuthorizationInfo;
import com.baijia.tianxiao.dal.wechat.po.AuthorizerInfo;
import com.baijia.tianxiao.dal.wechat.po.OrgWechatDataSyncLog;
import com.baijia.tianxiao.dal.wechat.po.OrgWechatMediaNews;
import com.baijia.tianxiao.dto.upload.UploadResult;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.exception.WebServiceException;
import com.baijia.tianxiao.exception.WechatException;
import com.baijia.tianxiao.sal.wechat.api.AuthorizationInfoService;
import com.baijia.tianxiao.sal.wechat.api.AuthorizerInfoService;
import com.baijia.tianxiao.sal.wechat.api.MediaService;
import com.baijia.tianxiao.sal.wechat.api.WechatDataSyncLogService;
import com.baijia.tianxiao.sal.wechat.constant.MediaType;
import com.baijia.tianxiao.sal.wechat.constant.SalWechatErrorCode;
import com.baijia.tianxiao.sal.wechat.constant.WechatApi;
import com.baijia.tianxiao.sal.wechat.dto.media.MediaNewsDto;
import com.baijia.tianxiao.sal.wechat.dto.media.MediaUploadResultDto;
import com.baijia.tianxiao.sal.wechat.dto.wechatapi.PermanentMediaApiDto;
import com.baijia.tianxiao.sal.wechat.dto.wechatapi.WechatApiResponse;
import com.baijia.tianxiao.sal.wechat.helper.WechatProperties;
import com.baijia.tianxiao.sal.wechat.helper.media.WechatMediaApiHelper;
import com.baijia.tianxiao.sal.wechat.task.MediaNewsSyncJob;
import com.baijia.tianxiao.sal.wechat.task.TaskHelper;
import com.baijia.tianxiao.sal.wechat.util.FileUploadUtils;
import com.baijia.tianxiao.sal.wechat.util.LocalFileHelper;
import com.baijia.tianxiao.sal.wechat.validator.WechatApiValidator;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.util.httpclient.HttpClientUtils;



@Slf4j
@Service
public class MediaServiceImpl implements MediaService{
	
	@Autowired
	private AuthorizationInfoService authorizationInfoService;
	@Autowired
	private AuthorizerInfoService authorizerInfoService;
	
    @Autowired
    private AuthorizationInfoDao authorizationInfoDao;
    @Autowired
    private AuthorizerInfoDao authorizerInfoDao;
    @Autowired
    private OrgWechatMediaNewsDao orgWechatMediaNewsDao;
    @Autowired
    private OrgInfoDao orgInfoDao;
    
    
    @Autowired(required = false)
	private RedisTemplate<String, Object> redisTemplate;
    @Autowired
	private WechatDataSyncLogService wechatDataSyncLogService;
    
    
    
    public boolean redisMediaSyncInterval(String appId) {
    	//(5分钟内只允许触发一次素材更新)
    	String key = String.format("tx_wechat_media_interval_%s", appId);
    	Long intervalSeconds = 5L;
    	
		RedisConnection connection = redisTemplate.getConnectionFactory().getConnection();
		try {
			connection.select(WechatProperties.getRedisDB());

			if (connection.exists(key.getBytes())) {
				return false;
			} else {
				connection.setEx(key.getBytes(), intervalSeconds, "".getBytes());
				return true;
			}
		} catch (Exception e) {
			log.error("Wechat - redisFansSyncInterval - exception - e:{}", e);
			return true;
		} finally {
			RedisConnectionUtils.releaseConnection(connection, redisTemplate.getConnectionFactory());
		}
	}
    
    
    /**
     * @Description  : 判断是否需要同步素材列表
     * 				   规则: 本地保存的素材数和微信保存的素材数是否相等
     * @Author       : zhenyujian
     * @Date         : 2016年1月7日 下午4:10:29 
     *
     * @Return       : boolean
     */
    @Transactional(readOnly = true)
    @Override
    public boolean isNeedToSync(int orgId){
    	try{
    		AuthorizerInfo authorizerInfo = authorizerInfoDao.getByOrgId(orgId);
    		
    		//判断是否已有同步任务在进行中
        	if( wechatDataSyncLogService.hasTaskOngoing(authorizerInfo.getAuthorizerAppId(), WechatDataSyncType.MEDIA_NEWS) ) {
    			log.info("wechat media - isNeedToSync - RETURN FALSE - orgId:{}",orgId);
    			return false;
    		}
        	
    		//判断两次同步间隔时间是否合法
        	if( !redisMediaSyncInterval(authorizerInfo.getAuthorizerAppId()) ){
        		log.info("wechat media - isNeedToSync - RETURN FALSE - orgId:{}",orgId);
        		return false;
        	}
        	
    		AuthorizationInfo authorizationInfo = authorizationInfoService.refreshAccessToken(orgId);
		
			WechatApiValidator._4CallApi(authorizationInfo, authorizerInfo, WechatApi.MEDIA, true);
		
	    	int mediaCount = orgWechatMediaNewsDao.count(authorizationInfo.getAuthorizerAppId());
	    	
	    	WechatApiResponse response = WechatMediaApiHelper.batchGetPermanentMediaNews(authorizationInfo.getAuthorizerAccessToken(), 0, 20);
	    	JSONObject rootNode = response.getRootJSONObj();
			
			if(mediaCount != rootNode.optInt("total_count",0)){
				log.info("wechat media - isNeedToSync - RETURN TRUE - orgId:{}",orgId);
				return true;
			}
		}catch(BussinessException | WechatException e){
			log.warn("wechat media - isNeedToSync - CustomException",e);
		}catch(Exception e){
			log.error("wechat media - isNeedToSync - Exception",e);
		}
    	
    	log.info("wechat media - isNeedToSync - RETURN FALSE - orgId:{}",orgId);
    	return false;
    }
    
    
    @Transactional(rollbackFor = Exception.class)
	@Override
	public void syncMediaNews(String authorizerAppId) {
    	
    	//启动素材同步
		log.info("wechat - MediaServiceImpl - mediaNewsSync - start - authorizerAppId:{}",authorizerAppId);
		Date now = new Date();
    	Date startTime = now;
    	Integer amountExpected = 0;
    	Integer amountCompleted = 0;
		
		AuthorizationInfo authorizationInfo = authorizationInfoService.refreshAccessToken(authorizerAppId);
		String accessToken = authorizationInfo.getAuthorizerAccessToken();
		
		//查询微信端图文素材数
		int count = WechatMediaApiHelper.countPermanentMedia(accessToken).getNewsCount();
		
		//检查任务log
		if(wechatDataSyncLogService.hasTaskOngoing(authorizerAppId, WechatDataSyncType.MEDIA_NEWS)){
    		log.info("wechat - MediaService - syncMediaNews - return - authorizerAppId:{}", authorizerAppId);
    		return;
    	}
    	
    	//创建任务Log
    	now = new Date();
    	amountExpected = count;
    	Date syncLogExpireTime = new Date( now.getTime()+amountExpected*500 ); //按每条记录执行时间0.5秒 计算任务超时时间
    	WechatDataSyncStatus syncLogEndStatus = WechatDataSyncStatus.END_SUCCESS;
    	OrgWechatDataSyncLog syncLog = wechatDataSyncLogService.buildDefaultLog(authorizerAppId, WechatDataSyncType.MEDIA_NEWS, startTime, amountExpected, syncLogExpireTime);
		    	
		try{
			Map<String,Date> oldMediaNewsMap = orgWechatMediaNewsDao.mapKeyMediaIdValueWechatUpdateTime( authorizerAppId ); //FIXME
			Map<String,OrgWechatMediaNews> newMediaNewsMap = new HashMap<String,OrgWechatMediaNews>();
			
			int num = 0;
			int pageSize = 20;
			int start = 0;
			List<OrgWechatMediaNews> mediaNewsToAdd = new ArrayList<OrgWechatMediaNews>();
			
			while(num<count){
				WechatApiResponse response = null;
				try{
					response = WechatMediaApiHelper.batchGetPermanentMediaNews(accessToken, start, start+pageSize);
				}catch(WechatException e){
					if(e.getErrorCode() == SalWechatErrorCode.WECHAT_ACCESS_TOKEN_INVALID){
						//数据同步过程中accesstoken失效 重新获取accesstoken
						accessToken = authorizationInfoService.forceRefreshAccessToken(authorizationInfo.getOrgId()).getAuthorizerAccessToken();
						response = WechatMediaApiHelper.batchGetPermanentMediaNews(accessToken, start, start+pageSize);
					}else{
						log.error("wechat - MediaServiceImpl - mediaNewsSync - batchGetPermanentMediaNews WechatException(all fail) - authorizerAppId:{},e:{}",authorizerAppId,e);
						throw e;
					}
				}catch(WebServiceException e){
					log.warn("wechat - MediaServiceImpl - mediaNewsSync - batchGetPermanentMediaNews Exception - authorizerAppId:{},e:{}",authorizerAppId,e);
					continue;
				}catch(Exception e){
					log.error("wechat - MediaServiceImpl - mediaNewsSync - batchGetPermanentMediaNews Exception - authorizerAppId:{},e:{}",authorizerAppId,e);
					continue;
				}
				
				JSONObject rootNode = response.getRootJSONObj();
			
				count = rootNode.optInt("total_count",0);
				num += rootNode.optInt("item_count",0);
				
				JSONArray mediaNewsArray = rootNode.getJSONArray("item");
				if(mediaNewsArray.size()>0){
					for(int i=0; i<mediaNewsArray.size(); i++){
						OrgWechatMediaNews mediaNews = new OrgWechatMediaNews();
						
						JSONObject mediaNewsJson = mediaNewsArray.getJSONObject(i);
						JSONObject contentJson = mediaNewsJson.getJSONObject("content");
						String mediaId = mediaNewsJson.getString("media_id");
						Date wechatUpdateTime = new Date(mediaNewsJson.getLong("update_time"));
						
						//判读素材需要插入或删除
						if(oldMediaNewsMap.get(mediaId)==null || !(oldMediaNewsMap.get(mediaId).getTime()/1000 == wechatUpdateTime.getTime()/1000)){
							//素材不存在 或素材更新时间发生变化
							mediaNews.setAuthorizerAppId( authorizerAppId );
							mediaNews.setMediaId( mediaId );
							mediaNews.setWechatUpdateTime( wechatUpdateTime );//FIXME
							
							JSONArray newsItemArray = contentJson.getJSONArray("news_item");
							
							if(newsItemArray.size()>0){
								for(int j=0; j<newsItemArray.size(); j++){
									JSONObject mediaItem = newsItemArray.getJSONObject(j);
									mediaItem.put("content", null);//清空content 太大
									String thumbMediaId = mediaItem.getString("thumb_media_id");
									
									//自行保存封面图片 获得封面图片url
									if(StringUtils.isNotBlank(thumbMediaId)){
										File localFile = null;
										try{
											byte[] data = WechatMediaApiHelper.downloadPermanentMedia(authorizationInfo.getAuthorizerAccessToken(), thumbMediaId);
											localFile = LocalFileHelper.saveToLocal(data, MIMEType.JPG);
											UploadResult result = FileUploadUtils.uploadToRemote(Long.parseLong(authorizationInfo.getOrgId()+""), localFile, false);
											mediaItem.put("thumb_media_url", result.getFiles().get(0).getUrl());
										}catch(Exception e){
											mediaItem.put("thumb_media_url", "");
										}finally{
											if (localFile != null) {
								            	LocalFileHelper.deleteFile(localFile);
								            }
										}
									}else{
										mediaItem.put("thumb_media_url", "");
									}
								}
								mediaNews.setJson(newsItemArray.toString());
							}else{
								mediaNews.setJson("[]");
							}
							
							if(oldMediaNewsMap.get(mediaId)==null){
								//加入插入列表
								mediaNews.setCreateTime(now);
								mediaNews.setUpdateTime(now);
								mediaNews.setType(WechatMediaNewsType.NORMAL.getValue());
								mediaNewsToAdd.add(mediaNews);
							}else{
								//更新
								OrgWechatMediaNews news = orgWechatMediaNewsDao.getByMediaId(authorizerAppId, mediaId);
								if(!news.getWechatUpdateTime().equals(wechatUpdateTime)){
									news.setJson(mediaNews.getJson());
									news.setWechatUpdateTime(wechatUpdateTime);
									news.setUpdateTime(now);
									orgWechatMediaNewsDao.update(news, true);
								}
							}
						}
						
						//记录所有素材id
						newMediaNewsMap.put(mediaId, mediaNews);
					}
				}
				start += pageSize;
			}
			
			
			//删除
			List<String> mediaIdsToDel = new ArrayList<String>();
			for(String mediaId:oldMediaNewsMap.keySet()){
				if(newMediaNewsMap.get(mediaId)==null){
					mediaIdsToDel.add(mediaId);
				}
			}
			if(mediaIdsToDel.size()>0){
				orgWechatMediaNewsDao.delByMediaIds(authorizerAppId, mediaIdsToDel);
			}
			
			//批量插入
			int batchNum = 20; //每次批量出入条数
			if(mediaNewsToAdd.size()>0){
				List<OrgWechatMediaNews> list = new ArrayList<OrgWechatMediaNews>();
				for(int i=0; i<mediaNewsToAdd.size(); i++){
					list.add( mediaNewsToAdd.get(i) );
					if( (i!=0 && i%batchNum==0) || i==mediaNewsToAdd.size()-1){
						try{
							batchSaveMediaNews(list);
						}catch(Exception e){
							log.warn("wechat - MediaServiceImpl - media batch insert fail",e);
						}
						list = new ArrayList<OrgWechatMediaNews>();
					}
				}
			}
			
			log.info("wechat - MediaServiceImpl - mediaNewsSync - end - authorizerAppId:{}",authorizerAppId);
			
		}catch(Exception e){
    		syncLogEndStatus = WechatDataSyncStatus.END_FAIL;
    		throw e;
    	}finally{
    		//跟新日志进度
            now = new Date();
            syncLog.setStatus(syncLogEndStatus.getValue());
            syncLog.setAmountCompleted(amountExpected);
    		syncLog.setStartTime(startTime);				
    		syncLog.setEndTime(now);
    		syncLog.setUpdateTime(now);
    		wechatDataSyncLogService.saveOrUpdateInNewTransaction(syncLog);
    	}
	}

    

    @Transactional(rollbackFor = Exception.class)
	@Override
	public List<MediaNewsDto> listPermanentMediaNews(int orgId, PageDto pageDto) {
		AuthorizationInfo authorizationInfo = authorizationInfoDao.getByOrgId(orgId);
		AuthorizerInfo authorizerInfo = authorizerInfoDao.getByOrgId(orgId);
		
		if(authorizationInfo==null){
			throw new BussinessException(SalWechatErrorCode.ORG_NUBIND_WECHATAPP);
		}
		
		//权限判断
		WechatApiValidator._4CallApi(authorizationInfo, authorizerInfo, WechatApi.MEDIA, true);
		
		if(pageDto.getPageNum()==1){
			//内置图文检测
			defaultMediaNews(orgId, authorizerInfo.getAuthorizerAppId());
			
			//微信数据同步检测
			if(isNeedToSync(orgId)){
		        try {
		        	log.info("wechat - listPermanentMediaNews - handle - add mediaNewsSync task - appId:{}",authorizerInfo.getAuthorizerAppId());
					TaskHelper.getInstance().addTask( new MediaNewsSyncJob(this, authorizerInfo.getAuthorizerAppId(), authorizerInfoService) );
				} catch (Exception e) {
					e.printStackTrace();
				}
			}			
		}
		
		String authorizerAppId = authorizerInfo.getAuthorizerAppId();
		
		//分页查询
		Integer count = orgWechatMediaNewsDao.count(authorizerAppId);
		List<OrgWechatMediaNews> mediaNewsList = null;
		if(count>0){
			mediaNewsList = orgWechatMediaNewsDao.search(authorizerAppId, pageDto);
		}else{
			mediaNewsList = new ArrayList<OrgWechatMediaNews>();
		}
		pageDto.setCurPageCount(mediaNewsList.size());
		pageDto.setCount(count);
		
		//结果转换
		List<MediaNewsDto> dtoList = new ArrayList<MediaNewsDto>();
		if(mediaNewsList!=null && !mediaNewsList.isEmpty()){
			MediaNewsDto dto;
			for(OrgWechatMediaNews mediaNews:mediaNewsList){
				dto = new MediaNewsDto();
				dto.setId( mediaNews.getId() );
				dto.setMediaId( mediaNews.getMediaId() );
				dto.setNewsItem( JSONArray.fromObject(mediaNews.getJson()) );
				dto.setWechatUpdateTime( mediaNews.getWechatUpdateTime().getTime() );
				dto.setUpdateTime( mediaNews.getUpdateTime().getTime() );
				dto.setCreateTime( mediaNews.getCreateTime().getTime() );
				dtoList.add(dto);
			}
		}
		
		return dtoList;
	}



    @Transactional(readOnly = true)
	@Override
	public MediaNewsDto getPermanentMediaNews(int orgId, String mediaId) {
		AuthorizationInfo authorizationInfo = authorizationInfoDao.getByOrgId(orgId);
		if(authorizationInfo==null){
			throw new BussinessException(SalWechatErrorCode.ORG_NUBIND_WECHATAPP);
		}
		
		OrgWechatMediaNews mediaNews = orgWechatMediaNewsDao.getByMediaId(authorizationInfo.getAuthorizerAppId(), mediaId);
		
		MediaNewsDto dto = null;
		if(mediaNews!=null){
			dto = new MediaNewsDto();
			dto.setId( mediaNews.getId() );
			dto.setMediaId( mediaNews.getMediaId() );
			dto.setNewsItem( JSONArray.fromObject(mediaNews.getJson()) );
			dto.setWechatUpdateTime( mediaNews.getWechatUpdateTime().getTime() );
			dto.setUpdateTime( mediaNews.getUpdateTime().getTime() );
			dto.setCreateTime( mediaNews.getCreateTime().getTime() );
		}
		
		return dto;
	}



    @Transactional(rollbackFor = Exception.class)
	@Override
	public MediaUploadResultDto uploadPermanentMediaImage(int orgId, String imageUrl, File imageFile) {
		AuthorizationInfo authorizationInfo = authorizationInfoService.refreshAccessToken(orgId);
		AuthorizerInfo authorizerInfo = authorizerInfoDao.getByOrgId(orgId);
		
		if(authorizationInfo==null){
			throw new BussinessException(SalWechatErrorCode.ORG_NUBIND_WECHATAPP);
		}
		
		//权限判断
		WechatApiValidator._4CallApi(authorizationInfo, authorizerInfo, WechatApi.MEDIA);

		PermanentMediaApiDto dto = WechatMediaApiHelper.uploadPermanentMediaImage(authorizationInfo.getAuthorizerAccessToken(), imageFile);
		
		MediaUploadResultDto result = new MediaUploadResultDto();
		result.setMediaId(dto.getMediaId());
		result.setMediaType(MediaType.IMAGE.getValue());
		result.setUrl(imageUrl);
		
		return result;
	}



    @Transactional(rollbackFor = Exception.class)
	@Override
	public MediaUploadResultDto uploadPermanentMediaImage(int orgId, String imageUrl) throws Exception {
		File localFile = null;
        try {
        	String extension = "jpg";
        	if(imageUrl.contains("wx_fmt=")){
        		Matcher matcher = Pattern.compile("wx_fmt=([^&]*)").matcher(imageUrl);
            	if(matcher.find()){
            		extension = matcher.group(1);
            	}
        	}else if(imageUrl.contains("\\.")){
        		String[] temp = imageUrl.split("\\.");
            	extension = temp[temp.length-1];
        	}
        	byte[] data = HttpClientUtils.download(imageUrl);
            localFile = LocalFileHelper.saveToLocal(data, extension);
            return uploadPermanentMediaImage(orgId, imageUrl,localFile);
            
        }catch(Exception e){
			throw e;
		}finally {
            if (localFile != null) {
            	LocalFileHelper.deleteFile(localFile);
            }
        }
	}
    


	@Override
	public MediaUploadResultDto uploadPermanentMediaNews(int orgId, String json) throws Exception {
		return uploadPermanentMediaNews(orgId, json, WechatMediaNewsType.NORMAL);
	}
	
	@Override
	public MediaUploadResultDto uploadPermanentMediaNews(int orgId, String json, WechatMediaNewsType type) throws Exception{
		AuthorizationInfo authorizationInfo = authorizationInfoService.refreshAccessToken(orgId);
		AuthorizerInfo authorizerInfo = authorizerInfoDao.getByOrgId(orgId);
		
		if(authorizationInfo==null){
			throw new BussinessException(SalWechatErrorCode.ORG_NUBIND_WECHATAPP);
		}
		
		//权限判断
		WechatApiValidator._4CallApi(authorizationInfo, authorizerInfo, WechatApi.MEDIA);

		//上传图文
		PermanentMediaApiDto dto = WechatMediaApiHelper.uploadPermanentNews(authorizationInfo.getAuthorizerAccessToken(), json);
		
		MediaUploadResultDto result = new MediaUploadResultDto();
		result.setMediaId(dto.getMediaId());
		result.setMediaType(MediaType.NEWS.getValue());
		
		WechatApiResponse response = WechatMediaApiHelper.getPermanentMediaNews(authorizationInfo.getAuthorizerAccessToken(), dto.getMediaId());
		JSONArray newsItemArray = response.getRootJSONObj().getJSONArray("news_item");
		
		
		//上传成功后 保存图文信息到数据库
		OrgWechatMediaNews mediaNews = new OrgWechatMediaNews();
		Date now = new Date();
		mediaNews.setAuthorizerAppId( authorizerInfo.getAuthorizerAppId() );
		mediaNews.setMediaId( dto.getMediaId() );
		mediaNews.setWechatUpdateTime( now );
		mediaNews.setCreateTime( now );
		mediaNews.setUpdateTime( now );
			
		for(int j=0; j<newsItemArray.size(); j++){
			JSONObject mediaItem = newsItemArray.getJSONObject(j);
			mediaItem.put("content", null);//清空content 太大
			String thumbMediaId = mediaItem.getString("thumb_media_id");
			
			//自行保存封面图片 获得封面图片url
			if(StringUtils.isNotBlank(thumbMediaId)){
				File localFile = null;
				try{
					byte[] data = WechatMediaApiHelper.downloadPermanentMedia(authorizationInfo.getAuthorizerAccessToken(), thumbMediaId);
					localFile = LocalFileHelper.saveToLocal(data, MIMEType.JPG);
					UploadResult uploadResult = FileUploadUtils.uploadToRemote(Long.parseLong(authorizationInfo.getOrgId()+""), localFile, false);
					mediaItem.put("thumb_media_url", uploadResult.getFiles().get(0).getUrl());
				}catch(Exception e){
					mediaItem.put("thumb_media_url", "");
				}finally{
					if (localFile != null) {
		            	LocalFileHelper.deleteFile(localFile);
		            }
				}
			}else{
				mediaItem.put("thumb_media_url", "");
			}
		}
		mediaNews.setJson(newsItemArray.toString());
		mediaNews.setType(type.getValue());
		orgWechatMediaNewsDao.save(mediaNews);
		
		return result;
	}


	
	@Override
	public void delMedia(Integer orgId, String mediaId) {
		AuthorizationInfo authorizationInfo = authorizationInfoService.refreshAccessToken(orgId);
		WechatMediaApiHelper.delPermanentMediaNews(authorizationInfo.getAuthorizerAccessToken(), mediaId);
		orgWechatMediaNewsDao.delByMediaId(mediaId);
	}


	@Transactional(propagation=Propagation.REQUIRES_NEW, rollbackFor = Exception.class)
	@Override
	public void batchSaveMediaNews(List<OrgWechatMediaNews> list) {
		orgWechatMediaNewsDao.saveAll(list,false);
	}
	
	
	public void defaultMediaNews(int orgId, String authorizerAppId){
		List<OrgWechatMediaNews> mediaNews = orgWechatMediaNewsDao.getByAppIdAndType(authorizerAppId, WechatMediaNewsType.BUILTIN.getValue());
		if(CollectionUtils.isEmpty(mediaNews)){
			AuthorizationInfo authorizationInfo = authorizationInfoService.refreshAccessToken(orgId);
			OrgInfo orgInfo = orgInfoDao.getOrgInfo(orgId);
			try{
				MediaUploadResultDto dto = uploadPermanentMediaImage(orgId, WechatProperties.getBuilitnMediaNewsImage());
				String content = HttpClientUtils.doGet( WechatProperties.getWebCRMUrlPrefix()+"view/weixin/guide.html", null);
				//机构名称替换
				content = content.replace("#机构名称#", orgInfo.getShortName());
				
				//图文中图片替换 (图片需要上传到微信)
				Pattern pattern = Pattern.compile("<img src=\"([^\"]*)");
		     	Matcher matcher = pattern.matcher( content );
		     	String imgSrc = null;
		     	String wechatImgSrc = null;
		     	while(matcher.find()){
		     		imgSrc = matcher.group(1);
		     		wechatImgSrc = uploadPermanentMediaNewsImage(authorizationInfo.getAuthorizerAccessToken(), imgSrc);
		     		content = content.replace(imgSrc, wechatImgSrc);
		     	}
				
				JSONObject json = new JSONObject();
				JSONArray articles = new JSONArray();
				JSONObject news = new JSONObject();
				news.put("thumb_media_id", dto.getMediaId());
				news.put("title", "新手引导");
				news.put("author", "天校");
				news.put("digest", "");
				news.put("show_cover_pic", "0");
				news.put("content", content);
				news.put("content_source_url", "");
				articles.add(news);
				json.put("articles", articles);
				uploadPermanentMediaNews(orgId, json.toString(), WechatMediaNewsType.BUILTIN);
			}catch(Exception e){
				log.error("wechat - defalutMediaNews - exception",e);
			}
		}else{
			if(mediaNews.size()>0){
				for(int i=1;i<mediaNews.size();i++){
					orgWechatMediaNewsDao.delById(mediaNews.get(i).getId());
				}
			}
		}
	}
	
	
	
	
	public String uploadPermanentMediaNewsImage(String accessToken, String imageUrl) throws Exception {
		File localFile = null;
        try {
        	String extension = "jpg";
        	if(imageUrl.contains("wx_fmt=")){
        		Matcher matcher = Pattern.compile("wx_fmt=([^&]*)").matcher(imageUrl);
            	if(matcher.find()){
            		extension = matcher.group(1);
            	}
        	}else if(imageUrl.contains("\\.")){
        		String[] temp = imageUrl.split("\\.");
            	extension = temp[temp.length-1];
        	}
        	byte[] data = HttpClientUtils.download(imageUrl);
            localFile = LocalFileHelper.saveToLocal(data, extension);
            return WechatMediaApiHelper.uploadPermanentMediaNewsImg(accessToken, localFile);
        }catch(Exception e){
			throw e;
		}finally {
            if (localFile != null) {
            	LocalFileHelper.deleteFile(localFile);
            }
        }
	}
	
}

