
/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2016 All Rights Reserved.
 */
package com.baijia.tianxiao.sal.upload.service.impl;

import com.baijia.commons.lang.utils.JacksonUtil;
import com.baijia.tianxiao.common.service.ImportDataProcessService;
import com.baijia.tianxiao.common.service.ImportDataProcessService.SingleSaveErrorResult;
import com.baijia.tianxiao.constants.DataProcType;
import com.baijia.tianxiao.dal.org.dao.OrgInfoDao;
import com.baijia.tianxiao.dal.org.po.OrgInfo;
import com.baijia.tianxiao.dal.upload.dao.TxUploadRecordsDao;
import com.baijia.tianxiao.dal.upload.po.TxUploadRecords;
import com.baijia.tianxiao.dto.upload.UploadResult.UploadFile;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.enums.RedisKeyEnums;
import com.baijia.tianxiao.excel.ExcelExporterUtils;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.filter.TianxiaoPCContext;
import com.baijia.tianxiao.sal.upload.dto.TaskStatus;
import com.baijia.tianxiao.sal.upload.dto.UploadRecordDto;
import com.baijia.tianxiao.sal.upload.service.UploadFileReaderService;
import com.baijia.tianxiao.sal.upload.service.UploadService;
import com.baijia.tianxiao.util.collection.CollectorUtil;
import com.baijia.tianxiao.util.storage.StorageUtil;
import com.baijia.tianxiao.util.upload.FileUploadUtils;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicInteger;
import javax.annotation.Resource;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.Cache;
import org.springframework.cache.Cache.ValueWrapper;
import org.springframework.cache.CacheManager;
import org.springframework.cache.concurrent.ConcurrentMapCache;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.data.redis.cache.RedisCacheManager;
import org.springframework.data.redis.core.RedisOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

/**
 * @title UploadServiceImpl
 * @desc TODO
 * @author cxm
 * @date 2016年3月15日
 * @version 1.0
 */
@Service
@Slf4j
public class UploadServiceImpl implements UploadService, InitializingBean, ApplicationContextAware {

    private Map<Integer, ImportDataProcessService> dataProcessServiceMap;

    private ApplicationContext context;

    @Autowired(required = false)
    private CacheManager cacheManager;

    /**
     * 创建一个固定10个线程大小的线程池,用来响应上传任务
     */
    private ExecutorService uploadTaskExecutor = Executors.newFixedThreadPool(10, new ThreadFactory() {
        private final AtomicInteger threadNumber = new AtomicInteger(1);

        @Override
        public Thread newThread(Runnable r) {
            Thread t = new Thread(r, "UPLOAD_TASK_" + threadNumber.getAndIncrement());
            return t;
        }
    });

    // private Cache<Integer, TaskStatus> taskCache = CacheBuilder.newBuilder()
    // // 设置并发级别为8，并发级别是指可以同时写缓存的线程数
    // .concurrencyLevel(8)
    // // 设置写缓存后5分钟过期
    // .expireAfterWrite(5, TimeUnit.MINUTES)
    // // 设置缓存容器的初始容量为10
    // .initialCapacity(10)
    // // 设置缓存最大容量为100，超过100之后就会按照LRU最近虽少使用算法来移除缓存项
    // .maximumSize(100)
    // // 设置要统计缓存的命中率
    // .recordStats().build();

    private Cache taskCache;

    @Resource
    private TxUploadRecordsDao txUploadRecordsDao;

    @Resource
    private OrgInfoDao orgInfoDao;

    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public int uploadFile(final Long orgId, final int uploadType, final boolean override, MultipartFile file)
        throws IOException {
        Preconditions.checkNotNull(uploadType, "upload type can not be null");
        Preconditions.checkArgument(file != null && !file.isEmpty(), "upload file is null");

        final TxUploadRecords uploadRecord = new TxUploadRecords();
        uploadRecord.setFileName(file.getOriginalFilename());
        uploadRecord.setOrgId(orgId);
        uploadRecord.setStatus(0);
        uploadRecord.setUploadType(uploadType);
        final Integer cascadeId =  TianxiaoPCContext.getTXCascadeId();
        txUploadRecordsDao.save(uploadRecord);

        final Integer taskId = uploadRecord.getId();

        final UploadFileReaderService service = getUploadFileService(file);
        uploadTaskExecutor.execute(new Runnable() {
            @Override
            public void run() {
                //新建的线程内取不到登陆信息，需要从外部传进来
                TianxiaoPCContext.setTXCascadeId(cascadeId);
                TianxiaoPCContext.setOrgId(orgId.intValue());
                log.info("start to execute upload task:{}", taskId);
                ImportDataProcessService processService = dataProcessServiceMap.get(uploadType);
                TaskStatus taskStatus = new TaskStatus();
                storeCache(taskId, taskStatus);
                try {
                    List<String> headers = tranHeader(service.readData());
                    taskStatus.setHeaders(headers);
                    taskStatus.setTotalCount(service.getDataLength());
                    if (CollectionUtils.isEmpty(headers) || !processService.validateHeader(headers)) {
                        taskStatus.setException(new BussinessException(CommonErrorCode.PARAM_ERROR, "上传文件的表头信息不正确!"));
                        log.warn("upload file header:{} is invalidate", taskStatus);
                        storeCache(taskId, taskStatus);
                        return;
                    }
                    Object[] lineData = null;
                    SingleSaveErrorResult saveResult = null;

                    List<Object[]> failDatas = Lists.newArrayList();
                    List<Object[]> repeatDatas = Lists.newArrayList();
                    int i = 0;
                    while ((lineData = service.readData()) != null) {
                        try {
                            saveResult = processService.saveSingleData(orgId, TianxiaoPCContext.getTXCascadeId()==null?-1L:TianxiaoPCContext.getTXCascadeId(), headers, lineData, override);
                            if (saveResult == null
                                || (!saveResult.isRepeat() && StringUtils.isBlank(saveResult.getErrorMsg()))
                                || saveResult.isSuccess()) {
                                taskStatus.increaseSuccessCount();
                            } else if (saveResult.isRepeat()) {
                                taskStatus.increaseRepeatCount();
                                repeatDatas.add(lineData);
                            } else {
                                taskStatus.increaseFailCount();
                                failDatas.add(lineData);
                            }
                        } catch (DuplicateKeyException e) {
                            log.warn("save data:{} ,is repeat", ToStringBuilder.reflectionToString(lineData));
                            taskStatus.increaseRepeatCount();
                            repeatDatas.add(lineData);
                        } catch (Exception e) {
                            log.warn("save data:{} ,catch error:{}", ToStringBuilder.reflectionToString(lineData),
                                e.getMessage());
                            failDatas.add(lineData);
                            log.error("save data catch error:", e);
                            taskStatus.increaseFailCount();
                        }
                        i++;
                        // 每10次保存一次状态
                        if (i % 10 == 0 && taskStatus.getCompleteCount() < taskStatus.getTotalCount()) {
                            processResultStore(uploadRecord, taskStatus, null, false);
                        }
                        storeCache(taskId, taskStatus);
                    }
                    if (taskStatus.getCompleteCount() == 0) {
                        taskStatus.setException(new BussinessException(CommonErrorCode.PARAM_ERROR, "文件数据为空"));
                    }else {
                        //processService.afterComplete();
                    }
                    taskStatus.setTotalCount(taskStatus.getCompleteCount());
                    processResultStore(uploadRecord, taskStatus,
                        generateErrorExcel(orgId, headers, repeatDatas, failDatas), true);

                } catch (Exception e) {
                    log.warn("read data catch error:", e);
                    taskStatus.setException(new BussinessException(CommonErrorCode.BUSINESS_ERROR, "读取文件错误"));
                } finally {
                    storeCache(taskId, taskStatus);
                    service.close();
                }

            }
        });
        return taskId;
    }

    private void storeCache(Integer taskId, TaskStatus taskStatus) {
        try {
            taskCache.put(taskId.toString(), JacksonUtil.obj2Str(taskStatus));
        } catch (IOException e1) {
            log.error("serialize taskstatus catch exception:", e1);
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "序列化进度信息失败");
        }
    }

    private String generateErrorExcel(Long orgId, List<String> headers, List<Object[]> repeatDatas,
        List<Object[]> failDatas) throws IOException {
        if (CollectionUtils.isEmpty(repeatDatas) && CollectionUtils.isEmpty(failDatas)) {
            return "";
        }
        Workbook workBook = null;
        OutputStream os = null;
        File excelFile = null;
        try {
            workBook = new SXSSFWorkbook(100);
            Map<String, CellStyle> cellStyleMap = Maps.newHashMap();
            if (CollectionUtils.isNotEmpty(repeatDatas)) {
                Sheet sheet = workBook.createSheet("重复信息表");
                ExcelExporterUtils.fillSheet(sheet, headers, repeatDatas, cellStyleMap);
            }
            if (CollectionUtils.isNotEmpty(failDatas)) {
                Sheet sheet = workBook.createSheet("错误信息表");
                ExcelExporterUtils.fillSheet(sheet, headers, failDatas, cellStyleMap);
            }

            File execelFolder = new File("/tmp/tianxiao-export");
            if (!execelFolder.exists()) {
                execelFolder.mkdirs();
            }
            excelFile = new File(execelFolder, System.currentTimeMillis() + ".xlsx");
            excelFile.createNewFile();
            os = new FileOutputStream(excelFile);
            workBook.write(os);

            UploadFile uploadFile = FileUploadUtils.uploadToRemote(orgId, excelFile, false);

            return StorageUtil.constructUrl(uploadFile);
        } catch (IOException e) {
            log.error("write error data to excel catch error:", e);
            throw e;
        } finally {
            IOUtils.closeQuietly(os);
            IOUtils.closeQuietly(workBook);
            if (excelFile != null) {
                excelFile.deleteOnExit();
            }
        }

    }

    private void processResultStore(TxUploadRecords uploadRecord, TaskStatus taskStatus, String fileUrl,
        boolean isComplete) {
        // TxUploadRecords uploadRecord = txUploadRecordsDao.getById(taskId);
        uploadRecord.setFailCount(taskStatus.getFailCount());
        uploadRecord.setRepeatCount(taskStatus.getRepeatCount());
        uploadRecord.setSuccessCount(taskStatus.getSuccessCount());
        uploadRecord.setTotalCount(taskStatus.getTotalCount() - 1);
        if (isComplete) {
            String errorFileUrl = fileUrl == null ? "" : fileUrl;
            uploadRecord.setErrorFileUrl(errorFileUrl);
            taskStatus.setErrorFileUrl(errorFileUrl);
            uploadRecord.setStatus(1);
        }
        txUploadRecordsDao.update(uploadRecord);
    }

    private List<String> tranHeader(Object[] headerObjs) {
        Preconditions.checkArgument(ArrayUtils.isNotEmpty(headerObjs), "头信息为空");
        List<String> headers = Lists.newArrayList();
        for (Object obj : headerObjs) {
            if (obj != null) {
                headers.add(obj.toString());
            } else {
                headers.add("-");
            }
        }
        return headers;

    }

    private UploadFileReaderService getUploadFileService(MultipartFile file) {
        UploadFileReaderService service = null;
        String fileName = file.getOriginalFilename();
        if (fileName.toLowerCase().endsWith(".xls") || fileName.toLowerCase().endsWith(".xlsx")) {
            service = new ExcelUploadFileReaderServiceImpl(file);
        } else if (fileName.toLowerCase().endsWith(".csv")) {
            // 默认不使用流的方式读取csv文件
            service = new CsvUploadFileReaderServiceImpl(file, false);
        } else {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, fileName + "文件类型不支持,只支持CSV和excel文件");
        }
        return service;
    }

    @Override
    public TaskStatus getUploadStatus(Long orgId, Integer taskId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(taskId != null && taskId > 0, "taskId is illegal");

        ValueWrapper value = taskCache.get(taskId.toString());
        if (value == null) {
            return new TaskStatus(null, 0);
        }
        TaskStatus taskStatus = null;
        try {
            taskStatus = JacksonUtil.str2Obj((String) value.get(), TaskStatus.class);
            taskStatus.getCompleteRate();
        } catch (IOException e) {
            log.error("deserilize error:", e);
            throw new RuntimeException("反序列化失败");
        }
        return taskStatus;
    }

    @Override
    public List<UploadRecordDto> listUploadTask(Long orgId, Date startTime, Date endTime, DataProcType uploadType,
        Integer status) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Integer type = uploadType != null ? uploadType.getType() : null;
        List<TxUploadRecords> records = txUploadRecordsDao.listUploadTask(orgId, startTime, endTime, type, status);
        if (CollectionUtils.isNotEmpty(records)) {
            return buildUploadRecordDto(records);
        }

        return Collections.emptyList();
    }

    private List<UploadRecordDto> buildUploadRecordDto(List<TxUploadRecords> records) {
        List<UploadRecordDto> result = Lists.newArrayList();
        Collection<Long> orgIds = CollectorUtil.collect(records, new Function<TxUploadRecords, Long>() {
            @Override
            public Long apply(TxUploadRecords input) {
                return input.getOrgId();
            }
        });
        Map<Long, String> orgNameMap = queryOrgNameMap(orgIds);
        for (TxUploadRecords record : records) {
            result.add(UploadRecordDto.transfer(record, orgNameMap.get(record.getOrgId())));
        }
        return result;
    }

    private Map<Long, String> queryOrgNameMap(Collection<Long> orgIds) {
        List<OrgInfo> orgInfos = orgInfoDao.getByIds(orgIds, "orgId", "shortName");
        return CollectorUtil.collectMap(orgInfos, new Function<OrgInfo, Long>() {
            @Override
            public Long apply(OrgInfo input) {
                return input.getOrgId().longValue();
            }
        }, new Function<OrgInfo, String>() {
            @Override
            public String apply(OrgInfo input) {
                return input.getShortName();
            }
        });
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.context = applicationContext;
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        dataProcessServiceMap = Maps.newHashMap();
        Map<String, ImportDataProcessService> serviceMap = context.getBeansOfType(ImportDataProcessService.class);
        for (ImportDataProcessService service : serviceMap.values()) {
            dataProcessServiceMap.put(service.getProcessType().getType(), service);
        }

        if (cacheManager == null) {
            try {
                @SuppressWarnings("unchecked")
                RedisOperations<String, String> template = this.context.getBean(RedisTemplate.class);
                this.cacheManager = new RedisCacheManager(template);
                this.taskCache = cacheManager.getCache(RedisKeyEnums.ERP.UPLOAD_TASK_CACHE.getRedisKey());
                log.info("try to init cache by redis template");
            } catch (Exception e) {
                this.taskCache = new ConcurrentMapCache(RedisKeyEnums.ERP.UPLOAD_TASK_CACHE.getRedisKey());
                log.info("use local cache because can not found redis config");
            }
        } else {
            this.taskCache = cacheManager.getCache(RedisKeyEnums.ERP.UPLOAD_TASK_CACHE.getRedisKey());
            log.info("use cache manager bean");
        }

    }

    @Override
    public String getErrorFileUrl(Long orgId, Integer taskId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(taskId != null && taskId > 0, "taskId is illegal");
        TxUploadRecords uploadRecord = txUploadRecordsDao.getById(taskId, "errorFileUrl");
        return uploadRecord.getErrorFileUrl();

    }

}
