
/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2016 All Rights Reserved.
 */
package com.baijia.tianxiao.sal.upload.service.impl;

import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.upload.service.UploadFileReaderService;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.List;

import lombok.extern.slf4j.Slf4j;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.web.multipart.MultipartFile;

/**
 * @title CsvUploadFileReaderServiceImpl
 * @desc TODO
 * @author cxm
 * @date 2016年3月16日
 * @version 1.0
 */
@Slf4j
public class CsvUploadFileReaderServiceImpl implements UploadFileReaderService {

    private MultipartFile csvFile;

    private BufferedReader br;

    private boolean isComplete;

    private int count;

    private boolean useStream;

    private int index;

    // 如果不使用流,这个就是总数据
    private List<Object[]> datas;

    public CsvUploadFileReaderServiceImpl(MultipartFile csvFile, boolean useStream) {
        Preconditions.checkArgument(csvFile != null && !csvFile.isEmpty(), "csv file is not exist");
        this.csvFile = csvFile;
        this.useStream = useStream;
        initReader();
    }

    private synchronized void initReader() {
        if (br == null && !isComplete) {
            try {
                br = new BufferedReader(new InputStreamReader(csvFile.getInputStream(), "GBK"));
                if (!useStream && this.datas == null) {
                    try {
                        this.datas = readDatas();
                    } catch (IOException e) {
                        throw new RuntimeException("get data length error");
                    }
                }
            } catch (FileNotFoundException e) {
                throw new BussinessException(CommonErrorCode.PARAM_ERROR, "上传的文件错误");
            } catch (IOException e) {
                log.error("init csv reader catch error:", e);
                IOUtils.closeQuietly(br);
                throw new BussinessException(CommonErrorCode.PARAM_ERROR, "读取文件信息错误");
            }
        }

    }

    @Override
    public int getDataLength() throws IOException {
        initReader();

        if (!isComplete) {
            throw new UnsupportedOperationException("can not get data length from csv file before read all data");
        }
        return count;
    }

    @Override
    public synchronized Object[] readData() throws IOException {
        String line = null;
        try {
            initReader();
            if (!useStream) {
                if (index >= this.count) {
                    return null;
                }
                return this.datas.get(index++);
            }

            while ((line = br.readLine()) != null) {
                if (StringUtils.isNoneBlank(line)) {
                    this.count++;
                    return StringUtils.split(line, ',');
                }
            }
        } catch (FileNotFoundException e) {
            log.error("init reader error,", e);
        } catch (Exception e) {
            log.error("read data get error:", e);
        } finally {
            if (line == null) {
                IOUtils.closeQuietly(br);
            }
        }
        this.isComplete = true;
        return null;
    }

    @Override
    public List<Object[]> readDatas() throws IOException {
        List<Object[]> results = Lists.newArrayList();
        try {
            initReader();
            if (this.datas != null) {
                return this.datas;
            }
            String line = null;
            while ((line = br.readLine()) != null) {
                if (StringUtils.isNoneBlank(line)) {
                    this.count++;
                    results.add(StringUtils.splitPreserveAllTokens(line, ','));
                }
            }
        } catch (FileNotFoundException e) {
            log.error("init reader error,", e);
        } catch (Exception e) {
            log.error("read data catch error:", e);
        } finally {
            IOUtils.closeQuietly(br);
        }
        this.isComplete = true;

        return results;

    }

    @Override
    public void close() {
        IOUtils.closeQuietly(br);
    }

    public static void main(String[] args) throws IOException {
        BufferedReader br = new BufferedReader(
            new InputStreamReader(new FileInputStream(new File("/Users/zhenyujian/Desktop/CRM-线索导入模板.xlsx")), "gbk"));
        String line = null;
        List<String[]> datas = Lists.newArrayList();
        while ((line = br.readLine()) != null) {
            if (StringUtils.isNoneBlank(line)) {
                String[] data = StringUtils.splitPreserveAllTokens(line, ',');
                System.out.println("line:" + line + " split data:" + ArrayUtils.toString(data));
                datas.add(data);

            }
        }
        System.out.println(datas);
        br.close();
    }

}
