
/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */

package com.baijia.tianxiao.sal.teacher.impl;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.baijia.tianxiao.dal.storage.dao.StorageDao;
import com.baijia.tianxiao.dal.storage.po.Storage;
import com.baijia.tianxiao.dal.user.dao.TeacherDao;
import com.baijia.tianxiao.dal.user.dao.UserDao;
import com.baijia.tianxiao.dal.user.po.Teacher;
import com.baijia.tianxiao.dal.user.po.User;
import com.baijia.tianxiao.sal.teacher.api.TeacherService;
import com.baijia.tianxiao.sal.teacher.dto.OrgTeacherResponseDto;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.util.BaseUtils;
import com.baijia.tianxiao.util.collection.CollectorUtil;
import com.baijia.tianxiao.util.properties.PropertiesReader;
import com.baijia.tianxiao.util.storage.StorageUtil;
import com.google.common.base.Function;
import com.google.common.collect.Lists;

/**
 * @author zhangbing
 * @version 1.0
 * @title TeacherServiceImpl
 * @desc TODO
 * @date 2015年12月25日
 */
@Service
public class TeacherServiceImpl implements TeacherService {

    private static final Logger logger = LoggerFactory.getLogger(TeacherServiceImpl.class);

    @Autowired
    private TeacherDao teacherDao;

    @Autowired
    private UserDao userDao;

    @Autowired
    private StorageDao storageDao;

    @Override
    public List<OrgTeacherResponseDto> getOrgTeacherList(Long orgId, PageDto pageDto) {
        List<Teacher> teachers = teacherDao.getTeachersByOrgId(orgId, pageDto);
        String host = PropertiesReader.getValue("wxm", "gsx-m.server");
        return buildTeacherResponse(teachers, host);
    }

    /**
     * 获取老师头像
     *
     * @param teachers
     * @return
     */
    @SuppressWarnings("unchecked")
    private Map<Long, Storage> getAndCacheStorage(List<Teacher> teachers) {
        List<Long> ids = BaseUtils.getListFilter(teachers, "avatar");
        Map<Long, Storage> cache = storageDao.getStorageMapByIds(ids);
        logger.debug("ids:{},cache:{}", ids, cache);
        return cache;
    }

    /**
     * 批量获取用户mobile，number信息
     *
     * @return
     */
    @SuppressWarnings("unchecked")
    private Map<Long, User> getAndCachedUser(List<Teacher> teachers) {
        Map<Long, User> cache = null;
        try {
            List<Long> userIds = BaseUtils.getListFilter(teachers, "userId");
            cache = userDao.getMapByIds(userIds);
            logger.debug("userIds:{},cache:{}", userIds, cache);
        } catch (Exception e) {
            logger.error("getAndCachedUser failed, e:{}", e);
        }

        return cache;
    }

    /**
     * 获取cdb.storage表数据对应图片url
     *
     * @param storage
     * @return
     */
    public String constructUrl(Storage storage, String imgServer) {
        StringBuilder sb = new StringBuilder(imgServer);
        String surfix = "";
        if (StringUtils.isNotBlank(storage.getMimetype())) {
            if (storage.getMimetype().contains("/")) {
                surfix = storage.getMimetype().split("/")[1];
            } else {
                surfix = storage.getMimetype();
            }
        }
        return sb.append(storage.getFid()).append("_").append(storage.getSn()).append(".").append(surfix).toString();
    }

    @Override
    public OrgTeacherResponseDto getOrgTeacherDetail(Long orgId, Long userId) {
        Teacher teacher = this.teacherDao.getByUserId(userId);
        OrgTeacherResponseDto dto = null;
        if (teacher != null && orgId != null && teacher.getOrgId() != null && teacher.getOrgId() == orgId.longValue()) {
            User user = this.userDao.getById(userId);
            if (user != null) {
                String host = PropertiesReader.getValue("wxm", "gsx-m.server");

                dto = new OrgTeacherResponseDto();
                Long storageId = teacher.getAvatar();
                if (storageId != null) {
                    Storage storage = this.storageDao.getById(teacher.getAvatar().longValue());
                    if (storage != null) {
                        dto.setAvatar(StorageUtil.constructUrl(storage.getFid(), storage.getMimetype(), storage.getSn()));
                    }
                } else {
                    dto.setAvatar("");
                }
                dto.setNumber(user.getNumber());
                dto.setTeacherName(teacher.getRealName());
                dto.setLink(host + "/" + user.getNumber());
                dto.setUserId(userId);
            }
        }
        return dto;
    }

    @Override
    public List<OrgTeacherResponseDto> getOrgTeacherList(Long orgId, Collection<Long> userIds) {
        List<Teacher> teachers = this.teacherDao.getByTeacherIds(userIds);
        return buildTeacherResponse(teachers, null);
    }

    private List<OrgTeacherResponseDto> buildTeacherResponse(List<Teacher> teachers, String host) {
        List<OrgTeacherResponseDto> data = Lists.newArrayList();
        if (teachers != null && !teachers.isEmpty()) {
            Map<Long, Storage> storageCache = this.getAndCacheStorage(teachers);
            Map<Long, User> userCache = this.getAndCachedUser(teachers);

            OrgTeacherResponseDto dto = null;
            for (Teacher teacher : teachers) {
                dto = new OrgTeacherResponseDto();
                Long userId = teacher.getUserId();
                if (userCache != null && userCache.containsKey(userId)) {
                    User user = userCache.get(userId);
                    dto.setNumber(user.getNumber());
                    dto.setLink(host + "/" + user.getNumber());
                }
                dto.setTeacherName(teacher.getRealName() == null ? "待定" : teacher.getRealName());
                Long storageId = teacher.getAvatar();
                if (storageId != null && storageCache.containsKey(storageId)) {
                    dto.setAvatar(StorageUtil.constructUrl(storageCache.get(storageId).getFid(),
                            storageCache.get(storageId).getMimetype(), storageCache.get(storageId).getSn()));
                } else {
                    dto.setAvatar("");
                }
                dto.setUserId(teacher.getUserId());
                dto.setDescr(teacher.getShortIntroduce() == null ? "" : teacher.getShortIntroduce());
                dto.setOtherInfo(teacher.getOtherInfo() == null ? "" : teacher.getOtherInfo());
                data.add(dto);
            }
        }

        logger.info("getOrgTeacherList.data:{}", data);
        return data;
    }

    @Override
    public Map<Long, OrgTeacherResponseDto> getOrgTeacherMap(Long orgId, Collection<Long> userIds) {
        return CollectorUtil.collectMap(this.getOrgTeacherList(orgId, userIds),
                new Function<OrgTeacherResponseDto, Long>() {
                    @Override
                    public Long apply(OrgTeacherResponseDto arg0) {
                        return arg0.getUserId();
                    }
                });
    }
}
