
/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */

package com.baijia.tianxiao.sal.teacher.impl;

import com.baijia.tianxiao.constant.OrgTeacherStatus;
import com.baijia.tianxiao.dal.course.dao.OrgCourseGroupRelateDao;
import com.baijia.tianxiao.dal.org.dao.*;
import com.baijia.tianxiao.dal.org.po.*;
import com.baijia.tianxiao.dal.storage.dao.StorageDao;
import com.baijia.tianxiao.dal.storage.po.Storage;
import com.baijia.tianxiao.dal.user.dao.TeacherDao;
import com.baijia.tianxiao.dal.user.dao.TeacherModifiedDao;
import com.baijia.tianxiao.dal.user.dao.UserDao;
import com.baijia.tianxiao.dal.user.po.Teacher;
import com.baijia.tianxiao.dal.user.po.TeacherModified;
import com.baijia.tianxiao.dal.user.po.User;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.teacher.api.OrgTeacherService;
import com.baijia.tianxiao.sal.teacher.dto.OrgTeacherInfoDto;
import com.baijia.tianxiao.sal.teacher.dto.OrgTeachersDto;
import com.baijia.tianxiao.sal.teacher.dto.request.TeacherInfoRequest;
import com.baijia.tianxiao.sal.teacher.util.OrgTeacherUtil;
import com.baijia.tianxiao.sal.teacher.util.TeacherErrorCode;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.util.ArithUtil;
import com.baijia.tianxiao.util.CollectorUtil;
import com.baijia.tianxiao.util.HtmlUtils;
import com.baijia.tianxiao.util.date.DateUtil;
import com.baijia.tianxiao.util.storage.StorageUtil;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;

import lombok.extern.slf4j.Slf4j;

@Service
@Slf4j
public class OrgTeacherServiceImpl implements OrgTeacherService {

    @Autowired
    private OrgTeacherDao orgTeacherDao;
    @Autowired
    private OrgCourseDao orgCourseDao;
    @Autowired
    private UserDao userDao;
    @Autowired
    private OrgCourseTeacherDao orgCourseTeacherDao;
    @Autowired
    private OrgAccountDao orgAccountDao;
    @Autowired
    private TeacherDao teacherDao;
    @Autowired
    private OrgCourseGroupRelateDao orgCourseGroupRelateDao;
    @Autowired
    private StorageDao storageDao;
    @Autowired
    private OrgTeacherLessonDao orgTeacherLessonDao;
    @Autowired
    private OrgLessonCommentDao orgLessonCommentDao;
    @Autowired
    private OrgClassLessonDao orgClassLessonDao;
    @Autowired
    private TeacherModifiedDao teacherModifiedDao;

    @Override
    public OrgTeacherInfoDto saveTeacher(Long orgId, TeacherInfoRequest request, Long szManagerId) {
        Preconditions.checkArgument(request.getName() != null && request.getName().length() <= 15, "姓名必填,而且不能大于15个字");

        OrgTeacher orgTeacher = null;
        if (request.getId() == null) {
            List<OrgTeacher> temp = orgTeacherDao.getTeacherByMobileAndOrgId(orgId, request.getMobile(), 1);
            if (temp != null && temp.size() > 0) {
                throw new BussinessException(TeacherErrorCode.MOBILE_DOUBLE);
            }
        } else {
            orgTeacher = orgTeacherDao.getById(request.getId());
            if (!request.getMobile().equals(orgTeacher.getMobile())) {
                List<OrgTeacher> temp = orgTeacherDao.getTeacherByMobileAndOrgId(orgId, request.getMobile(), 1);
                if (temp != null && temp.size() > 0) {
                    throw new BussinessException(TeacherErrorCode.MOBILE_DOUBLE);
                }
            }
        }

        if (request.getId() == null) {
            orgTeacher = new OrgTeacher();
            orgTeacher.setType(2);// 天校app添加来源
            OrgAccount orgAccount = orgAccountDao.getAccountById(orgId.intValue());
            if (orgAccount != null) {
                orgTeacher.setOrgNumber(orgAccount.getNumber());
            }

            orgTeacher.setOrgManagerId(szManagerId);
        }

        if (orgTeacher.getUserId() == null) {
            Map<String, Long> userInfoMap =
                OrgTeacherUtil.getUserIdAndNumber(orgId, request.getMobile(), request.getName());
            orgTeacher.setUserId(userInfoMap.get("id"));
        }

        Teacher teacher = teacherDao.getByUserId(orgTeacher.getUserId());
        TeacherModified teacherModified = teacherModifiedDao.getByUserId(orgTeacher.getUserId());

        // 保存orgTeacher
        saveOrUpdateOrgTeacher(orgTeacher, request, orgId);
        // 更新cdbTeacher
        SaveOrUpdateCdbTeacher(teacher, request, orgId);
        // 更新cdbTeacherModify
        SaveOrUpdateCdbTeacherModify(teacherModified, request, orgId);

        // 添加cdb.org_teacher 免审核
        if (request.getId() == null) {
            User user = userDao.getById(orgTeacher.getUserId());
            if (user != null) {
                orgTeacherDao.saveCdbOrgTeacher(orgTeacher.getUserId(), user.getNumber(), orgId);
            }
        }

        return detail(orgTeacher.getId());
    }

    @Override
    public void del(Long teacherId) {
        OrgTeacher orgTeacher = orgTeacherDao.getById(teacherId);
        if (orgTeacher != null) {
            orgTeacher.setStatus(OrgTeacherStatus.TERMINATED.getCode());
            orgTeacher.setFireTime(new Date());

            log.debug("del teacher == {}", orgTeacher);
            orgTeacherDao.update(orgTeacher);

            Teacher teacher = teacherDao.getByUserId(orgTeacher.getUserId());
            if (teacher != null) {
                teacher.setOrgId(null);
                teacher.setUpdatedAt(new Date());
                teacherDao.update(teacher, true, "orgId");
            }

            TeacherModified teacherModified = teacherModifiedDao.getByUserId(orgTeacher.getUserId());
            if (teacherModified != null) {
                teacherModified.setOrgId(null);
                teacherModified.setUpdatedAt(new Date());
                teacherModifiedDao.update(teacherModified, true, "orgId");
            }
        } else {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR);
        }

    }

    @Override
    public OrgTeacherInfoDto detail(Long teacherId) {
        OrgTeacher orgTeacher = orgTeacherDao.getById(teacherId);
        OrgTeacherInfoDto orgTeacherInfoDto = new OrgTeacherInfoDto();
        if (orgTeacher != null) {
            Teacher teacher = teacherDao.getByUserId(orgTeacher.getUserId());
            orgTeacherInfoDto.setUserId(teacher.getUserId());
            orgTeacherInfoDto.setId(orgTeacher.getId());
            orgTeacherInfoDto.setName(teacher.getRealName());
            orgTeacherInfoDto.setAvatar(teacher.getAvatar());
            if (teacher.getAvatar() != null) {
                Storage storage = storageDao.getById(teacher.getAvatar());
                if (storage != null) {
                    String avatarUrl =
                        StorageUtil.constructUrl(storage.getFid(), storage.getMimetype(), storage.getSn());
                    orgTeacherInfoDto.setAvatarUrl(avatarUrl);
                }
            }
            if (teacher.getBirthday() != null) {
                orgTeacherInfoDto.setBirthday(teacher.getBirthday().getTime());
            }
            User user = userDao.getById(teacher.getUserId());
            orgTeacherInfoDto.setNumber(user.getNumber());
            orgTeacherInfoDto.setDescr(teacher.getIntroduce() != null ? teacher.getIntroduce() : "");
            orgTeacherInfoDto.setMobile(orgTeacher.getMobile());
            orgTeacherInfoDto.setGender(teacher.getSex());
            orgTeacherInfoDto
                .setRemark(teacher.getOtherInfo() == null ? "" : HtmlUtils.delHTMLTag(teacher.getOtherInfo()));
            // orgTeacherInfoDto.setBackPictrueUrl(backPictrueUrl);
        }
        return orgTeacherInfoDto;
    }

    @Override
    public List<OrgTeacherInfoDto> list(Long orgId, Long lastId, Integer pageSize) {
        List<OrgTeacher> orgTeachers =
            orgTeacherDao.listByLastId(orgId, lastId, pageSize, OrgTeacherStatus.SIGNED.getCode());
        if (orgTeachers == null) {
            return Collections.EMPTY_LIST;
        }
        return buildOrgTeacherInfo(orgId, orgTeachers);
    }

    @Override
    public List<OrgTeacherInfoDto> listPage(Long orgId, PageDto pageDto) {
        List<OrgTeacher> orgTeachers = orgTeacherDao.getTeachersByOrgId(orgId, pageDto);
        if (orgTeachers == null) {
            return Collections.EMPTY_LIST;
        }
        return buildOrgTeacherInfo(orgId, orgTeachers);
    }

    @Override
    public List<OrgTeachersDto> getAndUpdateOrgTeacherByOpenId(Long orgId, String weixinOpenId) {
        List<OrgTeacher> teachers = this.orgTeacherDao.getTeacherByOpenIdAndOrgId(orgId, weixinOpenId, 1);

        List<OrgTeachersDto> result = Lists.newArrayList();
        if (teachers != null && !teachers.isEmpty()) {
            OrgTeachersDto item = null;
            for (OrgTeacher teacher : teachers) {
                Teacher realTeacher = teacherDao.getByUserId(teacher.getUserId());
                item = new OrgTeachersDto();
                org.springframework.beans.BeanUtils.copyProperties(teacher, item);
                item.setTeacherId(teacher.getUserId());
                item.setClassName(this.getClassName(orgId, teacher.getUserId()));
                item.setName(realTeacher.getNickName());

                result.add(item);
                // 将openId与student绑定
                if (!weixinOpenId.equals(teacher.getWeixin())) {
                    teacher.setWeixin(weixinOpenId);
                    teacher.setUpdateTime(new Date());
                    orgTeacherDao.update(teacher, "weixin", "updateTime");
                }
            }

        }
        return result;
    }

    @Override
    public OrgTeachersDto getAndUpdateOrgStudentByMobile(Long orgId, String mobile, String name, String weinOpenId)
        throws BussinessException {
        OrgTeachersDto result = null;

        List<OrgTeacher> teachers = this.orgTeacherDao.getTeacherByMobileAndOrgId(orgId, mobile, 1);

        if (teachers != null && !teachers.isEmpty()) {
            result = new OrgTeachersDto();
            org.springframework.beans.BeanUtils.copyProperties(teachers.get(0), result);
            result.setTeacherId(teachers.get(0).getId());

            for (OrgTeacher teacher : teachers) {
                String weixin = teacher.getWeixin();
                if (!weinOpenId.equals(weixin)) {
                    // teacher.setTeacherName(name);
                    teacher.setWeixin(weinOpenId);
                    teacher.setUpdateTime(new Date());

                    // this.orgTeacherDao.update(teacher, "name", "weixin", "updateTime");
                    this.orgTeacherDao.update(teacher, "weixin", "updateTime");
                }
            }
        } else {
            throw new BussinessException(CommonErrorCode.NOT_FOUND, "绑定失败，您所填写的手机号不是老师手机号");
        }
        return result;
    }

    private String getClassName(Long orgId, Long teacherId) {
        List<Long> courseIds = orgCourseTeacherDao.getOrgTeacherCourseIds(orgId, teacherId);
        String name = "";
        if (courseIds != null && !courseIds.isEmpty()) {
            OrgAccount orgAccount = orgAccountDao.getAccountById(Integer.parseInt(orgId + ""));
            List<OrgCourse> list =
                this.orgCourseDao.getCourseList(courseIds, Long.parseLong(orgAccount.getNumber() + ""), null);
            if (list != null && !list.isEmpty()) {
                name = list.get(0).getName();
            }
        }
        return StringUtils.isNotBlank(name) ? name : "未加班";
    }

    private void saveOrUpdateOrgTeacher(OrgTeacher teacher, TeacherInfoRequest request, Long orgId) {

        teacher.setMobile(request.getMobile());
        teacher.setOrgId(orgId);
        teacher.setId(request.getId());
        Date now = new Date();
        if (teacher.getId() == null) {
            teacher.setSignTime(now);
            teacher.setCreateTime(now);
            teacher.setInviteTime(now);
        }
        teacher.setUpdateTime(now);
        teacher.setStatus(OrgTeacherStatus.SIGNED.getCode());

        log.debug("save or update org teacher = {}", teacher);
        orgTeacherDao.saveOrUpdate(teacher);
    }

    private void SaveOrUpdateCdbTeacher(Teacher teacher, TeacherInfoRequest request, Long orgId) {
        if (teacher == null) {
            teacher = new Teacher();
            teacher.setCreatedAt(new Date());
        }
        teacher.setRealName(request.getName());
        teacher.setSex(request.getGender());
        teacher.setAvatar(request.getAvatar());
        if (StringUtils.isNotEmpty(request.getRemark())) {
            if (request.getRemark().length() > 20) {
                teacher.setShortIntroduce(request.getRemark().substring(0, 20) + "...");
            } else {
                teacher.setShortIntroduce(request.getRemark());
            }
            teacher.setOtherInfo(request.getRemark());
        }
        teacher.setOrgId(orgId);
        if (request.getBirthday() != null) {
            teacher.setBirthday(new Date(request.getBirthday()));
        }
        teacher.setUpdatedAt(new Date());
        log.debug("save or update cdb teacher = {}", teacher);
        teacherDao.saveOrUpdate(teacher);
    }

    private void SaveOrUpdateCdbTeacherModify(TeacherModified teacherModified, TeacherInfoRequest request, Long orgId) {
        if (teacherModified == null) {
            teacherModified = new TeacherModified();
            teacherModified.setCreatedAt(new Date());
        }
        teacherModified.setRealName(request.getName());
        teacherModified.setSex(request.getGender());
        teacherModified.setAvatar(request.getAvatar());
        if (StringUtils.isNotEmpty(request.getRemark())) {
            if (request.getRemark().length() > 20) {
                teacherModified.setShortIntroduce(request.getRemark().substring(0, 20) + "...");
            } else {
                teacherModified.setShortIntroduce(request.getRemark());
            }
            teacherModified.setOtherInfo(request.getRemark());
        }
        teacherModified.setOrgId(orgId);
        if (request.getBirthday() != null) {
            teacherModified.setBirthday(new Date(request.getBirthday()));
        }
        teacherModified.setUpdatedAt(new Date());
        log.debug("save or update cdb teacherModified = {}", teacherModified);
        teacherModifiedDao.saveOrUpdate(teacherModified);
    }

    private List<OrgTeacherInfoDto> buildOrgTeacherInfo(Long orgId, List<OrgTeacher> list) {
        List<OrgTeacherInfoDto> result = Lists.newArrayList();
        Map<Long, OrgTeacher> orgTeacherMap = CollectorUtil.collectMap(list, new Function<OrgTeacher, Long>() {
            @Override
            public Long apply(OrgTeacher teacher) {
                return teacher.getUserId();
            }
        });
        List<Teacher> teachers = teacherDao.getByUserIds(orgTeacherMap.keySet());
        Map<Long, Teacher> teacherMap = CollectorUtil.collectMap(teachers, new Function<Teacher, Long>() {
            @Override
            public Long apply(Teacher teacher) {
                return teacher.getUserId();
            }
        });

        Map<Long, User> userMap = userDao.getMapByIds(orgTeacherMap.keySet());

        List<ClassHour> teacherLessoId = orgClassLessonDao.queryFinishedLessonCountByTeacherIds(orgId,
            orgTeacherMap.keySet(), DateUtil.getFirstDate(new Date()), DateUtil.getDayDiff(0));
        Map<Long, ClassHour> classHourMap = CollectorUtil.collectMap(teacherLessoId, new Function<ClassHour, Long>() {
            @Override
            public Long apply(ClassHour classHour) {
                return classHour.getTeacherId();
            }
        });

        Map<Long, Long> scoreMap = orgLessonCommentDao.getAveCommentByTeacherIds(orgTeacherMap.keySet());

        Map<Long, Integer> recommendMap = orgCourseGroupRelateDao.isRecommendTeacher(orgTeacherMap.keySet());

        Map<Long, Teacher> storageIds = CollectorUtil.collectMap(teachers, new Function<Teacher, Long>() {
            @Override
            public Long apply(Teacher teacher) {
                return teacher.getAvatar();
            }
        });

        List<Storage> storages = storageDao.getByIds(storageIds.keySet());
        Map<Long, Storage> storageMap = CollectorUtil.collectMap(storages, new Function<Storage, Long>() {
            @Override
            public Long apply(Storage storage) {
                return storage.getId();
            }
        });

        for (OrgTeacher orgTeacher : list) {
            OrgTeacherInfoDto dto = new OrgTeacherInfoDto();
            Teacher teacher = teacherMap.get(orgTeacher.getUserId());
            dto.setId(orgTeacher.getId());
            if (teacher != null) {
                dto.setName(teacher.getRealName());
            }
            dto.setClassNum(classHourMap.get(orgTeacher.getUserId()) == null ? 0
                : classHourMap.get(orgTeacher.getUserId()).getLessonCount());
            Long commentScore = scoreMap.get(orgTeacher.getUserId());
            if (commentScore == null) {
                dto.setCommentNum(0.0d);
            } else {
                dto.setCommentNum(ArithUtil.round(((double) (commentScore)) / 100, 1));
            }
            Integer count = recommendMap.get(orgTeacher.getUserId());
            if (count == null || count == 0) {
                dto.setRecommended(false);
            } else {
                dto.setRecommended(true);
            }
            dto.setCreateDate(
                orgTeacher.getSignTime() == null ? new Date().getTime() : orgTeacher.getSignTime().getTime());

            if (teacher.getAvatar() != null) {
                Storage storage = storageMap.get(teacher.getAvatar());
                if (storage != null) {
                    String avatarUrl =
                        StorageUtil.constructUrl(storage.getFid(), storage.getMimetype(), storage.getSn());
                    dto.setAvatarUrl(avatarUrl);
                }
            }
            dto.setUserId(teacher.getUserId());
            User user = userMap.get(teacher.getUserId());
            dto.setNumber(user == null ? 0 : user.getNumber());
            dto.setDescr(teacher.getShortIntroduce() == null ? "" : teacher.getShortIntroduce());
            dto.setOtherInfo(teacher.getOtherInfo() == null ? "" : HtmlUtils.delHTMLTag(teacher.getOtherInfo()));
            result.add(dto);
        }
        return result;
    }

    @Override
    public OrgTeacher getByUserId(long parseLong) {
        return orgTeacherDao.getTeachersByUserId(parseLong);
    }

    @Override
    public int getTeacherIds(long id) {
        List<Long> list = orgTeacherDao.getTeacherIds(id, OrgTeacherStatus.SIGNED.getCode());
        return list.size();
    }

}
