
/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2016 All Rights Reserved.
 */
package com.baijia.tianxiao.sal.student.pc.impl;

import com.baijia.tianxiao.common.service.ImportDataProcessService;
import com.baijia.tianxiao.constant.Relatives;
import com.baijia.tianxiao.constants.DataProcType;
import com.baijia.tianxiao.dal.org.dao.AreaDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentDao;
import com.baijia.tianxiao.dal.org.po.Area;
import com.baijia.tianxiao.dal.org.po.OrgStudent;
import com.baijia.tianxiao.dal.util.AreaUtils;
import com.baijia.tianxiao.dto.baidu.Location;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.enums.CrmErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.student.dto.response.pc.StudentListResponseDto;
import com.baijia.tianxiao.sal.student.pc.StudentUserService;
import com.baijia.tianxiao.util.ExcelUtils;
import com.baijia.tianxiao.util.baidu.BaiduMapUtils;
import com.baijia.tianxiao.validation.ParamValidateUtils;

import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;

import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.hssf.usermodel.HSSFCellStyle;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.IndexedColors;
import org.apache.poi.xssf.usermodel.XSSFCellStyle;
import org.apache.poi.xssf.usermodel.XSSFColor;
import org.apache.poi.xssf.usermodel.XSSFRow;
import org.apache.poi.xssf.usermodel.XSSFSheet;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.awt.Color;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.InvocationTargetException;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.regex.Pattern;

import lombok.extern.slf4j.Slf4j;

/**
 * @author zhangbing
 * @version 1.0
 * @title UploadServiceImpl
 * @desc TODO
 * @date 2016年3月18日
 */
@Service
@Slf4j
public class ImportStudentServiceImpl implements ImportDataProcessService {

	public static final String STUDENT_IMPORT_TEMPLATE_PATH = "/excel/student_import.xlsx";
	public static final String STUDENT_RESULT_TEMPLATE_PATH = "/excel/student_result.xlsx";
	
    // 表头配置映射关系
    private static final Map<String, Map<String, Object>> cache = Maps.newHashMap();
    private final static SingleSaveErrorResult IMPORT_SUCCESS = SingleSaveErrorResult.createSuccessResult("导入成功!");
    private final static SingleSaveErrorResult IMPORT_FAIL = SingleSaveErrorResult.createFailResult("导入失败");
    private final static SingleSaveErrorResult STUDENT_EXSIST = new SingleSaveErrorResult(true, "学员档案已存在");



    static {
        // ResourceBundle rs = ResourceBundle.getBundle("upload");
        // String header = rs.getString("upload.student.header");
        String header = "姓名:name:true,手机号:mobile:true,性别:genderStr:false,标签:tagsStr:false,亲属关系:relationshipStr:false,家长姓名:parentName:false,家长手机号:parentMobile:false,生日:birthday:false,公立学校:school:false,年级:degreeClass:false,省:province:false,市:city:false,区:county:false,详细地址:address:false,QQ:qq:false,邮箱:mail:false";
        //String header =
        //        "学员姓名:name:true,学员手机:mobile:true,家长姓名:parentName:false,家长手机:parentMobile:false,生日:birthday:false,QQ:qq:false,邮箱:mail:false,学校:school:false,班级:degreeClass:false,省:province:false,市:city:false,区:county:false,详细地址:address:false,父亲职业:fatherOccupation:false,母亲职业:matherOccupation:false";
        if (StringUtils.isNotBlank(header) && header.contains(":")) {
            String[] cells = header.split(",");

            int index = 0;// 顺序
            Map<String, Object> subMap = null;
            for (String cell : cells) {
                subMap = Maps.newHashMap();
                String column[] = cell.split(":");
                String name = column[0].trim();
                String propName = column[1];
                boolean empty = Boolean.parseBoolean(column[2]);
                subMap.put("propName", propName);
                subMap.put("index", index);
                subMap.put("empty", empty);
                cache.put(name, subMap);
                index++;
            }
        }
    }

    @Autowired
    private StudentUserService studentUserService;
    @Autowired
    private OrgStudentDao orgStudentDao;
    @Autowired
    private AreaDao areaDao;

    
    @Override
    public DataProcType getProcessType() {
        return DataProcType.ORG_STUDENT;
    }

    /**
     * 上传文件格式校验 必须保证上传文件的表头与upload.properties中配置的 表头名称和顺序一致
     *
     * @param headers 用户上传文件的表头
     */
    @Override
    public boolean validateHeader(List<String> headers) {
        if (CollectionUtils.isEmpty(headers)) {
            return false;
        }
        int index = 0;
        for (String header : headers) {
            if (!cache.containsKey(header.trim())) {
                return false;
            } else {
                Map<String, Object> item = cache.get(header);
                int sequence = (int) item.get("index");
                if (index != sequence) {
                    return false;
                }
            }
            index++;
        }
        return true;
    }

    @Override
    public SingleSaveErrorResult saveSingleData(Long orgId, Long cascadeId, List<String> headers, Object[] lineData,
                                                boolean updateRepeat) {
    	log.info("^************************** lineData: "+lineData);
        Preconditions.checkArgument(orgId != null && orgId > 0, "illegal orgId.");
        Preconditions.checkArgument(lineData != null && lineData.length > 0, "lineData 为空");
        // 保存一行数据
        try {
            StudentListResponseDto studentInfo = new StudentListResponseDto();
            buildOrgStudentDto(studentInfo, lineData, headers, updateRepeat,orgId);
            //wrapDto(studentInfo);
            //CRM1.6需求 不设置默认跟进时间
//            Date nextRemindTime = new Date(); // 见 http://bug.baijiahulian.com/issues/42923
//            nextRemindTime = new Date(nextRemindTime.getTime()-5*60*1000);
//            studentInfo.setNextRemindTimeStr(nextRemindTime);
//            studentInfo.setNextRemindTime(nextRemindTime.getTime());
            
            if(StringUtils.isNotBlank(studentInfo.getTagsStr())){
            	String[] tags = studentInfo.getTagsStr().replaceAll("，", ",").split(",");
            	Map<String,String> map = new LinkedHashMap<String,String>();
            	for(String t:tags){
            		if(StringUtils.isNotBlank(t)){
            			map.put(t, "");
            		}
            	}
            	StringBuilder builder = new StringBuilder();
            	for(String t:map.keySet()){
            		builder.append(t).append(",");
            	}
            	studentInfo.setTagsStr( builder.toString().substring(0,builder.length()-1));
            }
            
            studentUserService.addStudentInfo(orgId, cascadeId==null?0:cascadeId.intValue(),studentInfo, updateRepeat);
            
        } catch (BussinessException ex) {
            if (ex.getErrorCode().getSubsystemErrorCode() == CrmErrorCode.STUDENT_HAS_EXISTS.getSubsystemErrorCode()) {
            	return SingleSaveErrorResult.createFailResult("学员已存在");
            } else {
                log.warn("save clue error:{}", ex);
                //throw ex;
                return SingleSaveErrorResult.createFailResult(ex.getMessage());
            }
        } catch (Exception ex) {
            log.warn("save clue error:{}", ex);
            //throw new BussinessException(CrmErrorCode.HANDLER_FAILED);
            return IMPORT_FAIL;
        }
        return IMPORT_SUCCESS;
    }


    private void buildOrgStudentDto(StudentListResponseDto dto, Object[] lineData, List<String> headers, boolean updateRepeat, Long orgId) {
        Preconditions.checkArgument(lineData.length == headers.size(), "数据数目和表头不匹配");
        Preconditions.checkArgument(validateHeader(headers), "表头格式错误:" + headers.toString());
        
        SimpleDateFormat dateFormat = new SimpleDateFormat("EEE MMM d HH:mm:ss 'CST' yyyy", Locale.ENGLISH);
        
        for (int i = 0; i < lineData.length; i++) {
            String fieldName = cache.get(headers.get(i)).get("propName").toString();
            
            if (fieldName != null && fieldName.equals("birthday") && lineData[i]!=null && !lineData[i].equals("")){
        		try{
        			Date date = dateFormat.parse(lineData[i].toString());
        			dto.setBirthdayStr( dateFormat.parse(lineData[i].toString()) );
        			dto.setBirthday(date.getTime());
        			lineData[i] = lineData[i].toString();
        			break;
        		}catch(Exception e){
        		}
            }
        }
        
        for (int i = 0; i < lineData.length; i++) {
            String fieldName = cache.get(headers.get(i)).get("propName").toString();
            boolean empty = Boolean.parseBoolean(cache.get(headers.get(i)).get("empty").toString());
            String name = headers.get(i);
            Object value = lineData[i];
            if (fieldName != null) {
            	if(fieldName.equals("birthday") && lineData[i]!=null && !lineData[i].equals("")){
            		try{
            			Date date = dateFormat.parse(lineData[i].toString());
            			dto.setBirthdayStr( dateFormat.parse(lineData[i].toString()) );
            			dto.setBirthday(date.getTime());
            			lineData[i] = lineData[i].toString();
            			continue;
            		}catch(Exception e){
            			throw new BussinessException(CommonErrorCode.PARAM_ERROR, "【生日】格式有误，需为2001/1/1或者2001-1-1样式");
            		}
            	}
            	
                if (empty && (value == null || "".equals(value))) {
                    throw new BussinessException(CrmErrorCode.FORMAT_ERROR, "【" + name + "】不能为空");
                }
                try {
                    if (value != null && !(value instanceof String && StringUtils.isBlank((String) value))) {
                    	if(value instanceof String){
                    		value = value.toString().trim();
                    	}
                        BeanUtils.setProperty(dto, fieldName, value);
                    }
                } catch (IllegalAccessException | InvocationTargetException e) {
                    log.warn("set StudentListResponseDto.{} = {} failed!", fieldName, value);
                    throw new BussinessException(CrmErrorCode.FORMAT_ERROR);
                }
            }
        }
        //1 姓名
        if( StringUtils.isBlank(dto.getName()) ){
    		throw new BussinessException(CommonErrorCode.PARAM_ERROR, "【姓名】不能为空");
    	}
        if( dto.getName().length()>30 ){
    		throw new BussinessException(CommonErrorCode.PARAM_ERROR, "【姓名】最多输入30个字符");
    	}
//        if( !TxConsultUser.validateStr(dto.getName())){
//        	throw new BussinessException(CommonErrorCode.PARAM_ERROR, "【姓名】文本包含特殊字符，请输入中英文、数字、括号、下划线、中划线");
//        }
    	
    	
    	//2 手机号
    	if( StringUtils.isBlank(dto.getMobile()) ){
    		throw new BussinessException(CommonErrorCode.PARAM_ERROR, "【手机号】不能为空");
    	}
    	if( dto.getMobile().length()!=11 ){
    		throw new BussinessException(CommonErrorCode.PARAM_ERROR, "【手机号】需为11位数字");
    	}
    	if( !dto.getMobile().startsWith("1")){
    		throw new BussinessException(CommonErrorCode.PARAM_ERROR, "【手机号】必须以1开头");
    	}
    	if( !ParamValidateUtils.validateMobile(dto.getMobile())){
    		throw new BussinessException(CommonErrorCode.PARAM_ERROR, "【手机号】不能包含特殊字符或空格");
    	}
    	
    	//3 性别
    	if(StringUtils.isNotBlank(dto.getGenderStr())){
    		if ( dto.getGenderStr().equals("男")) {
            	dto.setGender(0L);
            } else if ( dto.getGenderStr().equals("女")) {
            	dto.setGender(1L);
            }else{
            	throw new BussinessException(CommonErrorCode.PARAM_ERROR, "【性别】所填内容与系统选项不匹配");
            }
    	}else{
    		dto.setGender(-1L);
    	}
        
    	
       
    	//7 标签
    	if(StringUtils.isNotBlank(dto.getTagsStr())){
    		String[] tags = dto.getTagsStr().split("[,|，]");
    		Map<String,String> map = new LinkedHashMap<String,String>();
        	for(String t:tags){
        		if(StringUtils.isNotBlank(t)){
        			map.put(t, "");
        		}
        	}
    		if(map.keySet().size() > 30){
    			throw new BussinessException(CommonErrorCode.PARAM_ERROR, "【标签】最多输入30个标签");
    		}
    		for(String tag:map.keySet()){
//    			if(tag.equals("") || !TxConsultUser.validateStr(tag)){
//    				throw new BussinessException(CommonErrorCode.PARAM_ERROR, "【标签】多个标签用逗号进行间隔，不能包含其他特殊字符或空格");
//    			}else 
    			if(tag.length()>15){
    				throw new BussinessException(CommonErrorCode.PARAM_ERROR, "【标签】单个标签最多输入15个字符");
    			}
    		}
    	}
    	
    	//8 亲属关系
    	if(StringUtils.isNotBlank(dto.getRelationshipStr())){
	    	if( Relatives.getValue(dto.getRelationshipStr()) == null){
	    		throw new BussinessException(CommonErrorCode.PARAM_ERROR, "【亲属关系】所填内容与系统选项不匹配");
	    	}else{
	    		dto.setRelationship(Relatives.getValue(dto.getRelationshipStr()));
	    	}
    	}
    	
    	//9 家长姓名
    	if( StringUtils.isNotBlank(dto.getParentName()) ){
    		if( dto.getParentName().length()>30 ){
        		throw new BussinessException(CommonErrorCode.PARAM_ERROR, "【家长姓名】最多输入30个字符");
        	}
//            if( !TxConsultUser.validateStr(dto.getParentName())){
//            	throw new BussinessException(CommonErrorCode.PARAM_ERROR, "【家长姓名】文本包含特殊字符，请输入中英文、数字、括号、下划线、中划线");
//            }
    	}
    	
    	//10 家长手机号
    	if( StringUtils.isNotBlank(dto.getParentMobile()) ){
    		if( StringUtils.isBlank(dto.getParentMobile()) ){
        		throw new BussinessException(CommonErrorCode.PARAM_ERROR, "【家长手机号】不能为空");
        	}
        	if( dto.getParentMobile().length()!=11 ){
        		throw new BussinessException(CommonErrorCode.PARAM_ERROR, "【家长手机号】需为11位数字");
        	}
        	if( !dto.getParentMobile().startsWith("1")){
        		throw new BussinessException(CommonErrorCode.PARAM_ERROR, "【家长手机号】必须以1开头");
        	}
        	if( !ParamValidateUtils.validateMobile(dto.getParentMobile())){
        		throw new BussinessException(CommonErrorCode.PARAM_ERROR, "【家长手机号】不能包含特殊字符或空格");
        	}
    	}
    	

    	
    	//12 学校
    	if( StringUtils.isNotBlank(dto.getSchool()) ){
    		if( dto.getSchool().length()>30 ){
        		throw new BussinessException(CommonErrorCode.PARAM_ERROR, "【公立学校】最多输入30个字符");
        	}
//            if( !TxConsultUser.validateStr(dto.getSchool())){
//            	throw new BussinessException(CommonErrorCode.PARAM_ERROR, "【公立学校】文本包含特殊字符，请输入中英文、数字、括号、下划线、中划线");
//            }
    	}
    	
    	//13 年级
    	if( StringUtils.isNotBlank(dto.getDegreeClass()) ){
    		if( dto.getDegreeClass().length()>30 ){
        		throw new BussinessException(CommonErrorCode.PARAM_ERROR, "【年级】最多输入30个字符");
        	}
//            if( !TxConsultUser.validateStr(dto.getDegreeClass())){
//            	throw new BussinessException(CommonErrorCode.PARAM_ERROR, "【年级】文本包含特殊字符，请输入中英文、数字、括号、下划线、中划线");
//            }
    	}
    	
    	//14 qq
    	if( StringUtils.isNotBlank(dto.getQq()) ){
    		if( dto.getQq().length()>30 ){
        		throw new BussinessException(CommonErrorCode.PARAM_ERROR, "【QQ】最多输入30个字符");
        	}
    		if( !Pattern.matches("^[0-9]*$", dto.getQq())){
            	throw new BussinessException(CommonErrorCode.PARAM_ERROR, "【QQ】需填写数字");
            }
    	}
    	
    	//15 邮箱
    	if( StringUtils.isNotBlank(dto.getMail()) ){
    		if(dto.getMail().length()>50){
    			throw new BussinessException(CommonErrorCode.PARAM_ERROR, "【邮箱】最多输入50个字符");
    		}
    		if( !Pattern.matches("^([a-z0-9A-Z]+[-|_|\\.]?)+[a-z0-9A-Z]@([a-z0-9A-Z]+(-[a-z0-9A-Z]+)?\\.)+[a-zA-Z]{2,}$", dto.getMail())){
            	throw new BussinessException(CommonErrorCode.PARAM_ERROR, "【邮箱】格式不正确");
            }
    	}
    	
        //16 省、市、区
        String province = dto.getProvince();
        String city = dto.getCity();
        String county = dto.getCounty();

        String address = dto.getAddress();
        String region = "";
        int level = 0;
        if (StringUtils.isNotBlank(county)) {
            region = county;
            level = AreaUtils.AreaLevel.COUNTY.ordinal();
        } else if (StringUtils.isNotBlank(city)) {
            region = city;
            level = AreaUtils.AreaLevel.CITY.ordinal();
        } else if (StringUtils.isNotBlank(province)) {
            region = province;
            level = AreaUtils.AreaLevel.PROVINCE.ordinal();
        }
        
        int areaId = this.getAreaId(region, level);
        Location location;
		try {
			location = BaiduMapUtils.getLngAndLat(address, region);
			if (location != null) {
	        	dto.setLatitude(location.getLat());
	        	dto.setLongitude(location.getLng());
	        }
	        dto.setAreaId(areaId);
		} catch (UnsupportedEncodingException e) {
			e.printStackTrace();
		}
		
		if (!updateRepeat) {
            OrgStudent orgStudent =
                    this.orgStudentDao.getStudentByMobileAndName(orgId, dto.getMobile(), dto.getName());
            if (orgStudent != null) {
            	throw new BussinessException(CrmErrorCode.CUSTOM_HAS_EXISTS, "【冲突】已存在名称、手机号相同的学员");
            }
        }
		
		
    }


    /**
     * 封装areaId 、经纬度信息
     *
     * @param
     * @throws UnsupportedEncodingException
     */
    private void wrapDto(StudentListResponseDto student) throws UnsupportedEncodingException {
        String province = student.getProvince();
        String city = student.getCity();
        String county = student.getCounty();

        String address = student.getAddress();
        String region = "";
        int level = 0;
        if (StringUtils.isNotBlank(county)) {
            region = county;
            level = AreaUtils.AreaLevel.COUNTY.ordinal();
        } else if (StringUtils.isNotBlank(city)) {
            region = city;
            level = AreaUtils.AreaLevel.CITY.ordinal();
        } else if (StringUtils.isNotBlank(province)) {
            region = province;
            level = AreaUtils.AreaLevel.PROVINCE.ordinal();
        }
        int areaId = this.getAreaId(region, level);
        Location location = BaiduMapUtils.getLngAndLat(address, region);
        if (location != null) {
            student.setLatitude(location.getLat());
            student.setLongitude(location.getLng());
        }
        student.setAreaId(areaId);

        //男女
        if (student.getGenderStr() == null || student.getGenderStr().equals("")) {
            student.setGender(-1l);
        } else {
            if (student.getGenderStr().equals("女")) {
                student.setGender(1l);
            } else {
                student.setGender(0l);
            }
        }

        if (student.getRelationshipStr() != null) {
            student.setRelationship(Relatives.getValue(student.getRelationshipStr()));
        }
    }

    private int getAreaId(String region, int level) {
        if (StringUtils.isBlank(region) || level == 0) {
            return 0;
        }
        List<Area> areas = this.areaDao.getAreaByBname(region, level);
        if (CollectionUtils.isNotEmpty(areas)) {
            return areas.get(0).getId();
        }
        return 0;
    }

    @Override
    public BatchSaveResult batchImportDatas(Long orgId, Long cascadeId, List<String> headers, List<Object[]> datas,
                                            boolean updateRepeat) {
        BatchSaveResult batchSaveResult = new BatchSaveResult();
        for (int i = 0; i < datas.size(); i++) {
            Object[] data = datas.get(i);
            try {
                SingleSaveErrorResult singleSaveErrorResult = saveSingleData(orgId, cascadeId, headers, data, updateRepeat);
                if (singleSaveErrorResult.isRepeat()) {
                    batchSaveResult.getRepeatIndexs().add(i);
                }
            } catch (Throwable ex) {
                batchSaveResult.getFailIndexs().add(i);
            }
        }
        return batchSaveResult;
    }

    @Override
    public void afterComplete() {
        log.info("[Upload] upload complete!");
    }

	@Override
	public int[] getUniqueIndex() {
		return new int[]{0,1};
	}

	@Override
	public SingleSaveErrorResult validate(Long orgId, Long cascadeId, List<String> headers, SingleSaveErrorResult result,
										  List<SingleSaveErrorResult> dataList, boolean updateRepeat) {
		result.setSuccess(true);
		try{
			
			buildOrgStudentDto(new StudentListResponseDto(), result.getLineData(), headers, updateRepeat, orgId);
			
		}catch(BussinessException e){
			result.setSuccess(false);
			result.setErrorMsg(e.getMessage());
			log.warn(e.getMessage(),e);
		}catch(Exception e){
			log.error("",e);
			result.setSuccess(false);
			result.setErrorMsg(CommonErrorCode.SYSTEM_ERROR.getMessage());
		}
        
		return result;
	}

	@Override
	public void downloadValidateResult(OutputStream os, Long orgId, String taskId, Collection<SingleSaveErrorResult> data) {
		InputStream in = null;
		
		try{
			SimpleDateFormat dateFormat = new SimpleDateFormat("EEE MMM d HH:mm:ss 'CST' yyyy", Locale.ENGLISH);
			SimpleDateFormat dateFormat2 = new SimpleDateFormat("yyyy/MM/dd");
			in = Thread.currentThread().getContextClassLoader().getResourceAsStream(STUDENT_RESULT_TEMPLATE_PATH);
	        XSSFWorkbook workbook = new XSSFWorkbook(in);
	        XSSFSheet sheet = workbook.getSheetAt(1);
	        XSSFRow row = null;
	        int demoRowNo = 2;
	        int rowNo = demoRowNo;
	        int cellNo = 0;
	        Map<Integer,ExcelUtils.CellInfo> demoFormatMap = ExcelUtils.getCellInfoMap(sheet, demoRowNo, 0, 25);
	        Object value = null;
	        
	        
	        sheet.getRow(demoRowNo).createCell(0);
	        XSSFCellStyle rightCellStyle = sheet.getRow(demoRowNo).getCell(0).getCellStyle();
	        rightCellStyle.setWrapText(true);
	        rightCellStyle.setLocked(false);
	        rightCellStyle.setAlignment(HSSFCellStyle.ALIGN_LEFT);
	        ExcelUtils.fillCellStyleWithBGColor(rightCellStyle, new XSSFColor(Color.WHITE), CellStyle.SOLID_FOREGROUND);
			ExcelUtils.fillCellStyleWithFullBorder(rightCellStyle, CellStyle.BORDER_THIN, IndexedColors.GREY_25_PERCENT.getIndex());
			
			XSSFCellStyle errorCellStyle = (XSSFCellStyle)rightCellStyle.clone();
			ExcelUtils.fillCellStyleWithBGColor(errorCellStyle, new XSSFColor(Color.YELLOW), CellStyle.SOLID_FOREGROUND);
			
	        //1. 填充 错误数据
	        for(SingleSaveErrorResult result:data){
	        	if(result.isSuccess() == false){
	        		if(sheet.getRow(rowNo) == null){
		    			row = sheet.createRow(rowNo);
		    		}
	        		cellNo = 0;
	        		for(int i=0;i<=result.getLineData().length;i++){
	        			if(i==0){
	        				value = result.getErrorMsg();
	        			}else{
	        				value = result.getLineData()[i-1];
	        			}
	        			if(sheet.getRow(demoRowNo).getCell(cellNo) == null){
	        				sheet.getRow(demoRowNo).createCell(cellNo);
	        			}
	        			
	        			if(i==8 && result.getLineData()[i-1]!=null && !result.getLineData()[i-1].toString().equals("")){
	        				try{
		        				Date date = dateFormat.parse(result.getLineData()[i-1].toString());
		        				ExcelUtils.fillTheXSSFCellWithDateValue(sheet, rowNo,cellNo, ExcelUtils.decorateStyleWithDataFormatMap(cellNo,errorCellStyle,demoFormatMap), date);
		        				cellNo++;
		        				continue;
	        				}catch(Exception e){}
	        			}
	        			ExcelUtils.fillTheXSSFCellWithStringValue(sheet, rowNo,cellNo, errorCellStyle, value==null? "" : value.toString());
	        			
	        			cellNo++;
	        		}
	        		rowNo++;
	        	}
	        }
	        
	        //2. 填充 正确数据
	        Object rightValue = null;
	        for(SingleSaveErrorResult result:data){
	        	if(result.isSuccess() == true){
	        		if(sheet.getRow(rowNo) == null){
		    			row = sheet.createRow(rowNo);
		    		}
	        		cellNo = 0;
	        		for(int i=0;i<=result.getLineData().length;i++){
	        			if(i==0){
	        				rightValue = "";
	        			}else{
	        				rightValue = result.getLineData()[i-1];
	        			}
	        			if(sheet.getRow(demoRowNo).getCell(cellNo) == null){
	        				sheet.getRow(demoRowNo).createCell(cellNo);
	        			}
	        			
	        			if(i==8 && result.getLineData()[i-1]!=null && !result.getLineData()[i-1].toString().equals("")){
	        				try{
		        				Date date = dateFormat.parse(result.getLineData()[i-1].toString());
		        				ExcelUtils.fillTheXSSFCellWithDateValue(sheet, rowNo,cellNo, ExcelUtils.decorateStyleWithDataFormatMap(cellNo,rightCellStyle,demoFormatMap), date);
		        				cellNo++;
		        				continue;
	        				}catch(Exception e){}
	        			}
                    		
	        			ExcelUtils.fillTheXSSFCellWithStringValue(sheet, rowNo,cellNo, rightCellStyle, rightValue==null? "" : rightValue.toString());
	        			cellNo++;
	        		}
	        		rowNo++;
	        	}
	        }
	        
	        workbook.write(os);
		}catch(Exception e){
			log.error("",e);
			e.printStackTrace();
		}finally{
			try {
				if(os!=null){
					os.close();
				}
				if(in!=null){
					in.close();
				}
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
	}

	@Override
	public void downloadImportResult(OutputStream os, Long orgId, String taskId, Collection<SingleSaveErrorResult> data) {
		InputStream in = null;
		
		try{
			in = Thread.currentThread().getContextClassLoader().getResourceAsStream(STUDENT_RESULT_TEMPLATE_PATH);
	        XSSFWorkbook workbook = new XSSFWorkbook(in);
	        XSSFSheet sheet = workbook.getSheetAt(1);
	        
	        XSSFRow row = null;
	        int demoRowNo = 2;
	        int rowNo = demoRowNo;
	        int cellNo = 0;
	        Map<Integer,ExcelUtils.CellInfo> demoFormatMap = ExcelUtils.getCellInfoMap(sheet, demoRowNo, 0, 25);
	        Object value = null;
	        SimpleDateFormat dateFormat = new SimpleDateFormat("EEE MMM d HH:mm:ss 'CST' yyyy", Locale.ENGLISH);
	        
	        sheet.getRow(demoRowNo).createCell(0);
	        XSSFCellStyle errorCellStyle = sheet.getRow(demoRowNo).getCell(0).getCellStyle();
	        errorCellStyle.setWrapText(true);
	        errorCellStyle.setLocked(false);
	        errorCellStyle.setAlignment(HSSFCellStyle.ALIGN_LEFT);
	        ExcelUtils.fillCellStyleWithFullBorder(errorCellStyle, CellStyle.BORDER_THIN, IndexedColors.GREY_25_PERCENT.getIndex());
	        ExcelUtils.fillCellStyleWithBGColor(errorCellStyle, new XSSFColor(Color.YELLOW) ,CellStyle.SOLID_FOREGROUND);
			
	        
	        //1. 填充 错误数据
	        for(SingleSaveErrorResult result:data){
	        	if(result.isSuccess() == false){
	        		if(sheet.getRow(demoRowNo).getCell(cellNo) == null){
        				sheet.getRow(demoRowNo).createCell(cellNo);
        			}
	        		cellNo = 0;
	        		for(int i=0;i<=result.getLineData().length;i++){
	        			if(i==0){
	        				value = result.getErrorMsg();
	        			}else{
	        				value = result.getLineData()[i-1];
	        			}
	        			if(sheet.getRow(demoRowNo).getCell(cellNo) == null){
	        				sheet.getRow(demoRowNo).createCell(cellNo);
	        			}
	        			
	        			if(i==8 && result.getLineData()[i-1]!=null && !result.getLineData()[i-1].toString().equals("")){
	        				try{
		        				Date date = dateFormat.parse(result.getLineData()[i-1].toString());
		        				ExcelUtils.fillTheXSSFCellWithDateValue(sheet, rowNo,cellNo, ExcelUtils.decorateStyleWithDataFormatMap(cellNo,errorCellStyle,demoFormatMap), date);
		        				cellNo++;
		        				continue;
	        				}catch(Exception e){}
	        			}
	        			ExcelUtils.fillTheXSSFCellWithStringValue(sheet, rowNo,cellNo, errorCellStyle, value==null? "" : value.toString());
	        			cellNo++;
	        		}
	        		rowNo++;
	        	}
	        }
	        
	        workbook.write(os);
		}catch(Exception e){
			e.printStackTrace();
		}finally{
			try {
				if(os!=null){
					os.close();
				}
				if(in!=null){
					in.close();
				}
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
	}

	@Override
	public void downloadImportTemplate(OutputStream os) {
		InputStream in = null;
		
		try{
			in = Thread.currentThread().getContextClassLoader().getResourceAsStream(STUDENT_IMPORT_TEMPLATE_PATH);
	        XSSFWorkbook workbook = new XSSFWorkbook(in);
	        workbook.write(os);
		}catch(Exception e){
			e.printStackTrace();
		}finally{
			try {
				if(os!=null){
					os.close();
				}
				if(in!=null){
					in.close();
				}
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
	}

    @Override
    public String validateResult(List<SingleSaveErrorResult> list) {

        if(CollectionUtils.isNotEmpty(list)){
            Map<String,SingleSaveErrorResult> map = new HashMap<String,SingleSaveErrorResult>();
            String key = null;
            for(SingleSaveErrorResult s:list){
                key = String.format("%s_%s", s.getLineData()[0], s.getLineData()[1]);

                if(s.isSuccess() && map.get(key)!=null){

                    s.setSuccess(false);
                    s.setErrorMsg("【冲突】已存在名称、手机号相同的学员");
                }
                map.put(key, s);
            }
        }
        return null;
    }
	
}
