package com.baijia.tianxiao.sal.student.impl;

import com.baijia.tianxiao.constants.TianXiaoConstant;
import com.baijia.tianxiao.dal.enums.CourseTypeEnum;
import com.baijia.tianxiao.dal.org.dao.*;
import com.baijia.tianxiao.dal.org.po.*;
import com.baijia.tianxiao.dal.solr.constant.SolrConstant;
import com.baijia.tianxiao.dal.solr.enums.StudentLessonStatus;
import com.baijia.tianxiao.dal.solr.query.CrmStudentQuery;
import com.baijia.tianxiao.enums.StudentCourseStatus;
import com.baijia.tianxiao.redis.AbstractBaseRedisDao;
import com.baijia.tianxiao.sal.student.api.OrgStudentCourseService;
import com.baijia.tianxiao.sal.student.api.StudentLessonService;
import com.baijia.tianxiao.sal.student.enums.LessonStatus;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.util.ListUtil;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.solr.client.solrj.SolrServerException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataAccessException;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.io.IOException;
import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.Semaphore;

/**
 * Created by liuxp on 16/7/22.
 */
@Service
@Slf4j
public class StudentLessonServiceImpl extends AbstractBaseRedisDao<String, Object> implements StudentLessonService {

    private static final int PAGE_SIZE = 1000;
    private static final int UPDATE_TIME_TYPE = 1;
    private static final int START_TIME_TYPE = 2;
    private static final String TX_ORG_STU_LESSON_UPDATE_HSET = "#tx_org_stu_lesson_update_hset";

    @Autowired
    private OrgStudentDao studentDao;
    @Autowired
    private OrgClassLessonDao classLessonDao;
    @Autowired
    private OrgStudentLessonDao studentLessonDao;
    @Autowired
    private OrgStudentCourseDao orgStudentCourseDao;
    @Autowired
    private CrmStudentQuery studentQuery;
    @Autowired
    private OrgSubAccountDao subAccountDao;
    @Autowired
    private OrgCourseDao orgCourseDao;
    @Autowired
    private OrgStudentCourseService courseService;

    private static final Semaphore updateTimeStuLessonStatusSemaphore = new Semaphore(1);
    private static final Semaphore startTimeStuLessonStatusSemaphore = new Semaphore(1);
    private static final Semaphore updateSolrSemaphore = new Semaphore(1);

    private static final BigDecimal TO_CHARGE_VALUE = new BigDecimal("0.2");

    /**
     * 同步排课状态
     *
     * @param beforeMinutes
     */
    @Transactional
    public void syncClassLessonStatus(int beforeMinutes,int type) {
        log.info("[ClassLessonStatus] Start===============");
        long begin = System.currentTimeMillis();
        if(type==UPDATE_TIME_TYPE){
            boolean isUpdateTimeAllowed = updateTimeStuLessonStatusSemaphore.tryAcquire();
            if(!isUpdateTimeAllowed){
                log.info("[StudentLesson] Other thread is execute.");
                return;
            }
        }else {
            boolean isStartTimeAllowed = startTimeStuLessonStatusSemaphore.tryAcquire();
            if(!isStartTimeAllowed){
                log.info("[StudentLesson] Other thread is execute.");
                return;
            }
        }
        try {
            Calendar calendar = Calendar.getInstance();
            calendar.add(Calendar.MINUTE, -1 * beforeMinutes);
            PageDto pageDto = new PageDto();
            pageDto.setPageSize(PAGE_SIZE);
            Date now = new Date();
            List<OrgClassLesson> lessons = null;
            if (type == UPDATE_TIME_TYPE) {
                lessons = classLessonDao.getOrgClassLessonsByUpdateTime(calendar.getTime(), now, pageDto);//按跟新时间同步数据
            } else {
                lessons = classLessonDao.getOrgClassLessonsByStartTime(calendar.getTime(), now, pageDto);//按修改时间同步数据
            }
            while (lessons != null && lessons.size() > 0) {
                log.info("[ClassLessonStatus]Executing===============");
                updateClassLessonStatus(lessons);
                pageDto.setPageNum(pageDto.getPageNum() + 1);
                if (type == UPDATE_TIME_TYPE) {
                    lessons = classLessonDao.getOrgClassLessonsByUpdateTime(calendar.getTime(), now, pageDto);
                } else {
                    lessons = classLessonDao.getOrgClassLessonsByStartTime(calendar.getTime(), now, pageDto);
                }
            }
        }finally {
            if(type==UPDATE_TIME_TYPE){
                updateTimeStuLessonStatusSemaphore.release();
            }else {
                startTimeStuLessonStatusSemaphore.release();
            }
        }
        log.info("[ClassLessonStatus] cost={}",(System.currentTimeMillis()-begin));
    }

    private void updateClassLessonStatus(List<OrgClassLesson> lessons){
        Date now = new Date();
        Set<Long> unStartIds = new HashSet<>();
        Set<Long> finishedIds = new HashSet<>();
        for (OrgClassLesson lesson : lessons) {
            // 已到开课时间
            if (lesson.getStartTime().compareTo(now) <= 0) {
                finishedIds.add(lesson.getId());
            }else {
                unStartIds.add(lesson.getId());
            }
        }
        if (unStartIds.size() > 0) {
            studentLessonDao.batchUpdateStatus(unStartIds, LessonStatus.UN_START.getStatus());
        }
        if (finishedIds.size() > 0) {
            studentLessonDao.batchUpdateStatus(finishedIds, LessonStatus.FINISHED.getStatus());
        }
    }

    @Override
    @Transactional
    public void syncStudentStatus(long orgId) {
        log.info("[SyncStudentStatus] Start===============");
        long begin = System.currentTimeMillis();
        PageDto pageDto = new PageDto();
        pageDto.setPageSize(PAGE_SIZE);
        List<OrgStudentLessonCounter> list = studentLessonDao.getStudentLessonCounterList(orgId, pageDto);
        if(list==null || list.size()<1){
            studentDao.updateStudentStatusByOrgId(orgId, StudentLessonStatus.STUDYING.getStatus());
        }

        List<Long> allCourseIds =  ListUtil.toKeyList(list, "courseId", OrgStudentLessonCounter.class);
        Set<Long> oneToOneCourseIds = new HashSet<>();//1对1
        Set<Long> classCourseIds = new HashSet<>();//班课
        courseService.splitCourseIdsByType(allCourseIds,oneToOneCourseIds,classCourseIds);

        Long lastId = null;
        StudentLessonCount lastStuLessonCount = null;
        boolean isToCharge = false;// 是否是待续费学员
        while (list.size() > 0) {
            log.info("[StudentLesson] sync data size={},orgId={}",list.size(),orgId);
            Set<Long> toChargeIds = new HashSet<>();// 待续费
            Set<Long> pastIds = new HashSet<>();// 往期
            Set<Long> unLessonIds = new HashSet<>();// 未排课
            Set<Long> studyingIds = new HashSet<>();// 在读
            Map<Long, StudentLessonCount> countMap = new HashMap<>();
            if (lastId != null) {
                countMap.put(lastId, lastStuLessonCount);
            }

            //1v1课按合同课次计算总课次
            for (OrgStudentLessonCounter counter : list) {
                if(oneToOneCourseIds.contains(counter.getCourseId())){
                    OrgStudentCourse course = orgStudentCourseDao.getStudentCourse(counter.getOrgId(), counter.getCourseId(), counter.getUserId());
                    if(course!=null){
                        if(course.getStatus()== StudentCourseStatus.NORMAL.getCode()) {
                            counter.setTotalCount(Math.max(course.getLessonCount(), counter.getTotalCount()));
                        }else {
                            counter.setTotalCount(counter.getFinishedCount());
                        }
                    }
                }
            }

            isToCharge = false;
            for (OrgStudentLessonCounter counter : list) {
                StudentLessonCount count = countMap.get(counter.getUserId());
                if (count == null) {
                    count = new StudentLessonCount();
                    countMap.put(counter.getUserId(), count);
                }
                count.add(counter);
                if (counter.getLeftCount()>0 && counter.getLeftCountRatio().compareTo(TO_CHARGE_VALUE) < 0) {// 待续费
                    toChargeIds.add(counter.getUserId());
                    isToCharge = true;
                }
                lastId = counter.getUserId();
                lastStuLessonCount = count;
            }

            // 每页的最后一条数据与下一页数据一起处理
            toChargeIds.remove(lastId);
            countMap.remove(lastId);

            for (Long uerId : countMap.keySet()) {
                StudentLessonCount count = countMap.get(uerId);
                log.info("[StudentLesson] userId={},courseIds={}", uerId,count.getCourseIds());
                if (count.getFinish() == count.getTotal() && count.getTotal()>0) {
                    pastIds.add(uerId);
                }
            }

            studyingIds = countMap.keySet();

            Map<Long, List<Long>> stuLessonMap = getStuCourseId(pastIds, orgId);
            log.info("[StudentLesson] pass studentId enroll map={}", stuLessonMap);
            for (Long uerId : pastIds) {
                Set<Long> lessonCourseIds = countMap.get(uerId).getCourseIds();// 所有排课的课程Id
                List<Long> enrollCourseId = stuLessonMap.get(uerId);// 所有报名的课程ID
                if(enrollCourseId==null){
                    log.warn("[StudentLesson] uerId {} no enroll record",uerId);
                    continue;
                }
                enrollCourseId.removeAll(lessonCourseIds);
                if (enrollCourseId.size() > 0) {
                    List<OrgCourse> courses = orgCourseDao.getNormalCourseList(enrollCourseId);
                    if(courses!=null && courses.size()>0) {
                        unLessonIds.add(uerId);
                    }
                }
            }
            // 删除往期学员中有未排课课程的学员
            pastIds.removeAll(unLessonIds);
            // 在读等于所有减往期
            studyingIds.removeAll(pastIds);
            // 移除待续费
            studyingIds.removeAll(toChargeIds);
            log.info("[StudentLesson] toChargeIds={},pastIds={},studyingIds={},lastId={}",
                    toChargeIds, pastIds, studyingIds, lastId);
            if(toChargeIds.size()>0) {
                studentDao.batchUpdateStudentStatus(toChargeIds, StudentLessonStatus.TO_CHARGE.getStatus());
            }
            if(pastIds.size()>0) {
                studentDao.batchUpdateStudentStatus(pastIds, StudentLessonStatus.PAST.getStatus());
            }
            if(studyingIds.size()>0) {
                studentDao.batchUpdateStudentStatus(studyingIds, StudentLessonStatus.STUDYING.getStatus());
            }

            pageDto.setPageNum(pageDto.getPageNum() + 1);
            list = studentLessonDao.getStudentLessonCounterList(orgId, pageDto);
        }

        if (lastId != null) {
            StudentLessonStatus status = null;
            if (isToCharge) {
                status = StudentLessonStatus.TO_CHARGE;
            } else {
                if (lastStuLessonCount.getTotal() == lastStuLessonCount.getFinish()) {
                    List<Long> courseIds = orgStudentCourseDao.getStudentCourseIds(orgId, lastId, null);
                    courseIds.removeAll(lastStuLessonCount.getCourseIds());
                    if (courseIds.size() == 0) {
                        status = StudentLessonStatus.PAST;
                    } else {
                        status = StudentLessonStatus.STUDYING;
                    }
                } else {
                    status = StudentLessonStatus.STUDYING;
                }
            }
            studentDao.batchUpdateStudentStatus(Arrays.asList(lastId),status.getStatus());
        }
        log.info("[ClassLessonStatus] Org({}) Update finished.cost={}",orgId,(System.currentTimeMillis()-begin));
    }

    private Map<Long, List<Long>> getStuCourseId(Collection<Long> stuIds, Long orgId) {
        if (stuIds == null || stuIds.size() < 1) {
            return Collections.EMPTY_MAP;
        }
        return orgStudentCourseDao.getCourseIdMapByStuIds(stuIds, orgId);
    }

    @Data
    private class StudentLessonCount {
        private int finish = 0;
        private int total = 0;
        private Set<Long> courseIds = new HashSet<>();

        public void add(OrgStudentLessonCounter counter) {
            finish += counter.getFinishedCount();
            total += counter.getTotalCount();
            courseIds.add(counter.getCourseId());
        }
    }

    @Override
    public List<Integer> getAllTxOrgIds() {
        List<OrgSubAccount> accounts = subAccountDao.getAll("orgId");
        List<Integer> list = ListUtil.toKeyList(accounts, "orgId", OrgSubAccount.class);
        return list;
    }

    public void setUpdateTime(final String updateTime) {
        redisTemplate.execute(new RedisCallback<Integer>() {
            @Override
            public Integer doInRedis(RedisConnection connection) throws DataAccessException {
                connection.select(TianXiaoConstant.TX_PV_REDIS_DB);
                RedisSerializer<String> serializer = getRedisSerializer();
                byte[] keyBytes = serializer.serialize(TX_ORG_STU_LESSON_UPDATE_HSET);
                byte[] value = serializer.serialize(updateTime);
                connection.set(keyBytes, value);
                log.info("[Redis] Set student lesson updateTime={}", updateTime);
                return null;
            }
        });
    }

    public Date getUpdateTime() {
        String dateStr = redisTemplate.execute(new RedisCallback<String>() {
            @Override
            public String doInRedis(RedisConnection connection) throws DataAccessException {
                connection.select(TianXiaoConstant.TX_PV_REDIS_DB);
                RedisSerializer<String> serializer = getRedisSerializer();
                byte[] keyBytes = serializer.serialize(TX_ORG_STU_LESSON_UPDATE_HSET);
                byte[] value = connection.get(keyBytes);
                String updateTime = serializer.deserialize(value);
                log.info("[Redis] Get student lesson updateTime={}", updateTime);
                return updateTime;
            }
        });
        if(StringUtils.isNotBlank(dateStr)){
            return new Date(Long.parseLong(dateStr));
        }else {
            return null;
        }
    }

    @Override
    public void updateStudentLesson() {
        boolean isAllowed = updateSolrSemaphore.tryAcquire();
        if(!isAllowed){
            log.info("[Solr] Other thread is execute.");
            return;
        }
        try {
            Date date = getUpdateTime();
            long currentTime = new Date().getTime();
            if (date == null) {
                Calendar calendar = Calendar.getInstance();
                calendar.add(Calendar.MINUTE, -30);
                date = calendar.getTime();
            }
            PageDto pageDto = new PageDto();
            pageDto.setPageSize(100);
            List<OrgStudentLesson> studentLessons = studentLessonDao.getStudentLessonsByPage(date, pageDto);
            while (studentLessons != null && studentLessons.size() > 0) {
                Map<Long, List<Long>> map = new HashMap<>();
                for (OrgStudentLesson lesson : studentLessons) {
                    List<Long> userIds = map.get(lesson.getLessonId());
                    if (userIds == null) {
                        userIds = new ArrayList<>();
                        map.put(lesson.getLessonId(), userIds);
                    }
                    userIds.add(lesson.getUserId());
                }
                log.info("[Solr] update lesson and userId={}", map);
                List<OrgClassLesson> classLessons = classLessonDao.getByIds(map.keySet());
                if (classLessons != null) {
                    for (OrgClassLesson classLesson : classLessons) {
                        List<Long> userIds = map.get(classLesson.getId());
                        if (userIds != null && userIds.size() > 0) {
                            for (Long userId : userIds) {
                                OrgStudentLessonCounter counter = studentLessonDao.getStudentLessonCounter(classLesson.getOrgId().longValue(), userId, classLesson.getCourseId());
                                if (counter != null) {
                                    try {
                                        studentQuery.add(SolrConstant.CRM_STUDENT_COLLECTION, toMap(counter));
                                    } catch (SolrServerException e) {
                                        log.error("[Solr] SolrServerException", e);
                                    } catch (IOException e) {
                                        log.error("[Solr] IOException", e);
                                    }
                                }
                            }
                        }
                    }
                }
                pageDto.setPageNum(pageDto.getPageNum() + 1);
                studentLessons = studentLessonDao.getStudentLessonsByPage(date, pageDto);
            }
            setUpdateTime(String.valueOf(currentTime));
        }finally {
            updateSolrSemaphore.release();
        }
    }

    private Map<String,Object> toMap(OrgStudentLessonCounter counter){
        Map<String,Object> map = new HashMap<>();
        map.put("id",counter.getId());
        map.put("org_id",counter.getOrgId());
        map.put("student_id",counter.getUserId());
        map.put("course_id",counter.getCourseId());
        map.put("total",counter.getTotalCount());
        map.put("finished",counter.getFinishedCount());
        map.put("left_count",counter.getLeftCount());
        return map;
    }
}