package com.baijia.tianxiao.sal.student.impl;

import com.baijia.tianxiao.constant.Flag;
import com.baijia.tianxiao.dal.org.constant.DeleteStatus;
import com.baijia.tianxiao.dal.roster.constant.RecordTemplateType;
import com.baijia.tianxiao.dal.roster.dao.TxCustomRecordTemplateDao;
import com.baijia.tianxiao.dal.roster.dao.TxStudentCommentDao;
import com.baijia.tianxiao.dal.roster.po.TxCustomRecordTemplate;
import com.baijia.tianxiao.dal.roster.po.TxStudentComment;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.sal.organization.constant.ClientType;
import com.baijia.tianxiao.sal.organization.org.service.TxAccountModuleAuthService;
import com.baijia.tianxiao.sal.student.api.TxCustomRecordTemplateService;
import com.baijia.tianxiao.sal.student.dto.TxCustomRecordTemplateDto;
import com.baijia.tianxiao.util.exception.BussinessPreconditions;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Date;
import java.util.List;
import java.util.Map;

import lombok.extern.slf4j.Slf4j;

/**
 * @author weihongyan
 * @implNote <(▰˘◡˘▰)>
 * @since 02/08/2017 4:45 PM
 * @see com.baijia.tianxiao.sal.student.dto.TxCustomRecordTemplateDto
 */
@Slf4j
@Service
@Transactional
public class TxCustomRecordTemplateServiceImpl implements TxCustomRecordTemplateService {

    @Autowired
    private TxCustomRecordTemplateDao txCustomRecordTemplateDao;

    @Autowired
    private TxStudentCommentDao txStudentCommentDao;

    @Autowired
    private TxAccountModuleAuthService txAccountModuleAuthService;

    @Override
    public List<TxCustomRecordTemplateDto> listTemplates(Long orgId, Integer disable) {
        List<TxCustomRecordTemplateDto> result = Lists.newArrayList();
        if (!txAccountModuleAuthService.hasClientType(orgId, ClientType.CUSTOM_RECORD_TEMPLATE)) {
            return result;
        }
        List<TxCustomRecordTemplate> templates = txCustomRecordTemplateDao.getTemplatesByOrgId(orgId, disable);
        for (TxCustomRecordTemplate template : templates) {
            result.add(transform(template));
        }
        return result;
    }

    private TxCustomRecordTemplateDto transform(TxCustomRecordTemplate template) {
        TxCustomRecordTemplateDto dto = new TxCustomRecordTemplateDto();
        dto.setTemplateId(template.getId());
        dto.setName(template.getName());
        dto.setType(template.getType());
        dto.setDisable(template.getDisable());
        dto.setSort(template.getSort());
        return dto;
    }

    @Override
    public TxCustomRecordTemplateDto addTemplate(Long orgId, String name, Integer type) {
        BussinessPreconditions.checkArgument(
            txAccountModuleAuthService.hasClientType(orgId, ClientType.CUSTOM_RECORD_TEMPLATE), "无自定义记录模板权限");
        BussinessPreconditions.checkArgument(StringUtils.isNotBlank(name) && name.length() <= 16,
            CommonErrorCode.PARAM_ERROR, "自定义记录模板名称不能为空或者超过16个字符");
        String numStr = txAccountModuleAuthService.getModuleAuthValueStr(orgId, ClientType.CUSTOM_RECORD_TEMPLATE);
        Integer num = Integer.valueOf(numStr);
        BussinessPreconditions.checkArgument(txCustomRecordTemplateDao.countByOrgId(orgId) < num,
            "自定义记录不能超过%s个, 请删除一些记录项后再添加", num);
        BussinessPreconditions.checkArgument(RecordTemplateType.getByCode(type) != null, "type error!");

        TxCustomRecordTemplate template = new TxCustomRecordTemplate();
        template.setName(name);
        template.setOrgId(orgId);
        template.setType(type);
        template.setSort(txCustomRecordTemplateDao.getMaxSortByOrgId(orgId) + 1);
        template.setDisable(Flag.FALSE.getInt());
        template.setIsdel(DeleteStatus.NORMAL.getValue());
        template.setCreateTime(new Date());
        template.setUpdateTime(template.getCreateTime());
        txCustomRecordTemplateDao.save(template);

        return transform(template);
    }

    @Override
    public TxCustomRecordTemplateDto editTemplate(Long orgId, Long templateId, String name, Integer disable,
        Integer type) {
        BussinessPreconditions.checkArgument(
            txAccountModuleAuthService.hasClientType(orgId, ClientType.CUSTOM_RECORD_TEMPLATE), "无自定义记录模板权限");
        TxCustomRecordTemplate template = txCustomRecordTemplateDao.getById(templateId);
        BussinessPreconditions.checkArgument(null != template && orgId.longValue() == template.getOrgId(),
            "模板信息已经改变，请退出当前页面重试");

        if (null != name) {
            BussinessPreconditions.checkArgument(StringUtils.isNotBlank(name) && name.length() <= 16,
                CommonErrorCode.PARAM_ERROR, "自定义记录模板名称不能为空或者超过16个字符");
            template.setName(name);
        }

        if (null != disable) {
            BussinessPreconditions.checkArgument(disable == Flag.FALSE.getInt() || disable == Flag.TRUE.getInt(),
                "disable value error!");
            template.setDisable(disable);
            template.setSort(txCustomRecordTemplateDao.getMaxSortByOrgId(orgId) + 1);

        }
        
        if (null != type) {
            BussinessPreconditions.checkArgument(RecordTemplateType.getByCode(type) != null, "type error!");
            template.setType(type);
        }
        template.setUpdateTime(new Date());
        txCustomRecordTemplateDao.update(template, "name", "updateTime", "disable", "type", "sort");

        return this.transform(template);
    }

    @Override
    public void deleteByTemplateId(Long orgId, Long templateId) {
        BussinessPreconditions.checkArgument(
            txAccountModuleAuthService.hasClientType(orgId, ClientType.CUSTOM_RECORD_TEMPLATE), "无自定义记录模板权限");
        TxCustomRecordTemplate template = txCustomRecordTemplateDao.getById(templateId);
        BussinessPreconditions.checkArgument(null != template && orgId.longValue() == template.getOrgId(),
            "can not find comment!");
        template.setIsdel(DeleteStatus.DELETED.getValue());
        template.setUpdateTime(new Date());
        txCustomRecordTemplateDao.update(template, "isdel", "updateTime");
        log.info("delete template by id:{}, template:{}", templateId, template);

        List<TxStudentComment> commentList =
            txStudentCommentDao.getTemplateCommentHistory(orgId, templateId, null, null, null);
        commentList.parallelStream().forEach(txStudentComment -> {
            txStudentComment.setIsdel(DeleteStatus.DELETED.getValue());
            txStudentComment.setUpdateTime(new Date());
            txStudentCommentDao.update(txStudentComment, "isdel", "updateTime");
        });
        log.info("delete comments by templateId:{}, size:{}", templateId, commentList.size());
    }

    @Override
    public void batchSetSort(Long orgId, Map<Long, Integer> map) {
        BussinessPreconditions.checkArgument(
            txAccountModuleAuthService.hasClientType(orgId, ClientType.CUSTOM_RECORD_TEMPLATE), "无自定义记录模板权限");
        if (MapUtils.isEmpty(map)) {
            return;
        } else {
            BussinessPreconditions.checkArgument(map.size() == Sets.newHashSet(map.values()).size(),
                "map contains duplicated values!");
            List<TxCustomRecordTemplate> list = txCustomRecordTemplateDao.getByIds(map.keySet(), "id", "sort");
            Date updateTime = new Date();
            for (TxCustomRecordTemplate template : list) {
                Integer sort = map.get(template.getId());
                BussinessPreconditions.checkArgument(null != sort, CommonErrorCode.PARAM_ERROR,
                    "sort value can not be null!");
                if (sort.intValue() == template.getSort()) {
                    continue;
                }
                template.setSort(sort);
                template.setUpdateTime(updateTime);
                txCustomRecordTemplateDao.update(template, "sort", "updateTime");
            }
        }
    }

    @Override
    public TxCustomRecordTemplateDto getTemplate(Long orgId, Long templateId) {
        BussinessPreconditions.checkArgument(
            txAccountModuleAuthService.hasClientType(orgId, ClientType.CUSTOM_RECORD_TEMPLATE), "无自定义记录模板权限");
        TxCustomRecordTemplate template = txCustomRecordTemplateDao.getByTemplateId(templateId);
        BussinessPreconditions.checkArgument(null != template, "模板信息已经改变，请退出当前页面重试");
        return transform(template);
    }
}
