package com.baijia.tianxiao.sal.student.impl;

import com.baijia.tianxiao.constant.Flag;
import com.baijia.tianxiao.dal.org.constant.DeleteStatus;
import com.baijia.tianxiao.dal.roster.dao.TxCustomRecordKeyDao;
import com.baijia.tianxiao.dal.roster.po.TxCustomRecordKey;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.sal.organization.constant.ClientType;
import com.baijia.tianxiao.sal.organization.org.service.TxAccountModuleAuthService;
import com.baijia.tianxiao.sal.student.api.TxCustomRecordKeyService;
import com.baijia.tianxiao.sal.student.dto.TxCustomRecordKeyDto;
import com.baijia.tianxiao.util.exception.BussinessPreconditions;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.transaction.annotation.Transactional;

import java.util.Date;
import java.util.List;
import java.util.Map;

import lombok.extern.slf4j.Slf4j;

/**
 * @author weihongyan
 * @implNote <(▰˘◡˘▰)>
 * @since 02/08/2017 4:46 PM
 * @see com.baijia.tianxiao.sal.student.dto.TxCustomRecordKeyDto
 */
@Slf4j
@Controller
@Transactional
public class TxCustomRecordKeyServiceImpl implements TxCustomRecordKeyService {
    @Autowired
    private TxCustomRecordKeyDao txCustomRecordKeyDao;
    
    @Autowired
    private TxAccountModuleAuthService txAccountModuleAuthService;

    @Override
    public List<TxCustomRecordKeyDto> listKeys(Long orgId, Long templateId, Integer disable) {
        List<TxCustomRecordKeyDto> result = Lists.newArrayList();
        if (!txAccountModuleAuthService.hasClientType(orgId, ClientType.CUSTOM_RECORD_TEMPLATE)) {
            return result;
        }
        List<TxCustomRecordKey> keys = txCustomRecordKeyDao.getKeysByOrgId(orgId, templateId, disable);
        for (TxCustomRecordKey key : keys) {
            result.add(transform(key));
        }
        return result;
    }

    private TxCustomRecordKeyDto transform(TxCustomRecordKey key) {
        TxCustomRecordKeyDto dto = new TxCustomRecordKeyDto();
        dto.setKeyId(key.getId());
        dto.setKeyName(key.getKeyName());
        dto.setDisable(key.getDisable());
        dto.setSort(key.getSort());
        return dto;
    }

    @Override
    public TxCustomRecordKeyDto addKey(Long orgId, Long templateId, String keyName) {
        BussinessPreconditions.checkArgument(
            txAccountModuleAuthService.hasClientType(orgId, ClientType.CUSTOM_RECORD_TEMPLATE), "无自定义记录模板权限");
        BussinessPreconditions.checkArgument(StringUtils.isNotBlank(keyName) && keyName.length() <= 20,
            CommonErrorCode.PARAM_ERROR, "自定义记录字段名称不能为空或者超过20个字符");
        // TODO
        BussinessPreconditions.checkArgument(txCustomRecordKeyDao.countByOrgId(orgId, templateId) < 50,
            "自定义记录字段不能超过50个, 请删除一些字段项后再添加");

        TxCustomRecordKey key = new TxCustomRecordKey();
        key.setTemplateId(templateId);
        key.setKeyName(keyName);
        key.setOrgId(orgId);
        key.setSort(txCustomRecordKeyDao.getMaxSortByOrgId(orgId, templateId) + 1);
        key.setDisable(Flag.FALSE.getInt());
        key.setIsdel(DeleteStatus.NORMAL.getValue());
        key.setCreateTime(new Date());
        key.setUpdateTime(key.getCreateTime());
        txCustomRecordKeyDao.save(key);

        return transform(key);
    }

    @Override
    public TxCustomRecordKeyDto editKey(Long orgId, Long keyId, String keyName, Integer disable) {
        BussinessPreconditions.checkArgument(
            txAccountModuleAuthService.hasClientType(orgId, ClientType.CUSTOM_RECORD_TEMPLATE), "无自定义记录模板权限");
        TxCustomRecordKey key = txCustomRecordKeyDao.getById(keyId);
        BussinessPreconditions.checkArgument(null != key && orgId.longValue() == key.getOrgId(), "can not find key!");

        if (null != keyName) {
            BussinessPreconditions.checkArgument(StringUtils.isNotBlank(keyName) && keyName.length() <= 20,
                CommonErrorCode.PARAM_ERROR, "自定义记录字段名称不能为空或者超过20个字符");
            key.setKeyName(keyName);
        }

        if (null != disable) {
            BussinessPreconditions.checkArgument(disable == Flag.FALSE.getInt() || disable == Flag.TRUE.getInt(),
                "disable value error!");
            key.setDisable(disable);
            if (disable == Flag.FALSE.getInt() && disable.intValue() != key.getDisable()) {
                key.setSort(txCustomRecordKeyDao.getMaxSortByOrgId(orgId, key.getTemplateId()) + 1);
            }
        }
        
        key.setUpdateTime(new Date());
        txCustomRecordKeyDao.update(key, "keyName", "updateTime", "disable", "sort");

        return this.transform(key);
    }

    @Override
    public void deleteByKeyId(Long orgId, Long keyId) {
        TxCustomRecordKey key = txCustomRecordKeyDao.getById(keyId);
        BussinessPreconditions.checkArgument(null != key && orgId.longValue() == key.getOrgId(), "can not find key!");
        key.setIsdel(DeleteStatus.DELETED.getValue());
        key.setUpdateTime(new Date());
        txCustomRecordKeyDao.update(key, "isdel", "updateTime");
    }

    @Override
    public void batchSetSort(Long orgId, Map<Long, Integer> map) {
        BussinessPreconditions.checkArgument(
            txAccountModuleAuthService.hasClientType(orgId, ClientType.CUSTOM_RECORD_TEMPLATE), "无自定义记录模板权限");
        if (MapUtils.isEmpty(map)) {
            return;
        } else {
            BussinessPreconditions.checkArgument(map.size() == Sets.newHashSet(map.values()).size(),
                "map contains duplicated values!");
            List<TxCustomRecordKey> list = txCustomRecordKeyDao.getByIds(map.keySet(), "id", "sort");
            Date updateTime = new Date();
            for (TxCustomRecordKey key : list) {
                Integer sort = map.get(key.getId());
                BussinessPreconditions.checkArgument(null != sort, CommonErrorCode.PARAM_ERROR,
                    "sort value can not be null!");
                if (sort.intValue() == key.getSort()) {
                    continue;
                }
                key.setSort(sort);
                key.setUpdateTime(updateTime);
                txCustomRecordKeyDao.update(key, "sort", "updateTime");
            }
        }
    }
    
}
