/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */
package com.baijia.tianxiao.sal.student.impl;

import java.util.*;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.baijia.tianxiao.dal.org.constant.DeleteStatus;
import com.baijia.tianxiao.dal.org.constant.StudentType;
import com.baijia.tianxiao.dal.org.dao.OrgStudentDao;
import com.baijia.tianxiao.dal.org.po.OrgStudent;
import com.baijia.tianxiao.dal.roster.dao.TxConsultUserDao;
import com.baijia.tianxiao.dal.roster.dao.TxStudentTagDao;
import com.baijia.tianxiao.dal.roster.po.TxConsultUser;
import com.baijia.tianxiao.dal.roster.po.TxStudentTag;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.student.api.OrgStudentTagService;
import com.baijia.tianxiao.sal.student.constant.Biz;
import com.baijia.tianxiao.sal.student.dto.TagInfoDto;
import com.baijia.tianxiao.sal.student.dto.request.StudentCommenRequestDto;
import com.baijia.tianxiao.sal.student.dto.response.OrgTagListResopnseDto;
import com.baijia.tianxiao.sal.student.enums.StudentErrorCode;
import com.google.common.collect.Lists;

import lombok.extern.slf4j.Slf4j;

/**
 * @title StudentTagServiceImpl
 * @desc TODO
 * @author shanyu
 * @date 2015年12月4日
 * @version 1.0
 */
@Slf4j
@Service
public class OrgStudentTagServiceImpl implements OrgStudentTagService {

    @Autowired
    private OrgStudentDao orgStudentsDao;

    @Autowired
    private TxConsultUserDao txConsultUserDao;

    @Autowired
    private TxStudentTagDao txStudentTagDao;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public List<TagInfoDto> addStudentTag(List<TagInfoDto> tagDtos, Integer type, Long studentId, Long orgId) {
      //学员时studentId传的是学员id，consult学员studentId传的是consult id主键
        if (type == null || studentId == null || CollectionUtils.isEmpty(tagDtos)) {
            log.warn("add comments error: type ,studentId or tags is null!");
            throw new BussinessException(CommonErrorCode.PARAM_ERROR);
        }
        OrgStudent student = null;
        //校验
        if (type.intValue() == StudentType.ORG_STUDENTS.getCode()) {// 机构学员
            student = orgStudentsDao.getById(studentId);
            if (student == null || student.getOrgId().longValue() != orgId.longValue()) {
                throw new BussinessException(StudentErrorCode.STUDENT_NOT_EXIST);
            }
        } else {
            student = new OrgStudent();
            TxConsultUser user = txConsultUserDao.getById(studentId);
            if (user == null || user.getOrgId().longValue() != orgId.longValue()) {
                throw new BussinessException(StudentErrorCode.CONSULT_USER_NOT_EXIST);
            }
            //目的为了设置orgId
            student.setOrgId(user.getOrgId());
        }
        
        //校验标签个数是否大于30
        if (tagDtos.size() > Biz.MAX_TAGS) {
            throw new BussinessException(StudentErrorCode.MAX_TAGS);
        }
        
        List<TxStudentTag> tags = Lists.newArrayList();
        if(type == StudentType.ORG_STUDENTS.getCode()){
            tags = getTagsByTagDtos(tagDtos, type, student, null);
            this.txStudentTagDao.delTags(student.getUserId(), orgId);
        }else{
            //咨询学员studentId为consult主键
            tags = getTagsByTagDtos(tagDtos, type, student, studentId);
            this.txStudentTagDao.delTagsByConsulterId(studentId, orgId);
        }
        
        log.info("tags insert param:{}", tags);
        this.txStudentTagDao.saveAll(tags);
        List<TagInfoDto> result = Lists.newArrayList();
        for (TxStudentTag txStudentTag : tags) {
            TagInfoDto dto = new TagInfoDto();  
            dto.setContent(txStudentTag.getContent());
            dto.setTagId(txStudentTag.getId());
            result.add(dto);
        }
        return result;
    }
    
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void delStudentTag(StudentCommenRequestDto studentCommenRequestDto, Long orgId) {
        if (studentCommenRequestDto.getTagId() == null) {
            log.warn("del tag error: tagId is null!");
            throw new BussinessException(CommonErrorCode.PARAM_ERROR);
        }
        TxStudentTag tag = this.txStudentTagDao.getById(studentCommenRequestDto.getTagId());
        if (tag == null || tag.getOrgId().longValue() != orgId) {
            throw new BussinessException(StudentErrorCode.TAG_NOT_EXIST);
        }
        this.txStudentTagDao.delById(tag.getId());
    }

    /**
     * dto to po
     * 
     * @param dto
     * @param po
     */
    private void tagDto2Po(TagInfoDto dto, TxStudentTag po, Long consultId, Integer type, OrgStudent student) {
        if (type == StudentType.ORG_STUDENTS.getCode()) {
            po.setUserId(student.getUserId());
        } else {
            po.setConsultUserId(consultId);
        }
        po.setOrgId(student.getOrgId());
        po.setContent(dto.getContent());

    }

    @Override
    @Transactional(readOnly = true)
    public OrgTagListResopnseDto getTags(Integer type, Long studentId, Long orgId) {
        if (type == null || studentId == null) {
            log.warn("add comments error: type ,studentId is null!");
            throw new BussinessException(CommonErrorCode.PARAM_ERROR);
        }
        OrgStudent student = null;
        if (type.intValue() == StudentType.ORG_STUDENTS.getCode()) {// 机构学员
            student = orgStudentsDao.getById(studentId, "orgId", "userId");
            if (student == null || student.getOrgId().longValue() != orgId.longValue()) {
                throw new BussinessException(StudentErrorCode.STUDENT_NOT_EXIST);
            }
        } else {
            TxConsultUser user = txConsultUserDao.getById(studentId);
            if (user == null || user.getOrgId().longValue() != orgId.longValue()) {
                throw new BussinessException(StudentErrorCode.CONSULT_USER_NOT_EXIST);
            }
        }

        OrgTagListResopnseDto resopnse = new OrgTagListResopnseDto();
        Long userId = studentId;
        if (student != null) {
            userId = student.getUserId();
        }
        List<TxStudentTag> tags = this.txStudentTagDao.getTags(userId, orgId, type);
        List<TagInfoDto> tagsDto = Lists.newArrayList();
        for (TxStudentTag tag : tags) {
            TagInfoDto tagDto = new TagInfoDto();
            tagPo2Dto(tagDto, tag);
            tagsDto.add(tagDto);
        }
        resopnse.setTags(tagsDto);
        return resopnse;
    }

    /**
     * po to dto
     * 
     * @param dto
     * @param po
     */
    private void tagPo2Dto(TagInfoDto dto, TxStudentTag po) {
        dto.setTagId(po.getId());
        dto.setContent(po.getContent());
    }

    /**
     * 如果source中tx_student_tag.tag_id有与dest中tx_student_tag.tag_id重合的，删除
     * 
     * @param sourceConsultUser
     * @param destConsultUser
     * @param orgId
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void mergeTags(TxConsultUser sourceConsultUser, TxConsultUser destConsultUser, Long orgId) {
        Map<String, TxStudentTag> cache =
            this.txStudentTagDao.getAndCachedTags(destConsultUser.getId(), orgId, StudentType.CONSULT_USER.getCode());
        List<TxStudentTag> sourceTags =
            this.txStudentTagDao.getTags(sourceConsultUser.getId(), orgId, StudentType.CONSULT_USER.getCode());
        if (sourceTags != null && !sourceTags.isEmpty()) {
            for (TxStudentTag tag : sourceTags) {
                if (cache != null && !cache.isEmpty()) {
                    if (cache.containsKey(tag.getContent())) {
                        log.info("delete student tag:orgId:{}, sourceConsultUser:{}, destConsultUser:{}, tag:{}", orgId,
                            sourceConsultUser, destConsultUser, tag);
                        this.txStudentTagDao.delById(tag.getId());
                    } else {
                        log.info("update tag from tag:{}", tag);
                        tag.setConsultUserId(destConsultUser.getId());
                        tag.setUserId(destConsultUser.getUserId());
                        tag.setUpdateTime(new Date());
                        log.info("to --------- tag:{}", tag);
                        this.txStudentTagDao.update(tag, new String[] { "consultUserId", "userId", "updateTime" });
                    }
                } else {
                    log.info("update tag from tag:{}", tag);
                    tag.setConsultUserId(destConsultUser.getId());
                    tag.setUserId(destConsultUser.getUserId());
                    tag.setUpdateTime(new Date());
                    log.info("to --------- tag:{}", tag);
                    this.txStudentTagDao.update(tag, new String[] { "consultUserId", "userId", "updateTime" });
                }
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void batchHandleTags(Integer type, Long orgId, Long studentId, List<TagInfoDto> tags) {
        if (CollectionUtils.isEmpty(tags)) {
            return;
        }
        if (null == orgId || 0 == orgId || null == studentId || 0 == studentId) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR);
        }
        if (null == type
            || (type != StudentType.CONSULT_USER.getCode() && type != StudentType.ORG_STUDENTS.getCode())) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "学员类型错误");
        }

        OrgStudent orgStudent = null;
        // 学员id参数: 根据type代表consulterId或userId
        Long paramId = studentId;
        if (type == StudentType.ORG_STUDENTS.getCode()) {
            // 机构正式学员，则将studentId替换为userId
            orgStudent = this.orgStudentsDao.getById(studentId, "id", "userId", "delStatus", "orgId");
            if (null == orgStudent || orgStudent.getOrgId().longValue() != orgId.longValue()
                || DeleteStatus.DELETED.getValue() == orgStudent.getDelStatus().intValue()) {
                throw new BussinessException(StudentErrorCode.STUDENT_NOT_EXIST);
            } else {
                paramId = orgStudent.getUserId();
            }
        } else {
            orgStudent = new OrgStudent();
            TxConsultUser user = txConsultUserDao.getById(studentId);
            if (user == null || user.getOrgId().longValue() != orgId.longValue()) {
                throw new BussinessException(StudentErrorCode.CONSULT_USER_NOT_EXIST);
            }
            orgStudent.setOrgId(user.getOrgId());
        }

        for (TagInfoDto tagInfo : tags) {
            if (null != tagInfo.getTagId() && tagInfo.getTagId() > 0) {
                TxStudentTag tag = this.txStudentTagDao.getUniqueTag(tagInfo.getTagId(), orgId, type, paramId);
                if (null != tag) {
                    if (StringUtils.isNotBlank(tagInfo.getContent())) { // 修改
                        tag.setContent(tagInfo.getContent());
                        tag.setUpdateTime(new Date());
                        this.txStudentTagDao.update(tag, "content", "updateTime");
                    } else { // 删除
                        this.txStudentTagDao.delById(tagInfo.getTagId());
                    }
                }
            } else if (StringUtils.isNoneBlank(tagInfo.getContent())) { // 新增
                TxStudentTag tag = new TxStudentTag();
                this.tagDto2Po(tagInfo, tag, studentId, type, orgStudent);
                tag.setCreateTime(new Date());
                tag.setUpdateTime(new Date());
                this.txStudentTagDao.save(tag, false);
            }
        }
    }

	@Override
	public List<TagInfoDto> getHotTags(Long orgId, Integer listSize, Integer minTagTimes) {
		List<TagInfoDto> list = new ArrayList<TagInfoDto>();
		List<TxStudentTag> studentTags = txStudentTagDao.getHotTags(orgId, listSize, minTagTimes);
		
		if( !CollectionUtils.isEmpty(studentTags) ){
			TagInfoDto dto = null;
			for (TxStudentTag txStudentTag : studentTags) {
				dto = new TagInfoDto();
	            dto.setContent(txStudentTag.getContent());
	            dto.setTagId(-1L);
	            list.add(dto);
	        }
		}
		return list;
	}

	@Override
	public void delTagsByConsulterId(Long consulterId, Long orgId) {
        this.txStudentTagDao.delTagsByConsulterId(consulterId,orgId);
	}
	
	private List<TxStudentTag> getTagsByTagDtos(List<TagInfoDto> tagDtos, int type, OrgStudent orgStudent, Long consultId){
        List<TxStudentTag> tags = new ArrayList<>();
        for (TagInfoDto tagInfoDto : tagDtos) {
            String content = tagInfoDto.getContent();
            if (content.length() > 15) {
                throw new BussinessException(StudentErrorCode.MAX_TAG_CONTENT_LENGTH);
            }
            TxStudentTag tag = new TxStudentTag();
            tagDto2Po(tagInfoDto, tag, consultId, type, orgStudent);
            tags.add(tag);
        }
        return tags;
	}

    @Override
    public Map<Long, String> getTagsMap(Long orgId, Collection<Long> ids, StudentType type) {
        Map<Long, String> map = new HashMap<>();
        List<TxStudentTag> studentTags = txStudentTagDao.getTags(ids, orgId, type.getCode());
        if (CollectionUtils.isNotEmpty(studentTags)) {
            String tagsStr = null;
            for (TxStudentTag tag : studentTags) {
                Long crmId;
                if (type.getCode() == StudentType.CONSULT_USER.getCode()) {
                    crmId = tag.getConsultUserId();
                } else {
                    crmId = tag.getUserId();
                }
                if (StringUtils.isNotBlank(tag.getContent())) {
                    tagsStr = map.get(crmId);
                    if (StringUtils.isNotBlank(tagsStr)) {
                        tagsStr += ("," + tag.getContent());
                    } else {
                        tagsStr = new String();
                        tagsStr += tag.getContent();
                    }
                    map.put(crmId, tagsStr);
                }
            }
        }
        return map;
    }
}
