/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */
package com.baijia.tianxiao.sal.storage.service.impl;

import java.io.File;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.springframework.stereotype.Service;

import com.baijia.tianxiao.constants.MIMEType;
import com.baijia.tianxiao.dal.org.dao.OrgStorageDao;
import com.baijia.tianxiao.dal.org.po.OrgStorage;
import com.baijia.tianxiao.dal.storage.dao.StorageDao;
import com.baijia.tianxiao.dal.storage.po.Storage;
import com.baijia.tianxiao.dto.upload.UploadResult;
import com.baijia.tianxiao.dto.upload.UploadResult.UploadFile;
import com.baijia.tianxiao.sal.storage.dto.UploadResultDto;
import com.baijia.tianxiao.sal.storage.exception.UploadException;
import com.baijia.tianxiao.sal.storage.service.StorageService;
import com.baijia.tianxiao.util.file.FileUtil;
import com.baijia.tianxiao.util.storage.StorageUtil;
import com.baijia.tianxiao.util.upload.FileUploadUtils;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;

import lombok.extern.slf4j.Slf4j;

/**
 * @title StorageServiceImpl
 * @desc TODO
 * @author cxm
 * @date 2015年12月13日
 * @version 1.0
 */
@Service
@Slf4j
public class StorageServiceImpl implements StorageService {

    @Resource
    private OrgStorageDao orgStorageDao;

    @Resource
    private StorageDao storageDao;

    @Override
    public List<UploadResultDto> storage(Map<String, File> files, Long uid, boolean watermark, boolean saveToCdb) {
        Preconditions.checkArgument(MapUtils.isNotEmpty(files), "files is null");
        Preconditions.checkArgument(uid != null, "uid is null");
        UploadResult uploadResult = FileUploadUtils.uploadFiles(uid, files, watermark);
        if (CollectionUtils.isNotEmpty(uploadResult.getFiles())) {
            List<UploadResultDto> result = Lists.newArrayList();
            for (UploadFile uploadFile : uploadResult.getFiles()) {
                String extension = FileUtil.getFormatedExtension(uploadFile.getUrl());
                MIMEType mimeType = MIMEType.getType(extension);

                result.add(saveToStorage(uploadFile, mimeType, saveToCdb));
            }
            return result;
        }
        return Collections.emptyList();
    }
    @Override
    public List<UploadResultDto> storageNoCheckMIMEType(Map<String, File> files, Long uid) {
        Preconditions.checkArgument(MapUtils.isNotEmpty(files), "files is null");
        Preconditions.checkArgument(uid != null, "uid is null");
        UploadResult uploadResult = FileUploadUtils.uploadFiles(uid, files, false);
        if (CollectionUtils.isNotEmpty(uploadResult.getFiles())) {
            List<UploadResultDto> result = Lists.newArrayList();
            for (UploadFile uploadFile : uploadResult.getFiles()) {
                String extension = FileUtil.getFormatedExtension(uploadFile.getUrl());
                result.add(saveFileToStorage(uploadFile, extension));
            }
            return result;
        }
        return Collections.emptyList();
    }

    @Override
    public UploadResultDto storage(File file, Long uid, boolean watermark) {
        return storage(file, uid, watermark, false);
    }

    @Override
    public UploadResultDto storage(File file, Long uid, boolean watermark, boolean saveToCdb) {
        Preconditions.checkArgument(file != null, "file is null");
        Preconditions.checkArgument(uid != null, "uid is null");
        String extension = FileUtil.getFormatedExtension(file.getName());
        MIMEType mimeType = MIMEType.getType(extension);
        Preconditions.checkArgument(mimeType != null, "can not get mimetype from file %s", file.getName());

        UploadFile uploadFile = FileUploadUtils.uploadToRemote(uid, file, watermark);
        if (uploadFile == null) {
            log.warn("upload fail result is empty");
            throw new UploadException();
        }

        return saveToStorage(uploadFile, mimeType, saveToCdb);
    }

    private UploadResultDto saveToStorage(UploadFile uploadFile, MIMEType mimeType, boolean saveToCdb) {
        UploadResultDto result = new UploadResultDto();
        if (!saveToCdb) {
            OrgStorage storage = new OrgStorage();
            storage.setCreateTime(new Date());
            storage.setFid(uploadFile.getFid());
            storage.setSize(uploadFile.getSize());
            storage.setSn(uploadFile.getSn());
            storage.setMimeType(mimeType.getCode());
            storage.setUpdateTime(new Date());
            orgStorageDao.save(storage);
            log.info("success save upload result:{}", storage);
            result.setStorageId(storage.getId().longValue());
        } else {
            Storage storage = new Storage();
            storage.setFid(Long.parseLong(uploadFile.getFid()));
            storage.setSn(uploadFile.getSn());
            storage.setSize(uploadFile.getSize());
            storage.setMimetype(mimeType.getType());
            storageDao.save(storage);
            result.setStorageId(storage.getId());
        }
        result.setUrl(StorageUtil.constructUrl(uploadFile.getFid(), uploadFile.getSn(), mimeType));
        return result;
    }
    
    private UploadResultDto saveFileToStorage(UploadFile uploadFile, String mimeType) {
        UploadResultDto result = new UploadResultDto();

        Storage storage = new Storage();
        storage.setFid(Long.parseLong(uploadFile.getFid()));
        storage.setSn(uploadFile.getSn());
        storage.setSize(uploadFile.getSize());
        storage.setMimetype(mimeType);
        storageDao.save(storage);
        result.setStorageId(storage.getId());
        result.setUrl(StorageUtil.constructFileUrl(uploadFile.getFid(), uploadFile.getSn(), mimeType));
        return result;
    }
}
