
/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */
package com.baijia.tianxiao.sal.room.service.impl;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import javax.annotation.Resource;

import com.google.common.collect.Sets;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Service;

import com.baijia.tianxiao.dal.org.dao.OrgClassLessonDao;
import com.baijia.tianxiao.dal.org.dao.OrgClassRoomDao;
import com.baijia.tianxiao.dal.org.dao.OrgCourseRoomDao;
import com.baijia.tianxiao.dal.org.po.OrgClassLesson;
import com.baijia.tianxiao.dal.org.po.OrgClassRoom;
import com.baijia.tianxiao.dal.org.po.OrgCourseRoom;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.room.dto.ClassRoomDto;
import com.baijia.tianxiao.sal.room.service.ClassRoomService;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;

import lombok.extern.slf4j.Slf4j;

/**
 * @author cxm
 * @version 1.0
 * @title ClassRoomServiceImpl
 * @desc TODO
 * @date 2015年12月24日
 */
@Service
@Slf4j
public class ClassRoomServiceImpl implements ClassRoomService {

    @Resource
    private OrgClassRoomDao orgClassRoomDao;

    @Resource
    private OrgCourseRoomDao orgCourseRoomDao;

    @Resource
    private OrgClassLessonDao orgClassLessonDao;

    @Override
    public ClassRoomDto saveOrUpdateClassRoom(Long orgId, ClassRoomDto params) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        params.setOrgId(orgId);
        params.validate();
        OrgClassRoom classRoom = ClassRoomDto.converPo(params);
        if (params.getRoomId() != null && params.getRoomId() > 0) {
            orgClassRoomDao.update(classRoom);
        } else {
            if (classRoom.getBranchId() == null) {
                classRoom.setBranchId(0l);
            }
            orgClassRoomDao.save(classRoom);
            params.setRoomId(classRoom.getId());
        }
        return params;
    }

    @Override
    public void delClassRoom(Long orgId, Long roomId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(roomId != null && roomId > 0, "roomId is illegal");
        List<OrgClassLesson> lessons = orgClassLessonDao.getLeftRoomIdLessons(orgId, roomId, null);
        if (CollectionUtils.isNotEmpty(lessons)) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "该教室已安排课节,请先变更课节教室");
        }

        orgCourseRoomDao.delCourseRoom(orgId, null, Lists.newArrayList(roomId));

        orgClassRoomDao.delById(roomId);
    }

    @Override
    public List<ClassRoomDto> listOrgClassRooms(Long orgId, Long branchId, Long courseId, PageDto page) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        List<OrgClassRoom> classRooms = orgClassRoomDao.getClassRoom(orgId, branchId, false, page);
        if (CollectionUtils.isNotEmpty(classRooms)) {
            List<ClassRoomDto> result = Lists.newArrayList();
            Set<Long> courseRoomIds = Sets.newHashSet();
            if (courseId != null && courseId > 0) {
                courseRoomIds.addAll(orgCourseRoomDao.getCourseRoomIds(orgId, courseId));
            }
            for (OrgClassRoom classRoom : classRooms) {
                if (courseRoomIds.contains(classRoom.getId())) {
                    continue;
                }
                result.add(ClassRoomDto.covert2Dto(classRoom));
            }
            return result;
        }
        return Collections.emptyList();
    }

    @Override
    public void addClassRoomCourse(Long orgId, Long courseId, Collection<Long> roomIds) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(courseId != null && courseId > 0, "courseId is illegal");
        if (CollectionUtils.isEmpty(roomIds)) {
            log.warn("room ids is empty.");
            return;
        }
        log.info("add roomids:{} into course :{},orgId:{}", roomIds, courseId, orgId);
        List<OrgCourseRoom> courseRooms = orgCourseRoomDao.getCourseRoomList(orgId, courseId);
        boolean hasDefault = false;
        if (CollectionUtils.isNotEmpty(courseRooms)) {
            for (OrgCourseRoom courseRoom : courseRooms) {
                roomIds.remove(courseRoom.getRoomId());
                hasDefault |= courseRoom.isDefault();
            }
        }
        hasDefault = !hasDefault;
        if (CollectionUtils.isNotEmpty(roomIds)) {
            List<OrgCourseRoom> saveCourseRooms = Lists.newArrayList();

            List<OrgClassRoom> classRooms = orgClassRoomDao.getByIds(roomIds, "orgId", "id", "delStatus");
            OrgCourseRoom courseRoom = null;
            for (OrgClassRoom room : classRooms) {
                if (!orgId.equals(room.getOrgId()) || room.isDelStatus()) {
                    log.warn("room id :{} is not in current orgId:{}", room.getId(), room.getOrgId());
                    continue;
                }
                courseRoom = new OrgCourseRoom();
                courseRoom.setCourseId(courseId);
                courseRoom.setIsDefault(hasDefault);
                courseRoom.setOrgId(orgId);
                courseRoom.setRoomId(room.getId());
                hasDefault = false;
                saveCourseRooms.add(courseRoom);
            }
            log.debug("save course room :{}", saveCourseRooms);
            orgCourseRoomDao.saveAll(saveCourseRooms, "courseId", "isDefault", "orgId", "roomId");
        }
    }

    @Override
    public void delClassRoomFromCourse(Long orgId, Long courseId, Collection<Long> roomIds) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(courseId != null && courseId > 0, "courseId is illegal");
        if (CollectionUtils.isEmpty(roomIds)) {
            log.warn("room ids is empty.");
            return;
        }
        orgCourseRoomDao.delCourseRoom(orgId, courseId, roomIds);

        List<Long> lessonIds = orgClassLessonDao.getLeftLessonIds(orgId, courseId);
        if (CollectionUtils.isNotEmpty(lessonIds)) {
            int result = orgClassLessonDao.updateLessonClassRoom(orgId, lessonIds, null);
            log.info("delete class room from lessonIds:{},result:{}", lessonIds, result);
        }
    }

    @Override
    public List<ClassRoomDto> listOrgCourseRooms(Long orgId, Long courseId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(courseId != null && courseId > 0, "courseId is illegal");
        List<OrgCourseRoom> courseRooms = orgCourseRoomDao.getCourseRoomList(orgId, courseId);
        if (CollectionUtils.isEmpty(courseRooms)) {
            return Collections.emptyList();
        }
        Set<Long> roomIds = Sets.newHashSet();
        Long defaultRoomId = null;
        for (OrgCourseRoom courseRoom : courseRooms) {
            roomIds.add(courseRoom.getRoomId());
            if (courseRoom.isDefault()) {
                defaultRoomId = courseRoom.getRoomId();
            }
        }
        List<OrgClassRoom> classRooms = orgClassRoomDao.getByIds(roomIds);

        if (CollectionUtils.isNotEmpty(classRooms)) {
            List<ClassRoomDto> result = Lists.newArrayList();
            for (OrgClassRoom classRoom : classRooms) {
                ClassRoomDto dto = ClassRoomDto.covert2Dto(classRoom);
                if (classRoom.getId().equals(defaultRoomId)) {
                    dto.setIsDefault(1);
                }
                result.add(dto);
            }
            return result;
        }
        return Collections.emptyList();
    }

    @Override
    public void courseRoomSeting(Long orgId, Long courseId, Long roomId, boolean enable) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(courseId != null && courseId > 0, "courseId is illegal");
        Preconditions.checkArgument(roomId != null && roomId > 0, "roomId is illegal");
        log.info("set orgId:{} course:{} default room:{} : {}", orgId, courseId, roomId, enable);
        orgCourseRoomDao.courseDefaultRoomSet(orgId, courseId, roomId, enable);
    }

    @Override
    public ClassRoomDto info(Long orgId, Long roomId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(roomId != null && roomId > 0, "roomId is illegal");
        OrgClassRoom classRoom = orgClassRoomDao.getById(roomId);
        Preconditions.checkArgument(classRoom != null && classRoom.getOrgId().equals(orgId), "教室ID不正确");

        return ClassRoomDto.covert2Dto(classRoom);

    }

}
