/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */
package com.baijia.tianxiao.sal.organization.org.service.impl;

import com.baijia.tianxiao.consants.DataStatus;
import com.baijia.tianxiao.dal.org.constant.AuditStatus;
import com.baijia.tianxiao.dal.org.constant.OrgImgType;
import com.baijia.tianxiao.dal.org.dao.OrgPhotoDao;
import com.baijia.tianxiao.dal.org.dao.OrgStorageDao;
import com.baijia.tianxiao.dal.org.po.OrgPhoto;
import com.baijia.tianxiao.dal.org.po.OrgStorage;
import com.baijia.tianxiao.enums.OrgErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.organization.index.dto.OrgPhotoDto;
import com.baijia.tianxiao.sal.organization.org.service.OrgPhotoService;
import com.baijia.tianxiao.util.storage.StorageUtil;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Service;

import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

/**
 * @author cxm
 * @version 1.0
 * @title OrgPhotoServiceImpl
 * @desc TODO
 * @date 2015年10月27日
 */
@Service
@Slf4j
public class OrgPhotoServiceImpl implements OrgPhotoService {

    @Resource
    private OrgPhotoDao orgPhotoDao;

    @Resource
    private OrgStorageDao orgStorageDao;

    @Override
    public String getPhotoUrlByOrgIdAndCategory(@NonNull Long orgId, @NonNull OrgImgType imgType) {
        log.info("get org photo by orgId:{},imgType:{}", orgId, imgType);
        List<OrgPhoto> photoes = orgPhotoDao.getByOrgIdAndCategory(orgId.intValue(), imgType.getValue());
        String photoUrl = imgType.getDefaultUrl();
        if (CollectionUtils.isNotEmpty(photoes)) {
            OrgStorage storage = orgStorageDao.getById(photoes.get(0).getOrgStorageId(), "fid", "sn", "mimeType");
            photoUrl = StorageUtil.constructUrl(storage.getFid(), storage.getSn(), storage.getMimeType());
        }
        log.info("get org photo url by orgId:{},imgType:{},url:{}", orgId, imgType, photoUrl);
        return photoUrl;
    }

    @Override
    public Map<String, Object> getPhotoByOrgIdAndCategory(Long orgId, @NonNull OrgImgType imgType) {
        Map map = Maps.newHashMap();

        log.info("get org photo by orgid = {},imgType = {}", orgId, imgType);
        List<OrgPhoto> photoes = orgPhotoDao.getByOrgIdAndCategory(orgId.intValue(), imgType.getValue());
        String photoUrl = imgType.getDefaultUrl();
        if (CollectionUtils.isNotEmpty(photoes)) {
            OrgStorage storage = orgStorageDao.getById(photoes.get(0).getOrgStorageId(), "id", "fid", "sn", "mimeType");
            photoUrl = StorageUtil.constructUrl(storage.getFid(), storage.getSn(), storage.getMimeType());
            map.put("logoStorageId", storage.getId());
            map.put("logoUrl", photoUrl);
        }
        log.info("get org photo url by map = {}", map);
        return map;
    }

    @Override
    public int saveOrUpdateOrgPhoto(Long orgId, OrgImgType imgType, int logoId) {
        int count = orgPhotoDao.countOrgIdAndCategory(orgId, imgType.getValue());
        if (count >= 50) {//照片数量限制
            throw new BussinessException(OrgErrorCode.ORG_INFO_PHOTO_SIZE);
        }
        if (logoId <= 0) {
            throw new BussinessException(OrgErrorCode.ORG_INFO_PHOTO_ID);
        }

        log.info("updateOrgPhoto orgId = {},logoId={},imgType={}", orgId, logoId, imgType);
        List<OrgPhoto> photoes = Lists.newArrayList();
        if (imgType.getValue() == OrgImgType.ORG_LOGO.getValue()) {
            photoes = orgPhotoDao.getByOrgIdAndCategory(orgId.intValue(), imgType.getValue());
        }
        Date now = new Date();
        OrgPhoto orgPhoto = null;
        if (CollectionUtils.isNotEmpty(photoes)) {
            orgPhoto = photoes.get(0);
            orgPhoto.setUpdateTime(now);
            orgPhoto.setOrgStorageId(logoId);
        } else {
            orgPhoto = new OrgPhoto();
            orgPhoto.setOrgId(orgId.intValue());
            orgPhoto.setOrgStorageId(logoId);
            orgPhoto.setAuditstatus(AuditStatus.PASS.getValue());
            orgPhoto.setCategory(imgType.getValue());
            orgPhoto.setGroupId(imgType.getGroupId());
            orgPhoto.setCreateTime(now);
            orgPhoto.setUpdateTime(now);
            orgPhoto.setIsdel(DataStatus.NORMAL.getValue());
        }

        log.debug("save and update orgPhoto =={}", orgPhoto);
        orgPhotoDao.saveOrUpdate(orgPhoto);

        return orgPhoto.getId();
    }

    @Override
    public List<OrgPhotoDto> listAllPhotoByType(Long orgId, OrgImgType imgType) {
        List<OrgPhoto> photoes = orgPhotoDao.getByOrgIdAndCategory(orgId.intValue(), imgType.getValue());
        if (photoes.isEmpty()) {
            return Collections.emptyList();
        }
        List<OrgPhotoDto> result = Lists.newArrayList();
        for (OrgPhoto orgPhoto : photoes) {
            OrgStorage storage = orgStorageDao.getById(orgPhoto.getOrgStorageId(), "fid", "sn", "mimeType");
            String photoUrl = StorageUtil.constructUrl(storage.getFid(), storage.getSn(), storage.getMimeType());
            orgPhoto.setLink(photoUrl);

            OrgPhotoDto orgPhotoDto = new OrgPhotoDto();
            orgPhotoDto.setId(orgPhoto.getId());
            orgPhotoDto.setOrgStorageId(orgPhoto.getOrgStorageId());
            orgPhotoDto.setLink(orgPhoto.getLink());
            result.add(orgPhotoDto);
        }
        return result;
    }

    @Override
    public void batchDeleteByIds(String ids) {
        String[] idstr = ids.split(",");
        List<Integer> photoIds = Lists.newArrayList();
        for (String id : idstr) {
            photoIds.add(Integer.parseInt(id));
        }

        orgPhotoDao.delByIds(photoIds);
    }
}
