
/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2016 All Rights Reserved.
 */

package com.baijia.tianxiao.sal.organization.todo.service.impl;

import com.baijia.tianxiao.constants.org.BizConf;
import com.baijia.tianxiao.dal.org.constant.DeleteStatus;
import com.baijia.tianxiao.dal.org.dao.OrgAccountDao;
import com.baijia.tianxiao.dal.org.dao.OrgCrontabTaskDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentDao;
import com.baijia.tianxiao.dal.org.po.OrgAccount;
import com.baijia.tianxiao.dal.org.po.OrgCrontabTaskLog;
import com.baijia.tianxiao.dal.org.po.OrgStudent;
import com.baijia.tianxiao.dal.push.constant.NoticeType;
import com.baijia.tianxiao.dal.push.dto.content.NoticeMsgContent;
import com.baijia.tianxiao.dal.push.utils.ActionUtil;
import com.baijia.tianxiao.dal.roster.dao.TxConsultUserDao;
import com.baijia.tianxiao.dal.roster.po.TxConsultUser;
import com.baijia.tianxiao.dal.todo.dao.TxBacklogDao;
import com.baijia.tianxiao.dal.todo.po.TxBacklog;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.filter.TianxiaoMContext;
import com.baijia.tianxiao.sal.organization.org.service.impl.RequestSourceDesc;
import com.baijia.tianxiao.sal.organization.todo.constant.GroupType;
import com.baijia.tianxiao.sal.organization.todo.constant.RemindType;
import com.baijia.tianxiao.sal.organization.todo.constant.TaskStatus;
import com.baijia.tianxiao.sal.organization.todo.constant.TodoErrorCode;
import com.baijia.tianxiao.sal.organization.todo.dto.BacklogDto;
import com.baijia.tianxiao.sal.organization.todo.dto.BacklogListResponseDto;
import com.baijia.tianxiao.sal.organization.todo.dto.RelatedStudent;
import com.baijia.tianxiao.sal.organization.todo.service.TxBacklogService;
import com.baijia.tianxiao.sal.organization.utils.DataAuthority;
import com.baijia.tianxiao.sal.push.dto.ConsultAvatarUrlAndNameDto;
import com.baijia.tianxiao.sal.push.dto.PushConfig;
import com.baijia.tianxiao.sal.push.service.ConsultAvatarUrlService;
import com.baijia.tianxiao.sal.push.service.ConsultMessageService;
import com.baijia.tianxiao.sal.push.utils.NativeUrlUtil;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import lombok.extern.slf4j.Slf4j;

/**
 * @title TxBacklogServiceImpl
 * @desc TODO
 * @author zhangbing
 * @date 2016年2月25日
 * @version 1.0
 */

@Service
@Slf4j
public class TxBacklogServiceImpl implements TxBacklogService {

    // 待办事项
    private static final String BACKLOG_ACTION = "backlog_info";

    @Autowired
    private OrgAccountDao orgAccountDao;

    @Autowired
    private TxBacklogDao txBacklogDao;

    @Autowired
    private TxConsultUserDao txConsultUserDao;

    @Autowired
    private OrgStudentDao orgStudentDao;

    @Autowired
    private OrgCrontabTaskDao orgCrontabTaskDao;

    @Autowired
    private TxConsultUserDao consultUserDao;
    @Autowired
    private ConsultMessageService messageService;
    @Autowired
    private ConsultAvatarUrlService consultAvatarUrlService;

    @Override
    @DataAuthority(resourceTypes = { RequestSourceDesc.BACK_LOG_LIST })
    public BacklogListResponseDto getBacklogList(Long orgId, int groupType, PageDto page) {
        Preconditions.checkArgument(orgId != null, "orgId can not be null");
        Preconditions.checkArgument(groupType >= 0 && groupType <= 3, "groupType is illegal");
        OrgAccount account = this.orgAccountDao.getById(orgId);
        if (account == null) {
            throw new BussinessException(TodoErrorCode.ORG_NOT_EXIST);
        }

        Integer txCascadeId = TianxiaoMContext.getTXCascadeId();

        BacklogListResponseDto responseDto = new BacklogListResponseDto();
        List<BacklogDto> backlogListDto = Lists.newArrayList();
        List<TxBacklog> list = Lists.newArrayList();
        responseDto.setGroupType(groupType);
        responseDto.setList(backlogListDto);

        // 如果判断当前的登录角色不是员工，不对数据获取进行限制
        boolean canAccess = RequestSourceDesc.BACK_LOG_LIST.canAccess("getBacklogList", this.getClass(),
            new Class<?>[] { Long.class, int.class, PageDto.class });
        if (canAccess) {
            txCascadeId = null;
        }

        list = this.txBacklogDao.getBacklogListByGroup(orgId, txCascadeId, groupType, page);
        log.info("getBacklogList-------orgId={},groupType={},page={},list={}", orgId,
            GroupType.getNoteWithValue(groupType), page, list);
        List<Long> consultUserIds = new ArrayList<>();
        List<Long> studentIds = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(list)) {
            for (TxBacklog po : list) {
                BacklogDto dto = new BacklogDto();
                backlogPo2Dto(po, dto);
                if (po.getStudentId() != null && po.getStudentId() > 0) {
                    studentIds.add(po.getStudentId());
                } else {
                    consultUserIds.add(po.getConsultUserId());
                }
                backlogListDto.add(dto);
            }

            Map<Long, RelatedStudent> studentMap = buildStudents(studentIds, orgId);
            Map<Long, RelatedStudent> consultUserMap = buildConsultUsers(consultUserIds, orgId);

            for (BacklogDto dto : backlogListDto) {
                RelatedStudent relatedStudent = null;
                if (dto.getRelatedStudent() == null) {
                    log.warn("RelatedStudent is null!");
                    continue;
                }
                if (dto.getRelatedStudent().getType() == 1) {
                    relatedStudent = studentMap.get(dto.getRelatedStudent().getStudentId());
                } else {
                    relatedStudent = consultUserMap.get(dto.getRelatedStudent().getStudentId());
                }
                if (relatedStudent != null) {
                    dto.setRelatedStudent(relatedStudent);
                } else {
                    log.warn("[Backlog] student is not exist.{}", dto.getRelatedStudent());
                }
            }

        }
        return responseDto;
    }

    @Override
    @Transactional(rollbackFor = { Exception.class, BussinessException.class })
    public Long saveBacklog(Long orgId, BacklogDto backlogDto) {
        Preconditions.checkArgument(orgId != null, "orgId can not be null");
        Long backlogId = backlogDto.getBacklogId();
        log.info("saveBacklog---------backlogDto={},backlogId={}", backlogDto, backlogId);
        if (null != backlogId && backlogId > 0) {
            backlogId = this.editBacklog(backlogDto, orgId);
        } else {
            backlogId = this.addBacklog(backlogDto, orgId);
        }
        return backlogId;
    }

    @Override
    @Transactional(rollbackFor = { Exception.class, BussinessException.class })
    public void delBacklog(Long orgId, Long backlogId) {
        Preconditions.checkArgument(orgId != null, "orgId can not be null");
        Preconditions.checkArgument(backlogId != null && backlogId > 0, "backlogId is illegal");

        TxBacklog txBacklog = this.txBacklogDao.getById(backlogId, "id", "orgId", "delStatus");
        log.info("delBacklog---------backlogId={},orgId={},txBacklog={}", backlogId, orgId, txBacklog);

        if (null != txBacklog && DeleteStatus.NORMAL.getValue() == txBacklog.getDelStatus()) {
            if (null != txBacklog.getOrgId() && txBacklog.getOrgId().longValue() == orgId.longValue()) {
                txBacklog.setDelStatus(DeleteStatus.DELETED.getValue());
                txBacklog.setUpdateTime(new Date());
                txBacklogDao.update(txBacklog, false, "delStatus", "updateTime");
            } else {
                throw new BussinessException(TodoErrorCode.STUDENT_NOT_IN_ORG);
            }
        } else {
            throw new BussinessException(TodoErrorCode.BACKLOG_NOT_EXIST);
        }
    }

    @Override
    @Transactional(rollbackFor = { Exception.class, BussinessException.class })
    public void setBacklogStatus(Long orgId, Long backlogId, int finish) {
        Preconditions.checkArgument(orgId != null, "orgId can not be null");
        Preconditions.checkArgument(backlogId != null && backlogId > 0, "backlogId is illegal");

        TxBacklog txBacklog = txBacklogDao.getBacklogByIdAndOrgId(backlogId, orgId, "id", "finish");
        log.info("setBacklogStatus------txBacklog={}", txBacklog);
        if (null != txBacklog) {
            if (finish != txBacklog.getFinish()) {
                txBacklog.setFinish(finish);
                txBacklog.setUpdateTime(new Date());
                if (finish == 0) {
                    this.txBacklogDao.update(txBacklog, "finish", "updateTime");
                } else if (finish == 1) {
                    // txBacklog.setFinishTime(new Date());
                    // log.info("完成时间=========" + txBacklog.getFinishTime());
                    this.txBacklogDao.update(txBacklog, "finish", "updateTime");
                }
            } else {
                throw new BussinessException(
                    0 == finish ? TodoErrorCode.BACKLOG_HAS_UNFINISH : TodoErrorCode.BACKLOG_HAS_FINISH);
            }
        } else {
            throw new BussinessException(TodoErrorCode.BACKLOG_NOT_EXIST);
        }
    }

    private void backlogPo2Dto(TxBacklog po, BacklogDto dto) {
        dto.setIsSys(po.getIsSys());
        dto.setBacklogId(po.getId());
        dto.setContent(StringUtils.isNotBlank(po.getContent()) ? po.getContent() : "");
        if (null != po.getEndTime()) {
            dto.setEndTime(po.getEndTime().getTime());
            if (null != po.getRemindTime()) {
                dto.setRemindType(
                    RemindType.getByDiff(po.getRemindTime().getTime() - po.getEndTime().getTime()).getValue());
            } else {
                dto.setRemindType(RemindType.NO_REMIND.getValue());
            }
        } else {
            dto.setEndTime(System.currentTimeMillis());
            dto.setRemindType(RemindType.NO_REMIND.getValue());
        }
        dto.setFinish(po.getFinish());
        dto.setFinishString(TaskStatus.getNoteByValue(po.getFinish()));
        dto.setRelatedStudent(buildStudent(po.getConsultUserId(), po.getStudentId()));
    }

    private RelatedStudent buildStudent(Long consultUserId, Long studentId) {
        RelatedStudent relatedStudent = null;
        if (null != studentId && studentId > 0) {
            relatedStudent = new RelatedStudent();
            relatedStudent.setType(1);
            relatedStudent.setStudentId(studentId);
        } else if (null != consultUserId && consultUserId > 0) {
            relatedStudent = new RelatedStudent();
            relatedStudent.setType(0);
            relatedStudent.setStudentId(consultUserId);
        }
        log.info("buildStudent------relatedStudent={}", relatedStudent);
        return relatedStudent;
    }

    private Map<Long, RelatedStudent> buildStudents(Collection<Long> studentIds, Long orgId) {
        Map<Long, RelatedStudent> retMap = new HashMap<>();
        if (null != studentIds && studentIds.size() > 0) {

            List<OrgStudent> students = orgStudentDao.getStudentByIds(orgId, studentIds, "id", "userId");

            Map<Long, Long> map = new HashMap<>();

            if (students != null) {
                for (OrgStudent stu : students) {
                    map.put(stu.getUserId(), stu.getId());
                }
            }

            Map<Long, ConsultAvatarUrlAndNameDto> avatarMap =
                consultAvatarUrlService.batchStudentAvatarUrlAndNameDMap(map.keySet(), orgId);
            Set<Long> keySet = avatarMap.keySet();
            for (Long id : keySet) {
                ConsultAvatarUrlAndNameDto dto = avatarMap.get(id);
                RelatedStudent relatedStudent = new RelatedStudent();
                relatedStudent.setType(1);
                relatedStudent.setStudentId(map.get(id));
                relatedStudent.setName(dto.getName());
                relatedStudent.setAvatar(dto.getAvatarUrl());
                retMap.put(map.get(id), relatedStudent);
            }
        }
        return retMap;
    }

    private Map<Long, RelatedStudent> buildConsultUsers(Collection<Long> consultUserIds, Long orgId) {
        Map<Long, RelatedStudent> retMap = new HashMap<>();
        if (null != consultUserIds && consultUserIds.size() > 0) {
            List<TxConsultUser> consultUsers =
                this.txConsultUserDao.batchTxConsultUserByIds(consultUserIds, null, null);
            if (null != consultUsers) {
                Map<Long, ConsultAvatarUrlAndNameDto> avatarMap =
                    consultAvatarUrlService.batchConsultAvatarUrlAndNameDtoMap(consultUserIds, orgId);
                Set<Long> keySet = avatarMap.keySet();
                for (Long id : keySet) {
                    ConsultAvatarUrlAndNameDto dto = avatarMap.get(id);
                    RelatedStudent relatedStudent = new RelatedStudent();
                    relatedStudent.setType(0);
                    relatedStudent.setStudentId(id);
                    relatedStudent.setName(dto.getName());
                    relatedStudent.setAvatar(dto.getAvatarUrl());
                    retMap.put(id, relatedStudent);
                }
            }
        }
        return retMap;
    }

    private Long addBacklog(BacklogDto backlogDto, Long orgId) {
        log.info("addBacklog------orgId={}, backlogDto={}", orgId, backlogDto);
        TxBacklog txBacklog = new TxBacklog();
        backlogDto.setCascadeId(TianxiaoMContext.getTXCascadeId() == null ? 0 : TianxiaoMContext.getTXCascadeId()); // 设置子账号ID
        this.backlogDto2Po(backlogDto, txBacklog, orgId);
        this.txBacklogDao.save(txBacklog, true);
        log.info("addBacklog-------txBacklog={}", txBacklog.toString());
        return txBacklog.getId();
    }

    private Long editBacklog(BacklogDto backlogDto, Long orgId) {
        Long backlogId = backlogDto.getBacklogId();
        TxBacklog txBacklog = txBacklogDao.getBacklogByIdAndOrgId(backlogId, orgId);
        log.info("editBacklog-------backlogDto={},txBacklog={}", backlogDto, txBacklog);
        if (null != txBacklog) {
            this.verifyParaAndStatus(txBacklog, backlogDto);
            this.syncStudentInfo(txBacklog, backlogDto, orgId);
            this.backlogDto2Po(backlogDto, txBacklog, orgId);
            this.txBacklogDao.update(txBacklog, true);
        } else {
            throw new BussinessException(TodoErrorCode.BACKLOG_NOT_EXIST);
        }
        log.info("editBacklog-------txBacklog={}", txBacklog);
        return txBacklog.getId();
    }

    private void backlogDto2Po(BacklogDto dto, TxBacklog po, Long orgId) {
        po.setStudentId(0l);
        po.setConsultUserId(0l);
        po.setOrgId(orgId);
        if (dto.getCascadeId() != null) {
            po.setCascadeId(dto.getCascadeId());
        }
        po.setContent(StringUtils.isNotEmpty(dto.getContent()) ? dto.getContent() : "");
        po.setEndTime(new Date(dto.getEndTime()));
        po.setCreateTime(null == po.getCreateTime() ? new Date() : po.getCreateTime());
        po.setUpdateTime(new Date());

        if (dto.getRemindType() >= RemindType.ON_TIME.getValue()
            && dto.getRemindType() <= RemindType.TWO_DAY_AHEAD.getValue()) {
            po.setRemindTime(new Date((dto.getEndTime() - RemindType.getDiffByValue(dto.getRemindType()))));
        } else {
            po.setRemindTime(null);
        }

        if (null != dto.getRelatedStudent() && null != dto.getRelatedStudent().getStudentId()
            && dto.getRelatedStudent().getStudentId() > 0) {
            if (0 == dto.getRelatedStudent().getType()) {
                Long consultUserId = dto.getRelatedStudent().getStudentId();
                TxConsultUser consultUser =
                    this.txConsultUserDao.getById(consultUserId, "id", "orgId", "delStatus", "studentId");
                if (null != consultUser && DeleteStatus.NORMAL.getValue() == consultUser.getDelStatus()) {
                    if (orgId.longValue() == consultUser.getOrgId().longValue()) {
                        po.setConsultUserId(consultUserId);
                    } else {
                        throw new BussinessException(TodoErrorCode.STUDENT_NOT_IN_ORG);
                    }
                } else {
                    throw new BussinessException(TodoErrorCode.STUDENT_NOT_EXIST);
                }
            } else if (1 == dto.getRelatedStudent().getType()) {
                Long studentId = dto.getRelatedStudent().getStudentId();
                OrgStudent student = this.orgStudentDao.getById(studentId, "org_id", "delStatus");
                if (null != student && DeleteStatus.NORMAL.getValue() == student.getDelStatus()) {
                    if (orgId.longValue() == student.getOrgId().longValue()) {
                        po.setStudentId(studentId);
                        List<TxConsultUser> list =
                            this.txConsultUserDao.lookByStudentId(orgId, studentId, "id", "studentId");
                        if (CollectionUtils.isNotEmpty(list)) {
                            po.setConsultUserId(list.get(0).getId());
                        }
                    } else {
                        throw new BussinessException(TodoErrorCode.STUDENT_NOT_IN_ORG);
                    }
                } else {
                    throw new BussinessException(TodoErrorCode.STUDENT_NOT_EXIST);
                }
            }
        }
    }

    /**
     * 编辑待办事项之前进行参数及状态验证
     *
     * @param txBacklog 待办事项记录
     * @param backlogDto 编辑传入参数
     */
    private void verifyParaAndStatus(TxBacklog txBacklog, BacklogDto backlogDto) {
        if (txBacklog.getEndTime().before(new Date())) {
            throw new BussinessException(TodoErrorCode.CAN_NOT_EDIT_EXPIRED_BACKLOG);
        }

        if (BizConf.TRUE.intValue() == txBacklog.getFinish()) {
            throw new BussinessException(TodoErrorCode.CAN_NOT_EDIT_FINISH_BACKLOG);
        }

        if (BizConf.TRUE.intValue() == txBacklog.getIsSys()) {
            if (null != backlogDto.getRelatedStudent()) {
                if (0 == backlogDto.getRelatedStudent().getType()) {
                    if (null != txBacklog.getConsultUserId() && txBacklog.getConsultUserId().longValue() != backlogDto
                        .getRelatedStudent().getStudentId().longValue()) {
                        throw new BussinessException(TodoErrorCode.CAN_NOT_EDIT_SYSTEM_BACKLOG_WITH_STUDENT);
                    }
                } else {
                    if (null != txBacklog.getStudentId() && txBacklog.getStudentId().longValue() != backlogDto
                        .getRelatedStudent().getStudentId().longValue()) {
                        throw new BussinessException(TodoErrorCode.CAN_NOT_EDIT_SYSTEM_BACKLOG_WITH_STUDENT);
                    }
                }
            }
        }
    }

    /**
     * 编辑系统待办事项时若截止时间改变，则需要同步咨询学员(正式学员)的下次跟进时间信息
     *
     * @param txBacklog 待办事项记录
     * @param backlogDto 编辑传入参数
     * @param orgId 机构id
     */
    private void syncStudentInfo(TxBacklog txBacklog, BacklogDto backlogDto, Long orgId) {
        if (BizConf.TRUE.intValue() == txBacklog.getIsSys()) {
            if (backlogDto.getEndTime().longValue() != txBacklog.getEndTime().getTime()) {
                if (0 == backlogDto.getRelatedStudent().getType()) {
                    if (backlogDto.getRelatedStudent().getStudentId().longValue() == txBacklog.getConsultUserId()
                        .longValue()) {
                        TxConsultUser consulterUser =
                            this.consultUserDao.getOrgConsultUser(orgId, txBacklog.getConsultUserId());
                        if (null != consulterUser) {
                            consulterUser.setNextRemindTime(new Date(backlogDto.getEndTime()));
                            consulterUser.setUpdateTime(new Date());
                            this.txConsultUserDao.update(consulterUser, "nextRemindTime", "updateTime");
                        } else {
                            throw new BussinessException(TodoErrorCode.STUDENT_NOT_EXIST);
                        }
                    } else {
                        throw new BussinessException(TodoErrorCode.BACKLOG_NOT_EXIST);
                    }
                } else if (1 == backlogDto.getRelatedStudent().getType()) {
                    if (backlogDto.getRelatedStudent().getStudentId().longValue() == txBacklog.getStudentId()
                        .longValue()) {
                        OrgStudent orgStudent = this.orgStudentDao.getById(txBacklog.getStudentId());
                        if (null != orgStudent
                            && orgStudent.getDelStatus().intValue() == DeleteStatus.NORMAL.getValue()) {
                            orgStudent.setNextRemindTime(new Date(backlogDto.getEndTime()));
                            orgStudent.setUpdateTime(new Date());
                            this.orgStudentDao.update(orgStudent, "nextRemindTime", "updateTime");
                        } else {
                            throw new BussinessException(TodoErrorCode.STUDENT_NOT_EXIST);
                        }
                    } else {
                        throw new BussinessException(TodoErrorCode.BACKLOG_NOT_EXIST);
                    }
                }
            }
        }
    }

    /**
     * 待办事项提醒
     *
     * @param startTime 系统第一次启动,数据库org_schedule_task_log中没有执行记录,则从第一条开始处理 如果org_schedule_task_log中有记录，从上次处理的截止时间开始处理
     * @param
     */
    @Override
    public void remind(Date startTime, Date endTime, OrgCrontabTaskLog taskLog, boolean isNew) {
        List<TxBacklog> result = txBacklogDao.getBacklogListByRemindTime(startTime, endTime);

        if (result != null && !result.isEmpty()) {
            prepared(result, taskLog, isNew);
        } else {
            taskLog.setUpdateTime(new Timestamp(System.currentTimeMillis()));
            if (isNew) {
                orgCrontabTaskDao.save(taskLog);
            } else {
                orgCrontabTaskDao.update(taskLog, "executeTime", "updateTime");
            }
        }
    }

    private void prepared(List<TxBacklog> result, OrgCrontabTaskLog taskLog, boolean isNew) {
        for (TxBacklog backlog : result) {
            pushBacklog(backlog);
        }

        taskLog.setUpdateTime(new Timestamp(System.currentTimeMillis()));
        taskLog.setValue(result.get(result.size() - 1).getId());
        if (isNew) {
            orgCrontabTaskDao.save(taskLog);
        } else {
            orgCrontabTaskDao.update(taskLog, "executeTime", "updateTime", "value");
        }
    }

    private void pushBacklog(TxBacklog backlog) {
        Map<String, Object> param = new HashMap<>();
        param.put("bid", backlog.getId());
        log.info("[Notice] Todo notice");
        try {
            String content = NoticeType.TO_DO.getContent() + ":" + backlog.getContent();
            messageService.sendNotice(backlog.getOrgId(), backlog.getCascadeId(), NoticeMsgContent.createNoticeContent(
                NoticeType.TO_DO, ActionUtil.getAction(ActionUtil.ACTION_TO_CRM_TODO_TASK, param), content));
        } catch (Exception e) {
            log.error("[Notice] exception ", e);
        }

    }

    private PushConfig getPushConfig(TxBacklog backlog) {
        PushConfig config = new PushConfig();
        config.setTip("您有一个待办事项需要处理");
        String url = NativeUrlUtil.getUrl(BACKLOG_ACTION);
        url = NativeUrlUtil.addParam(url, "bid", backlog.getId());
        config.setJumpUrl(url);
        return config;
    }

    /**
     * 查询待办事项详情
     */
    @Override
    public BacklogDto getBacklogDetail(Long orgId, Long bid) throws BussinessException {
        TxBacklog backlog = this.txBacklogDao.getById(bid);
        BacklogDto dto = null;
        if (backlog != null && backlog.getOrgId() == orgId.longValue()) {
            dto = new BacklogDto();
            this.backlogPo2Dto(backlog, dto);
            RelatedStudent relatedStudent = null;
            if (dto.getRelatedStudent() != null) {
                if (dto.getRelatedStudent().getType() == 1) {
                    Map<Long, RelatedStudent> studentMap =
                        buildStudents(Arrays.asList(dto.getRelatedStudent().getStudentId()), orgId);
                    relatedStudent = studentMap.get(dto.getRelatedStudent().getStudentId());
                } else {
                    Map<Long, RelatedStudent> consultUserMap =
                        buildConsultUsers(Arrays.asList(dto.getRelatedStudent().getStudentId()), orgId);
                    relatedStudent = consultUserMap.get(dto.getRelatedStudent().getStudentId());
                }
                dto.setRelatedStudent(relatedStudent);
            }
        } else {
            throw new BussinessException(TodoErrorCode.BACKLOG_NOT_EXIST);
        }

        return dto;
    }
}
