
/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */

package com.baijia.tianxiao.sal.organization.org.service.impl;

import com.baijia.tianxiao.constants.CourseType;
import com.baijia.tianxiao.constants.OrgCourseStatus;
import com.baijia.tianxiao.constants.TeacherClassCourseStatus;
import com.baijia.tianxiao.constants.org.BizConf;
import com.baijia.tianxiao.dal.constant.ClassCourseStatus;
import com.baijia.tianxiao.dal.course.dao.ClassCoursePhotoDao;
import com.baijia.tianxiao.dal.course.dao.OrgCourseGroupDao;
import com.baijia.tianxiao.dal.course.dao.OrgCourseGroupRelateDao;
import com.baijia.tianxiao.dal.course.dao.OrgRecommendCourseDao;
import com.baijia.tianxiao.dal.course.dao.SubjectDao;
import com.baijia.tianxiao.dal.course.dao.TeacherClassCourseDao;
import com.baijia.tianxiao.dal.course.po.ClassCoursePhoto;
import com.baijia.tianxiao.dal.course.po.OrgCourseGroup;
import com.baijia.tianxiao.dal.course.po.OrgCourseGroupRelate;
import com.baijia.tianxiao.dal.course.po.OrgRecommendCourse;
import com.baijia.tianxiao.dal.course.po.TeacherClassCourse;
import com.baijia.tianxiao.dal.enums.CourseTypeEnum;
import com.baijia.tianxiao.dal.org.constant.AuditColumType;
import com.baijia.tianxiao.dal.org.constant.MIMEType;
import com.baijia.tianxiao.dal.org.constant.OrgImgType;
import com.baijia.tianxiao.dal.org.constant.TemplateStatus;
import com.baijia.tianxiao.dal.org.dao.OrgAccountDao;
import com.baijia.tianxiao.dal.org.dao.OrgBranchsDao;
import com.baijia.tianxiao.dal.org.dao.OrgCourseDao;
import com.baijia.tianxiao.dal.org.dao.OrgHomepageConfigDao;
import com.baijia.tianxiao.dal.org.dao.OrgHomepageTemplateDao;
import com.baijia.tianxiao.dal.org.dao.OrgInfoDao;
import com.baijia.tianxiao.dal.org.dao.OrgPhotoDao;
import com.baijia.tianxiao.dal.org.dao.OrgStorageDao;
import com.baijia.tianxiao.dal.org.dao.OrgTxtMsgDao;
import com.baijia.tianxiao.dal.org.po.OrgAccount;
import com.baijia.tianxiao.dal.org.po.OrgBranchs;
import com.baijia.tianxiao.dal.org.po.OrgCourse;
import com.baijia.tianxiao.dal.org.po.OrgHomepageConfig;
import com.baijia.tianxiao.dal.org.po.OrgHomepageTemplate;
import com.baijia.tianxiao.dal.org.po.OrgInfo;
import com.baijia.tianxiao.dal.org.po.OrgPhoto;
import com.baijia.tianxiao.dal.org.po.OrgStorage;
import com.baijia.tianxiao.dal.org.po.OrgTemplateCourse;
import com.baijia.tianxiao.dal.org.po.OrgTxtMsg;
import com.baijia.tianxiao.dal.storage.dao.StorageDao;
import com.baijia.tianxiao.dal.storage.po.Storage;
import com.baijia.tianxiao.enums.OrgErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.organization.constant.TemplatModuleType;
import com.baijia.tianxiao.sal.organization.dto.OptionalParams;
import com.baijia.tianxiao.sal.organization.org.dto.OrgBaseInfoModel;
import com.baijia.tianxiao.sal.organization.org.dto.OrgRecommendCourseList;
import com.baijia.tianxiao.sal.organization.org.dto.init.DefaultTemplateData;
import com.baijia.tianxiao.sal.organization.org.dto.init.DefaultTemplateDataList;
import com.baijia.tianxiao.sal.organization.org.dto.k12.K12TemplateData;
import com.baijia.tianxiao.sal.organization.org.dto.k12.OrgHotCourseDto;
import com.baijia.tianxiao.sal.organization.org.dto.kaixue.KaixueTemplateDate;
import com.baijia.tianxiao.sal.organization.org.service.OrgIndexService;
import com.baijia.tianxiao.sal.organization.org.service.OrgTemplateCourseService;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.util.BaseUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * @author zhangbing
 * @version 1.0
 * @title IndexServiceImpl
 * @desc TODO
 * @date 2015年12月4日
 */
@Service("orgIndexService")
public class OrgIndexServiceImpl implements OrgIndexService {

    private static final Logger logger = LoggerFactory.getLogger(OrgIndexServiceImpl.class);

    @Autowired
    private OrgAccountDao orgAccountDao;

    @Autowired
    private OrgHomepageConfigDao orgHomepageConfigDao;

    @Autowired
    private OrgHomepageTemplateDao orgHomepageTemplateDao;

    @Autowired
    private TeacherClassCourseDao teacherClassCourseDao;

    @Autowired
    private OrgRecommendCourseDao orgRecommendCourseDao;

    @Autowired
    private OrgCourseGroupDao orgCourseGroupDao;

    @Autowired
    private OrgCourseDao orgCourseDao;

    @Autowired
    private OrgInfoDao orgInfoDao;

    @Autowired
    private OrgTxtMsgDao orgTxtMsgDao;

    @Autowired
    private OrgPhotoDao orgPhotoDao;

    @Autowired
    private OrgStorageDao orgStorageDao;

    @Autowired
    private StorageDao storageDao;

    @Autowired
    private SubjectDao subjectDao;

    @Autowired
    private OrgBranchsDao orgBranchsDao;

    @Autowired
    private OrgCourseGroupRelateDao orgCourseGroupRelateDao;

    @Autowired
    private ClassCoursePhotoDao classCoursePhotoDao;

    @Autowired
    private OrgTemplateCourseService orgTemplateCourseService;

    /**
     * 渲染机构模板
     *
     * @param orgId 机构ID
     */
    @Override
    public JSONObject render(Integer orgId, Boolean readOnly, OptionalParams params) throws BussinessException {
        String scheme = "default";
        JSONObject content = null;
        if (params.getTemplateId() == null) {
            OrgHomepageConfig usingTemplate =
                orgHomepageConfigDao.getTemplateConfByStatus(orgId, TemplateStatus.USING.getCode());
            if (usingTemplate != null) {
                params.setTemplateId(usingTemplate.getTemplateId());
            }

            if (usingTemplate != null) {
                String config = usingTemplate.getContent();
                OrgHomepageTemplate template =
                    this.orgHomepageTemplateDao.getById(usingTemplate.getTemplateId(), new String[] {});
                if (template != null) {
                    JSONObject templateJson = JSONObject.fromObject(template.getContent());
                    templateJson.put("config", JSONObject.fromObject(config));
                    content = templateJson;
                }
            } else {
                // OrgHomepageTemplate defaultTemplate =
                // orgHomepageTemplateDao.getTemplate(BooleanStatus.TRUE.getStatus());
                // content = defaultTemplate.getContent();
                throw new BussinessException(OrgErrorCode.NO_TEMPLATE_SELECTED);
            }
        } else {
            OrgHomepageTemplate template = this.orgHomepageTemplateDao.getById(params.getTemplateId());
            if (template == null) {
                throw new BussinessException(OrgErrorCode.NO_TEMPLATE_SELECTED);
            } else {
                OrgHomepageConfig usingTemplate =
                    orgHomepageConfigDao.getTemplateConfByStatus(orgId, TemplateStatus.USING.getCode());
                if (usingTemplate != null && usingTemplate.getTemplateId() == params.getTemplateId()) {
                    String config = usingTemplate.getContent();
                    content = JSONObject.fromObject(template.getContent());
                    content.put("config", JSONObject.fromObject(config));
                } else {
                    usingTemplate = orgHomepageConfigDao.getTemplateConf(orgId, params.getTemplateId());
                    if (usingTemplate != null) {
                        String config = usingTemplate.getContent();
                        content = JSONObject.fromObject(template.getContent());
                        content.put("config", JSONObject.fromObject(config));
                    } else {
                        content = JSONObject.fromObject(template.getContent());
                    }
                }
            }
        }

        scheme = content.getString("scheme");
        return dispatch(orgId, scheme, content.toString(), readOnly, params);
    }

    /**
     * 渲染模板分发
     *
     * @param orgId 机构ID
     * @param scheme 模板类型
     * @param content 模板内容
     * @return 模板填充的内容
     */
    private JSONObject dispatch(Integer orgId, String scheme, String content, Boolean readOnly, OptionalParams params) {
        JSONObject templateJson = null;
        if ("abroad".equalsIgnoreCase(scheme)) {
            templateJson = wrapperAbroadTemplate(orgId, content, params);
        } else if ("art".equalsIgnoreCase(scheme)) {
            templateJson = wrapperArtTemplate(orgId, content, params);
        } else if ("k12".equalsIgnoreCase(scheme)) {
            templateJson = wrapperK12Template(orgId, content, params, readOnly);
        } else if ("kaixue".equalsIgnoreCase(scheme)) {
            templateJson = wrapperKaixueTemplate(orgId, content, params, readOnly);
        } else if ("zaojiao".equalsIgnoreCase(scheme)) {
            templateJson = wrapperZaojiaoTemplate(orgId, content, params, readOnly);
        } else if ("default".equalsIgnoreCase(scheme)) {
            templateJson = wrapperDefaultTemplate(orgId, content, params);
        } else if ("life".equalsIgnoreCase(scheme)) {
            templateJson = wrapperK12Template(orgId, content, params, readOnly);
        } else if ("language".equalsIgnoreCase(scheme)) {
            templateJson = wrapperK12Template(orgId, content, params, readOnly);
        } else {
            throw new BussinessException(OrgErrorCode.TEMPLATE_NOT_EXISTS);
        }

        templateJson.put("scheme", scheme);
        templateJson.put("templateId", params.getTemplateId());
        templateJson.put("readOnly", readOnly ? BizConf.TRUE.toString() : BizConf.FALSE.toString());
        logger.info("dispatch.templateScheme:{}", templateJson.toString());
        return templateJson;
    }

    /**
     * 渲染默认机构主页模板
     *
     * @param orgId
     * @param content 模板内容
     * @return
     */
    private JSONObject wrapperDefaultTemplate(Integer orgId, String content, OptionalParams params) {
        OrgBaseInfoModel model = this.getAndSetOrgInfoModel(orgId, params);

        DefaultTemplateData data = new DefaultTemplateData();
        List<DefaultTemplateDataList> subList =
            getAboutUsLinks(model.getNumber(), params.getMonkeySiteHost(), params.getGsxSiteHost());
        data.setGrid(subList);

        JSONObject defaultJson = JSONObject.fromObject(content);
        defaultJson.put("orgInfo", JSONObject.fromObject(model));

        JSONObject dataJson = JSONObject.fromObject(data);
        defaultJson.put("data", dataJson);

        return defaultJson;
    }

    /**
     * 获取机构基本信息
     *
     * @param orgId
     * @return
     */
    private OrgBaseInfoModel getAndSetOrgInfoModel(Integer orgId, OptionalParams params) {
        OrgBaseInfoModel model = new OrgBaseInfoModel();
        this.getAndSetOrgNumber(orgId, model, params.getMonkeySiteHost());
        getAndSetSummary(orgId, model);
        getAndSetAddress(orgId, model);
        getAndSetTags(orgId, model);
        getAndSetOrgLogo(orgId, model, params.getImgServer());

        // JSONObject links = JSONObject.fromObject(BizConf.ORG_INFO_LINKS.replaceAll("host",
        // params.getMonkeySiteHost()).replaceAll("number", model.getNumber().toString()));

        OrgHomepageConfig usingTemplate =
            orgHomepageConfigDao.getTemplateConfByStatus(orgId, TemplateStatus.USING.getCode());
        if (usingTemplate != null) {
            model.setTemplateId(usingTemplate.getTemplateId());
        }

        model.setLinks(getFooterLinks(model.getNumber(), params.getMonkeySiteHost(), params.getGsxSiteHost()));
        return model;
    }

    private JSONObject getFooterLinks(Integer number, String monkeySizeHost, String gsxSiteHost) {
        JSONObject links = new JSONObject();
        links.put("course", getLink(BizConf.ORG_COURSE_LINK, number, monkeySizeHost, gsxSiteHost));
        links.put("brief", getLink(BizConf.ABOUT_US, number, monkeySizeHost, gsxSiteHost));
        links.put("comment", getLink(BizConf.ORG_COMMENT_LINK, number, monkeySizeHost, gsxSiteHost));
        links.put("album", getLink(BizConf.ORG_PHOTO_LINK, number, monkeySizeHost, gsxSiteHost));
        links.put("branch", getLink(BizConf.ORG_BRANCH_LINK, number, monkeySizeHost, gsxSiteHost));
        links.put("teacher", getLink(BizConf.ORG_TEACHER_LINK, number, monkeySizeHost, gsxSiteHost));
        links.put("coupon", getLink(BizConf.ORG_COUPON_LINK, number, monkeySizeHost, gsxSiteHost));

        return links;
    }

    /**
     * 渲染艺体模板
     *
     * @param orgId
     * @return
     */
    private JSONObject wrapperArtTemplate(Integer orgId, String content, OptionalParams params) {
        OrgBaseInfoModel model = this.getAndSetOrgInfoModel(orgId, params);
        JSONObject artJson = JSONObject.fromObject(content);

        List<String> subjects = getOrgCourseSubject(orgId, model.getNumber(), BizConf.MAX_SUBJECT_DISPLAY);
        List<DefaultTemplateDataList> kinds =
            this.getCourseSubjects(subjects, BizConf.MAX_SUBJECT_DISPLAY, model, params);
        JSONObject dataJson = artJson.getJSONObject("data");
        dataJson.put("kinds", JSONArray.fromObject(kinds));

        artJson.put("orgInfo", JSONObject.fromObject(model));
        artJson.put("data", dataJson);
        return artJson;
    }

    /**
     * 渲染K12模板
     *
     * @param orgId
     * @return
     */
    private JSONObject wrapperK12Template(Integer orgId, String content, OptionalParams params, Boolean readOnly) {
        OrgBaseInfoModel model = getAndSetOrgInfoModel(orgId, params);

        K12TemplateData data = new K12TemplateData();
        List<OrgRecommendCourseList> courseList = this.getAndSetRecommentCourseList(orgId, params);
        data.setCourseList(courseList);
        List<DefaultTemplateDataList> grid =
            this.getOrgCourseSubjects(orgId, model.getNumber(), params.getMonkeySiteHost(), params.getGsxSiteHost());
        data.setGrid(grid);

        // 如果是编辑状态，九宫格不显示分类，但下面的分类要显示
        if (readOnly) {
            if (grid != null && !grid.isEmpty()) {
                data.setClassifyCourseList(getOrgHotCourseList(model.getNumber(), grid, params.getImgServer(),
                    params.getMonkeySiteHost(), readOnly));
            } else {
                data.setClassifyCourseList(Collections.<OrgHotCourseDto> emptyList());
            }
        } else {
            List<DefaultTemplateDataList> tempGrid = getTempGrids(orgId, grid);
            data.setClassifyCourseList(getOrgHotCourseList(model.getNumber(), tempGrid, params.getImgServer(),
                params.getMonkeySiteHost(), readOnly));
        }

        JSONObject k12Json = JSONObject.fromObject(content);
        k12Json.put("orgInfo", JSONObject.fromObject(model));

        JSONObject dataJson = JSONObject.fromObject(data);
        k12Json.put("data", dataJson);

        return k12Json;
    }

    /**
     * 渲染开学季模板
     *
     * @param orgId
     * @return
     */
    private JSONObject wrapperKaixueTemplate(Integer orgId, String content, OptionalParams params, Boolean readOnly) {
        OrgBaseInfoModel model = getAndSetOrgInfoModel(orgId, params);

        KaixueTemplateDate data = new KaixueTemplateDate();
        logger.info("params=={},", params);

        data.setMiaoshaList(convertToKaixue(params, orgTemplateCourseService.getByOrgIdTemplate(orgId,
            params.getTemplateId(), TemplatModuleType.MIAOSHA.getValue()), readOnly));
        data.setTehuiList(convertToKaixue(params, orgTemplateCourseService.getByOrgIdTemplate(orgId,
            params.getTemplateId(), TemplatModuleType.TEHUI.getValue()), readOnly));

        logger.info("data=={},", data);

        List<DefaultTemplateDataList> grid =
            this.getOrgCourseSubjects(orgId, model.getNumber(), params.getMonkeySiteHost(), params.getGsxSiteHost());

        List<DefaultTemplateDataList> tempGrid = getTempGrids(orgId, grid);
        data.setClassifyCourseList(getOrgHotCourseList(model.getNumber(), tempGrid, params.getImgServer(),
            params.getMonkeySiteHost(), readOnly));

        JSONObject kaixueJson = JSONObject.fromObject(content);
        kaixueJson.put("orgInfo", JSONObject.fromObject(model));

        JSONObject dataJson = JSONObject.fromObject(data);
        kaixueJson.put("data", dataJson);

        return kaixueJson;
    }

    /**
     * 渲染早教模板
     *
     * @param orgId
     * @return
     */
    private JSONObject wrapperZaojiaoTemplate(Integer orgId, String content, OptionalParams params, Boolean readOnly) {
        OrgBaseInfoModel model = getAndSetOrgInfoModel(orgId, params);

        K12TemplateData data = new K12TemplateData();

        List<DefaultTemplateDataList> grid =
            this.getOrgCourseSubjects(orgId, model.getNumber(), params.getMonkeySiteHost(), params.getGsxSiteHost());

        List<DefaultTemplateDataList> tempGrid = getTempGrids(orgId, grid);
        data.setClassifyCourseList(getOrgHotCourseList(model.getNumber(), tempGrid, params.getImgServer(),
            params.getMonkeySiteHost(), readOnly));

        JSONObject zaojiaoJson = JSONObject.fromObject(content);
        zaojiaoJson.put("orgInfo", JSONObject.fromObject(model));

        JSONObject dataJson = JSONObject.fromObject(data);
        zaojiaoJson.put("data", dataJson);

        return zaojiaoJson;
    }

    /**
     * 编辑状态时，空的分类也要显示
     *
     * @param orgId
     * @return
     */
    private List<DefaultTemplateDataList> getTempGrids(Integer orgId, List<DefaultTemplateDataList> grids) {
        List<DefaultTemplateDataList> tempGrids = Lists.newArrayList();
        Map<Integer, DefaultTemplateDataList> existGrids = Maps.newHashMap();
        // 有课的分类
        if (CollectionUtils.isNotEmpty(grids)) {
            for (DefaultTemplateDataList grid : grids) {
                if (grid.getCourseGroupId() != null) {
                    existGrids.put(grid.getCourseGroupId(), grid);
                }
            }
            if (existGrids.size() >= BizConf.MAX_COURSE_GROUP) {
                return grids;
            }
        }
        List<OrgCourseGroup> orgCourseGroupList = this.orgCourseGroupDao.getOrgCourseGroupList(orgId);
        if (CollectionUtils.isNotEmpty(orgCourseGroupList)) {
            for (OrgCourseGroup group : orgCourseGroupList) {
                if (tempGrids.size() >= BizConf.MAX_COURSE_GROUP) {
                    break;
                }
                // 已经存在的，要保持顺序
                if (!existGrids.containsKey(group.getId())) {
                    DefaultTemplateDataList item = new DefaultTemplateDataList();
                    item.setCourseGroupId(group.getId());
                    item.setName(group.getName());
                    tempGrids.add(item);
                } else {
                    tempGrids.add(existGrids.get(group.getId()));
                }
            }
        }
        logger.info("####################### tempGrids:{}", tempGrids);
        return tempGrids;
    }

    /**
     * 获取机构自定义课程分类
     *
     * @param orgId
     * @return
     */
    private List<DefaultTemplateDataList> getOrgCourseSubjects(Integer orgId, Integer number, String monkeySiteHost,
        String gsxSiteHost) {
        List<OrgCourseGroup> orgCourseGroupList = this.orgCourseGroupDao.getOrgCourseGroupList(orgId);
        List<DefaultTemplateDataList> gridList = Lists.newArrayList();
        String link = getLink(BizConf.ORG_COURSE_LINK, number, monkeySiteHost, gsxSiteHost);

        DefaultTemplateDataList item = null;
        if (orgCourseGroupList != null && !orgCourseGroupList.isEmpty()) {
            for (OrgCourseGroup courseGroup : orgCourseGroupList) {
                boolean valid = isValidSubject(number, courseGroup.getId(), CourseType.CLASS.getCode());
                if (!valid) {
                    valid = isValidSubject(number, courseGroup.getId(), CourseType.ORG_COURSE.getCode());
                }
                if (valid) {
                    item = new DefaultTemplateDataList();
                    item.setLink(link);
                    item.setName(courseGroup.getName());
                    item.setCourseGroupId(courseGroup.getId());
                    gridList.add(item);
                }
                if (gridList.size() == BizConf.MAX_COURSE_GROUP) {
                    break;
                }
            }
        }

        gridList.add(new DefaultTemplateDataList("名师推荐",
            getLink(BizConf.ORG_TEACHER_LINK, number, monkeySiteHost, gsxSiteHost)));
        gridList.add(new DefaultTemplateDataList("学员评价",
            getLink(BizConf.ORG_COMMENT_LINK, number, monkeySiteHost, gsxSiteHost)));
        gridList.add(
            new DefaultTemplateDataList("校区查询", getLink(BizConf.ORG_BRANCH_LINK, number, monkeySiteHost, gsxSiteHost)));
        return gridList;
    }

    /**
     * 验证课程分类是否需要显示
     *
     * @return
     */
    private boolean isValidSubject(Integer orgNumber, Integer courseGroupId, Integer courseType) {
        int count = 0;
        List<Long> courseIds = this.orgCourseGroupRelateDao.getCourseIdsByGroup(courseGroupId, courseType);
        if (courseIds != null && !courseIds.isEmpty()) {
            if (courseType == CourseType.CLASS.getCode()) {
                count = this.teacherClassCourseDao.getValidClassCourseCountByIds(orgNumber, courseIds);
            } else if (courseType == CourseType.ORG_COURSE.getCode()) {
                count = this.orgCourseDao.getValidOrgCourseCount(orgNumber, courseIds);
            }
        }
        return count > 0;
    }

    /**
     * 获取m站具体tab页链接地址
     *
     * @param url
     * @param number
     * @param monkeySizeHost
     * @return
     */
    private String getLink(String url, Integer number, String monkeySizeHost, String gsxSiteHost) {
        if (url.contains("{host}")) {
            return url.replaceAll("\\{host\\}", monkeySizeHost).replaceAll("\\{number\\}", number.toString());
        } else {
            return url.replaceAll("\\{host1\\}", gsxSiteHost).replaceAll("\\{number\\}", number.toString());
        }
    }

    /**
     * 获取机构推荐课程列表
     *
     * @param orgId 机构ID
     * @return
     */
    private List<OrgRecommendCourseList> getAndSetRecommentCourseList(Integer orgId, OptionalParams params) {
        List<OrgRecommendCourse> recommendCourses = orgRecommendCourseDao.getRecommendCourseList(orgId);

        String imgServer = params.getImgServer();
        String monkeySite = params.getMonkeySiteHost();
        String gsxSite = params.getGsxSiteHost();
        List<OrgRecommendCourseList> data = Lists.newArrayList();
        if (recommendCourses != null && !recommendCourses.isEmpty()) {
            OrgRecommendCourseList subData = null;

            Map<Long, OrgCourse> orgCourseCache = this.getAndCacheOrgCourseInfo(recommendCourses);
            Map<Long, TeacherClassCourse> classCourseCache = this.getAndCacheClassCourseInfo(recommendCourses);
            for (OrgRecommendCourse course : recommendCourses) {
                Integer courseId = course.getCourseid();
                Integer courseType = course.getCourseType();

                subData = new OrgRecommendCourseList();
                if (courseType == CourseType.CLASS.getCode()) {
                    if (classCourseCache.containsKey(courseId.longValue())) {
                        TeacherClassCourse po = classCourseCache.get(courseId.longValue());
                        // TeacherClassCourse po = this.teacherClassCourseDao.getById(courseId);
                        if (po.getStatus() == TeacherClassCourseStatus.ENROLLING.getStatus()
                            && po.getBeginTime().getTime() > System.currentTimeMillis()) {
                            subData.setCourseType(courseType);
                            subData.setName(po.getName());
                            subData.setNumber(po.getNumber());
                            subData.setPrice(po.getPrice().toString());
                            subData.setPreface("");
                            subData.setCourseId(po.getId());
                            if (po.getCover() != null) {
                                Storage storage = this.getClassCourseStorage(po.getCover());
                                subData.setPreface(constructUrl(storage, imgServer));
                            }
                            subData.setLink(BizConf.CLASS_COURSE_DETAIL.replace("{host1}", gsxSite).replace("{number}",
                                po.getNumber().toString()));
                            data.add(subData);
                        }
                    }
                } else if (courseType == CourseType.ORG_COURSE.getCode()) {
                    if (orgCourseCache.containsKey(courseId.longValue())) {
                        OrgCourse po = orgCourseCache.get(courseId.longValue());
                        //OrgCourse po = this.orgCourseDao.getById(courseId);
                        if (po != null && po.getStatus() == OrgCourseStatus.IN_PROGRESS.getCode()) {
                            subData.setCourseType(courseType);
                            subData.setName(po.getName());
                            subData.setNumber(po.getNumber());
                            subData.setPrice(po.getPrice().toString());
                            subData.setPreface("");
                            subData.setCourseId(po.getId());
                            if (po.getCover() != null) {
                                Storage storage = this.storageDao.getById(po.getCover());
                                if (storage != null) {
                                    subData.setPreface(constructUrl(storage, imgServer));
                                }
                            }
                            subData.setLink(BizConf.ORG_COURSE_DETAIL.replace("{host}", monkeySite).replace("{number}",
                                po.getNumber().toString()));
                            data.add(subData);
                        }
                    }
                } else {
                    continue;
                }
            }
        }

        return data;
    }

    /**
     * 批量查询3810课程信息并缓存
     *
     * @param recommendCourses
     */
    private Map<Long, OrgCourse> getAndCacheOrgCourseInfo(List<OrgRecommendCourse> recommendCourses) {
        List<Long> orgCourseIds = Lists.newArrayList();
        for (OrgRecommendCourse course : recommendCourses) {
            Long courseId = course.getCourseid().longValue();
            Integer courseType = course.getCourseType();
            if (courseType == CourseType.ORG_COURSE.getCode()) {
                orgCourseIds.add(courseId);
            }
        }

        return this.getOrgCourseCache(orgCourseIds);
    }

    /**
     * 批量查询班课信息并缓存
     *
     * @param recommendCourses
     */
    private Map<Long, TeacherClassCourse> getAndCacheClassCourseInfo(List<OrgRecommendCourse> recommendCourses) {
        List<Long> classCourseIds = Lists.newArrayList();
        for (OrgRecommendCourse course : recommendCourses) {
            Long courseId = course.getCourseid().longValue();
            Integer courseType = course.getCourseType();
            if (courseType == CourseType.CLASS.getCode()) {
                classCourseIds.add(courseId);
            }
        }

        return getClassCourseCache(classCourseIds);
    }

    /**
     * 批量查询3810课程信息
     */
    private Map<Long, OrgCourse> getOrgCourseCache(List<Long> ids) {
        return this.orgCourseDao.getOrgCourseMap(ids);
    }

    /**
     * 批量查询班课信息
     */
    private Map<Long, TeacherClassCourse> getClassCourseCache(List<Long> ids) {
        return this.teacherClassCourseDao.getTeacherClassCourses(ids);
    }

    /**
     * 1.如果机构有自定义分类，取10个机构的自定义分类 2.如果机构没有自定义分类，取机构班课+3810课的10个分类
     *
     * @param orgId
     * @param number
     * @return
     */
    private List<String> getOrgCourseSubject(Integer orgId, Integer number, int limit) {
        List<String> subjects = this.getOrgCustomClassfication(orgId);

        // Set<String> result = Sets.newHashSet();
        // if (subjects != null && !subjects.isEmpty()) {
        //
        // if (subjects.size() < limit) {
        //// Set<String> names = getClassCourseAndOrgCourseSubjects(number);
        //// result.addAll(subjects);
        //// if (names != null && !names.isEmpty()) {
        // for (String name : subjects) {
        // result.add(name);
        // if (result.size() == limit) {
        // break;
        // }
        // }
        //// }
        // }
        // } else {
        // Set<String> names = getClassCourseAndOrgCourseSubjects(number);
        // if (names != null && !names.isEmpty()) {
        // for (String name : names) {
        // result.add(name);
        // if (result.size() == limit) {
        // break;
        // }
        // }
        // }
        // }

        logger.info("getOrgCourseSubject.orgId:{}, number:{}, limit:{}, subjects:{}", orgId, number, limit, subjects);
        return subjects;
    }

    /**
     * 获取机构班课、3810课程id集合
     *
     * @param courseType
     * @return
     */
    private <T> Collection<Long> getPropertiesVals(List<T> objs, CourseType courseType, String cType, String courseId) {
        Set<Long> excludeIds = Sets.newHashSet();
        switch (courseType) {
            case CLASS:
                for (T obj : objs) {
                    Integer type = BaseUtils.getValue(obj, cType);
                    if (type == CourseType.CLASS.getCode()) {
                        excludeIds.add((Long) BaseUtils.getValue(obj, courseId));
                    }
                }
                break;
            case ORG_COURSE:
                for (T obj : objs) {
                    Integer type = BaseUtils.getValue(obj, cType);
                    if (type == CourseType.ORG_COURSE.getCode()) {
                        excludeIds.add((Long) BaseUtils.getValue(obj, courseId));
                    }
                }
                break;
            default:
                break;
        }
        return excludeIds;
    }

    /**
     * 获取机构热门推荐课程
     *
     * @param grids
     * @return
     */
    private List<OrgHotCourseDto> getOrgHotCourseList(Integer orgNumber, List<DefaultTemplateDataList> grids,
        String imgServer, String monkeySite, Boolean readOnly) {
        List<OrgHotCourseDto> result = Lists.newArrayList();

        List<OrgRecommendCourseList> subResult = null;
        OrgHotCourseDto subItem = null;
        for (DefaultTemplateDataList grid : grids) {
            subItem = new OrgHotCourseDto();
            Integer courseGroupId = grid.getCourseGroupId();
            if (courseGroupId == null) {
                continue;
            }

            subItem.setGroupId(courseGroupId);

            // 机构班课
            List<Long> ids =
                this.orgCourseGroupRelateDao.getCourseIdsByGroup(courseGroupId, CourseType.CLASS.getCode());

            subResult = Lists.newArrayList();
            if (ids != null && !ids.isEmpty()) {
                subResult.addAll(this.getHotClassCourse(ids, imgServer, monkeySite));
            }

            List<Long> orgCourseIds =
                this.orgCourseGroupRelateDao.getCourseIdsByGroup(courseGroupId, CourseType.ORG_COURSE.getCode());
            if (orgCourseIds != null && !orgCourseIds.isEmpty()) {
                subResult.addAll(this.getHotOrgCourse(orgCourseIds, imgServer, monkeySite));
            }

            subItem.setName(grid.getName());
            if (subResult != null && !subResult.isEmpty()) {
                if (readOnly) {
                    // 大于四个显示四个，2、3个时显示2个
                    if (subResult.size() >= BizConf.MAX_COURSE_DISPLAY) {
                        subItem.setList(subResult.subList(0, BizConf.MAX_COURSE_DISPLAY));
                        result.add(subItem);
                    } else if (subResult.size() >= BizConf.MAX_COURSE_DISPLAY / 2
                        && subResult.size() < BizConf.MAX_COURSE_DISPLAY) {
                        subItem.setList(subResult.subList(0, BizConf.MAX_COURSE_DISPLAY / 2));
                        result.add(subItem);
                    } else {
                        continue;
                    }
                } else {
                    // 编辑模式下，不大于四个就给显示了
                    if (subResult.size() >= BizConf.MAX_COURSE_DISPLAY) {
                        subItem.setList(subResult.subList(0, BizConf.MAX_COURSE_DISPLAY));
                        result.add(subItem);
                    } else {
                        subItem.setList(subResult);
                        result.add(subItem);
                    }
                }
            } else {
                if (readOnly) {
                    continue;
                } else {
                    // 编辑模式下，空的也要显示
                    result.add(subItem);
                }
            }

            logger.info("<<<<------------------------------->>>>");
            logger.info("orgNumber:{}, courseGroupId:{}, ids:{}, orgCourseIds:{}, subResult:{}, result:{}", orgNumber,
                courseGroupId, ids, orgCourseIds, JSONArray.fromObject(subResult).toString(),
                JSONArray.fromObject(result).toString());
            logger.info("<<<<------------------------------->>>>");
        }

        return result;
    }

    private List<OrgRecommendCourseList> getOrgHotCourseList(Integer orgId, Integer number, OptionalParams params) {
        PageDto pageDto = new PageDto();
        pageDto.setPageSize(BizConf.MAX_SUBJECT_DISPLAY);

        String imgServer = params.getImgServer();
        String monkeySite = params.getMonkeySiteHost();
        List<OrgRecommendCourseList> courses = Lists.newArrayList();
        List<OrgCourseGroupRelate> recommendCourseLists = this.getOrgCustomGroupCourse(orgId, pageDto);
        List<TeacherClassCourse> classCourses = Lists.newArrayList();
        List<OrgCourse> orgCourses = Lists.newArrayList();
        if (recommendCourseLists != null && !recommendCourseLists.isEmpty()) {
            int size = recommendCourseLists.size();
            if (BizConf.MAX_SUBJECT_DISPLAY > size) {
                pageDto.setPageSize(BizConf.MAX_SUBJECT_DISPLAY - size);
                classCourses = getOrgTeacherClassCourseList(number,
                    getPropertiesVals(recommendCourseLists, CourseType.CLASS, "courseType", "courseId"), pageDto);
                if (BizConf.MAX_SUBJECT_DISPLAY > (size + classCourses.size())) {
                    orgCourses = getOrgCourseList(number,
                        getPropertiesVals(recommendCourseLists, CourseType.ORG_COURSE, "courseType", "courseId"),
                        pageDto);
                }
            }
        } else {
            pageDto.setPageSize(BizConf.MAX_SUBJECT_DISPLAY);
            classCourses = getOrgTeacherClassCourseList(number, null, pageDto);
            int size = classCourses.size();
            if (classCourses != null && !classCourses.isEmpty() && BizConf.MAX_SUBJECT_DISPLAY > size) {
                pageDto.setPageSize(BizConf.MAX_SUBJECT_DISPLAY - size);
                orgCourses = this.getOrgCourseList(number, null, pageDto);
            }
        }

        if (recommendCourseLists != null && !recommendCourseLists.isEmpty()) {
            for (OrgCourseGroupRelate recommendCourse : recommendCourseLists) {
                OrgRecommendCourseList dto = null;
                if (recommendCourse.getCourseType() == CourseType.ORG_COURSE.getCode()) {
                    dto = this.getHotOrgCourse(recommendCourse.getCourseId(), imgServer, monkeySite);
                } else if (recommendCourse.getCourseType() == CourseType.CLASS.getCode()) {
                    dto = this.getHotClassCourse(recommendCourse.getCourseId(), imgServer, monkeySite);
                }
                if (dto != null) {
                    courses.add(dto);
                }
            }
        }

        if (classCourses != null && !classCourses.isEmpty()) {
            for (TeacherClassCourse classCourse : classCourses) {
                OrgRecommendCourseList dto = this.getHotClassCourse(classCourse.getId(), imgServer, monkeySite);
                if (dto != null) {
                    courses.add(dto);
                }
            }
        }

        if (orgCourses != null && !orgCourses.isEmpty()) {
            for (OrgCourse orgCourse : orgCourses) {
                OrgRecommendCourseList dto = this.getHotOrgCourse(orgCourse.getId(), imgServer, monkeySite);
                if (dto != null) {
                    courses.add(dto);
                }
            }
        }
        return courses;
    }

    /**
     * 组装3810课程信息
     *
     * @param courseId
     * @param imgServer
     * @return
     */
    private OrgRecommendCourseList getHotOrgCourse(Long courseId, String imgServer, String monkeySite) {
        OrgCourse orgCourse = this.orgCourseDao.getById(courseId);
        OrgRecommendCourseList dto = null;
        if (orgCourse != null) {
            if (orgCourse.getStatus() == OrgCourseStatus.IN_PROGRESS.getCode()) {
                dto = new OrgRecommendCourseList();
                dto.setCourseType(CourseType.ORG_COURSE.getCode());
                dto.setName(orgCourse.getName());
                dto.setNumber(orgCourse.getNumber());
                dto.setPrice(orgCourse.getPrice().toString());
                dto.setCourseId(orgCourse.getId());
                Integer cover = orgCourse.getCover();
                if (cover != null) {
                    Storage storage = this.getStorage(cover.longValue());
                    String url = constructUrl(storage, imgServer);
                    dto.setPreface(url);
                }

                dto.setLink(BizConf.ORG_COURSE_DETAIL.replace("{host}", monkeySite).replace("{number}",
                    orgCourse.getNumber().toString()));
            }
        }

        return dto;
    }

    /**
     * 组装机构班课信息
     *
     * @param courseId
     * @param imgServer
     * @return
     */
    private OrgRecommendCourseList getHotClassCourse(Long courseId, String imgServer, String monkeySite) {
        TeacherClassCourse classCourse = this.teacherClassCourseDao.getById(courseId);
        OrgRecommendCourseList dto = null;
        if (classCourse != null && classCourse.getStatus() == ClassCourseStatus.ENROLLING.getState()
            && classCourse.getBeginTime().getTime() > System.currentTimeMillis()) {
            dto = new OrgRecommendCourseList();
            dto.setCourseType(CourseType.CLASS.getCode());
            dto.setName(classCourse.getName());
            dto.setNumber(classCourse.getNumber());
            dto.setPrice(classCourse.getPrice().toString());
            dto.setCourseId(classCourse.getId());
            Long cover = classCourse.getCover();
            if (cover != null) {
                Storage storage = this.getClassCourseStorage(cover.longValue());
                String url = constructUrl(storage, imgServer);
                dto.setPreface(url);
            }
            dto.setLink(BizConf.CLASS_COURSE_DETAIL.replace("{host}", monkeySite).replace("{number}",
                classCourse.getNumber().toString()));
        }

        return dto;
    }

    /**
     * 组装机构班课信息
     *
     * @param imgServer
     * @return
     */
    private List<OrgRecommendCourseList> getHotClassCourse(List<Long> courseIds, String imgServer, String monkeySite) {
        List<OrgRecommendCourseList> result = Lists.newArrayList();
        if (courseIds != null && !courseIds.isEmpty()) {
            for (Long courseId : courseIds) {
                OrgRecommendCourseList ret = this.getHotClassCourse(courseId, imgServer, monkeySite);
                if (ret != null) {
                    result.add(ret);
                }
            }
        }
        return result;
    }

    /**
     * 组装机构班课信息
     *
     * @param imgServer
     * @return
     */
    private List<OrgRecommendCourseList> getHotOrgCourse(List<Long> courseIds, String imgServer, String monkeySite) {
        List<OrgRecommendCourseList> result = Lists.newArrayList();
        if (courseIds != null && !courseIds.isEmpty()) {
            for (Long courseId : courseIds) {
                OrgRecommendCourseList ret = this.getHotOrgCourse(courseId, imgServer, monkeySite);
                if (ret != null) {
                    result.add(ret);
                }
            }
        }
        return result;
    }

    /**
     * 获取机构自定义课程分类
     *
     * @param orgId
     * @return
     */
    private List<OrgCourseGroupRelate> getOrgCustomGroupCourse(Integer orgId, PageDto pageDto) {
        return orgCourseGroupRelateDao.getOrgCourseGroup(orgId, pageDto);
    }

    /**
     * 获取机构正在招生的班课列表
     *
     * @param orgNumber 机构number
     * @param pageDto
     * @return
     */
    private List<TeacherClassCourse> getOrgTeacherClassCourseList(Integer orgNumber, Collection<Long> excludeIds,
        PageDto pageDto) {
        return this.teacherClassCourseDao.getOrgCourseList(orgNumber, excludeIds, pageDto);
    }

    /**
     * 获取机构未下架的3810课程
     *
     * @param orgNumber 机构number
     * @param pageDto
     * @return
     */
    private List<OrgCourse> getOrgCourseList(Integer orgNumber, Collection<Long> excludeIds, PageDto pageDto) {
        return this.orgCourseDao.getAndFilterOrgCourse(orgNumber, excludeIds, CourseTypeEnum.IS_COURSE_TRUE.getCode(),
            null, null, pageDto);
    }

    /**
     * 1:首先取用户自定义课程分类4个 1.1:如果自定义分类等于4个,返回这四个分类
     *
     * @param subjects
     * @param limit
     * @return
     */
    public List<DefaultTemplateDataList> getCourseSubjects(List<String> subjects, int limit, OrgBaseInfoModel model,
        OptionalParams params) {
        List<DefaultTemplateDataList> data = Lists.newArrayList();

        DefaultTemplateDataList item = null;

        String courseUrl = BizConf.ORG_COURSE_LINK.replace("{host}", params.getMonkeySiteHost()).replace("{number}",
            model.getNumber().toString());
        if (subjects != null && !subjects.isEmpty()) {
            int index = 0;
            for (String name : subjects) {
                item = new DefaultTemplateDataList();
                item.setName(name);
                item.setLink(courseUrl);
                data.add(item);
                index++;
                if (index == limit) {
                    break;
                }
            }
        } else {
            int index = 1;
            for (String link : BizConf.PHOTO_LINKS) {
                item = new DefaultTemplateDataList();
                item.setLink(courseUrl);
                item.setName("分类" + index);
                data.add(item);
                index++;
            }
        }

        return data;
    }

    /**
     * 获取机构班课和3810课程分类列表
     *
     * @param number
     * @return
     */
    private Set<String> getClassCourseAndOrgCourseSubjects(Integer number) {
        List<Integer> classCourseSubjectIds = getTeacherClassCourseSubjects(number);
        List<Integer> orgCourseSubjectIds = getOrgCourseSubjects(number);

        Set<String> result = Sets.newHashSet();
        Set<Integer> subjectIds = Sets.newHashSet();
        subjectIds.addAll(classCourseSubjectIds);
        subjectIds.addAll(orgCourseSubjectIds);

        if (subjectIds != null && !subjectIds.isEmpty()) {
            List<String> subLists = this.subjectDao.getSubjectNames(subjectIds, "name");
            result.addAll(subLists);
        }
        return result;
    }

    /**
     * 获取机构自定义分类列表
     *
     * @param orgId jigou Id
     */
    private List<String> getOrgCustomClassfication(Integer orgId) {
        PageDto page = new PageDto();
        page.setPageSize(BizConf.MAX_SUBJECT_DISPLAY);
        return this.orgCourseGroupDao.getDistinctSubject(orgId, page);
    }

    /**
     * 获取机构3810课程分类列表
     *
     * @param number
     * @return
     */
    private List<Integer> getOrgCourseSubjects(Integer number) {
        return orgCourseDao.getOrgCourseSubjectIds(number, null, null, null);
    }

    /**
     * 获取机构班课课程分类列表
     *
     * @param number
     * @return
     */
    private List<Integer> getTeacherClassCourseSubjects(Integer number) {
        return teacherClassCourseDao.getOrgDistinctSubject(number);
    }

    Storage getClassCourseStorage(Long storageId) {
        ClassCoursePhoto cover = this.classCoursePhotoDao.getById(storageId);
        return this.storageDao.getById(cover.getStorageId());
    }

    Storage getStorage(Long storageId) {
        return this.storageDao.getById(storageId);
    }

    /**
     * 渲染出国留学模板
     *
     * @param orgId
     * @return
     */
    private JSONObject wrapperAbroadTemplate(Integer orgId, String content, OptionalParams params) {
        OrgBaseInfoModel model = this.getAndSetOrgInfoModel(orgId, params);
        JSONObject abroadJson = JSONObject.fromObject(content);

        abroadJson.put("orgInfo", JSONObject.fromObject(model));
        return abroadJson;
    }

    /**
     * 设置机构number和domain
     *
     * @param orgId
     * @param model
     */
    private void getAndSetOrgNumber(Integer orgId, OrgBaseInfoModel model, String host) {
        OrgAccount orgAccount = this.orgAccountDao.getAccountById(orgId);
        if (orgAccount != null) {
            model.setNumber(orgAccount.getNumber());
            model.setDomain(host + "/i/" + model.getNumber());
            getAndSetOrgInfo(orgId, model);
            getAndSetOrgCourse(orgAccount.getNumber(), model);
        } else {
            throw new BussinessException(OrgErrorCode.USER_NOT_FOUND);
        }
    }

    /**
     * 设置机构基本信息
     *
     * @param orgId
     * @param model
     */
    private void getAndSetOrgInfo(Integer orgId, OrgBaseInfoModel model) {
        OrgInfo orgInfo = orgInfoDao.getOrgInfo(orgId, new String[] {});
        if (orgInfo != null) {
            String _400_phone = orgInfo.getExtension();
            if (StringUtils.isNotBlank(_400_phone)) {
                model.setExtension(BizConf._400_PHONE_PREFIX + "," + orgInfo.getExtension());
            } else {
                model.setExtension(BizConf._400_PHONE_PREFIX);
            }
            model.setId(orgId);
            model.setShortName(orgInfo.getShortName());
        } else {
            throw new BussinessException(OrgErrorCode.USER_NOT_FOUND);
        }
    }

    /**
     * 设置机构简介
     *
     * @param orgId
     * @param model
     */
    private void getAndSetSummary(Integer orgId, OrgBaseInfoModel model) {
        OrgTxtMsg summary = orgTxtMsgDao.getOrgTxtByIdAndType(orgId, AuditColumType.SUMMARY.getValue());
        if (summary != null) {
            model.setBrief(summary.getValue());
        }
    }

    /**
     * 设置机构分校区地址
     *
     * @param orgId
     * @param model
     */
    private void getAndSetAddress(Integer orgId, OrgBaseInfoModel model) {
        List<OrgBranchs> branchs = this.orgBranchsDao.listBranchsByOrgId(orgId);
        if (branchs != null && !branchs.isEmpty()) {
            OrgBranchs branch = branchs.get(0);
            if (branch != null) {
                branch.setProvince(branch.getProvince() == null ? "" : branch.getProvince());
                branch.setCounty(branch.getCounty() == null ? "" : branch.getCounty());
                branch.setDistrict(branch.getDistrict() == null ? "" : branch.getDistrict());
                branch.setAddress(branch.getAddress() == null ? "" : branch.getAddress());
                model
                    .setAddress(branch.getProvince() + branch.getCounty() + branch.getDistrict() + branch.getAddress());
            } else {
                model.setAddress("");
            }
        } else {
            model.setAddress("");
        }
    }

    /**
     * 设置机构特色标签
     *
     * @param orgId
     * @param model
     */
    private void getAndSetTags(Integer orgId, OrgBaseInfoModel model) {
        OrgTxtMsg tags = orgTxtMsgDao.getOrgTxtByIdAndType(orgId, AuditColumType.TAGS.getValue());
        if (tags != null) {
            String value = tags.getValue();
            if (StringUtils.isNotBlank(value)) {
                model.setTags(value.split("\\s+"));
            }
        }
    }

    /**
     * 设置机构logo
     *
     * @param orgId
     * @param model
     */
    private void getAndSetOrgLogo(Integer orgId, OrgBaseInfoModel model, String imgServer) {
        List<OrgPhoto> imgs = this.orgPhotoDao.getByOrgIdAndCategory(orgId, OrgImgType.ORG_LOGO.getValue());

        logger.info("getAndSetOrgLogo.orgId:{}, imgServer:{}, logogs:{}", orgId, imgServer,
            JSONArray.fromObject(imgs).toString());
        if (imgs != null && !imgs.isEmpty()) {
            OrgPhoto logo = imgs.get(0);
            Integer storageId = logo.getOrgStorageId();
            OrgStorage storage = this.orgStorageDao.getById(storageId);
            if (storage != null) {
                model.setLogo(constructUrl(storage, imgServer));
            }
        } else {
            model.setLogo(BizConf.DEFAULT_LOGO_URL);
        }
    }

    /**
     * 获取机构可报名的3810课程和班课数
     *
     * @param orgNumber
     * @param model
     */
    private void getAndSetOrgCourse(Integer orgNumber, OrgBaseInfoModel model) {
        int classCourseCount = this.teacherClassCourseDao.getVliadClassCourseCount(orgNumber);
        int microCourseCount =
            this.orgCourseDao.getValidOrgCourseCount(orgNumber, CourseTypeEnum.IS_COURSE_TRUE.getCode(), null, null);
        model.setCourseCount(microCourseCount);
    }

    /**
     * 获取机构logo地址
     *
     * @param storage
     * @return
     */
    public String constructUrl(OrgStorage storage, String imgServer) {
        StringBuilder sb = new StringBuilder(imgServer);
        String surfix = ((MIMEType.values())[storage.getMimeType() - 1]).getExtension();
        return sb.append(storage.getFid()).append("_").append(storage.getSn()).append(".").append(surfix).toString();
    }

    /**
     * 获取cdb.storage表数据对应图片url
     *
     * @param storage
     * @return
     */
    public String constructUrl(Storage storage, String imgServer) {
        StringBuilder sb = new StringBuilder(imgServer);
        String surfix = "";
        if (storage == null) {
            return "";
        }
        if (StringUtils.isNotBlank(storage.getMimetype())) {
            if (storage.getMimetype().contains("/")) {
                surfix = storage.getMimetype().split("/")[1];
            } else {
                surfix = storage.getMimetype();
            }
        }
        return sb.append(storage.getFid()).append("_").append(storage.getSn()).append(".").append(surfix).toString();
    }

    // 机构m关于我们链接地址
    private List<DefaultTemplateDataList> getAboutUsLinks(Integer number, String monkeySiteHost, String gsxSiteHost) {
        DefaultTemplateDataList[] list = new DefaultTemplateDataList[] {
            new DefaultTemplateDataList("关于我们",
                getRealAddress(BizConf.ABOUT_US, monkeySiteHost, gsxSiteHost, number.toString())),
            new DefaultTemplateDataList("最新课程",
                getRealAddress(BizConf.ORG_COURSE_LINK, monkeySiteHost, gsxSiteHost, number.toString())),
            new DefaultTemplateDataList("老师团队",
                getRealAddress(BizConf.ORG_TEACHER_LINK, monkeySiteHost, gsxSiteHost, number.toString())),
            new DefaultTemplateDataList("在线预约", "javascript:;"),
            new DefaultTemplateDataList("学员之声",
                getRealAddress(BizConf.ORG_COMMENT_LINK, monkeySiteHost, gsxSiteHost, number.toString())),
            new DefaultTemplateDataList("优惠活动",
                getRealAddress(BizConf.ORG_COUPON_LINK, monkeySiteHost, gsxSiteHost, number.toString())), };

        return Arrays.asList(list);
    }

    private String getRealAddress(String source, String monkeySiteHost, String gsxSiteHost, String number) {
        if (source.contains("{host}")) {
            return source.replace("{host}", monkeySiteHost).replace("{number}", number);
        } else {
            return source.replace("{host1}", gsxSiteHost).replace("{number}", number);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.baijia.tianxiao.sal.organization.org.service.OrgIndexService#preview(java.lang.Integer)
     */

    @Override
    public OrgBaseInfoModel preview(Integer orgId, OptionalParams params) {
        return this.getAndSetOrgInfoModel(orgId, params);
    }

    private List<OrgRecommendCourseList> convertToKaixue(OptionalParams params, List<OrgTemplateCourse> list,
        boolean readonly) {
        String imgServer = params.getImgServer();
        String monkeySite = params.getMonkeySiteHost();

        List<OrgRecommendCourseList> result = Lists.newArrayList();
        for (OrgTemplateCourse templateCourse : list) {
            if (templateCourse.getCourseType().intValue() == CourseType.CLASS.getCode()) {
                OrgRecommendCourseList dto =
                    this.getHotClassCourse(Long.valueOf(templateCourse.getCourseId()), imgServer, monkeySite);
                if (dto != null) {
                    result.add(dto);
                }
            } else if (templateCourse.getCourseType().intValue() == CourseType.ORG_COURSE.getCode()) {
                OrgRecommendCourseList dto =
                    this.getHotOrgCourse(Long.valueOf(templateCourse.getCourseId()), imgServer, monkeySite);
                if (dto != null) {
                    result.add(dto);
                }
            }
        }
        if (readonly) {
            if (result.size() == 1 || result.size() == 3) {
                result.remove(result.size() - 1);
            }
        }
        logger.info("convertToKaixue ==params={},result={}", params, result);

        return result;
    }
}
