package com.baijia.tianxiao.sal.kexiao.service.impl;

import com.baijia.tianxiao.dal.constant.ChargeUnit;
import com.baijia.tianxiao.dal.finance.dao.TxTransferClassInfoDao;
import com.baijia.tianxiao.dal.finance.po.TxTransferClassInfo;
import com.baijia.tianxiao.dal.signup.dao.OrgSignupCourseDao;
import com.baijia.tianxiao.dal.signup.dao.OrgSignupCourseLessonDao;
import com.baijia.tianxiao.dal.signup.dao.OrgSignupRefundDao;
import com.baijia.tianxiao.dal.signup.po.OrgSignupCourse;
import com.baijia.tianxiao.dal.signup.po.OrgSignupCourseLesson;
import com.baijia.tianxiao.dal.signup.po.OrgSignupRefund;
import com.baijia.tianxiao.sal.kexiao.dto.LessonKexiaoStatisticsDto;
import com.baijia.tianxiao.sal.kexiao.service.KexiaoStatisticsService;
import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author : fulinyu
 * @title : KexiaoStatisticsServiceImpl
 * @description : 课消相关统计(按校区/班级/学员统计:报名,超排,退费,转出,已消数据===> 求对应的待消数据)
 * @date : 2017年8月23日
 */
@Service
@Slf4j
public class KexiaoStatisticsServiceImpl implements KexiaoStatisticsService {

    private static final int HOUR = 60;

    private static final int HALF_HOUR = 30;

    @Autowired
    private OrgSignupCourseDao orgSignupCourseDao;

    @Autowired
    private OrgSignupCourseLessonDao orgSignupCourseLessonDao;

    @Autowired
    private OrgSignupRefundDao orgSignupRefundDao;

    @Autowired
    private TxTransferClassInfoDao txTransferClassInfoDao;

    @Override
    public Map<Long, LessonKexiaoStatisticsDto> getSignupCourseStatisticsMapOfOrg(Collection<Long> orgIds) {
        Map<Long, LessonKexiaoStatisticsDto> result = Maps.newHashMap();
        List<OrgSignupCourse> signupList = this.orgSignupCourseDao.getSignupCourseSuccessedList(orgIds, null, null);
        this.filterSignupList(signupList);//过滤lesson_count为0的订单
        if (CollectionUtils.isNotEmpty(signupList)) {
            Map<Long, List<OrgSignupCourse>> signupOrgMap = signupList.stream().collect(Collectors.groupingBy(OrgSignupCourse::getOrgId));//按校区分组
            result = this.calculateSignupStatisticsOfGroup(signupOrgMap);
        }
        return result;
    }

    @Override
    public Map<Long, LessonKexiaoStatisticsDto> getSignupCourseStatisticsMapOfClass(Collection<Long> classIds) {
        Map<Long, LessonKexiaoStatisticsDto> result = Maps.newHashMap();
        List<OrgSignupCourse> signupList = this.orgSignupCourseDao.getSignupCourseSuccessedList(null, classIds, null);
        this.filterSignupList(signupList);
        if (CollectionUtils.isNotEmpty(signupList)) {
            Map<Long, List<OrgSignupCourse>> signupClassMap = signupList.stream().collect(Collectors.groupingBy(OrgSignupCourse::getClassId));//按班级分组
            result = this.calculateSignupStatisticsOfGroup(signupClassMap);
        }
        return result;
    }

    @Override
    public Map<Long, LessonKexiaoStatisticsDto> getSignupCourseStatisticsMapOfStudentClass(Long userId, Collection<Long> classIds) {
        Map<Long, LessonKexiaoStatisticsDto> result = Maps.newHashMap();
        List<OrgSignupCourse> signupList = this.orgSignupCourseDao.getSignupCourseSuccessedList(null, classIds, Sets.newHashSet(userId));
        this.filterSignupList(signupList);
        if (CollectionUtils.isNotEmpty(signupList)) {
            Map<Long, List<OrgSignupCourse>> signupClassMap = signupList.stream().collect(Collectors.groupingBy(OrgSignupCourse::getClassId));//按班级分组
            result = this.calculateSignupStatisticsOfGroup(signupClassMap);
        }
        return result;
    }

    @Override
    public Map<Long, LessonKexiaoStatisticsDto> getSignupCourseStatisticsMapOfStudent(Collection<Long> userIds) {
        Map<Long, LessonKexiaoStatisticsDto> result = Maps.newHashMap();
        List<OrgSignupCourse> signupList = this.orgSignupCourseDao.getSignupCourseSuccessedList(null, null, userIds);
        this.filterSignupList(signupList);
        if (CollectionUtils.isNotEmpty(signupList)) {
            Map<Long, List<OrgSignupCourse>> signupStudentMap = signupList.stream().collect(Collectors.groupingBy(OrgSignupCourse::getUserId));//按学员分组
            result = this.calculateSignupStatisticsOfGroup(signupStudentMap);
        }
        return result;
    }

    @Override
    public Map<Long, LessonKexiaoStatisticsDto> getSignupCourseStatisticsMapOfClassStudent(Long classId, Collection<Long> userIds) {
        Map<Long, LessonKexiaoStatisticsDto> result = Maps.newHashMap();
        List<OrgSignupCourse> signupList = this.orgSignupCourseDao.getSignupCourseSuccessedList(null, Sets.newHashSet(classId), userIds);
        this.filterSignupList(signupList);
        if (CollectionUtils.isNotEmpty(signupList)) {
            Map<Long, List<OrgSignupCourse>> signupStudentMap = signupList.stream().collect(Collectors.groupingBy(OrgSignupCourse::getUserId));//按学员分组
            result = this.calculateSignupStatisticsOfGroup(signupStudentMap);
        }
        return result;
    }

    /**
     * 分组计算报名的统计数据
     *
     * @param signupMap
     * @return
     */
    private Map<Long, LessonKexiaoStatisticsDto> calculateSignupStatisticsOfGroup(Map<Long, List<OrgSignupCourse>> signupMap) {
        Map<Long, LessonKexiaoStatisticsDto> result = Maps.newHashMap();
        if (MapUtils.isNotEmpty(signupMap)) {
            for (Long id : signupMap.keySet()) {
                List<OrgSignupCourse> orgSignupList = signupMap.get(id);
                LessonKexiaoStatisticsDto kexiaoStatisticsDto = new LessonKexiaoStatisticsDto();
                for (OrgSignupCourse signup : orgSignupList) {
                    this.upLessonCountAndMoney(kexiaoStatisticsDto, signup.getChargeUnit(), false, signup.getLessonCount(), signup.getTotalPayPrice());
                }
                result.put(id, kexiaoStatisticsDto);
            }
        }
        return result;
    }

    /**
     * 过滤出lessonCount>0的订单[对于过滤出lessonCount=0的订单不统计课消及待消]
     *
     * @param signupList
     */
    private void filterSignupList(List<OrgSignupCourse> signupList) {
        if (CollectionUtils.isNotEmpty(signupList)) {
            signupList = signupList.stream().filter(o -> o.getLessonCount() > 0).collect(Collectors.toList());
        }
    }


    /**
     * 累计课节的数量以及金额<br/>
     * 使用情况: <br/>
     * 1.报名(时长分单位 ignoreDuration = false),<br/>
     * 2.退费(时长不分单位 ignoreDuration = true),<br/>
     * 3.转出(时长不分单位 ignoreDuration = true) <br/>
     *
     * @param kexiaoStatisticsDto
     * @param chargeUnitInt
     * @param ignoreDuration      忽略时长分单位(小时,半小时,分钟)
     * @param count
     * @param money
     */
    private void upLessonCountAndMoney(LessonKexiaoStatisticsDto kexiaoStatisticsDto, Integer chargeUnitInt, boolean ignoreDuration, long count, long money) {
        Preconditions.checkArgument(kexiaoStatisticsDto != null, "课消统计参数为空");
        ChargeUnit chargeUnit = ChargeUnit.getByCode(chargeUnitInt);

        Long minutes = 0l;
        Long times = 0l;
        Long minutesMoney = 0l;
        Long timesMoney = 0l;

        if (ChargeUnit.isByDuration(chargeUnitInt)) {
            minutesMoney = money;
            if (ignoreDuration) {
                minutes = count;
            } else {
                switch (chargeUnit) {
                    case BY_HOUR: {
                        minutes = count * HOUR;
                        break;
                    }
                    case BY_HALF_HOUR: {
                        minutes = count * HALF_HOUR;
                        break;
                    }
                    case BY_MINUTE: {
                        minutes = count;
                        break;
                    }
                }
            }
        } else {
            times = count;
            timesMoney = money;
        }
        kexiaoStatisticsDto.addFromMinutesAndTimes(minutes, minutesMoney, times, timesMoney);
    }


    @Override
    public Map<Long, LessonKexiaoStatisticsDto> getLessonFreeMapOfOrg(Collection<Long> orgIds) {
        Map<Long, LessonKexiaoStatisticsDto> result = Maps.newHashMap();
        List<OrgSignupCourseLesson> lessonList = this.orgSignupCourseLessonDao.getLessonFreeList(orgIds, null, null);
        if (CollectionUtils.isNotEmpty(lessonList)) {
            Map<Long, List<OrgSignupCourseLesson>> lessonFreeOrgMap = lessonList.stream().collect(Collectors.groupingBy(OrgSignupCourseLesson::getOrgId));//按校区分组
            result = this.calculateLessonFreeStatisticsOfGroup(lessonFreeOrgMap);
        }
        return result;
    }


    @Override
    public Map<Long, LessonKexiaoStatisticsDto> getLessonFreeMapOfClass(Collection<Long> classIds) {
        Map<Long, LessonKexiaoStatisticsDto> result = Maps.newHashMap();
        List<OrgSignupCourseLesson> lessonList = this.orgSignupCourseLessonDao.getLessonFreeList(null, classIds, null);
        if (CollectionUtils.isNotEmpty(lessonList)) {
            Map<Long, List<OrgSignupCourseLesson>> lessonFreeClassMap = lessonList.stream().collect(Collectors.groupingBy(OrgSignupCourseLesson::getClassId));//按班级分组
            result = this.calculateLessonFreeStatisticsOfGroup(lessonFreeClassMap);
        }
        return result;
    }

    @Override
    public Map<Long, LessonKexiaoStatisticsDto> getLessonFreeMapOfStudentClass(Long userId, Collection<Long> classIds) {
        Map<Long, LessonKexiaoStatisticsDto> result = Maps.newHashMap();
        List<OrgSignupCourseLesson> lessonList = this.orgSignupCourseLessonDao.getLessonFreeList(null, classIds, Sets.newHashSet(userId));
        if (CollectionUtils.isNotEmpty(lessonList)) {
            Map<Long, List<OrgSignupCourseLesson>> lessonFreeClassMap = lessonList.stream().collect(Collectors.groupingBy(OrgSignupCourseLesson::getClassId));//按班级分组
            result = this.calculateLessonFreeStatisticsOfGroup(lessonFreeClassMap);
        }
        return result;
    }

    @Override
    public Map<Long, LessonKexiaoStatisticsDto> getLessonFreeMapOfStudent(Collection<Long> userIds) {
        Map<Long, LessonKexiaoStatisticsDto> result = Maps.newHashMap();
        List<OrgSignupCourseLesson> lessonList = this.orgSignupCourseLessonDao.getLessonFreeList(null, null, userIds);
        if (CollectionUtils.isNotEmpty(lessonList)) {
            Map<Long, List<OrgSignupCourseLesson>> lessonFreeStudentMap = lessonList.stream().collect(Collectors.groupingBy(OrgSignupCourseLesson::getUserId));//按学员分组
            result = this.calculateLessonFreeStatisticsOfGroup(lessonFreeStudentMap);
        }
        return result;
    }

    @Override
    public Map<Long, LessonKexiaoStatisticsDto> getLessonFreeMapOfClassStudent(Long classId, Collection<Long> userIds) {
        Map<Long, LessonKexiaoStatisticsDto> result = Maps.newHashMap();
        List<OrgSignupCourseLesson> lessonList = this.orgSignupCourseLessonDao.getLessonFreeList(null, Sets.newHashSet(classId), userIds);
        if (CollectionUtils.isNotEmpty(lessonList)) {
            Map<Long, List<OrgSignupCourseLesson>> lessonFreeStudentMap = lessonList.stream().collect(Collectors.groupingBy(OrgSignupCourseLesson::getUserId));//按学员分组
            result = this.calculateLessonFreeStatisticsOfGroup(lessonFreeStudentMap);
        }
        return result;
    }

    /**
     * 分组统计超排(赠送课)
     *
     * @param lessonFreeMap
     * @return
     */
    private Map<Long, LessonKexiaoStatisticsDto> calculateLessonFreeStatisticsOfGroup(Map<Long, List<OrgSignupCourseLesson>> lessonFreeMap) {
        Map<Long, LessonKexiaoStatisticsDto> result = Maps.newHashMap();
        if (MapUtils.isNotEmpty(lessonFreeMap)) {
            for (Long id : lessonFreeMap.keySet()) {
                List<OrgSignupCourseLesson> lessonFreeList = lessonFreeMap.get(id);
                LessonKexiaoStatisticsDto kexiaoStatisticsDto = new LessonKexiaoStatisticsDto();
                for (OrgSignupCourseLesson lessonFree : lessonFreeList) {
                    this.upLessonCountAndMoney(kexiaoStatisticsDto, lessonFree, false);
                }
                result.put(id, kexiaoStatisticsDto);
            }
        }
        return result;
    }


    /**
     * 累计课节的数量以及金额 使用情况: 1.超排(赠送),2.已消
     *
     * @param kexiaoStatisticsDto
     * @param studentLesson       超排(赠送) false;  已消 true
     * @param calculateMoney
     */
    private void upLessonCountAndMoney(LessonKexiaoStatisticsDto kexiaoStatisticsDto, OrgSignupCourseLesson studentLesson, boolean calculateMoney) {
        Preconditions.checkArgument(kexiaoStatisticsDto != null, "课消统计参数为空");
        Preconditions.checkArgument(studentLesson != null, "学员课节参数为空");

        Integer minutes = 0;
        Integer times = 0;
        Long minutesMoney = 0l;
        Long timesMoney = 0l;
        if (ChargeUnit.isByDuration(studentLesson.getChargeUnit())) {//按时长算
            minutes = studentLesson.getLessonDuration();
            if (calculateMoney) {
                minutesMoney = studentLesson.getAmount();
            }
        } else {//按次数算
            times = 1;
            if (calculateMoney) {
                timesMoney = studentLesson.getAmount();
            }
        }
        kexiaoStatisticsDto.addFromMinutesAndTimes(minutes, minutesMoney, times, timesMoney);

    }


    @Override
    public Map<Long, LessonKexiaoStatisticsDto> getKexiaoFinishedMapOfOrg(Collection<Long> orgIds) {
        Map<Long, LessonKexiaoStatisticsDto> result = Maps.newHashMap();
        List<OrgSignupCourseLesson> kexiaoList = this.orgSignupCourseLessonDao.getKexiaoFinishedList(orgIds, null, null);
        if (CollectionUtils.isNotEmpty(kexiaoList)) {
            Map<Long, List<OrgSignupCourseLesson>> kexiaoMap = kexiaoList.stream().collect(Collectors.groupingBy(OrgSignupCourseLesson::getOrgId));//按校区分组
            result = this.calculateKexiaoFinishedStatisticsOfGroup(kexiaoMap);
        }
        return result;
    }

    @Override
    public Map<Long, LessonKexiaoStatisticsDto> getKexiaoFinishedMapOfClass(Collection<Long> classIds) {
        Map<Long, LessonKexiaoStatisticsDto> result = Maps.newHashMap();
        List<OrgSignupCourseLesson> kexiaoList = this.orgSignupCourseLessonDao.getKexiaoFinishedList(null, classIds, null);
        if (CollectionUtils.isNotEmpty(kexiaoList)) {
            Map<Long, List<OrgSignupCourseLesson>> kexiaoMap = kexiaoList.stream().collect(Collectors.groupingBy(OrgSignupCourseLesson::getClassId));//按班级分组
            result = this.calculateKexiaoFinishedStatisticsOfGroup(kexiaoMap);
        }
        return result;
    }

    @Override
    public Map<Long, LessonKexiaoStatisticsDto> getKexiaoFinishedMapOfStudentClass(Long userId, Collection<Long> classIds) {
        Map<Long, LessonKexiaoStatisticsDto> result = Maps.newHashMap();
        List<OrgSignupCourseLesson> kexiaoList = this.orgSignupCourseLessonDao.getKexiaoFinishedList(null, classIds, Sets.newHashSet(userId));
        if (CollectionUtils.isNotEmpty(kexiaoList)) {
            Map<Long, List<OrgSignupCourseLesson>> kexiaoMap = kexiaoList.stream().collect(Collectors.groupingBy(OrgSignupCourseLesson::getClassId));//按班级分组
            result = this.calculateKexiaoFinishedStatisticsOfGroup(kexiaoMap);
        }
        return result;
    }

    @Override
    public Map<Long, LessonKexiaoStatisticsDto> getKexiaoFinishedMapOfStudent(Collection<Long> userIds) {
        Map<Long, LessonKexiaoStatisticsDto> result = Maps.newHashMap();
        List<OrgSignupCourseLesson> kexiaoList = this.orgSignupCourseLessonDao.getKexiaoFinishedList(null, null, userIds);
        if (CollectionUtils.isNotEmpty(kexiaoList)) {
            Map<Long, List<OrgSignupCourseLesson>> kexiaoMap = kexiaoList.stream().collect(Collectors.groupingBy(OrgSignupCourseLesson::getUserId));//按学员分组
            result = this.calculateKexiaoFinishedStatisticsOfGroup(kexiaoMap);
        }
        return result;
    }

    @Override
    public Map<Long, LessonKexiaoStatisticsDto> getKexiaoFinishedMapOfClassStudent(Long classId, Collection<Long> userIds) {
        Map<Long, LessonKexiaoStatisticsDto> result = Maps.newHashMap();
        List<OrgSignupCourseLesson> kexiaoList = this.orgSignupCourseLessonDao.getKexiaoFinishedList(null, Sets.newHashSet(classId), userIds);
        if (CollectionUtils.isNotEmpty(kexiaoList)) {
            Map<Long, List<OrgSignupCourseLesson>> kexiaoMap = kexiaoList.stream().collect(Collectors.groupingBy(OrgSignupCourseLesson::getUserId));//按学员分组
            result = this.calculateKexiaoFinishedStatisticsOfGroup(kexiaoMap);
        }
        return result;
    }

    /**
     * 分组统计课消
     *
     * @param kexiaoMap
     * @return
     */
    private Map<Long, LessonKexiaoStatisticsDto> calculateKexiaoFinishedStatisticsOfGroup(Map<Long, List<OrgSignupCourseLesson>> kexiaoMap) {
        Map<Long, LessonKexiaoStatisticsDto> result = Maps.newHashMap();
        if (MapUtils.isNotEmpty(kexiaoMap)) {
            for (Long id : kexiaoMap.keySet()) {
                List<OrgSignupCourseLesson> kexiaoList = kexiaoMap.get(id);
                LessonKexiaoStatisticsDto kexiaoStatisticsDto = new LessonKexiaoStatisticsDto();
                for (OrgSignupCourseLesson kexiaoRecord : kexiaoList) {
                    this.upLessonCountAndMoney(kexiaoStatisticsDto, kexiaoRecord, true);
                }
                result.put(id, kexiaoStatisticsDto);
            }
        }
        return result;
    }

    @Override
    public Map<Long, LessonKexiaoStatisticsDto> getRefundMapOfOrg(Collection<Long> orgIds) {
        Map<Long, LessonKexiaoStatisticsDto> result = Maps.newHashMap();
        List<OrgSignupRefund> refundList = this.orgSignupRefundDao.getRefundList(orgIds, null, null);
        if (CollectionUtils.isNotEmpty(refundList)) {
            Map<Long, List<OrgSignupRefund>> refundMap = refundList.stream().collect(Collectors.groupingBy(OrgSignupRefund::getOrgId));//按校区分组
            result = this.calculateRefundStatisticsOfGroup(refundMap);
        }
        return result;
    }

    @Override
    public Map<Long, LessonKexiaoStatisticsDto> getRefundMapOfClass(Collection<Long> classIds) {
        Map<Long, LessonKexiaoStatisticsDto> result = Maps.newHashMap();
        List<OrgSignupRefund> refundList = this.orgSignupRefundDao.getRefundList(null, classIds, null);
        if (CollectionUtils.isNotEmpty(refundList)) {
            Map<Long, List<OrgSignupRefund>> refundMap = refundList.stream().collect(Collectors.groupingBy(OrgSignupRefund::getClassId));//按班级分组
            result = this.calculateRefundStatisticsOfGroup(refundMap);
        }
        return result;
    }

    @Override
    public Map<Long, LessonKexiaoStatisticsDto> getRefundMapOfStudentClass(Long userId, Collection<Long> classIds) {
        Map<Long, LessonKexiaoStatisticsDto> result = Maps.newHashMap();
        List<OrgSignupRefund> refundList = this.orgSignupRefundDao.getRefundList(null, classIds, Sets.newHashSet(userId));
        if (CollectionUtils.isNotEmpty(refundList)) {
            Map<Long, List<OrgSignupRefund>> refundMap = refundList.stream().collect(Collectors.groupingBy(OrgSignupRefund::getClassId));//按班级分组
            result = this.calculateRefundStatisticsOfGroup(refundMap);
        }
        return result;
    }

    @Override
    public Map<Long, LessonKexiaoStatisticsDto> getRefundMapOfStudent(Collection<Long> userIds) {
        Map<Long, LessonKexiaoStatisticsDto> result = Maps.newHashMap();
        List<OrgSignupRefund> refundList = this.orgSignupRefundDao.getRefundList(null, null, userIds);
        if (CollectionUtils.isNotEmpty(refundList)) {
            Map<Long, List<OrgSignupRefund>> refundMap = refundList.stream().collect(Collectors.groupingBy(OrgSignupRefund::getUserId));//按学员分组
            result = this.calculateRefundStatisticsOfGroup(refundMap);
        }
        return result;
    }

    @Override
    public Map<Long, LessonKexiaoStatisticsDto> getRefundMapOfClassStudent(Long classId, Collection<Long> userIds) {
        Map<Long, LessonKexiaoStatisticsDto> result = Maps.newHashMap();
        List<OrgSignupRefund> refundList = this.orgSignupRefundDao.getRefundList(null, Sets.newHashSet(classId), userIds);
        if (CollectionUtils.isNotEmpty(refundList)) {
            Map<Long, List<OrgSignupRefund>> refundMap = refundList.stream().collect(Collectors.groupingBy(OrgSignupRefund::getUserId));//按学员分组
            result = this.calculateRefundStatisticsOfGroup(refundMap);
        }
        return result;
    }

    /**
     * 分组计算退费的统计数据
     *
     * @param refundMap
     * @return
     */
    private Map<Long, LessonKexiaoStatisticsDto> calculateRefundStatisticsOfGroup(Map<Long, List<OrgSignupRefund>> refundMap) {
        Map<Long, LessonKexiaoStatisticsDto> result = Maps.newHashMap();
        if (MapUtils.isNotEmpty(refundMap)) {
            for (Long id : refundMap.keySet()) {
                List<OrgSignupRefund> refundList = refundMap.get(id);
                LessonKexiaoStatisticsDto kexiaoStatisticsDto = new LessonKexiaoStatisticsDto();
                for (OrgSignupRefund refund : refundList) {
                    this.upLessonCountAndMoney(kexiaoStatisticsDto, refund.getChargeUnit(), true, refund.getRefundLessonCount(), refund.getRefundPrice() + refund.getRefundFee());
                }
                result.put(id, kexiaoStatisticsDto);
            }
        }
        return result;
    }

    @Override
    public Map<Long, LessonKexiaoStatisticsDto> getTransferMapOfOrg(Collection<Long> orgIds) {
        Map<Long, LessonKexiaoStatisticsDto> result = Maps.newHashMap();
        List<TxTransferClassInfo> transferList = this.txTransferClassInfoDao.getTransferSuccessList(orgIds, null, null);
        if (CollectionUtils.isNotEmpty(transferList)) {
            Map<Long, List<TxTransferClassInfo>> transferMap = transferList.stream().collect(Collectors.groupingBy(TxTransferClassInfo::getOrgId));//按校区分组
            result = this.calculateTransferStatisticsOfGroup(transferMap);
        }
        return result;
    }

    @Override
    public Map<Long, LessonKexiaoStatisticsDto> getTransferMapOfClass(Collection<Long> classIds) {
        Map<Long, LessonKexiaoStatisticsDto> result = Maps.newHashMap();
        List<TxTransferClassInfo> transferList = this.txTransferClassInfoDao.getTransferSuccessList(null, classIds, null);
        if (CollectionUtils.isNotEmpty(transferList)) {
            Map<Long, List<TxTransferClassInfo>> transferMap = transferList.stream().collect(Collectors.groupingBy(TxTransferClassInfo::getTransferOutClassId));//按班级分组
            result = this.calculateTransferStatisticsOfGroup(transferMap);
        }
        return result;
    }

    @Override
    public Map<Long, LessonKexiaoStatisticsDto> getTransferMapOfStudentClass(Long userId, Collection<Long> classIds) {
        Map<Long, LessonKexiaoStatisticsDto> result = Maps.newHashMap();
        List<TxTransferClassInfo> transferList = this.txTransferClassInfoDao.getTransferSuccessList(null, classIds, Sets.newHashSet(userId));
        if (CollectionUtils.isNotEmpty(transferList)) {
            Map<Long, List<TxTransferClassInfo>> transferMap = transferList.stream().collect(Collectors.groupingBy(TxTransferClassInfo::getTransferOutClassId));//按班级分组
            result = this.calculateTransferStatisticsOfGroup(transferMap);
        }
        return result;
    }

    @Override
    public Map<Long, LessonKexiaoStatisticsDto> getTransferMapOfStudent(Collection<Long> userIds) {
        Map<Long, LessonKexiaoStatisticsDto> result = Maps.newHashMap();
        List<TxTransferClassInfo> transferList = this.txTransferClassInfoDao.getTransferSuccessList(null, null, userIds);
        if (CollectionUtils.isNotEmpty(transferList)) {
            Map<Long, List<TxTransferClassInfo>> transferMap = transferList.stream().collect(Collectors.groupingBy(TxTransferClassInfo::getUserId));//按学员分组
            result = this.calculateTransferStatisticsOfGroup(transferMap);
        }
        return result;
    }

    @Override
    public Map<Long, LessonKexiaoStatisticsDto> getTransferMapOfClassStudent(Long classId, Collection<Long> userIds) {
        Map<Long, LessonKexiaoStatisticsDto> result = Maps.newHashMap();
        List<TxTransferClassInfo> transferList = this.txTransferClassInfoDao.getTransferSuccessList(null, Sets.newHashSet(classId), userIds);
        if (CollectionUtils.isNotEmpty(transferList)) {
            Map<Long, List<TxTransferClassInfo>> transferMap = transferList.stream().collect(Collectors.groupingBy(TxTransferClassInfo::getUserId));//按学员分组
            result = this.calculateTransferStatisticsOfGroup(transferMap);
        }
        return result;
    }

    /**
     * 分组计算转出的统计数据
     *
     * @param
     * @return
     */
    private Map<Long, LessonKexiaoStatisticsDto> calculateTransferStatisticsOfGroup(Map<Long, List<TxTransferClassInfo>> transferMap) {
        Map<Long, LessonKexiaoStatisticsDto> result = Maps.newHashMap();
        if (MapUtils.isNotEmpty(transferMap)) {
            for (Long id : transferMap.keySet()) {
                List<TxTransferClassInfo> transferList = transferMap.get(id);
                LessonKexiaoStatisticsDto kexiaoStatisticsDto = new LessonKexiaoStatisticsDto();
                for (TxTransferClassInfo transfer : transferList) {
                    this.upLessonCountAndMoney(kexiaoStatisticsDto, transfer.getChargeUnit(), true, transfer.getLessonCount(), transfer.getLessonMoney());
                }
                result.put(id, kexiaoStatisticsDto);
            }
        }
        return result;
    }

}
