package com.baijia.tianxiao.sal.kexiao.service.impl;

import com.baijia.tianxiao.constant.KexiaoChangeEvent;
import com.baijia.tianxiao.constant.LessonStatus;
import com.baijia.tianxiao.dal.org.dao.*;
import com.baijia.tianxiao.dal.org.po.*;
import com.baijia.tianxiao.dal.signup.dao.OrgSignupCourseLessonDao;
import com.baijia.tianxiao.sal.kexiao.dto.BatchSignInfo;
import com.baijia.tianxiao.sal.kexiao.dto.FastSignInfo;
import com.baijia.tianxiao.sal.kexiao.dto.RuleChangeInfo;
import com.baijia.tianxiao.sal.kexiao.dto.LessonStartInfo;
import com.baijia.tianxiao.sal.kexiao.service.KexiaoMonitorService;
import com.baijia.tianxiao.sal.kexiao.utils.KexiaoUtil;
import com.baijia.tianxiao.util.CollectionHelper;
import com.baijia.tianxiao.util.ListUtil;
import com.google.gson.Gson;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.*;

/**
 * Created by liuxp on 16/12/17.
 * 课消监视程序，需要校验的内容：
 * 1、课消状态是否一致
 *
 */
@Service
@Slf4j
public class KexiaoMonitorServiceImpl implements KexiaoMonitorService {

    @Autowired
    private KexiaoChangeLogDao changeLogDao;
    @Autowired
    private OrgCourseConsumeRuleDao courseConsumeRuleDao;
    @Autowired
    private OrgStudentKexiaoRecordDao kexiaoRecordDao;
    @Autowired
    private OrgStudentLessonDao studentLessonDao;
    @Autowired
    private OrgClassLessonDao classLessonDao;
    @Autowired
    private OrgSignupCourseLessonDao signupCourseLessonDao;
    @Autowired
    private OrgLessonSignDao lessonSignDao;


    @Override
    public void validate(Long orgId, Date startTime, Date endTime) {
        List<KexiaoChangeLog> changeLogs = changeLogDao.selectByTime(orgId, startTime, endTime);
        Set<Long> stuLessonIds = new HashSet<>();
        Set<Long> courseIds = new HashSet<>();
        Gson gson = new Gson();
        if (changeLogs!=null && changeLogs.size()>0){
            for (KexiaoChangeLog changeLog:changeLogs){
                KexiaoChangeEvent event = KexiaoChangeEvent.getByType(changeLog.getEventType());
                switch (event){
                    case KEXIAO_REUL:
                        RuleChangeInfo ruleChange = gson.fromJson(changeLog.getInfo(), RuleChangeInfo.class);
                        courseIds.add(ruleChange.getCourseId());
                        break;
                    case START:
                        LessonStartInfo start = gson.fromJson(changeLog.getInfo(),LessonStartInfo.class);
                        stuLessonIds.addAll(start.getStuLessonIds());
                    case BATCH_SIGN:
                        BatchSignInfo batchSign = gson.fromJson(changeLog.getInfo(),BatchSignInfo.class);
                        List<OrgStudentLesson> batchSignLessons = studentLessonDao.getByLessonIdsStudentIds(batchSign.getClassLessonIds(), batchSign.getUserIds(), "id");
                        List<Long> ids = ListUtil.toKeyList(batchSignLessons,"id",OrgStudentLesson.class);
                        stuLessonIds.addAll(ids);
                        break;
                    case FAST_SIGN:
                        FastSignInfo sign = gson.fromJson(changeLog.getInfo(),FastSignInfo.class);
                        List<OrgStudentLesson> lessons = studentLessonDao.getByLessonIdsStudentIds(sign.getClassLessonIds(), sign.getStuStatus().keySet(), "id");
                        List<Long> lessonIds = ListUtil.toKeyList(lessons,"id",OrgStudentLesson.class);
                        stuLessonIds.addAll(lessonIds);
                        break;
                    default:
                        log.warn("[Kexiao] KexiaoChangeLog={}",changeLog);
                        throw new RuntimeException("KexiaoChangeEvent error");
                }
            }
        }
        if(courseIds.size()>0) {
            validateByCourseId(orgId, courseIds);
        }
        if(stuLessonIds.size()>0) {
            validateByLessonIds(orgId, stuLessonIds);
        }
    }

    private void validateByCourseId(Long orgId,Set<Long> courseIds){
        List<OrgClassLesson> classLessons = classLessonDao.getLessonByCourseIds(courseIds, 0,"id");
        List<Long> lessonIds = ListUtil.toKeyList(classLessons, "id", OrgClassLesson.class);
        List<OrgStudentLesson> lessons = studentLessonDao.getOrgStudentLessonsByLessonIds(lessonIds);
        validateLessonStatus(orgId,lessons);
    }

    private void validateByLessonIds(Long orgId,Set<Long> stuLessonIds){
        List<OrgStudentLesson> studentLessons = studentLessonDao.getByIds(stuLessonIds);
        validateLessonStatus(orgId,studentLessons);
    }

    private void validateLessonStatus(Long orgId,List<OrgStudentLesson> studentLessons){
        List<Long> stuLessonIds = ListUtil.toKeyList(studentLessons,"id",OrgStudentLesson.class);

        List<Long> classLessonIds = ListUtil.toKeyList(studentLessons, "lessonId", OrgStudentLesson.class);
        List<OrgClassLesson> classLessons = classLessonDao.getByIds(classLessonIds);
        Map<Long,OrgClassLesson> classLessonMap = CollectionHelper.toIdMap(classLessons);

        List<Long> courseIds = ListUtil.toKeyList(classLessons, "courseId", OrgClassLesson.class);
        List<OrgCourseConsumeRule> rules = courseConsumeRuleDao.queryConsumRuleListByCourseIds(orgId, courseIds);
        Map<Long,OrgCourseConsumeRule> ruleMap = CollectionHelper.toKeyMap(rules, "courseId");

        List<OrgLessonSign> lessonSigns = lessonSignDao.getByStudentLessonIds(stuLessonIds);
        Map<Long,OrgLessonSign> signMap = CollectionHelper.toKeyMap(lessonSigns, "lessonId");

        List<OrgStudentKexiaoRecord> records = kexiaoRecordDao.getByStuLessonIds(orgId, stuLessonIds);
        Map<Long,OrgStudentKexiaoRecord> recordMap = CollectionHelper.toKeyMap(records,"studentLessonId");

        for (OrgStudentLesson studentLesson:studentLessons){
            OrgClassLesson classLesson = classLessonMap.get(studentLesson.getLessonId());
            OrgCourseConsumeRule rule = ruleMap.get(classLesson.getCourseId());
            LessonStatus status;
            if(rule==null||rule.getRuleValue()==0){//按开课时间算课消
                status = classLesson.getStartTime().compareTo(new Date())>0?LessonStatus.UN_START:LessonStatus.FINISHED;
            }else{
                OrgLessonSign sign = signMap.get(studentLesson.getId());
                if(sign!=null){
                    status = KexiaoUtil.getKexiaoStatus(sign.getStatus(),rule.getRuleValue());
                }else {
                    status = LessonStatus.UN_START;
                }
            }
            OrgStudentKexiaoRecord record = recordMap.get(studentLesson.getId());
            if(record==null){//未计算课消
                if(status!=LessonStatus.UN_START){
                    log.error("[Kexiao]Kexiao status is error.expect status={},current status={}",status,LessonStatus.UN_START);
                }
            }else {
                if(status!=LessonStatus.FINISHED){
                    log.error("[Kexiao]Kexiao status is error.expect status={},current status={}",status,LessonStatus.FINISHED);
                }
            }
        }

    }
}
