package com.baijia.tianxiao.sal.elastic.service.impl;


import com.baijia.commons.lang.utils.PropertiesReader;
import com.baijia.tianxiao.dal.finance.dao.TxStudentFinanceAccountDao;
import com.baijia.tianxiao.dal.finance.po.TxStudentFinanceAccount;
import com.baijia.tianxiao.dal.org.constant.DeleteStatus;
import com.baijia.tianxiao.dal.org.constant.StudentType;
import com.baijia.tianxiao.dal.org.dao.OrgStudentDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentLessonDao;
import com.baijia.tianxiao.dal.org.po.OrgStudent;
import com.baijia.tianxiao.dal.roster.UserLastFollowTime;
import com.baijia.tianxiao.dal.roster.dao.CustomFieldDao;
import com.baijia.tianxiao.dal.roster.dao.CustomFieldValueDao;
import com.baijia.tianxiao.dal.roster.dao.TxStudentCommentDao;
import com.baijia.tianxiao.dal.roster.dao.TxStudentTagDao;
import com.baijia.tianxiao.dal.roster.dao.TxTagDao;
import com.baijia.tianxiao.dal.roster.po.CustomFieldValue;
import com.baijia.tianxiao.dal.roster.po.TxConsultUser;
import com.baijia.tianxiao.dal.roster.po.TxStudentTag;
import com.baijia.tianxiao.dal.signup.dao.OrgSignupCourseDao;
import com.baijia.tianxiao.sal.common.api.KexiaoApiService;
import com.baijia.tianxiao.sal.common.dto.kexiao.KexiaoStudentStat;

import com.baijia.tianxiao.sal.elastic.service.StudentDataImportService;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.util.CollectionHelper;
import com.baijia.tianxiao.util.HanZiPinYinUtils;
import com.baijia.tianxiao.util.ListUtil;
import com.baijia.tianxiao.util.json.JacksonUtil;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.action.bulk.BulkProcessor;
import org.elasticsearch.action.bulk.BulkRequest;
import org.elasticsearch.action.bulk.BulkRequestBuilder;
import org.elasticsearch.action.bulk.BulkResponse;
import org.elasticsearch.action.delete.DeleteResponse;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.update.UpdateRequest;
import org.elasticsearch.client.transport.TransportClient;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.transport.InetSocketTransportAddress;
import org.elasticsearch.common.unit.ByteSizeUnit;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.transport.client.PreBuiltTransportClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.IOException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.*;
import java.util.concurrent.ExecutionException;

/**
 * Created by liuxp on 16/11/17.
 */
@Slf4j
@Service
public class StudentDataImportServiceImpl extends AbstractEsBaseServiceImpl implements StudentDataImportService {

    @Autowired
    private OrgStudentDao studentDao;

    @Autowired
    private CustomFieldValueDao customFieldValueDao;

    @Autowired
    private TxStudentTagDao tagDao;

    @Autowired
    private OrgStudentLessonDao lessonDao;

    @Autowired
    private OrgSignupCourseDao signupCourseDao;

    @Autowired
    private KexiaoApiService kexiaoApiService;

    @Autowired
    private TxStudentFinanceAccountDao studentFinanceAccountDao;
    @Autowired
    private TxStudentCommentDao studentCommentDao;

    @Override
    public void fullImport() {
        importByOrgId(null);
    }

    @Override
    public void importByOrgId(Long orgId) {

        long time = System.currentTimeMillis();
        BulkProcessor bulkProcessor = getDefaultBulkProcessor();

        PageDto page = new PageDto();
        page.setPageNum(1);
        page.setPageSize(3000);
        List<OrgStudent> students = null;
        if (orgId == null) {
            students = studentDao.getByPage(page);
        } else {
            students = studentDao.getOrgStudentByPage(orgId, page);
        }
        while (students.size() > 0) {
            fillBulkData(orgId,students,bulkProcessor);
            page.setPageNum(page.getPageNum() + 1);
            if (orgId == null) {
                students = studentDao.getByPage(page);
            } else {
                students = studentDao.getOrgStudentByPage(orgId, page);
            }
            log.info("import page={}", page);
            bulkProcessor.flush();
        }

        bulkProcessor.close();

        log.info("import {} data costs:{}", page.getCount(), System.currentTimeMillis() - time);
    }

    private void fillBulkData(long orgId,List<OrgStudent> students,BulkProcessor bulkProcessor ){
        if(students==null || students.size()<1){
            log.warn("[ES] students is null.orgId={}",orgId);
            return;
        }
        List<Long> studentIds = ListUtil.toKeyList(students, "id", OrgStudent.class);
        List<Long> userIds = ListUtil.toKeyList(students, "userId", OrgStudent.class);
        List<TxStudentFinanceAccount> accounts = studentFinanceAccountDao.listFinanceAccount(orgId, studentIds);
        Map<Long, TxStudentFinanceAccount> accountMap = CollectionHelper.toKeyMap(accounts, "studentId");
        Map<Long, KexiaoStudentStat> kexiaoStatMap = kexiaoApiService.queryUserKexiaoStat(orgId, userIds);

        Map<Long, List<CustomFieldValue>> valueMap = customFieldValueDao.batchGetValueMap(null, true, studentIds, null);
        List<TxStudentTag> tags = tagDao.getTags(userIds, null, StudentType.ORG_STUDENTS.getCode(), null);
        Map<Long, List<TxStudentTag>> studentTags = toMap(tags,StudentType.ORG_STUDENTS);

        Set<Long> scheduleUserIds = lessonDao.hasScheduleLesson(userIds);

        Set<Long> refundUserIds = signupCourseDao.hasRefund(userIds);

        for (OrgStudent student : students) {
            Map<String, Object> map = studentToMap(student, valueMap.get(student.getId()),
                    studentTags.get(student.getUserId()),
                    kexiaoStatMap.get(student.getUserId()),
                    accountMap.get(student.getId()));
            if (scheduleUserIds.contains(student.getUserId())) {
                map.put("scheduleStatus", 1);
            } else {
                map.put("scheduleStatus", 2);
            }
            if (refundUserIds.contains(student.getUserId())) {
                map.put("refundClassStatus", 1);
            } else {
                map.put("refundClassStatus", 2);
            }
            bulkProcessor.add(
                    new IndexRequest()
                            .index("students")
                            .type("student")
                            .source(map)
                            .id(student.getId().toString())
            );
        }
    }

    @Override
    public void importByOrgIdAndUpdateTime(Long orgId, int beforeHour) {
        long time = System.currentTimeMillis();

        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.HOUR_OF_DAY,beforeHour);
        List<OrgStudent> students = studentDao.listByUpdateTime(orgId,calendar.getTime());
        BulkProcessor bulkProcessor = getDefaultBulkProcessor();
        fillBulkData(orgId,students,bulkProcessor);
        bulkProcessor.flush();
        bulkProcessor.close();
        log.info("[ES] Import {} data costs:{}", students.size(), System.currentTimeMillis() - time);
    }

    @Override
    public void batchImport(Long orgId, Collection<Long> studentIds) {
        log.info("[ES] batch import studentIds={},orgId={}",studentIds,orgId);
        if(studentIds==null || studentIds.size()<1){
            return;
        }
        long time = System.currentTimeMillis();

        List<OrgStudent> students = studentDao.getByIds(studentIds);
        BulkProcessor bulkProcessor = getDefaultBulkProcessor();
        fillBulkData(orgId,students,bulkProcessor);
        bulkProcessor.flush();
        bulkProcessor.close();
        log.info("[ES] BatchImport {} data costs:{}", students.size(), System.currentTimeMillis() - time);
    }

    @Override
    public void batchImportByUserIds(Long orgId, Collection<Long> userIds) {
        log.info("[ES] batch import userIds={},orgId={}",userIds,orgId);
        if(userIds==null || userIds.size()<1){
            return;
        }
        long time = System.currentTimeMillis();

        List<OrgStudent> students = studentDao.getStudentByUserIds(orgId,userIds);
        BulkProcessor bulkProcessor = getDefaultBulkProcessor();
        fillBulkData(orgId,students,bulkProcessor);
        bulkProcessor.flush();
        bulkProcessor.close();
        log.info("[ES] BatchImport {} data costs:{}", students.size(), System.currentTimeMillis() - time);
    }

    @Override
    public void saveOrUpdateStudentById(Long studentId) {
        log.info("[ES] update studentId={}", studentId);
        OrgStudent student = studentDao.getById(studentId);
        int i = 1;
        while (student == null && i <= 10) {//防止主从延迟
            try {
                Thread.sleep(i * 5 * 1000);
                student = studentDao.getById(studentId);
                i++;
                log.warn("[ES] Retry count={},studentId={}", i, studentId);
            } catch (InterruptedException e) {
                log.error("[ES] InterruptedException error={}", e);
            }
        }

        if (student == null) {
            log.error("[ES] studentId={} is not exist", studentId);
            return;
        }

        saveOrUpdateStudent(student);
    }

    @Override
    public void saveOrUpdateStudentByUserId(Long orgId, Long userId) {
        OrgStudent student = studentDao.getStudentByUserId(orgId, userId);
        int i = 1;
        while (student == null && i <= 10) {//防止主从延迟
            try {
                Thread.sleep(i * 5 * 1000);
                student = studentDao.getStudentByUserId(orgId, userId);
                i++;
                log.warn("[ES] Retry count={},userId={},orgId={}", i, userId, orgId);
            } catch (InterruptedException e) {
                log.error("[ES] InterruptedException error={}", e);
            }
        }

        if (student == null) {
            log.error("[ES] orgId={},userId={} is not exist", orgId, userId);
            return;
        }
        saveOrUpdateStudent(student);
    }

    @Override
    public void importLastFollowTime() {
        long time = System.currentTimeMillis();
        PageDto page = new PageDto();
        page.setPageNum(1);
        page.setPageSize(3000);
        List<UserLastFollowTime> data=studentCommentDao.getLastFollowComment(page,StudentType.ORG_STUDENTS.getCode());
        while(data.size()>0){
            for(UserLastFollowTime user:data){
                OrgStudent student=studentDao.getStudent(user.getOrgId(),user.getId(), DeleteStatus.NORMAL.getValue());
                if(student!=null) {
                    student.setLastFollowTime(user.getLastFollowTime());
                    studentDao.update(student, "lastFollowTime");
                }

            }
            page.setCount((page.getPageNum()-1)*page.getPageSize()+data.size());
            page.setPageNum(page.getPageNum()+1);
            data=studentCommentDao.getLastFollowComment(page,StudentType.ORG_STUDENTS.getCode());
        }
        log.info("import {} data costs:{}", page.getCount(), System.currentTimeMillis() - time);
    }


    private void saveOrUpdateStudent(OrgStudent student) {
        log.info("[ES] update student={},userId={}", student.getId(), student.getUserId());
        DeleteResponse deleteresponse = getClient()
                .prepareDelete("students", "student", String.valueOf(student.getId()))
                .execute()
                .actionGet();
        Map<Long, List<CustomFieldValue>> valueMap = customFieldValueDao.batchGetValueMap(student.getOrgId(),
                true, Arrays.asList(student.getId()), null);
        List<TxStudentTag> tags = tagDao.getTags(student.getUserId(), student.getOrgId(), StudentType.ORG_STUDENTS.getCode());
        List<Long> userIds = Arrays.asList(student.getUserId());

        Map<Long, KexiaoStudentStat> kexiaoStatMap = kexiaoApiService.queryUserKexiaoStat(student.getOrgId(), userIds);
        TxStudentFinanceAccount account = studentFinanceAccountDao.getFinanceAccount(student.getOrgId(), student.getId());

        Map<String, Object> map = studentToMap(student, valueMap.get(student.getId()), tags,
                kexiaoStatMap.get(student.getUserId()), account);

        Set<Long> scheduleUserIds = lessonDao.hasScheduleLesson(userIds);
        if (scheduleUserIds.contains(student.getUserId())) {
            map.put("scheduleStatus", 1);
        } else {
            map.put("scheduleStatus", 2);
        }
        Set<Long> refundUserIds = signupCourseDao.hasRefund(userIds);
        if (refundUserIds.contains(student.getUserId())) {
            map.put("refundClassStatus", 1);
        } else {
            map.put("refundClassStatus", 2);
        }

        getClient().prepareIndex("students", "student")
                .setSource(map)
                .setId(student.getId().toString())
                .execute()
                .actionGet();
    }

    private Map<String, Object> studentToMap(OrgStudent student,
                                             List<CustomFieldValue> customFieldValues,
                                             List<TxStudentTag> tags,
                                             KexiaoStudentStat stat,
                                             TxStudentFinanceAccount account) {
        Map<String, Object> map = new HashMap<>();
        map.put("id", student.getId());
        map.put("user_id", student.getUserId());
        map.put("org_id", student.getOrgId());
        map.put("name", student.getName());
        map.put("mobile", student.getMobile());
        map.put("weixin", student.getWeixin());
        map.put("lesson_status", student.getLessonStatus());
        map.put("add_cascade_id", student.getAddCascadeId());
        map.put("parent_name", student.getParentName());
        map.put("parent_mobile", student.getParentMobile());
        map.put("source", student.getSource());
        String pinyin = HanZiPinYinUtils.getLowerCasePinYin(student.getName());
        if (StringUtils.isNotBlank(pinyin)) {
            char init = pinyin.charAt(0);
            if (init < 'a' || init > 'z') {
                pinyin = "~";
            }
        } else {
            pinyin = "~";
        }
        map.put("pinyin", pinyin);
        map.put("gender", student.getGender());
        map.put("del_status", student.getDelStatus());
        map.put("avatar", student.getAvatar());
        map.put("create_time", student.getCreateTime().getTime());
        map.put("update_time", student.getUpdateTime().getTime());
        map.put("show_time", student.getCreateTime().getTime());
        if (student.getNextRemindTime() != null) {
            map.put("next_remind_time", student.getNextRemindTime().getTime());
        }

        if (student.getLastRemindTime() != null) {
            map.put("last_remind_time", student.getLastRemindTime().getTime());
        }

        if (student.getLastFollowTime() != null) {
            map.put("last_follow_time", student.getLastFollowTime().getTime());
        }

        Map<String, Object> customFieldMap = new HashMap<>();
        if (customFieldValues != null) {
            for (CustomFieldValue value : customFieldValues) {
                try {
                    Map<String, Object> valueMap = JacksonUtil.str2Obj(value.getValue(),
                            new TypeReference<Map<String, Object>>() {
                            });
                    customFieldMap.put(value.getFieldId().toString(), valueMap);
                } catch (IOException e) {
                    log.error("IOException = {}", e);
                }
            }
        }
        map.put("custom_field", customFieldMap);

        String tagStr = "";
        if (tags != null && tags.size() > 0) {
            StringBuilder sb = new StringBuilder();
            for (TxStudentTag tag : tags) {
                sb.append(",").append(tag.getContent());
            }
            tagStr = sb.substring(1);
        }
        map.put("tag", tagStr);

        if (student.getBirthday() != null) {
            Calendar calendar = Calendar.getInstance();
            calendar.setTime(student.getBirthday());
            calendar.set(Calendar.HOUR, 0);
            calendar.set(Calendar.MINUTE, 0);
            calendar.set(Calendar.SECOND, 0);
            calendar.set(Calendar.MILLISECOND, 0);

            map.put("birthday", student.getBirthday().getTime());
            map.put("birthDayOfMonth", calendar.get(Calendar.DAY_OF_MONTH));
            map.put("birthMonth", calendar.get(Calendar.MONTH) + 1);
            map.put("birthYear", calendar.get(Calendar.YEAR));
        } else {
            map.put("birthday", -1);
            map.put("birthDayOfMonth", -1);
            map.put("birthMonth", -1);
            map.put("birthYear", -1);
        }
        if (StringUtils.isNotBlank(student.getWeixin())) {
            map.put("stuCenterBindStatus", 1);//绑定了个人中心
        } else {
            map.put("stuCenterBindStatus", 2);
        }

        if (stat != null) {
            if(stat.getCompleteStatus()==0) {
                map.put("leftAmount", stat.getLeftAmount());
                map.put("arrangeTime", stat.getArrangeNormalTime() + stat.getArrangeFreeTime());
                map.put("kexiaoTime", stat.getKexiaoNormalTime() + stat.getKexiaoFreeTime());
                map.put("totalTime", stat.getTotalTime());
                map.put("leftTime", stat.getTotalTime() - (stat.getKexiaoNormalTime() + stat.getKexiaoFreeTime()));
                map.put("arrangeCount", stat.getArrangeNormalCount() + stat.getArrangeFreeCount());
                map.put("kexiaoCount", stat.getKexiaoNormalCount() + stat.getKexiaoFreeCount());
                map.put("totalCount", stat.getTotalCount());
                map.put("leftCount", stat.getTotalCount() - (stat.getKexiaoNormalCount() + stat.getKexiaoFreeCount()));
            }else {
                map.put("leftAmount", -1);
                map.put("arrangeTime", -1);
                map.put("kexiaoTime", -1);
                map.put("totalTime", -1);
                map.put("leftTime", -1);
                map.put("arrangeCount", -1);
                map.put("kexiaoCount", -1);
                map.put("totalCount", -1);
                map.put("leftCount", -1);
            }
        } else {
            map.put("leftAmount", 0);
            map.put("arrangeTime", 0);
            map.put("kexiaoTime", 0);
            map.put("totalTime", 0);
            map.put("leftTime", 0);
            map.put("arrangeCount", 0);
            map.put("kexiaoCount", 0);
            map.put("totalCount", 0);
            map.put("leftCount", 0);
        }

        if (account != null) {
            map.put("banlance", account.getBalance());
        } else {
            map.put("banlance", 0);
        }

        return map;
    }
}
