package com.baijia.tianxiao.sal.elastic.service.impl;

import com.baijia.tianxiao.dal.org.dao.OrgStudentCourseDao;
import com.baijia.tianxiao.dal.org.po.OrgStudentCourse;
import com.baijia.tianxiao.dal.roster.dao.CustomFieldDao;
import com.baijia.tianxiao.dal.roster.po.CustomField;
import com.baijia.tianxiao.dal.solr.dto.StudentDto;
import com.baijia.tianxiao.dal.solr.dto.StudentQueryParam;
import com.baijia.tianxiao.dal.solr.dto.TimeRange;
import com.baijia.tianxiao.dal.solr.enums.PCStudentOrderEnum;
import com.baijia.tianxiao.dal.solr.enums.StudentLessonStatus;
import com.baijia.tianxiao.dal.solr.enums.TimeType;
import com.baijia.tianxiao.sal.elastic.enums.LessonStatType;
import com.baijia.tianxiao.sal.elastic.service.OrgStudentQueryService;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.util.GenericsUtils;
import com.baijia.tianxiao.util.ListUtil;
import com.baijia.tianxiao.util.NumberUtil;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.apache.solr.client.solrj.util.ClientUtils;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.index.query.*;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.SearchHits;
import org.elasticsearch.search.sort.SortBuilder;
import org.elasticsearch.search.sort.SortBuilders;
import org.elasticsearch.search.sort.SortOrder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.text.SimpleDateFormat;
import java.util.*;

@Slf4j
@Service
public class OrgStudentQueryServiceImpl extends AbstractEsBaseServiceImpl implements OrgStudentQueryService {

    @Autowired
    private OrgStudentCourseDao studentCourseDao;

    @Override
    public List<StudentDto> queryStudent(StudentQueryParam param, PageDto pageDto) {
        List<StudentDto> result = new ArrayList<>();
        SearchResponse response = buildQuery(param, pageDto);
        if (response == null) {
            return result;
        }
        SearchHits hits = response.getHits();
        long total = hits.getTotalHits();
        pageDto.setCount((int) total);
        log.info("[ES] query size={}", total);
        SearchHit[] searchHits = hits.hits();

        for (SearchHit s : searchHits) {
            StudentDto dto = buildStudent(s.getSource());
            result.add(dto);
        }
        return result;
    }

    private StudentDto buildStudent(Map<String, Object> doc) {
        StudentDto stu = new StudentDto();
        stu.setName((String) doc.get("name"));
        stu.setMobile((String) doc.get("mobile"));
        stu.setWeixin((String) doc.get("weixin"));
        stu.setStuLessonStatus((Integer) doc.get("lesson_status"));
        stu.setStudentId(((Integer) doc.get("id")).longValue());
        stu.setUserId(((Integer) doc.get("user_id")).longValue());
        stu.setPinyin((String) doc.get("pinyin"));
        stu.setAvatar(((Integer) doc.get("avatar")).longValue());
        return stu;
    }

    private SearchResponse buildQuery(StudentQueryParam param, PageDto page) {
        log.info("[ES] origin query param={}", param);
        BoolQueryBuilder builder = QueryBuilders.boolQuery()
                .must(getMatchQuery("org_id", param.getOrgId()))
                .must(getMatchQuery("del_status", 0));

        // for pc search
        if (StringUtils.isNotBlank(param.getQueryStr()) && StringUtils.isNotBlank(param.getQueryValue())) {
            param.setQueryValue(ClientUtils.escapeQueryChars(param.getQueryValue()));
            if (param.getQueryStr().equals("mobile")) {
                builder.must(getWildcardQuery("mobile", param.getQueryValue()));
            }
            if (param.getQueryStr().equals("name")) {
                builder.must(getWildcardQuery("name.keyword", param.getQueryValue()));
            }
            if (param.getQueryStr().equals("parentName")) {
                builder.must(getWildcardQuery("parent_name.keyword", param.getQueryValue()));
            }
            if (param.getQueryStr().equals("parentMobile")) {
                builder.must(getWildcardQuery("parent_mobile", param.getQueryValue()));
            }
            if (param.getQueryStr().equals("tag")) {
                builder.must(getWildcardQuery("tag.keyword", param.getQueryValue()));
            }

            if (StringUtils.isNotBlank(param.getQueryStr()) && param.getQueryStr().startsWith("customField")) {
                String idStr = param.getQueryStr().replace("customField", "");
                if (StringUtils.isNotBlank(param.getQueryValue())) {
                    String customKey = "custom_field." + idStr + ".content.keyword";
                    builder.must(getWildcardQuery(customKey, param.getQueryValue()));
                } else {
                    return null;
                }
            }
        }

        if (param.getSource() != null) {
            builder.must(getMatchQuery("source", param.getSource()));
        }

        if (StringUtils.isNotBlank(param.getSearchKey())) {
            BoolQueryBuilder shouldBuilder = QueryBuilders.boolQuery();
            shouldBuilder.should().add(getWildcardQuery("name.keyword", param.getSearchKey()));
            if (NumberUtils.isDigits(param.getSearchKey()) && param.getSearchKey().length() >= 3) {
                shouldBuilder.should().add(getWildcardQuery("mobile", param.getSearchKey()));
            }
            shouldBuilder.should().add(getWildcardQuery("tag.keyword", param.getSearchKey()));
            builder.must(shouldBuilder);
        }

        if (param.getGender() != null) {
            builder.must(getMatchQuery("gender", param.getGender()));
        }

        if (param.getScheduleStatus() != null && param.getScheduleStatus() != 0) {
            builder.must(getMatchQuery("scheduleStatus", param.getScheduleStatus()));
        }

        if (param.getRefundClassStatus() != null && param.getRefundClassStatus() != 0) {
            builder.must(getMatchQuery("refundClassStatus", param.getRefundClassStatus()));
        }

        if (param.getStuCenterBindStatus() != null && param.getStuCenterBindStatus() != 0) {
            builder.must(getMatchQuery("stuCenterBindStatus", param.getStuCenterBindStatus()));
        }

        //生日：年、月、日筛选
        if (param.getBirthYear() != null) {
            builder.must(getMatchQuery("birthYear", param.getBirthYear()));
        }
        if (param.getBirthMonth() != null) {
            builder.must(getMatchQuery("birthMonth", param.getBirthMonth()));
        }
        if (param.getBirthDayOfMonth() != null) {
            builder.must(getMatchQuery("birthDayOfMonth", param.getBirthDayOfMonth()));
        }

        if (param.getCreateTime() != null) {
            TimeRange range = param.getCreateTime().timeRange();
            builder.must(getTimestampRangeQueryBuilder("create_time", range.getStartTime(), range.getEndTime()));
        }

        if (param.getEnrollTime() != null) {
            TimeRange range = param.getEnrollTime().timeRange();
            List<OrgStudentCourse> courseList = studentCourseDao.listByTimeRange(param.getOrgId(), null, range.getStartTime(), range.getEndTime(), "userId", "id");
            if (courseList == null || courseList.size() < 1) {
                return null;
            } else {
                List<Long> userIds = ListUtil.toKeyList(courseList, "userId", OrgStudentCourse.class);
                if (param.getStudentIds() == null || param.getStudentIds().size() < 1) {
                    param.setStudentIds(new HashSet<>(userIds));
                } else {
                    param.getStudentIds().retainAll(userIds);
                    if (param.getStudentIds().size() < 1) {
                        return null;
                    }
                }
            }
        }

        // 按时间筛选
        if (param.getDateFieldKey() != null) {
            String dateFieldKey = ClientUtils.escapeQueryChars(param.getDateFieldKey());
            if (StringUtils.isNotBlank(dateFieldKey) && dateFieldKey.startsWith("customField")) {
                String idStr = dateFieldKey.replace("customField", "");
                String customDateKey = "custom_field." + idStr + ".content";
                builder.must(getTimestampRangeQueryBuilder(customDateKey, param.getStart(), param.getEnd()));
            } else {
                String keyFieldName = PCStudentOrderEnum.find(dateFieldKey);
                builder.must(getTimestampRangeQueryBuilder(keyFieldName, param.getStart(), param.getEnd()));
            }
        }

        List<Integer> lessonStatus = Lists.newArrayList();
        if ((param.getStatus() != null && param.getStatus() != StudentLessonStatus.ALL)) {
            lessonStatus.addAll(StudentLessonStatus.findSubStatus(param.getStatus().getStatus()));
            BoolQueryBuilder shouldBuilder = QueryBuilders.boolQuery();
            for (Integer status : lessonStatus) {
                shouldBuilder.should().add(getMatchQuery("lesson_status", status));
            }
            builder.must(shouldBuilder);
        }

        if (StringUtils.isNotBlank(param.getSelectFieldKey()) && StringUtils.isNotBlank(param.getSelectFieldValue())) {
            String selectKey = param.getSelectFieldKey().replace("customField", "");
            builder.must(getMatchQuery(getCustomFieldKey(selectKey, param.getOrgId()), param.getSelectFieldValue()));
        }

        if (param.getStudentIds() != null && param.getStudentIds().size() > 0) {
            builder.must(QueryBuilders.termsQuery("user_id", param.getStudentIds()));
        }

        if (param.getAddCascadeIds() != null && param.getAddCascadeIds().size() > 0) {
            builder.must(QueryBuilders.termsQuery("add_cascade_id", param.getAddCascadeIds()));
        }

        if (param.getLessonStatType() != null && param.getLessonStatType() > 0) {
            LessonStatType statType = LessonStatType.getByType(param.getLessonStatType());
            if (statType != null) {
                RangeQueryBuilder rangeQueryBuilder = QueryBuilders.rangeQuery(statType.getSearchKey());
                if (param.getMinNum() != null) {
                    rangeQueryBuilder.gte(param.getMinNum());
                }
                if (param.getMaxNum() != null) {
                    rangeQueryBuilder.lte(param.getMaxNum());
                }
                builder.must(rangeQueryBuilder);
            }
        }

        SortBuilder sortBuilder = SortBuilders.fieldSort("last_remind_time").order(SortOrder.DESC);
        if (GenericsUtils.notNullAndEmpty(param.getOrderName())) {
            String orderName = toEsField(param.getOrderName());
            if (orderName.equals("pinyin")) {
                orderName = orderName + ".keyword";
            }
            if (param.getOrderType() == 0) {
                sortBuilder = SortBuilders.fieldSort(orderName).order(SortOrder.ASC);
            }
            if (param.getOrderType() == 1) {
                sortBuilder = SortBuilders.fieldSort(orderName).order(SortOrder.DESC);
            }
        }

        log.info("[ES] query param={}", builder.toString());

        SearchResponse response = getClient().prepareSearch("students")
                .setTypes("student")
                .setQuery(builder)
                .addSort(sortBuilder)
                .setFrom((page.getPageNum() - 1) * page.getPageSize())
                .setSize(page.getPageSize())
                .get();
        return response;

    }

    private String toEsField(String orderName) {
        if (orderName.equals("addCascadeIdStr")) {
            return "add_cascade_id";
        } else if ("birthdayStr".equals(orderName)) {
            return "birthday";
        } else if ("createTimeStr".equals(orderName)) {
            return "create_time";
        } else if ("lastRemindTimeStr".equals(orderName)) {
            return "last_remind_time";
        } else if ("lastFollowTimeStr".equals(orderName)) {
            return "last_follow_time";
        } else {
            return orderName;
        }
    }
}
