package com.baijia.tianxiao.sal.elastic.service.impl;

import com.baijia.tianxiao.dal.es.constant.EsClientConfig;
import com.baijia.tianxiao.dal.es.dao.impl.AbstractEsBaseDao;
import com.baijia.tianxiao.dal.es.exceptions.EsException;
import com.baijia.tianxiao.dal.org.constant.StudentType;
import com.baijia.tianxiao.dal.roster.dao.CustomFieldDao;
import com.baijia.tianxiao.dal.roster.po.CustomField;
import com.baijia.tianxiao.dal.roster.po.TxStudentTag;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.sal.elastic.service.EsBaseService;

import org.elasticsearch.action.admin.indices.create.CreateIndexResponse;
import org.elasticsearch.action.admin.indices.delete.DeleteIndexResponse;
import org.elasticsearch.action.bulk.BulkProcessor;
import org.elasticsearch.action.bulk.BulkRequest;
import org.elasticsearch.action.bulk.BulkResponse;
import org.elasticsearch.client.transport.TransportClient;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.transport.InetSocketTransportAddress;
import org.elasticsearch.common.unit.ByteSizeUnit;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.index.query.FuzzyQueryBuilder;
import org.elasticsearch.index.query.MatchQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.index.query.RangeQueryBuilder;
import org.elasticsearch.index.query.WildcardQueryBuilder;
import org.elasticsearch.transport.client.PreBuiltTransportClient;
import org.springframework.beans.factory.annotation.Autowired;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import lombok.extern.slf4j.Slf4j;

@Slf4j
public abstract class AbstractEsBaseServiceImpl implements EsBaseService {

    @Autowired
    protected CustomFieldDao customFieldDao;



    private volatile static TransportClient transportClient;

    @Override
    public TransportClient getClient() {
        if (null == transportClient) {
            synchronized (AbstractEsBaseDao.class) {

                if (null == transportClient) {
                    // create client
                    long time = System.currentTimeMillis();
                    Settings settings =
                        Settings.builder().put("cluster.name", EsClientConfig.clusterNameString).build();
                    TransportClient creatingClient = new PreBuiltTransportClient(settings);
                    for (Map.Entry<String, Integer> entry : EsClientConfig.getAddressList()) {
                        try {
                            creatingClient.addTransportAddress(new InetSocketTransportAddress(
                                InetAddress.getByName(entry.getKey()), entry.getValue()));
                        } catch (UnknownHostException e) {
                            log.error("[AbstractEsBaseDao] invalide ip or port, and ignored! ip:{}, port:{}",
                                entry.getKey(), entry.getValue());
                            continue;
                        }
                    }
                    log.info("[{}] init transportClient done! costs:{}ms", this.getClass().getSimpleName(),
                        System.currentTimeMillis() - time);
                    this.transportClient = creatingClient;

                }

            }
        }
        return transportClient;
    }

    @Override
    public void createIndex(String index, int numberOfShards, int numberOfReplicas) {
        try {
            CreateIndexResponse createIndexResponse =
                this.getClient().admin()
                    .indices().prepareCreate(index).setSettings(Settings.builder()
                    .put("number_of_shards", numberOfShards).put("number_of_replicas", numberOfReplicas).build())
                    .get();
            log.warn("[{}] create es index:{} with {} Shards, {} Replicas result:{}", this.getClass().getSimpleName(),
                index, numberOfShards, numberOfReplicas, createIndexResponse.isAcknowledged());
        } catch (Exception e) {
            log.error("[{}] create es index:{} with {} Shards, {} Replicas error! e:{}",
                this.getClass().getSimpleName(), index, numberOfShards, numberOfReplicas, e);
            throw new EsException(CommonErrorCode.SYSTEM_ERROR, e.getMessage());
        }
    }

    @Override
    public void deleteIndex(String index) {
        try {
            DeleteIndexResponse deleteIndexResponse = this.getClient().admin().indices().prepareDelete(index).get();
            log.warn("[{}] delete es index:{} result:{}", this.getClass().getSimpleName(), index,
                deleteIndexResponse.isAcknowledged());
        } catch (Exception e) {
            log.error("[{}] delete es index:{} error! e:{}", this.getClass().getSimpleName(), index, e);
            throw new EsException(CommonErrorCode.SYSTEM_ERROR, e.getMessage());
        }
    }

    public BulkProcessor getDefaultBulkProcessor() {
        BulkProcessor bulkProcessor = BulkProcessor.builder(this.getClient(), new BulkProcessor.Listener() {
            @Override
            public void beforeBulk(long executionId, BulkRequest request) {
                log.info("going to bulk, executionId:{}, numOfActions:{}", executionId, request.numberOfActions());
            }

            @Override
            public void afterBulk(long executionId, BulkRequest request, BulkResponse response) {
                log.info("bulk ok, executionId:{}, numOfActions:{}, costs:{}ms, hasFailures:{}", executionId,
                    request.numberOfActions(), response.getTookInMillis(), response.hasFailures());
            }

            @Override
            public void afterBulk(long executionId, BulkRequest request, Throwable failure) {
                log.error("bulk failed!, error:{}", failure);
            }
        }).setConcurrentRequests(4).setBulkActions(20000).setBulkSize(new ByteSizeValue(8, ByteSizeUnit.MB))
            .setFlushInterval(TimeValue.timeValueSeconds(2)).build();
        return bulkProcessor;
    }

    protected RangeQueryBuilder getTimestampRangeQueryBuilder(String key, Date from, Date to) {
        long from_time = (null == from ? 0L : from.getTime());
        long to_time = (null == to ? 0L : to.getTime());

        if(key.equals("finally_hold_time")){
            if(0L == from_time){
                from_time = -1L;
            }
        }
        RangeQueryBuilder rangeQueryBuilder = QueryBuilders.rangeQuery(key);
        rangeQueryBuilder.from(from_time);
        rangeQueryBuilder.to(to_time);
        return rangeQueryBuilder;
    }

    protected MatchQueryBuilder getMatchQuery(String key, Object value) {
        return QueryBuilders.matchQuery(key, value);
    }

    protected WildcardQueryBuilder getWildcardQuery(String key, Object value) {
        return QueryBuilders.wildcardQuery(key, "*" + value + "*");
    }

    protected FuzzyQueryBuilder getFuzzyQuery(String key, Object value) {
        return QueryBuilders.fuzzyQuery(key, value);
    }

    protected RangeQueryBuilder getDateRangeQueryBuilder(String key, Date from, Date to) {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        RangeQueryBuilder rangeQueryBuilder = QueryBuilders.rangeQuery(key);
        rangeQueryBuilder.from(sdf.format(from));
        rangeQueryBuilder.to(sdf.format(to));
        return rangeQueryBuilder;
    }

    protected String getCustomFieldKey(String key, Long orgId) {
        StringBuilder keyBuilder = new StringBuilder();
        keyBuilder.append("custom_field.").append(key);
        CustomField field = customFieldDao.getCustomFieldById(orgId, Long.parseLong(key));
        if (field == null) {
            log.warn("[ES] Field is not exist.Field={}", key);
            return "-";
        }
        if (field.getType() == 3) {// 多选
            keyBuilder.append(".options");
        }
        return keyBuilder.append(".id").toString();
    }

    protected Map<Long, List<TxStudentTag>> toMap(List<TxStudentTag> tags, StudentType userType) {
        Map<Long, List<TxStudentTag>> retMap = new HashMap<>();
        for (TxStudentTag tag : tags) {
            long key = 0;
            if(userType==StudentType.CONSULT_USER){
                key = tag.getConsultUserId();
            }else {
                key = tag.getUserId();
            }
            List<TxStudentTag> list = retMap.get(key);
            if (list == null) {
                list = new ArrayList<>();
                retMap.put(key, list);
            }
            list.add(tag);
        }
        return retMap;
    }
}
