package com.baijia.tianxiao.display.service.impl;

import com.baijia.tianxiao.constants.DataProcType;
import com.baijia.tianxiao.constants.org.BizConf;
import com.baijia.tianxiao.dal.roster.dao.CustomFieldDao;
import com.baijia.tianxiao.dal.roster.po.CustomField;
import com.baijia.tianxiao.dal.show.dao.FieldShowInfoDao;
import com.baijia.tianxiao.dal.show.enums.FieldStatus;
import com.baijia.tianxiao.dal.show.enums.ShowStatus;
import com.baijia.tianxiao.dal.show.po.FieldShowInfo;
import com.baijia.tianxiao.display.dto.annotation.FieldProp;
import com.baijia.tianxiao.display.dto.response.crm.DefaultClueField;
import com.baijia.tianxiao.display.dto.response.crm.DefaultStudentField;
import com.baijia.tianxiao.display.service.FieldShowInfoService;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.field.FieldOption;
import com.baijia.tianxiao.util.BaseUtils;
import com.baijia.tianxiao.util.json.JacksonUtil;
import com.beust.jcommander.internal.Lists;
import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import java.lang.reflect.Field;
import java.util.*;

/**
 * Created by wangsixia on 2016/11/10.
 */
@Service
public class FieldShowInfoServiceImpl implements FieldShowInfoService {

    // 公共线索列表显示时过滤的字段属性名
    private static final Set<String> PUBLIC_CLUE_EXCLUDE = Sets.newHashSet("consultStatusStr");
    // 自定义字段
    private static final int CUSTOM_CODE = -1;
    private static final String CUSTOM_STR = "custom_";

    @Autowired
    private FieldShowInfoDao fieldShowInfoDao;

    @Autowired
    private CustomFieldDao customFieldDao;

    @Override
    @Transactional(rollbackFor = {Exception.class, BussinessException.class})
    public Map<String, Object> getShowConfig(Long orgId, DataProcType type) {
        if (orgId == null || orgId <= 0) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "机构id错误");
        }
        // 初始化字段信息:系统字段+自定义字段
        this.initFieldShow(orgId, type);

        List<FieldOption> all = new ArrayList<>();
        List<FieldOption> using = new ArrayList<>();
        Map<String, Object> map = new HashMap<>();
        // 显示字段+隐藏字段
        List<FieldShowInfo> fieldList = this.fieldShowInfoDao.getOrgFieldShow(orgId, type.getType(), FieldStatus.NORMAL.getCode(),
                Sets.<Integer>newHashSet(ShowStatus.SHOW.getCode(), ShowStatus.HIDDEN.getCode()));
        if (CollectionUtils.isNotEmpty(fieldList)) {
            // 显示字段map<preShowId, fieldShowInfo>,用于显示字段根据preShowId排序
            Map<Long, FieldShowInfo> preCurShowMap = new HashMap<>();
            int checkPreNull = 0;
            for (FieldShowInfo field : fieldList) {
                if (field.getShowStatus() == ShowStatus.SHOW.getCode()) {
                    if (field.getPreShowId() == null) {
                        // 显示字段中只有第一个的preShowId为null
                        if (checkPreNull > 0) {
                            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "显示字段异常");
                        }
                        checkPreNull++;
                    }
                    preCurShowMap.put(field.getPreShowId(), field);
                }
            }

            // 显示字段排序
            Long preId = null;
            while (!preCurShowMap.isEmpty()) {
                FieldShowInfo info = preCurShowMap.remove(preId);
                if (info != null) {
                    FieldOption fieldOption = new FieldOption();
                    fieldOption.setFieldShowId(info.getId()); // 重要,用于保存时回传
                    fieldOption.setLock(info.getLocked() == BizConf.TRUE.intValue() ? true : false);// 重要,用于保存时回传
                    fieldOption.setPreId(info.getPreShowId());
                    fieldOption.setName(info.getName());
                    fieldOption.setShowName(info.getShowName());
                    fieldOption.setHidden(false);
                    fieldOption.setSort(info.getSorted() == BizConf.TRUE.intValue() ? true : false);
                    fieldOption.setCustomId(info.getCustomFieldId());
                    fieldOption.setQueryProp(info.getQueryProp());
                    using.add(fieldOption);
                    all.add(fieldOption);
                    // 改变指针
                    preId = info.getId();
                } else {
                    throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "显示字段异常");
                }
            }

            // 将其他非显示字段加入all
            for (FieldShowInfo po : fieldList) {
                if (po.getShowStatus() != ShowStatus.SHOW.getCode()) {
                    FieldOption dto = new FieldOption();
                    dto.setFieldShowId(po.getId()); // 重要,用于保存时回传
                    dto.setLock(po.getLocked() == BizConf.TRUE.intValue() ? true : false); // 重要,用于保存时回传
                    dto.setName(po.getName());
                    dto.setShowName(po.getShowName());
                    dto.setPreId(po.getPreShowId());
                    dto.setSort(po.getSorted() == BizConf.TRUE.intValue() ? true : false);
                    dto.setCustomId(po.getCustomFieldId());
                    dto.setQueryProp(po.getQueryProp());
                    if (po.getShowStatus() == ShowStatus.NEVER_SHOW.getCode()) {
                        dto.setHidden(true);
                    } else {
                        dto.setHidden(false);
                    }
                    all.add(dto);
                }
            }
        }
        map.put("getTime", new Date().getTime());
        map.put("using", using);
        map.put("all", all);
        return map;
    }

    @Override
    @Transactional(rollbackFor = {Exception.class, BussinessException.class})
    public void setShowConfig(Long orgId, DataProcType type, Long getTime, String content) {
        if (orgId == null || orgId <= 0) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "机构id错误");
        }
        if (StringUtils.isBlank(content)) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "设置参数不能为空");
        }
        if (getTime == null || new Date(getTime).after(new Date())) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "获取设置时间参数错误");
        }
        boolean neverUpdate = this.fieldShowInfoDao.consistencyCheck(orgId, type.getType(), new Date(getTime));
        if (!neverUpdate) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "属性字段在此期间已被更新,请刷新页面重试");
        }

        List<FieldOption> optionList = null;
        try {
            optionList = JacksonUtil.str2List(content, FieldOption.class);
        } catch (Exception e) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "设置参数格式错误");
        }
        if (CollectionUtils.isNotEmpty(optionList)) {
            // 更新显示字段的preId,锁定字段保持不变
            for (int index = 0; index < optionList.size(); index++) {
                if (optionList.get(index).isLock()) {
                    continue;
                }
                optionList.get(index).setPreId(optionList.get(index - 1).getFieldShowId());
            }
            // 待更新的FieldShowInfo
            List<FieldShowInfo> updateList = new ArrayList<>();
            // 当前系统中显示/隐藏字段
            List<FieldShowInfo> curList = this.fieldShowInfoDao.getOrgFieldShow(orgId, type.getType(), FieldStatus.NORMAL.getCode(),
                    Sets.newHashSet(ShowStatus.SHOW.getCode(), ShowStatus.HIDDEN.getCode()));
            Map<Long, FieldShowInfo> curFieldMap = BaseUtils.listToMap(curList, "id");
            for (FieldOption option : optionList) {
                FieldShowInfo po = curFieldMap.remove(option.getFieldShowId());
                if (po == null) {
                    throw new BussinessException(CommonErrorCode.PARAM_ERROR, "字段参数异常");
                }
                // 隐藏-->显示、显示顺序改变
                if (po.getShowStatus() == ShowStatus.HIDDEN.getCode() || (po.getPreShowId() != null && po.getPreShowId().longValue() != option.getPreId().longValue())) {
                    po.setShowStatus(ShowStatus.SHOW.getCode());
                    po.setPreShowId(option.getPreId());
                    po.setUpdateTime(new Date());
                    updateList.add(po);
                }
            }
            if (!curFieldMap.isEmpty()) {
                for (FieldShowInfo otherField : curFieldMap.values()) {
                    // 显示-->隐藏
                    if (otherField.getShowStatus() == ShowStatus.SHOW.getCode()) {
                        otherField.setShowStatus(ShowStatus.HIDDEN.getCode());
                        otherField.setPreShowId(null);
                        otherField.setUpdateTime(new Date());
                        updateList.add(otherField);
                    }
                }
            }
            if (CollectionUtils.isNotEmpty(updateList)) {
                for (FieldShowInfo po : updateList) {
                    this.fieldShowInfoDao.update(po, true, "showStatus", "preShowId", "updateTime");
                }
            }
        } else {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "显示字段不能为空");
        }
    }

    @Override
    public List<FieldOption> getHeader(Long orgId, DataProcType type, Set<String> excludeProps, boolean filterCustom) {
        if (orgId == null || orgId <= 0) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "机构id错误");
        }
        // 初始化字段信息:系统字段+自定义字段
        this.initFieldShow(orgId, type);

        List<FieldOption> header = Lists.newArrayList();
        List<FieldOption> result = Lists.newArrayList();
        // 显示字段+隐藏属性
        List<FieldShowInfo> fieldList = this.fieldShowInfoDao.getOrgFieldShow(orgId, type.getType(), FieldStatus.NORMAL.getCode(),
                Sets.<Integer>newHashSet(ShowStatus.SHOW.getCode(), ShowStatus.NEVER_SHOW.getCode()));
        if (CollectionUtils.isNotEmpty(fieldList)) {
            // 显示字段map<preShowId, fieldShowInfo>,用于显示字段根据preShowId排序
            Map<Long, FieldShowInfo> preCurShowMap = new HashMap<>();
            int checkPreNull = 0;
            for (FieldShowInfo field : fieldList) {
                if (field.getShowStatus() == ShowStatus.SHOW.getCode()) {
                    if (field.getPreShowId() == null) {
                        // 显示字段中只有第一个的preShowId为null
                        if (checkPreNull > 0) {
                            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "显示字段异常");
                        }
                        checkPreNull++;
                    }
                    preCurShowMap.put(field.getPreShowId(), field);
                }
            }

            // 显示字段排序
            Long preId = null;
            while (!preCurShowMap.isEmpty()) {
                FieldShowInfo info = preCurShowMap.remove(preId);
                if (info != null) {
                    FieldOption fieldOption = new FieldOption();
                    fieldOption.setFieldShowId(info.getId());
                    fieldOption.setLock(info.getLocked() == BizConf.TRUE.intValue() ? true : false);
                    fieldOption.setPreId(info.getPreShowId());
                    fieldOption.setName(info.getName());
                    fieldOption.setShowName(info.getShowName());
                    fieldOption.setHidden(false);
                    fieldOption.setSort(info.getSorted() == BizConf.TRUE.intValue() ? true : false);
                    fieldOption.setCustomId(info.getCustomFieldId());// 重要,若为自定义字段,用于获取字段值
                    fieldOption.setQueryProp(info.getQueryProp());
                    header.add(fieldOption);
                    // 改变指针
                    preId = info.getId();
                } else {
                    throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "显示字段异常");
                }
            }

            // 将其他非显示字段加入header,如id
            for (FieldShowInfo po : fieldList) {
                if (po.getShowStatus() != ShowStatus.SHOW.getCode()) {
                    FieldOption dto = new FieldOption();
                    dto.setFieldShowId(po.getId());
                    dto.setLock(po.getLocked() == BizConf.TRUE.intValue() ? true : false);
                    dto.setName(po.getName());
                    dto.setShowName(po.getShowName());
                    dto.setPreId(po.getPreShowId());
                    dto.setSort(po.getSorted() == BizConf.TRUE.intValue() ? true : false);
                    dto.setHidden(true);
                    dto.setCustomId(po.getCustomFieldId());
                    dto.setQueryProp(po.getQueryProp());
                    header.add(dto);
                }
            }

            if (CollectionUtils.isNotEmpty(header)) {
                for (int index = 0; index < header.size(); index++) {
                    // 需要剔除的系统字段
                    if (CollectionUtils.isNotEmpty(excludeProps)) {
                        String name = header.get(index).getName();
                        if (excludeProps.contains(name)) {
                            excludeProps.remove(name);
                            continue;
                        }
                    }
                    // 导出时需要剔除自定义字段
                    if (filterCustom && header.get(index).getCustomId() != null && header.get(index).getCustomId() > 0) {
                        continue;
                    }
                    result.add(header.get(index));
                }
            }
        }
        return result;
    }

    @Override
    @Transactional(rollbackFor = {Exception.class, BussinessException.class})
    public void addCustomField(Long orgId, DataProcType type, Long customFieldId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "机构id错误");
        Preconditions.checkArgument(customFieldId != null && customFieldId > 0, "自定义字段id错误");

        CustomField customField = this.customFieldDao.getCustomFieldById(orgId, customFieldId);
        if (customField == null) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "自定义字段不存在或已被删除");
        }
        if (customField.getSystemFieldNum() != null && customField.getSystemFieldNum() != CUSTOM_CODE) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "该字段为系统字段");
        }
        if (customField.getIsPaused() != null && customField.getIsPaused().intValue() == BizConf.TRUE.intValue()) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "字段已存在且为停用状态");
        }

        // 初始化字段,避免重复添加自定义字段显示属性
        boolean hasInitCustom = this.initFieldShow(orgId, type);
        if (hasInitCustom) {
            return;
        }

        Long lastLockId = this.fieldShowInfoDao.getLastLockInfo(orgId, type.getType());
        if (lastLockId != null && lastLockId > 0) {
            FieldShowInfo nextField = this.fieldShowInfoDao.getShowBehind(orgId, type.getType(), lastLockId, "id");

            FieldShowInfo po = new FieldShowInfo();
            po.setOrgId(orgId);
            po.setCustomFieldId(customField.getId());
            po.setName(CUSTOM_STR + customField.getId());
            po.setShowName(customField.getLabel());
            po.setFieldStatus(FieldStatus.NORMAL.getCode());
            po.setType(type.getType());
            po.setShowStatus(ShowStatus.SHOW.getCode());
            po.setPreShowId(lastLockId);
            this.fieldShowInfoDao.save(po, false);

            // 更新最后一个锁定字段原来的后一个字段的preId
            if (nextField != null) {
                nextField.setPreShowId(po.getId());
                nextField.setUpdateTime(new Date());
                this.fieldShowInfoDao.update(nextField, "preShowId", "updateTime");
            }
        }else {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "未找到锁定字段");
        }
    }

    @Override
    @Transactional(rollbackFor = {Exception.class, BussinessException.class})
    public void pauseCustomField(Long orgId, DataProcType type, Long customFieldId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "机构id错误");
        Preconditions.checkArgument(customFieldId != null && customFieldId > 0, "自定义字段id错误");

        CustomField customField = this.customFieldDao.getCustomFieldById(orgId, customFieldId);
        if (customField == null) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "字段不存在或已被删除");
        }
        if (customField.getSystemFieldNum() != null && customField.getSystemFieldNum() != CUSTOM_CODE) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "该字段为系统字段");
        }
        if (customField.getIsPaused() != null && customField.getIsPaused().intValue() == BizConf.TRUE.intValue()) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "字段已存在且为停用状态");
        }

        this.initFieldShow(orgId, type);

        // 二次校验
        FieldShowInfo tobePause = this.fieldShowInfoDao.getByCustomId(orgId, type.getType(), customField.getId(), "id", "fieldStatus", "preShowId");
        if (tobePause == null || tobePause.getFieldStatus() == FieldStatus.DELETED.getCode()) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "字段不存在或已被删除");
        }

        if (tobePause.getFieldStatus() == FieldStatus.PAUSED.getCode()) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "字段已存在且为停用状态");
        }

        FieldShowInfo nextField = this.fieldShowInfoDao.getShowBehind(orgId, type.getType(), tobePause.getId(), "id");
        if (nextField != null) {
            nextField.setPreShowId(tobePause.getPreShowId());
            nextField.setUpdateTime(new Date());
            this.fieldShowInfoDao.update(nextField, false, "preShowId", "updateTime");
        }

        tobePause.setFieldStatus(FieldStatus.PAUSED.getCode());
        tobePause.setPreShowId(null);
        tobePause.setUpdateTime(new Date());
        this.fieldShowInfoDao.update(tobePause, true, "fieldStatus", "preShowId", "updateTime");
    }

    @Override
    @Transactional(rollbackFor = {Exception.class, BussinessException.class})
    public void enableCustomField(Long orgId, DataProcType type, Long customFieldId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "机构id错误");
        Preconditions.checkArgument(customFieldId != null && customFieldId > 0, "自定义字段id错误");

        CustomField customField = this.customFieldDao.getCustomFieldById(orgId, customFieldId);
        if (customField == null) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "字段不存在或已被删除");
        }
        if (customField.getSystemFieldNum() != null && customField.getSystemFieldNum() != CUSTOM_CODE) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "该字段为系统字段");
        }
        if (customField.getIsPaused() != null && customField.getIsPaused().intValue() == BizConf.FALSE.intValue()) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "字段已存在且为启用状态");
        }

        this.initFieldShow(orgId, type);

        // 二次校验
        FieldShowInfo tobeEnable = this.fieldShowInfoDao.getByCustomId(orgId, type.getType(), customField.getId(), "id", "fieldStatus");
        if (tobeEnable == null || tobeEnable.getFieldStatus() == FieldStatus.DELETED.getCode()) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "字段不存在或已被删除");
        }
        if (tobeEnable.getFieldStatus() == FieldStatus.NORMAL.getCode()) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "字段已存在且为启用状态");
        }

        Long lastLockId = this.fieldShowInfoDao.getLastLockInfo(orgId, type.getType());
        if (lastLockId != null && lastLockId > 0) {
            FieldShowInfo nextField = this.fieldShowInfoDao.getShowBehind(orgId, type.getType(), lastLockId, "id");

            tobeEnable.setShowStatus(ShowStatus.SHOW.getCode());
            tobeEnable.setFieldStatus(FieldStatus.NORMAL.getCode());
            tobeEnable.setPreShowId(lastLockId);
            tobeEnable.setUpdateTime(new Date());
            this.fieldShowInfoDao.update(tobeEnable, "showStatus", "fieldStatus", "preShowId", "updateTime");

            // 更新最后一个锁定字段原来的后一个字段的preId
            if (nextField != null) {
                nextField.setPreShowId(tobeEnable.getId());
                nextField.setUpdateTime(new Date());
                this.fieldShowInfoDao.update(nextField, "preShowId", "updateTime");
            }
        } else {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "未找到锁定字段");
        }
    }

    @Override
    @Transactional(rollbackFor = {Exception.class, BussinessException.class})
    public void deleteCustomField(Long orgId, DataProcType type, Long customFieldId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "机构id错误");
        Preconditions.checkArgument(customFieldId != null && customFieldId > 0, "自定义字段id错误");

        CustomField customField = this.customFieldDao.getCustomFieldById(orgId, customFieldId);
        if (customField == null) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "字段不存在或已被删除");
        }
        if (customField.getSystemFieldNum() != null && customField.getSystemFieldNum() != CUSTOM_CODE) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "该字段为系统字段");
        }

        this.initFieldShow(orgId, type);

        // 二次校验
        FieldShowInfo tobeDelete = this.fieldShowInfoDao.getByCustomId(orgId, type.getType(), customField.getId(), "id", "fieldStatus", "preShowId");
        if (tobeDelete == null || tobeDelete.getFieldStatus() == FieldStatus.DELETED.getCode()) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "字段不存在或已被删除");
        }

        FieldShowInfo nextField = this.fieldShowInfoDao.getShowBehind(orgId, type.getType(), tobeDelete.getId(), "id");
        if (nextField != null) {
            nextField.setPreShowId(tobeDelete.getPreShowId());
            nextField.setUpdateTime(new Date());
            this.fieldShowInfoDao.update(nextField, false, "preShowId", "updateTime");
        }

        tobeDelete.setFieldStatus(FieldStatus.DELETED.getCode());
        tobeDelete.setPreShowId(null);
        tobeDelete.setUpdateTime(new Date());
        this.fieldShowInfoDao.update(tobeDelete, true, "fieldStatus", "preShowId", "updateTime");
    }

    @Override
    @Transactional(rollbackFor = {Exception.class, BussinessException.class})
    public void modCustomShowName(Long orgId, DataProcType type, Long customFieldId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "机构id错误");
        Preconditions.checkArgument(customFieldId != null && customFieldId > 0, "自定义字段id错误");

        CustomField customField = this.customFieldDao.getCustomFieldById(orgId, customFieldId);
        if (customField == null) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "字段不存在或已被删除");
        }
        if (customField.getSystemFieldNum() != null && customField.getSystemFieldNum() != CUSTOM_CODE) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "该字段为系统字段");
        }
        if (customField.getIsPaused() != null && customField.getIsPaused().intValue() == BizConf.TRUE.intValue()) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "字段已存在且为停用状态");
        }

        this.initFieldShow(orgId, type);

        // 二次校验
        FieldShowInfo tobeEdit = this.fieldShowInfoDao.getByCustomId(orgId, type.getType(), customField.getId(), "id", "fieldStatus");
        if (tobeEdit == null || tobeEdit.getFieldStatus() == FieldStatus.DELETED.getCode()) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "字段不存在或已被删除");
        }
        if (tobeEdit.getFieldStatus() == FieldStatus.PAUSED.getCode()) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "字段已存在且为停用状态");
        }

        // 自定义字段名字发生改变时才更新
        if (!customField.getLabel().equals(tobeEdit.getShowName())) {
            tobeEdit.setShowName(customField.getLabel());
            tobeEdit.setUpdateTime(new Date());
            this.fieldShowInfoDao.update(tobeEdit, false, "showName", "updateTime");
        }
    }

    /**
     * 初始化字段信息
     * @param orgId
     * @param type
     * @return 返回此次是否初始化自定义字段
     */
    private boolean initFieldShow(Long orgId, DataProcType type) {
        boolean hasInit = this.fieldShowInfoDao.hasInit(orgId, type.getType());
        if (hasInit) {
            return false;
        }
        Class clazz = null;
        // 目前只有crm类型,即线索和学员
        if (type.getType() == DataProcType.CONSULT.getType()) {
            clazz = DefaultClueField.class;
        } else if (type.getType() == DataProcType.ORG_STUDENT.getType()) {
            clazz = DefaultStudentField.class;
        }
        if (clazz != null) {
            return this.initCrmFieldShow(orgId, type, clazz);
        } else {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "不支持的数据类型");
        }
    }

    /**
     * 初始化线索/学员字段属性
     * @param orgId
     * @param type
     * @param clazz
     * @param <T>
     */
    private<T> boolean initCrmFieldShow(Long orgId, DataProcType type, Class<T> clazz) {
        boolean hasInitCustom = false;
        Field[] fields = clazz.getDeclaredFields();
        if (fields != null && fields.length > 0) {
            // 系统显示字段的id列表,按先后插入顺序
            List<Long> sysShowIdList = Lists.newArrayList();
            // 系统字段显示顺序
            int sysShowIndex = 0;
            for (int index = 0; index < fields.length; index++) {
                FieldProp anno = fields[index].getAnnotation(FieldProp.class);
                if (anno != null) {
                    FieldShowInfo po = new FieldShowInfo();
                    po.setOrgId(orgId);
                    po.setType(type.getType());
                    po.setName(anno.name());
                    po.setShowName(anno.showName());
                    po.setQueryProp(anno.queryProp());
                    po.setFieldStatus(FieldStatus.NORMAL.getCode());
                    po.setShowStatus(anno.showStatus());
                    po.setLocked(anno.lock() ? BizConf.TRUE.intValue() : BizConf.FALSE.intValue());
                    po.setSorted(anno.sort() ? BizConf.TRUE.intValue() : BizConf.FALSE.intValue());
                    po.setIsSys(BizConf.TRUE.intValue());
                    po.setCustomFieldId(null);
                    // 显示字段需要记录显示顺序,用于给它后面的显示字段提供preShowId
                    if (anno.showStatus() == ShowStatus.SHOW.getCode()) {
                        // 第一个显示字段的preShowId为null
                        if (sysShowIndex > 0) {
                            po.setPreShowId(sysShowIdList.get(sysShowIndex - 1));
                        }
                        this.fieldShowInfoDao.save(po, false);
                        sysShowIdList.add(po.getId());
                        sysShowIndex++;
                    } else {
                        this.fieldShowInfoDao.save(po, false);
                    }
                }
            }
        }

        // 初始化机构已设置的自定义字段(兼容老数据),包括启用和停用
        List<CustomField> customFieldList = this.customFieldDao.getCustomFieldList(orgId, BizConf.FALSE);
        if (CollectionUtils.isNotEmpty(customFieldList)) {
            Long lastLockId = this.fieldShowInfoDao.getLastLockInfo(orgId, type.getType());
            if (lastLockId != null && lastLockId > 0) {
                FieldShowInfo nextField = this.fieldShowInfoDao.getShowBehind(orgId, type.getType(), lastLockId, "id");
                // 自定义启用字段的id列表,按先后插入顺序。第一个为最后一个锁定字段id
                List<Long> customShowIdList = Lists.newArrayList(lastLockId);
                // 自定义启用字段显示顺序
                int customShowIndex = 0;
                for (int index = 0; index < customFieldList.size(); index++) {
                    CustomField field = customFieldList.get(index);
                    FieldShowInfo po = new FieldShowInfo();
                    po.setOrgId(orgId);
                    po.setCustomFieldId(field.getId());
                    po.setName(CUSTOM_STR + field.getId());
                    po.setShowName(field.getLabel());
                    po.setType(type.getType());
                    po.setIsSys(BizConf.FALSE.intValue());
                    po.setShowStatus(ShowStatus.SHOW.getCode());
                    po.setLocked(BizConf.FALSE.intValue());
                    po.setSorted(BizConf.FALSE.intValue());
                    if (field.getIsPaused() != null && field.getIsPaused().intValue() == BizConf.TRUE.intValue()) {
                        // 停用字段
                        po.setFieldStatus(FieldStatus.PAUSED.getCode());
                        this.fieldShowInfoDao.save(po, false);
                    } else {
                        // 启用字段
                        po.setPreShowId(customShowIdList.get(customShowIndex));
                        po.setFieldStatus(FieldStatus.NORMAL.getCode());
                        this.fieldShowInfoDao.save(po, false);
                        customShowIdList.add(po.getId());
                        customShowIndex++;
                    }
                }
                // 更新最后一个锁定字段原来的后一个字段的preId为添加的最后一个启用字段的id
                if (nextField != null) {
                    nextField.setPreShowId(customShowIdList.get(customShowIndex));
                    nextField.setUpdateTime(new Date());
                    this.fieldShowInfoDao.update(nextField, false, "preShowId", "updateTime");
                }
            } else {
                throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "未找到锁定字段");
            }
            hasInitCustom = true;
        }
        return hasInitCustom;
    }
}
