package com.baijia.tianxiao.sal.course.service.impl;

import com.baijia.commons.lang.utils.PropertiesReader;
import com.baijia.tianxiao.consants.UserRole;
import com.baijia.tianxiao.constants.sms.SmsMessageType;
import com.baijia.tianxiao.constants.sms.SmsSendResult;
import com.baijia.tianxiao.dal.org.dao.*;
import com.baijia.tianxiao.dal.org.po.OrgCourseSms;
import com.baijia.tianxiao.dal.org.po.OrgStudent;
import com.baijia.tianxiao.dal.org.po.OrgTeacher;
import com.baijia.tianxiao.dal.org.po.TtsSms;
import com.baijia.tianxiao.dal.user.dao.TeacherDao;
import com.baijia.tianxiao.dal.user.po.Teacher;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.course.service.SmsService;
import com.baijia.tianxiao.util.CourseSmsTokenUtil;
import com.baijia.tianxiao.util.ShortUrlUtil;
import com.baijia.tianxiao.util.SmsSendUtil;
import com.baijia.tianxiao.util.mobile.MaskUtil;
import com.baijia.tianxiao.validation.ParamValidateUtils;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

/**
 * 发送课表
 * 
 * @title SmsServiceImpl
 * @desc TODO
 * @author shizuwei
 * @date 2015年12月29日
 * @version 1.0
 */
@Slf4j
@Service("SmsService1")
public class SmsServiceImpl implements SmsService {

    // private final static String STUDENT_SMS_TEMPLATE = "『%s』您好，您在『%s』的%s已经完成课程安排。课表查看地址：%s";
    // private final static String TEACHER_SMS_TEMPLATE = "『%s』您好，您在『%s』的%s已经完成课程安排。课表查看地址：%s";

    private final static String STUDENT_SMS_TEMPLATE = "%s您好，您在%s的%s已经完成课程安排。课表查看地址：%s";
    private final static String TEACHER_SMS_TEMPLATE = "%s您好，您在%s的%s已经完成课程安排。课表查看地址：%s";

    @Resource
    private OrgCourseDao orgCourseDao;

    @Resource
    private TeacherDao teacherDao;

    @Resource
    private OrgStudentDao orgStudentDao;

    @Resource
    private OrgCourseSmsDao orgCourseSmsDao;

    @Resource
    private OrgInfoDao orgInfoDao;

    @Resource
    private TtsSmsDao ttsSmsDao;

    @Resource
    private OrgTeacherDao orgTeacherDao;

    @Override
    public boolean saveAndSendSms(Long operatorId, Long smsKey, Integer userRole, Long userId, String mobile,
        String content) {
        if (!ParamValidateUtils.validateMobile(mobile) || StringUtils.isBlank(content)) {
            return false;
        }
        TtsSms sms = new TtsSms();
        sms.setContent(content);
        sms.setMessageType(SmsMessageType.TIANXIAO_NOTIFY.getCode());
        sms.setMobile(mobile);
        sms.setOperatorId(operatorId);
        sms.setSendSmsKey(smsKey != null ? smsKey : 0l);
        sms.setUserRole(userRole);
        boolean sendResult = false;
        try {
            sendResult =
                SmsSendUtil.sendSms(mobile, content, SmsMessageType.TIANXIAO_NOTIFY.getCode(), operatorId.intValue(), UserRole.ORGANIZATION.getRole(), true);
            sms.setSendResult(sendResult ? SmsSendResult.SUCCESS.getValue() : SmsSendResult.FAILED.getValue());
            ttsSmsDao.save(sms);
        } catch (Exception e) {
            log.warn("save sms record catch error:", e);
            // return false;
        }
        log.info("send sms to :{} is succ:{}", mobile, sendResult);
        return sendResult;
    }

    @Override
    public List<Long> sendCourseSms(@NonNull Long orgId, @NonNull Long courseId, @NonNull Integer userRole,
        @NonNull List<Long> userIds) {
        log.debug("userIds = {} orgId ={} courseId = {},userRole = {}", userIds, orgId, courseId, userRole);
        if (UserRole.TEACHER.getRole() == userRole) {
            return sendTeacherCourseSms(orgId, courseId, userRole, userIds);
        } else if (UserRole.STUDENT.getRole() == userRole) {
            return sendStudentCourseSms(orgId, courseId, userRole, userIds);
        } else {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR);
        }
    }

    /**
     * 生成token并保存
     * 
     * @param orgId
     * @param courseId
     * @param userRole
     * @param userId
     * @return
     */
    private String getSmsToken(Long orgId, Long courseId, Integer userRole, Long userId, Long studentId) {

        String token = null;
        // 查找是否存在
        OrgCourseSms orgCourseSms = this.orgCourseSmsDao.getOrgCourseSms(orgId, courseId, userRole, userId);
        Map<String, Object> params = new HashMap<>();
        if (orgCourseSms == null) {
            orgCourseSms = new OrgCourseSms();
            orgCourseSms.setCourseId(courseId);
            orgCourseSms.setOrgId(orgId);
            orgCourseSms.setUserRole(userRole);
            orgCourseSms.setUserId(userId);
            orgCourseSms.setSend(0);
            this.orgCourseSmsDao.save(orgCourseSms);
        }
        params.put("id", orgCourseSms.getId());
        params.put("orgId", orgId);
        params.put("courseId", courseId);

        if (UserRole.STUDENT.getRole() == userRole) {
            params.put("studentId", studentId);
        } else if (UserRole.TEACHER.getRole() == userRole) {
            params.put("teacherId", userId);
        }
        orgCourseSms.setStudentId(studentId);
        try {
            log.debug("orgCourseSms = {}", orgCourseSms);
            token = CourseSmsTokenUtil.encodeToken(params);
        } catch (Exception e) {
            log.debug("token = {} ", token);
            return null;
        }
        return token;
    }

    private List<Long> sendStudentCourseSms(Long orgId, Long courseId, Integer userRole, List<Long> studentIds) {

        Map<Long, Long> stuUserMap = orgStudentDao.getStudentIdUserIdMap(studentIds);

        List<Long> succeedStudentIds = Lists.newArrayList();
        boolean sendResult = false;
        String content = null;
        List<OrgStudent> students = this.orgStudentDao.getStudentByIds(orgId, studentIds);
        String orgName = this.orgInfoDao.getOrgShortNameByOrgId(orgId.intValue());
        String courseName = this.orgCourseDao.getCourseNameById(courseId);
        String longUrlFormat = PropertiesReader.getProperties("url.properties").getProperty("sms.course.student.url");
        String longUrl;
        Map<Long, OrgStudent> stuMap = Maps.newHashMap();
        for (OrgStudent student : students) {
            stuMap.put(student.getId(), student);
        }
        for (Long stuId : studentIds) {
            OrgStudent student = stuMap.get(stuId);
            if (student != null) {
                // 生成token
                String token = getSmsToken(orgId, courseId, userRole, stuUserMap.get(stuId), stuId);
                if (StringUtils.isEmpty(token)) {
                    continue;
                }
                longUrl = String.format(longUrlFormat, token);
                String url = ShortUrlUtil.getShortUrl(longUrl);
                log.debug("send sms: longUrl = {}, url = {}", longUrl, url);

                content = String.format(STUDENT_SMS_TEMPLATE, getStudentName(student), orgName, courseName, url);

                sendResult = saveAndSendSms(orgId, null, userRole, student.getUserId(),
                    StringUtils.isNoneEmpty(student.getShowMobile()) && student.getShowMobile().indexOf("*") < 0
                        ? student.getShowMobile() : student.getMobile(),
                    content);
                if (sendResult == true) {
                    this.orgCourseSmsDao.increaseSendTime(orgId, courseId, userRole, student.getUserId());
                    succeedStudentIds.add(stuId);
                    log.debug("send stu sms mobile = {}, content = {} ", student.getMobile(), content);
                } else {
                    log.warn("send stu sms mobile = {}, content = {}", student.getMobile(), content);
                }
            }
        }

        return succeedStudentIds;
    }

    private static String getStudentName(OrgStudent student) {
        if (student != null) {
            if (StringUtils.isNotEmpty(student.getName())) {
                return student.getName();
            } else if (StringUtils.isNotEmpty(student.getNickName())) {
                return student.getNickName();
            } else {
                return MaskUtil.maskMobile(student.getMobile());
            }
        }
        return "";
    }

    private List<Long> sendTeacherCourseSms(Long orgId, Long courseId, Integer userRole, List<Long> userIds) {
        List<Long> succeedUserIds = Lists.newArrayList();
        boolean sendResult = false;
        String content = null;
        List<Teacher> teachers = this.teacherDao.getByUserIds(userIds);

        log.debug("teachers = {}", teachers);

        Map<Long, OrgTeacher> idUserMap = this.orgTeacherDao.getMapByIds(orgId,userIds);

        log.debug("idUserMap = {}", idUserMap);

        String orgName = this.orgInfoDao.getOrgShortNameByOrgId(orgId.intValue());
        String courseName = this.orgCourseDao.getCourseNameById(courseId);
        String longUrlFormat = PropertiesReader.getProperties("url.properties").getProperty("sms.course.teacher.url");
        String longUrl;
        Map<Long, Teacher> teacherMap = Maps.newHashMap();
        for (Teacher teacher : teachers) {
            teacherMap.put(teacher.getUserId(), teacher);
        }
        for (Long userId : userIds) {
            Teacher teacher = teacherMap.get(userId);
            if (teacher != null) {
                // 生成token
                String token = getSmsToken(orgId, courseId, userRole, userId, null);
                if (StringUtils.isEmpty(token)) {
                    continue;
                }
                // 生成url
                longUrl = String.format(longUrlFormat, token);
                String url = ShortUrlUtil.getShortUrl(longUrl);
                log.debug("send sms: longUrl = {}, url = {}", longUrl, url);
                // 生成内容
                content = String.format(TEACHER_SMS_TEMPLATE,
                    StringUtils.isEmpty(teacher.getRealName()) ? teacher.getNickName() : teacher.getRealName(), orgName,
                    courseName, url);

                OrgTeacher user = idUserMap.get(userId);

                if (user != null) {
                    log.debug("teacher mobile = {}", user.getMobile());
                    sendResult = saveAndSendSms(orgId, null, userRole, userId, user.getMobile(), content);
                }

                if (sendResult == true) {
                    succeedUserIds.add(userId);
                    this.orgCourseSmsDao.increaseSendTime(orgId, courseId, userRole, userId);
                    log.debug("send teacher sms mobile = {}, content = {} ", user != null ? user.getMobile() : null,
                        content);
                } else {
                    log.warn("send teacher sms mobile = {}, content = {} ", user != null ? user.getMobile() : null,
                        content);
                }
            }
        }
        return succeedUserIds;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void viewSms(Long orgId, Long courseId, Long userId, Integer userRole) {
        log.debug("view sms info ,orgId={},courseId={},userId={},userRole={}", orgId, courseId, userId, userRole);
        OrgCourseSms sms = this.orgCourseSmsDao.getOrgCourseSms(orgId, courseId, userRole, userId);
        log.debug("sms={}", sms);
        if (sms != null) {
            sms.setViewStatus(1);
            this.orgCourseSmsDao.update(sms, "viewStatus");
        }
    }

}
