package com.baijia.tianxiao.sal.course.service.impl;

import java.io.OutputStream;
import java.lang.reflect.InvocationTargetException;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import javax.annotation.Resource;

import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import com.baijia.tianxiao.common.service.ImportDataProcessService;
import com.baijia.tianxiao.constants.DataProcType;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.course.constant.CourseErrorCode;
import com.baijia.tianxiao.sal.course.dto.OrgCourseInfoDto;
import com.baijia.tianxiao.sal.course.service.OrgCourseService;
import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;

import lombok.Data;
import lombok.extern.slf4j.Slf4j;

/**
 * 课程导入接口实现类
 * 
 * @title ImportOrgCourseProcessServiceImpl
 * @desc TODO
 * @author shizuwei
 * @date 2016年3月17日
 * @version 1.0
 */
@Slf4j
@Service(value = "ImportOrgCourseProcessService")
public class ImportOrgCourseProcessServiceImpl implements ImportDataProcessService {

    @Resource
    private OrgCourseService orgCourseService;

    private final static SingleSaveErrorResult IMPORT_SUCCESS = SingleSaveErrorResult.createSuccessResult("导入成功");
    private final static SingleSaveErrorResult COURSE_EXSIST = new SingleSaveErrorResult(true, "课程重复");

    @Data
    public static class CourseFormat {

        private String fieldName;
        private String formatRegex;
        private String formatMsg;

        public CourseFormat(String fieldName, String formatRegex, String formatMsg) {
            this.fieldName = fieldName;
            this.formatRegex = formatRegex;
            this.formatMsg = formatMsg;
        }

        String getFmtErrorMsg() {
            return formatMsg;
        }
//        String getFmtErrorMsg() {
//            return fieldName + ":" + formatMsg;
//        }
    }

    private final static Map<String, CourseFormat> colMap = Maps.newHashMap();

    static {
        colMap.put("课程名称", new CourseFormat("courseName", ".{0,20}", "【课程名称】必须是1-20个文字"));
        colMap.put("上课次数", new CourseFormat("freq", "^[0-9]*$", "【上课次数】需为大于0的整数"));
        colMap.put("班级人数", new CourseFormat("maxStudent", "^[0-9]*$", "【班级人数】需必须为大于0的整数"));
        colMap.put("价格", new CourseFormat("coursePrice", "^(?!0(\\d|\\.0+$|$))\\d+(\\.\\d{1,2})?$", "【价格】需为大于0的数字"));
//        colMap.put("价格", new CourseFormat("coursePrice", "^[0-9]+\\.{0,1}[0-9]{0,2}$", "【价格】需为大于0的数字"));
    }

    @Override
    public boolean validateHeader(List<String> headers) {
        for (String headName : headers) {
            if (colMap.get(headName.trim()) == null) {
                return false;
            }
        }
        return true;
    }

    public static void main(String[] args) throws IllegalAccessException, InvocationTargetException {
        OrgCourseInfoDto dto = new OrgCourseInfoDto();
        Object value = "9999999.99";
        BeanUtils.setProperty(dto, "coursePrice", value);
        log.info("dto={}", dto);

        log.info("={}", Pattern.matches("^[0-9]{1,9}\\.{0,1}[0-9]{0,2}$", "11.444"));
        log.info("={}", Pattern.matches(".{1,20}", "123456789012345678900"));
        log.info("={}", Pattern.matches("^[0-9]+$", "11.0"));
        BeanUtils.setProperty(dto, "courseNumber", null);
        System.out.println(dto);
        System.out.println(Pattern.matches("^[1-9]{1,9}\\.{0,1}[0-9]{0,2}$", "11.013"));
        System.out.println(Pattern.matches(".{1,20}", "123456789012345678900"));
        System.out.println(Pattern.matches("^[0-9]+$", "11"));
        System.out.println(Pattern.matches("^(?!0(\\d|\\.0+$|$))\\d+(\\.\\d{1,2})?$", "0.01"));
    }

    private void buildOrgCourseInfoDto(OrgCourseInfoDto dto, Object[] lineData, List<String> headers) {
        Preconditions.checkArgument(lineData.length == headers.size(), "数据数目和表头不匹配");
        Preconditions.checkArgument(validateHeader(headers), "表头格式错误:" + headers.toString());
        for (int i = 0; i < lineData.length; i++) {
            CourseFormat fmt = colMap.get(headers.get(i));
            String fieldName = fmt.getFieldName();
            Object value = lineData[i];
            if (fieldName != null) {
                try {
                    if (value != null
                        && !(value instanceof String && (StringUtils.isBlank((String) value) || !Pattern.matches(
                            fmt.getFormatRegex(), value.toString())))) {
                        BeanUtils.setProperty(dto, fieldName, value);
                    } else {
                        throw new BussinessException(CommonErrorCode.PARAM_ERROR, fmt.getFmtErrorMsg());
                    }
                } catch (IllegalAccessException | InvocationTargetException e) {
                    log.warn("set OrgCourseInfoDto.{} = {} failed!", fieldName, value);
                    throw new BussinessException(CourseErrorCode.COURSE_IMPORT_COL_FORMAT_ERROR);
                }
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class, propagation = Propagation.REQUIRES_NEW)
    public SingleSaveErrorResult saveSingleData(Long orgId, Long cascadeId, List<String> headers, Object[] lineData,
        boolean updateRepeat) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "illegal orgId.");
        Preconditions.checkArgument(lineData != null && lineData.length > 0, "lineData 为空");
        // 保存一行数据
        try {
            OrgCourseInfoDto course = new OrgCourseInfoDto();
            buildOrgCourseInfoDto(course, lineData, headers);
            orgCourseService.saveOrUpdateCourse(orgId, course, false, updateRepeat);
        } catch (BussinessException ex) {
            if (ex.getErrorCode().getSubsystemErrorCode() == CourseErrorCode.COURSE_EXSIST.getSubsystemErrorCode()) {
                return COURSE_EXSIST;
            } else {
                log.error("save course error:{}", ex);
                throw ex;
            }
        } catch (Throwable ex) {
            log.error("save course error:{}", ex);
            throw ex;
        }

        return IMPORT_SUCCESS;
    }

    @Override
    public BatchSaveResult batchImportDatas(Long orgId, Long cascadeId, List<String> headers, List<Object[]> datas, boolean updateRepeat) {
        BatchSaveResult batchSaveResult = new BatchSaveResult();
        for (int i = 0; i < datas.size(); i++) {
            Object[] data = datas.get(i);
            try {
                SingleSaveErrorResult singleSaveErrorResult = saveSingleData(orgId, cascadeId, headers, data, updateRepeat);
                if (singleSaveErrorResult.isRepeat()) {
                    batchSaveResult.getRepeatIndexs().add(i);
                }
            } catch (Throwable ex) {
                batchSaveResult.getFailIndexs().add(i);
            }
        }
        return batchSaveResult;

    }

    @Override
    public DataProcType getProcessType() {
        return DataProcType.COURSE;
    }

    @Override
    public void afterComplete(int successCount) {
        log.info("[Upload] upload complete!");
    }

	@Override
	public int[] getUniqueIndex() {
		// TODO Auto-generated method stub
		return null;
	}


	@Override
	public void downloadValidateResult(OutputStream os, Long orgId,
			String taskId, Collection<SingleSaveErrorResult> dataMap) {
		// TODO Auto-generated method stub
		
	}

	@Override
	public void downloadImportResult(OutputStream os, Long orgId,
			String taskId, Collection<SingleSaveErrorResult> data) {
		// TODO Auto-generated method stub
		
	}

	@Override
	public void downloadImportTemplate(OutputStream os) {
		// TODO Auto-generated method stub
		
	}

	@Override
	public SingleSaveErrorResult validate(Long orgId, Long cascadeId,
			List<String> headers, SingleSaveErrorResult result,
			List<SingleSaveErrorResult> dataMap, boolean updateRepeat) {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public String validateResult(List<SingleSaveErrorResult> list) {
		// TODO Auto-generated method stub
		return null;
	}

    @Override
    public boolean validateHeader(List<String> headers, Long orgId) {
        
        // TODO Auto-generated method stub
        return false;
            
    }
}
