/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */

package com.baijia.tianxiao.sal.course.service.impl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.env.Environment;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import com.baijia.tianxiao.consants.UserRole;
import com.baijia.tianxiao.constants.CourseType;
import com.baijia.tianxiao.dal.course.dao.OrgCourseGroupRelateDao;
import com.baijia.tianxiao.dal.course.dao.OrgCoursePhotoDao;
import com.baijia.tianxiao.dal.course.po.OrgCoursePhoto;
import com.baijia.tianxiao.dal.org.constant.DeleteStatus;
import com.baijia.tianxiao.dal.org.dao.CoursePurchaseDao;
import com.baijia.tianxiao.dal.org.dao.OrgAccountDao;
import com.baijia.tianxiao.dal.org.dao.OrgClassLessonDao;
import com.baijia.tianxiao.dal.org.dao.OrgClassRoomDao;
import com.baijia.tianxiao.dal.org.dao.OrgCourseDao;
import com.baijia.tianxiao.dal.org.dao.OrgCourseDao.OrgCourseListParam;
import com.baijia.tianxiao.dal.org.dao.OrgCourseRoomDao;
import com.baijia.tianxiao.dal.org.dao.OrgCourseSeatDao;
import com.baijia.tianxiao.dal.org.dao.OrgCourseSmsDao;
import com.baijia.tianxiao.dal.org.dao.OrgCourseTeacherDao;
import com.baijia.tianxiao.dal.org.dao.OrgLessonSignDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentCourseDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentLessonDao;
import com.baijia.tianxiao.dal.org.dao.OrgTeacherDao;
import com.baijia.tianxiao.dal.org.po.OrgAccount;
import com.baijia.tianxiao.dal.org.po.OrgCourse;
import com.baijia.tianxiao.dal.org.po.OrgCourseRoom;
import com.baijia.tianxiao.dal.org.po.OrgCourseSms;
import com.baijia.tianxiao.dal.org.po.OrgStudent;
import com.baijia.tianxiao.dal.org.po.OrgStudentCourse;
import com.baijia.tianxiao.dal.org.po.OrgTeacher;
import com.baijia.tianxiao.dal.storage.dao.StorageDao;
import com.baijia.tianxiao.dal.storage.po.Storage;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.course.constant.ArrangeStatus;
import com.baijia.tianxiao.sal.course.constant.CourseErrorCode;
import com.baijia.tianxiao.sal.course.constant.RestConfig;
import com.baijia.tianxiao.sal.course.constant.ViewStatus;
import com.baijia.tianxiao.sal.course.dto.request.WeikeRequestDto;
import com.baijia.tianxiao.sal.course.dto.response.CourseListReponseDto;
import com.baijia.tianxiao.sal.course.service.OrgCourseListService;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.util.BaseUtils;
import com.baijia.tianxiao.util.collection.CollectorUtil;
import com.baijia.tianxiao.util.properties.PropertiesReader;
import com.baijia.tianxiao.util.storage.StorageUtil;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

/**
 * @author bjhl
 * @version 1.0
 * @title OrgCourseListServiceImpl
 * @desc TODO
 * @date 2015年12月25日
 */
@Slf4j
@Service
public class OrgCourseListServiceImpl implements OrgCourseListService {

    @Autowired
    private OrgCourseDao orgCourseDao;
    @Autowired
    private OrgClassRoomDao orgClassRoomDao;

    @Autowired
    private OrgCourseTeacherDao orgCourseTeacherDao;

    @Resource
    private OrgTeacherDao orgTeacherDao;

    @Resource
    private CoursePurchaseDao coursePurchaseDao;

    @Resource
    private OrgAccountDao orgAccountDao;

    @Resource
    private OrgStudentCourseDao orgStudentCourseDao;

    @Resource
    private OrgStudentDao orgStudentDao;

    @Resource
    private OrgCourseSeatDao orgCourseSeatDao;

    @Resource
    private OrgClassLessonDao orgClassLessonDao;

    @Resource
    private OrgCourseRoomDao orgCourseRoomDao;

    @Resource
    private OrgCourseSmsDao orgCourseSmsDao;

    @Resource
    private StorageDao storageDao;

    @Resource
    private Environment environment;

    @Resource
    private OrgStudentLessonDao orgStudentLessonDao;

    @Resource
    private OrgLessonSignDao orgLessonSignDao;

    @Resource
    private OrgCourseGroupRelateDao orgCourseGroupRelateDao;

    @Resource
    private OrgCoursePhotoDao orgCoursePhotoDao;

    @Override
    public List<CourseListReponseDto> getCourseList(List<OrgCourse> courses, Long orgId, boolean needLesson,
        boolean needSms, boolean needTeacher, Boolean needStudent, boolean needRoom, boolean needCover,
        boolean needUrl) {
        CourseListParam param = new CourseListParam();
        param.setCourses(courses).setNeedLesson(needLesson).setNeedSms(needSms).setNeedTeacher(needTeacher)
            .setNeedStudent(needStudent).setNeedRoom(needRoom).setNeedCover(needCover).setNeedUrl(needUrl)
            .setOrgId(orgId);
        return getCourseList(param);
    }

    @Override
    public List<CourseListReponseDto> getWeikeList(Long orgId, WeikeRequestDto requestDto) {
        log.info("{},{}", orgId, requestDto);
        String key = requestDto.getSearchKey();
        List<Long> teacherIds = requestDto.getTeacherIdList();
        List<Integer> groupIds = requestDto.getGroupIdList();
        List<Integer> exGroupIds = requestDto.getExGroupIdList();
        Long orgNumber = getOrgNumber(orgId);
        List<Long> courseIdsOfTeachers = null;
        if (CollectionUtils.isNotEmpty(teacherIds)) {
            courseIdsOfTeachers = this.orgCourseTeacherDao.getCourseIdsByteacherIds(teacherIds);
        }
        Collection<Long> courseOfGroups = null;
        if (CollectionUtils.isNotEmpty(groupIds)) {
            courseOfGroups =
                this.orgCourseGroupRelateDao.getCourseIdsByGroups(groupIds, CourseType.ORG_COURSE.getCode());
        }

        Collection<Long> exCourseOfGroups = null;
        if (CollectionUtils.isNotEmpty(exGroupIds)) {
            exCourseOfGroups =
                this.orgCourseGroupRelateDao.getCourseIdsByGroups(exGroupIds, CourseType.ORG_COURSE.getCode());
        }

        Collection<Long> filteredCourses = null;
        if (courseIdsOfTeachers != null) {
            filteredCourses = courseIdsOfTeachers;
            if (courseOfGroups != null) {
                filteredCourses.retainAll(courseOfGroups);
            }

        } else {
            filteredCourses = courseOfGroups;
        }

        // 过滤：老师，价格区间，课程分类，key
        List<Long> courseIds = this.orgCourseDao.getCourseList(orgNumber, filteredCourses, requestDto.getLPrice(),
            requestDto.getHPrice(), null, key, requestDto.getStatus());
        if (CollectionUtils.isNotEmpty(courseIds) && CollectionUtils.isNotEmpty(exCourseOfGroups)) {
            courseIds.removeAll(exCourseOfGroups);
        }

        Map<String, String> orderConditionMap = requestDto.getOrderConditionMap();
        OrgCourseListParam param = new OrgCourseListParam();
        param.setPage(requestDto);
        param.setStatus(requestDto.getStatus());
        if (MapUtils.isNotEmpty(orderConditionMap)) {
            String studentCountOrder = orderConditionMap.get("studentCount");
            if (StringUtils.isNotBlank(studentCountOrder)) {
                if ("ASC".equals(studentCountOrder)) {
                    courseIds = this.orgStudentCourseDao.getOrgCourseIds(orgId, courseIds, true, requestDto);
                } else if ("DESC".equals(studentCountOrder)) {
                    courseIds = this.orgStudentCourseDao.getOrgCourseIds(orgId, courseIds, false, requestDto);
                }
                param.setPage(null);
            } else {
                if (StringUtils.isNotBlank(orderConditionMap.get("beginTime"))) {
                    param.setOrderType(orderConditionMap.get("beginTime"));
                    param.setOrderField("beginTime");
                }
                if (StringUtils.isNotBlank(orderConditionMap.get("createTime"))) {
                    param.setOrderType(orderConditionMap.get("createTime"));
                    param.setOrderField("createTime");
                }
            }
        } else {
            param.setOrderType("DESC");
            param.setOrderField("createTime");
        }

        List<OrgCourse> orgCourses = this.orgCourseDao.getCourseList(orgNumber, courseIds, param);
        // Map<Long, Integer> studentCntMap = this.orgStudentCourseDao.getStudentCount(orgId, courseIds);

        CourseListParam courseListParam = new CourseListParam();
        courseListParam.setCourses(orgCourses).setNeedCover(true).setNeedUrl(true).setOrgId(orgId);
        return this.getCourseList(courseListParam);
    }

    /**
     * 批量设置颜色
     *
     * @param courseIds
     * @param courses 课节
     */
    private void batchUpdateColor(Set<Long> courseIds, List<OrgCourse> courses) {
        List<OrgCourse> courseList = Lists.newArrayList();
        String colorsStr = "";
        try {
            colorsStr = PropertiesReader.getValueNoCache("erp.properties", "course.colors");
        } catch (Exception e) {

        }

        String colors[] = null;
        if (StringUtils.isNotBlank(colorsStr)) {
            colors = colorsStr.split(",");
        }
        Random rd = new Random();
        for (OrgCourse orgCourse : courses) {
            if (StringUtils.isBlank(orgCourse.getColor())) {
                if (colors != null) {
                    orgCourse.setColor(colors[rd.nextInt(colors.length - 1)]);
                }
                courseList.add(orgCourse);
            }
            courseIds.add(orgCourse.getId());
        }
        if (colors != null) {
            this.orgCourseDao.changeCourseColor(courseList);// 批量修改颜色
        }
    }

    /**
     * 获取短信查看数量
     *
     * @param studentSms
     * @param teacherSms
     * @param studentView
     * @param teacherView
     * @param courseSms
     */
    private void getSmsCount(Map<Long, Integer> studentSms, Map<Long, Integer> teacherSms,
        Map<Long, Integer> studentView, Map<Long, Integer> teacherView, List<OrgCourseSms> courseSms,
        Map<Long, List<Long>> courseStudentMap, Map<Long, List<Long>> courseTeacherMap) {
        for (OrgCourseSms orgCourseSms : courseSms) {
            if (!studentSms.containsKey(orgCourseSms.getCourseId())) {
                studentSms.put(orgCourseSms.getCourseId(), 0);
            }
            if (!teacherSms.containsKey(orgCourseSms.getCourseId())) {
                teacherSms.put(orgCourseSms.getCourseId(), 0);
            }
            if (!studentView.containsKey(orgCourseSms.getCourseId())) {
                studentView.put(orgCourseSms.getCourseId(), 0);
            }
            if (!teacherView.containsKey(orgCourseSms.getCourseId())) {
                teacherView.put(orgCourseSms.getCourseId(), 0);
            }
            if (orgCourseSms.getUserRole().intValue() == UserRole.STUDENT.getRole()) {// 学生
                List<Long> studentIds = courseStudentMap.get(orgCourseSms.getCourseId());
                if (studentIds != null && studentIds.contains(orgCourseSms.getUserId())) {
                    if (orgCourseSms.getSend() > 0) {// 已发送短信
                        studentSms.put(orgCourseSms.getCourseId(), studentSms.get(orgCourseSms.getCourseId()) + 1);
                    }
                    if (orgCourseSms.getViewStatus().intValue() == ViewStatus.VIEW.getCode()) {// 已查看
                        studentView.put(orgCourseSms.getCourseId(), studentView.get(orgCourseSms.getCourseId()) + 1);
                    }
                }
            } else {// 老师
                List<Long> teacherIds = courseTeacherMap.get(orgCourseSms.getCourseId());
                if (teacherIds != null && teacherIds.contains(orgCourseSms.getUserId())) {
                    if (orgCourseSms.getSend() > 0) {// 已发送短信
                        teacherSms.put(orgCourseSms.getCourseId(), teacherSms.get(orgCourseSms.getCourseId()) + 1);
                    }
                    if (orgCourseSms.getViewStatus().intValue() == ViewStatus.VIEW.getCode()) {// 已查看
                        teacherView.put(orgCourseSms.getCourseId(), teacherView.get(orgCourseSms.getCourseId()) + 1);
                    }
                }
            }
        }
    }

    /**
     * 获取课程封面urlmap
     *
     * @param courses
     */
    public Map<Integer, String> getCoverMap(List<OrgCourse> courses) {
        Collection<Integer> coverList = CollectorUtil.collect(courses, new Function<OrgCourse, Integer>() {
            @Override
            public Integer apply(OrgCourse arg0) {
                return arg0.getCover();
            }
        });
        List<Storage> storages = this.storageDao.getByIds(coverList);
        return CollectorUtil.collectMap(storages, new Function<Storage, Integer>() {
            @Override
            public Integer apply(Storage arg0) {
                return arg0.getId().intValue();
            }
        }, new Function<Storage, String>() {
            @Override
            public String apply(Storage arg0) {
                if (StringUtils.isNotBlank(arg0.getAttach_url())) {
                    return arg0.getAttach_url();
                }
                return StorageUtil.constructUrl(arg0.getFid(), arg0.getMimetype(), arg0.getSn());
            }
        });
    }

    public Map<Long, OrgCoursePhoto> getPhoteMap(List<Long> courseIds) {
        Map<String, Object> condition = new HashMap<String, Object>();
        condition.put("orgCourseId", courseIds);
        // orgCoursePhotoDao.countByCondition(condition);
        List<OrgCoursePhoto> list = orgCoursePhotoDao.queryByCourseIds(courseIds);
        Map<Long, OrgCoursePhoto> map = new HashMap<Long, OrgCoursePhoto>();
        for (OrgCoursePhoto coursePhoto : list) {
            map.put(coursePhoto.getOrgCourseId(), coursePhoto);
        }
        return map;
    }

    /**
     * 构造返回数据
     *
     * @param po
     * @param dto
     */
    private void buildCourseListResponseDto(OrgCourse po, CourseListReponseDto dto) {
        dto.setOrgCourseId(po.getId());
        dto.setOrgCourseNumber(po.getNumber());
        dto.setColor(po.getColor());
        dto.setCourseName(po.getName());
        dto.setMaxStudent(po.getMaxStudent());
        dto.setFreq(po.getFreq());
        dto.setStartTime(po.getBeginTime());
        dto.setEndTime(po.getEndTime());
        dto.setPrice(po.getPrice());
        dto.setStatus(po.getStatus());
        String intro = BaseUtils.blobToString(po.getIntroduction(), "UTF-8");
        dto.setIntroduction(intro);
    }

    /**
     * 构造 课程排课课节数，课节完成数
     *
     * @param po
     * @param dto
     * @param arrangeMap
     * @param finishMap
     */
    private void buildLessonCount(OrgCourse po, CourseListReponseDto dto, Map<Long, Integer> arrangeMap,
        Map<Long, Integer> finishMap) {
        if (arrangeMap.containsKey(po.getId())) {
            dto.setLessonCount(arrangeMap.get(po.getId()));
        } else {
            dto.setLessonCount(0);
        }
        if (finishMap.containsKey(po.getId())) {
            dto.setLessonFinish(finishMap.get(po.getId()));
        } else {
            dto.setLessonFinish(0);
        }
        if (dto.getLessonCount() != null && dto.getLessonFinish() != null
            && dto.getLessonCount() > dto.getLessonFinish() && dto.getLessonFinish() > 0) {
            dto.setArrageStatus(ArrangeStatus.BEGIN.getCode());
        } else if (dto.getLessonCount() != null && dto.getLessonCount() > 0) {
            dto.setArrageStatus(ArrangeStatus.ARRANGEED.getCode());
        } else {
            dto.setArrageStatus(ArrangeStatus.UNARRANGE.getCode());
        }

    }

    /**
     * 设置课程老师数量
     *
     * @param po
     * @param dto
     * @param teacherMap
     */
    private void buildTeacher(OrgCourse po, CourseListReponseDto dto, Map<Long, Integer> teacherMap) {
        Integer count = teacherMap.get(po.getId());
        if (count != null) {
            dto.setTeacherCount(count);
        } else {
            dto.setTeacherCount(0);
        }
    }

    /**
     * 设置课程学生数量
     *
     * @param po
     * @param dto
     * @param studentMap
     */
    private void buildStudent(OrgCourse po, CourseListReponseDto dto, Map<Long, Integer> studentMap) {
        Integer count = studentMap.get(po.getId());
        if (count != null) {
            dto.setStudentCount(count);
        } else {
            dto.setStudentCount(0);
        }
    }

    /**
     * 设置课程教室数量
     *
     * @param po
     * @param dto
     * @param roomMap
     */
    private void buildRoom(OrgCourse po, CourseListReponseDto dto, Map<Long, Integer> roomMap) {
        Integer count = roomMap.get(po.getId());
        if (count != null) {
            dto.setRoomCount(count);
        } else {
            dto.setRoomCount(0);
        }
    }

    /**
     * 构造课表短信发送及查看相关数据
     *
     * @param po
     * @param dto
     * @param studentSms
     * @param teacherSms
     * @param studentView
     * @param teacherView
     */
    private void buildSmsCount(OrgCourse po, CourseListReponseDto dto, Map<Long, Integer> studentSms,
        Map<Long, Integer> teacherSms, Map<Long, Integer> studentView, Map<Long, Integer> teacherView) {
        int studentSmsCount = 0;
        int teacherSmsCount = 0;
        int studentViewCount = 0;
        int teacherViewCount = 0;
        if (studentSms.get(po.getId()) != null) {
            studentSmsCount = studentSms.get(po.getId());
        }
        if (teacherSms.get(po.getId()) != null) {
            teacherSmsCount = teacherSms.get(po.getId());
        }
        if (studentView.get(po.getId()) != null) {
            studentViewCount = studentView.get(po.getId());
        }
        if (teacherView.get(po.getId()) != null) {
            teacherViewCount = teacherView.get(po.getId());
        }
        log.debug("studentSmsCount={}, teacherSmsCount={}", studentSmsCount, teacherSmsCount);
        dto.setSmsStudent(dto.getStudentCount() - studentSmsCount);
        dto.setViewStudent(dto.getStudentCount() - studentViewCount);
        dto.setSmsTeacher(dto.getTeacherCount() - teacherSmsCount);
        dto.setViewTeacher(dto.getTeacherCount() - teacherViewCount);
    }

    private Long getOrgNumber(Long orgId) {
        OrgAccount orgAccount = this.orgAccountDao.getAccountById(orgId.intValue(), "number");
        if (orgAccount == null) {
            log.error("orgId={} not exist in orgAccount.", orgId);
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "机构帐号错误");
        }
        return orgAccount.getNumber().longValue();
    }

    @Override
    @Transactional(readOnly = true)
    public List<CourseListReponseDto> getCourseList(@NonNull Long orgId, @NonNull Long studentId, PageDto pageDto) {
        Long userId = this.orgStudentDao.getUserId(studentId);
        Preconditions.checkArgument(userId != null && userId > 0, "student not exits!");
        List<OrgStudentCourse> stuCourses = this.orgStudentCourseDao.getOrgCourseIds(orgId, userId, pageDto);
        if (CollectionUtils.isEmpty(stuCourses)) {
            return Collections.emptyList();
        }

        Map<Long, OrgStudentCourse> idCourseMap = Maps.newHashMap();
        for (OrgStudentCourse course : stuCourses) {
            idCourseMap.put(course.getCourseId(), course);
        }
        OrgAccount orgAccount = this.orgAccountDao.getAccountById(orgId.intValue(), "number");
        if (orgAccount == null) {
            log.error("orgId={} not exist in orgAccount.", orgId);
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "机构帐号错误");
        }
        Long orgNumber = orgAccount.getNumber().longValue();
        List<OrgCourse> courses = this.orgCourseDao.getCourseList(idCourseMap.keySet(), orgNumber, null);
        log.debug("OrgStudentCourses = {}, courses = {}, page = {}", stuCourses, courses, pageDto);
        if (CollectionUtils.isEmpty(courses)) {
            return Collections.emptyList();
        }
        List<CourseListReponseDto> list =
            this.getCourseList(courses, orgId, true, false, false, false, false, true, false);
        if (CollectionUtils.isNotEmpty(list)) {
            Map<Long, Integer> lessonCountMap =
                orgStudentLessonDao.getStudentLessonCountOfCourses(orgId, userId, idCourseMap.keySet());
            log.debug("list={},lessonCountMap={}", list, lessonCountMap);
            for (CourseListReponseDto dto : list) {
                Integer lessonCount = lessonCountMap.get(dto.getOrgCourseId());
                if (lessonCount == null) {
                    lessonCount = 0;
                }
                dto.setLessonCount(lessonCount);
                dto.setStatus(idCourseMap.get(dto.getOrgCourseId()).getStatus());
            }
        }
        return list;
    }

    @Override
    @Transactional(readOnly = true)
    public CourseListReponseDto getCourseDetail(Long orgId, Long orgCourseId) {
        Preconditions.checkArgument(orgId != null, "orgId is null!");
        Preconditions.checkArgument(orgCourseId != null, "orgCourseId is null!");
        OrgCourse course = this.orgCourseDao.getByCourseId(orgCourseId);
        OrgAccount account = this.orgAccountDao.getAccountById(orgId.intValue());
        if (course == null || account == null || course.getOrgNumber().longValue() != account.getNumber().longValue()) {
            log.error("orgId or courseId not exist error,orgId={},courseId={}", orgId, orgCourseId);
            throw new BussinessException(CourseErrorCode.COURSE_NOT_EXIST);
        }
        Integer lessonCount = this.orgClassLessonDao.getLessonCount(orgId, orgCourseId, DeleteStatus.NORMAL.getValue());
        Integer finishCount =
            this.orgClassLessonDao.getFinishLessonCount(orgId, orgCourseId, DeleteStatus.NORMAL.getValue());

        // 教室数量
        List<Long> roomIds = this.orgCourseRoomDao.getCourseRoomIds(orgId, orgCourseId);
        // 老师数量
        List<Long> teacherIds = this.orgCourseTeacherDao.getTeacherIdsByCourseId(orgCourseId);

        // 学生数量
        List<Long> studentIds = this.orgStudentCourseDao.getStudents(orgId, orgCourseId, 0);

        // 学生数量和老师数量都要过滤机构需要过滤，删除的学员，解约的老师，等等
        List<OrgTeacher> teacherList = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(teacherIds)) {
            teacherList = orgTeacherDao.getTeacherByUserIdsAndOrgId(orgId, teacherIds, 1, "userId");
        }
        List<OrgStudent> studentlist = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(studentIds)) {
            studentlist = orgStudentDao.getStudentByUserIds(orgId, studentIds, "userId");
        }

        CourseListReponseDto dto = new CourseListReponseDto();
        dto.setCourseName(course.getName());
        dto.setOrgCourseId(orgCourseId);
        dto.setOrgCourseNumber(course.getNumber());
        dto.setFreq(course.getFreq());
        String courseUrl = this.environment.getProperty("org_class_course.url");
        dto.setCourseUrl(course.getNumber() == null ? "" : courseUrl + course.getNumber());
        dto.setLessonCount(lessonCount);
        dto.setLessonFinish(finishCount);
        dto.setRoomCount(roomIds.size());
        dto.setTeacherCount(teacherList.size());
        dto.setStudentCount(studentlist.size());
        dto.setStatus(course.getStatus());
        dto.setMaxStudent(course.getMaxStudent());
        dto.setColor(course.getColor());
        dto.setCreateTime(course.getCreateTime());
        dto.setCascadeId(course.getCascadeId());
        return dto;
    }

    @Override
    @Transactional(noRollbackFor = Exception.class, propagation = Propagation.REQUIRES_NEW)
    public List<CourseListReponseDto> getCourseList(CourseListParam param) {
        List<OrgCourse> courses = param.getCourses();
        Long studentId = param.getStudentId();
        Long teacherId = param.getTeacherId();
        Long orgId = param.getOrgId();
        boolean needLesson = param.isNeedLesson();
        boolean needSms = param.isNeedSms();
        boolean needTeacher = param.isNeedTeacher();
        boolean needStudent = param.isNeedStudent();
        boolean needRoom = param.isNeedRoom();
        boolean needCover = param.isNeedCover();
        boolean needUrl = param.isNeedUrl();
        log.debug("needLesson={},needSms={},needTeacher={},needStudent={},needRoom={},needCover={},needUrl={}",
            needLesson, needSms, needTeacher, needStudent, needRoom, needCover, needUrl);
        List<CourseListReponseDto> response = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(courses)) {
            Set<Long> courseIds = Sets.newHashSet();
            batchUpdateColor(courseIds, courses);

            log.debug("courseIds={}", courseIds);
            Map<Integer, String> coverMap = null;
            Map<Long, OrgCoursePhoto> photoMap = null;
            if (needCover) { // 需要封面
                coverMap = this.getCoverMap(courses); // 获取封面url
                photoMap = getPhoteMap(new ArrayList<Long>(courseIds));
                log.debug("orgCoursePhotoMap={}", photoMap);
            }

            Map<Long, Integer> arrangeMap = null;// 排课数
            Map<Long, Integer> finishMap = null; // 课节完成数

            if (needLesson) {// 需要课节统计
                arrangeMap = this.orgClassLessonDao.getCourseLessonCount(orgId, courseIds, studentId, teacherId, null);
                finishMap =
                    this.orgClassLessonDao.getCourseLessonCount(orgId, courseIds, studentId, teacherId, new Date());
                // arrangeMap = this.orgClassLessonDao.getLessonCount(orgId, courseIds, DeleteStatus.NORMAL.getValue());
                // finishMap =
                // this.orgClassLessonDao.getFinishLessonCount(orgId, courseIds, DeleteStatus.NORMAL.getValue());
            }

            Map<Long, List<Long>> courseTeacherIdsMap = null;
            if (needTeacher) {// 需要统计老师
                courseTeacherIdsMap = orgCourseTeacherDao.getTeacherMap(courseIds);
                List<Long> allTeacherUserIds = orgTeacherDao.getTeacherIds(orgId, 1);
                for (Map.Entry<Long, List<Long>> entry : courseTeacherIdsMap.entrySet()) {
                    List<Long> tempIds = entry.getValue();
                    tempIds.retainAll(allTeacherUserIds);
                }
            }

            Map<Long, Integer> studentMap = null;
            Map<Long, List<Long>> studentIdMap = null;
            if (needStudent) {// 需要统计学生
                studentIdMap = this.orgStudentCourseDao.getStudentIdMapBycourseIds(courseIds, orgId);
                studentMap = this.getCourseStudentMap(studentIdMap, orgId);
                log.debug("studentMap = {}", studentMap);
            }
            Map<Long, Integer> roomMap = null;
            Map<Long, List<OrgCourseRoom>> mapCourseRoom = null;
            Map<Long, String> roomNameMap = null;
            if (needRoom) {// 需要统计教室
                roomMap = this.orgCourseRoomDao.getCourseRoomCount(orgId, courseIds);
                mapCourseRoom = this.orgCourseRoomDao.getCourseRoomMap(orgId, courseIds);
                Set<Long> roomIds = Sets.newHashSet();
                for (Long k : mapCourseRoom.keySet()) {
                    List<OrgCourseRoom> rooms = mapCourseRoom.get(k);
                    for (OrgCourseRoom room : rooms) {
                        roomIds.add(room.getRoomId());
                    }
                }
                roomNameMap = this.orgClassRoomDao.getRoomName(orgId, roomIds);
            }

            Map<Long, Integer> studentSms = Maps.newHashMap(); // 学生短信已发送数
            Map<Long, Integer> teacherSms = Maps.newHashMap(); // 老师短信发送数
            Map<Long, Integer> studentView = Maps.newHashMap(); // 学生短信查看数
            Map<Long, Integer> teacherView = Maps.newHashMap(); // 老师短信查看数
            if (needSms) {// 需要统计短信发送
                if (studentIdMap == null) {
                    studentIdMap = this.orgStudentCourseDao.getStudentIdMapBycourseIds(courseIds, orgId);

                }
                Map<Long, List<Long>> courseTeacherMap = this.orgCourseTeacherDao.getTeacherMap(courseIds);
                List<OrgCourseSms> courseSms = this.orgCourseSmsDao.getOrgCourseSmsList(orgId, courseIds, null);
                getSmsCount(studentSms, teacherSms, studentView, teacherView, courseSms, studentIdMap,
                    courseTeacherMap);
                log.debug("student sms count ={},teacher sms count ={}", studentSms, teacherSms);
            }

            String url = environment.getProperty(RestConfig.org_class_course_url);
            Map<Long, List<String>> courseTeacherMap = this.orgCourseTeacherDao.getAllTeacherNames(courseIds);
            Map<Long, List<OrgStudentCourse>> orgStudentMap =
                this.orgStudentCourseDao.getStudentMapBycourseIds(courseIds, 0);
            for (OrgCourse course : courses) {
                CourseListReponseDto responseDto = new CourseListReponseDto();
                buildCourseListResponseDto(course, responseDto);
                if (needCover) {// 需要封面
                    responseDto.setCoverUrl(coverMap.get(course.getCover()));
                   // OrgCoursePhoto orgCoursePhoto = photoMap.get(course.getId());
                    // 这里暂时不需要 OrgCoursePhoto 了
                }
                if (needUrl) {// 需要url
                    responseDto.setCourseUrl(url + course.getNumber()); // 课程链接
                }
                if (needLesson) {// 需要统计课节
                    buildLessonCount(course, responseDto, arrangeMap, finishMap);
                }
                if (needTeacher) {// 需要统计老师
                    List<String> teacherList = courseTeacherMap.get(course.getId());
                    responseDto.setTeacherNames(StringUtils.join(teacherList, ","));
                    List<Long> teacherIdList = courseTeacherIdsMap.get(course.getId());
                    if (teacherIdList != null) {
                        responseDto.setTeacherCount(teacherIdList.size());
                    } else {
                        responseDto.setTeacherCount(0);
                    }
                }
                if (needStudent) {// 需要统计学生
                    List<OrgStudentCourse> courseList = orgStudentMap.get(course.getId());
                    if (courseList == null) {
                        courseList = Lists.newArrayList();
                    }
                    List<String> students = Lists.newArrayList();
                    for (OrgStudentCourse stuCourse : courseList) {
                        students.add(stuCourse.getStudentName());
                    }
                    responseDto.setStudentNames(StringUtils.join(students, ","));
                    List<Long> studentIdList = studentIdMap.get(course.getId());
                    if (studentIdList != null) {
                        responseDto.setStudentCount(studentIdList.size());
                    }else{
                        responseDto.setStudentCount(0);
                    }
                }
                if (needRoom) {// 需要统计教室
                    buildRoom(course, responseDto, roomMap);
                    List<OrgCourseRoom> roomsOfCourse = mapCourseRoom.get(course.getId());
                    if (CollectionUtils.isNotEmpty(roomsOfCourse)) {
                        String roomNames = "";
                        log.debug("roomsOfCourse ==={}", roomsOfCourse);
                        for (OrgCourseRoom room : roomsOfCourse) {
                            List<String> roomNamesList = Lists.newArrayList();
                            roomNamesList.add(roomNameMap.get(room.getRoomId()));
                            roomNames = StringUtils.join(roomNamesList, ",");
                            if (room.isDefault()) {
                                break;
                            }
                        }
                        responseDto.setRoomNames(roomNames);
                    }
                }
                if (needSms) {// 需要统计短信发送数
                    buildSmsCount(course, responseDto, studentSms, teacherSms, studentView, teacherView);
                }
                responseDto.setCreateTime(course.getCreateTime());
                responseDto.setCascadeId(course.getCascadeId());
                responseDto.setAddress(course.getAddress());
                responseDto.setStartTime(course.getBeginTime());
                responseDto.setEndTime(course.getEndTime());
                response.add(responseDto);
            }
        }
        return response;
    }

    /**
     * 获取课程学生数
     *
     * @param courseStudentMap
     * @param orgId
     * @return
     */
    private Map<Long, Integer> getCourseStudentMap(Map<Long, List<Long>> courseStudentMap, Long orgId) {
        Map<Long, Integer> result = Maps.newHashMap();
        Set<Long> userIds = Sets.newHashSet();
        Iterator<Map.Entry<Long, List<Long>>> iterator = courseStudentMap.entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry<Long, List<Long>> entry = iterator.next();
            List<Long> value = entry.getValue();
            userIds.addAll(value);
        }
        log.info("userIds in course ={}", userIds);
        Map<Long, Long> userIdStudentMap = this.orgStudentDao.getUserIdStudentIdMap(userIds, orgId, true);
        userIds.removeAll(userIdStudentMap.keySet());
        log.info("userIds not in org course ={}", userIds);
        Iterator<Map.Entry<Long, List<Long>>> iterator1 = courseStudentMap.entrySet().iterator();
        while (iterator1.hasNext()) {
            Map.Entry<Long, List<Long>> entry = iterator1.next();
            Long key = entry.getKey();
            List<Long> value = entry.getValue();
            value.removeAll(userIds);
            result.put(key, value.size());
        }
        return result;
    }

}
