/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */
package com.baijia.tianxiao.sal.course.service.impl;

import com.baijia.tianxiao.consants.UserRole;
import com.baijia.tianxiao.constant.OrgCourseStudentOpStatus;
import com.baijia.tianxiao.constant.SignStatus;
import com.baijia.tianxiao.dal.org.constant.DeleteStatus;
import com.baijia.tianxiao.dal.org.dao.CoursePurchaseDao;
import com.baijia.tianxiao.dal.org.dao.OrgAccountDao;
import com.baijia.tianxiao.dal.org.dao.OrgClassLessonDao;
import com.baijia.tianxiao.dal.org.dao.OrgCourseDao;
import com.baijia.tianxiao.dal.org.dao.OrgCourseRoomDao;
import com.baijia.tianxiao.dal.org.dao.OrgCourseSeatDao;
import com.baijia.tianxiao.dal.org.dao.OrgCourseSmsDao;
import com.baijia.tianxiao.dal.org.dao.OrgCourseStudentOpDao;
import com.baijia.tianxiao.dal.org.dao.OrgLessonCommentDao;
import com.baijia.tianxiao.dal.org.dao.OrgLessonSignDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentCourseDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentLessonDao;
import com.baijia.tianxiao.dal.org.po.OrgAccount;
import com.baijia.tianxiao.dal.org.po.OrgClassLesson;
import com.baijia.tianxiao.dal.org.po.OrgCourse;
import com.baijia.tianxiao.dal.org.po.OrgCourseSeat;
import com.baijia.tianxiao.dal.org.po.OrgCourseStudentOp;
import com.baijia.tianxiao.dal.org.po.OrgLessonComment;
import com.baijia.tianxiao.dal.org.po.OrgLessonSign;
import com.baijia.tianxiao.dal.org.po.OrgStudent;
import com.baijia.tianxiao.dal.org.po.OrgStudentCourse;
import com.baijia.tianxiao.dal.org.po.OrgStudentLesson;
import com.baijia.tianxiao.dal.storage.dao.StorageDao;
import com.baijia.tianxiao.dal.storage.po.Storage;
import com.baijia.tianxiao.dal.user.dao.StudentDao;
import com.baijia.tianxiao.dal.user.dao.TeacherDao;
import com.baijia.tianxiao.dal.user.po.Student;
import com.baijia.tianxiao.dto.PersonBaseDto;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.filter.TianxiaoMContext;
import com.baijia.tianxiao.sal.course.dto.response.ClassLessonInfoDto;
import com.baijia.tianxiao.sal.course.dto.response.CourseStudentDetailDto;
import com.baijia.tianxiao.sal.course.dto.response.CourseStudentSumDto;
import com.baijia.tianxiao.sal.course.dto.response.OrgStudentsChooseListDto;
import com.baijia.tianxiao.sal.course.dto.response.StudentResponseDto;
import com.baijia.tianxiao.sal.course.enums.CourseConsumeRuleEnum;
import com.baijia.tianxiao.sal.course.service.CourseStudentService;
import com.baijia.tianxiao.sal.course.service.OrgCourseConsumeRuleService;
import com.baijia.tianxiao.sal.course.util.StudentNameUtil;
import com.baijia.tianxiao.sal.organization.org.service.TxCascadeCredentialService;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.util.BaseUtils;
import com.baijia.tianxiao.util.HanZiPinYinUtils;
import com.baijia.tianxiao.util.mobile.MaskUtil;
import com.baijia.tianxiao.util.storage.StorageUtil;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Resource;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

/*
 * 班级学生
 * 
 * @title CourseStudentServiceImpl
 * 
 * @desc TODO
 * 
 * @author shizuwei
 * 
 * @date 2015年12月26日
 * 
 * @version 1.0
 */
@Slf4j
@Service
public class CourseStudentServiceImpl implements CourseStudentService {

    @Autowired
    private OrgCourseDao orgCourseDao;

    @Resource
    private TeacherDao teacherDao;

    @Resource
    private CoursePurchaseDao coursePurchaseDao;

    @Resource
    private OrgAccountDao orgAccountDao;

    @Resource
    private OrgStudentCourseDao orgStudentCourseDao;

    @Resource
    private OrgStudentDao orgStudentDao;

    @Resource
    private OrgCourseSeatDao orgCourseSeatDao;

    @Resource
    private OrgClassLessonDao orgClassLessonDao;

    @Resource
    private OrgCourseRoomDao orgCourseRoomDao;

    @Resource
    private OrgCourseSmsDao orgCourseSmsDao;

    @Resource
    private StorageDao storageDao;

    @Resource
    private OrgStudentLessonDao orgStudentLessonDao;

    @Resource
    private OrgCourseStudentOpDao orgCourseStudentOpDao;

    @Resource
    private StudentDao studentDao;

    @Resource
    private OrgLessonSignDao orgLessonSignDao;

    @Resource
    private OrgLessonCommentDao orgLessonCommentDao;

    @Autowired
    private TxCascadeCredentialService txCascadeCredentialService;
    
    @Resource
    private OrgCourseConsumeRuleService orgCourseConsumeRuleService;
    
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteOrgCourseStudent(Long orgId, Long courseId, Long studentId, String cause)
            throws BussinessException {

        log.debug("删除 orgId = {}, courseId = {}, studentId = {} cause = {} ", orgId, courseId, studentId, cause);

        Long userId = this.orgStudentDao.getUserId(studentId);

        if (userId == null) {
            log.error("userId == null, studentId = {}", studentId);
            throw new BussinessException(CommonErrorCode.PARAM_ERROR);
        }

        // 从课程中删除
        orgStudentCourseDao.deleteOrgCourseStudent(orgId, courseId, userId, OrgCourseStudentOpStatus.WITHDRAW);
        // 删除选座位
        OrgCourseSeat seat = orgCourseSeatDao.getCourseSeat(courseId, userId);
        if (seat != null) {
            orgCourseSeatDao.delById(seat.getId());
        }

        // 从未上完课节删除
        this.deleteStudentFromCourseLesson(orgId, courseId, userId);

        // 从未上完课节删除
        this.orgCourseSmsDao.delCourseSmsRecord(orgId, courseId, UserRole.STUDENT.getRole(),
                Lists.newArrayList(userId));
        // 添加退班标记
        OrgCourseStudentOp orgCourseStudentStatus = new OrgCourseStudentOp();
        orgCourseStudentStatus.setCause(cause);
        orgCourseStudentStatus.setFromCourseId(courseId);
        orgCourseStudentStatus.setStatus(OrgCourseStudentOpStatus.WITHDRAW);
        orgCourseStudentStatus.setStudentId(studentId);
        // orgCourseStudentStatus.setToCourseId(toCourseId);
        orgCourseStudentStatus.setOrgId(orgId);
        orgCourseStudentOpDao.save(orgCourseStudentStatus);
    }

    @Override
    public Map<Long, Integer> getCourseStudentCntMap(Long orgId, Collection<Long> courseIds, Boolean tradeComplete) {
        return this.orgStudentCourseDao.getStudentCount(orgId, courseIds, tradeComplete);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void transferOrgCourseStudent(Long orgId, Long studentId, Long fromCourseId, Long toCourseId, String cause)
            throws BussinessException {
        log.debug("course student transfer orgId = {}, from = {}, to = {}, åstudentId = {} cause = {} ", orgId,
                fromCourseId, toCourseId, studentId, cause);
        Long userId = this.orgStudentDao.getUserId(studentId);

        if (userId == null) {
            log.error("userId == null, studentId = {}", studentId);
            throw new BussinessException(CommonErrorCode.PARAM_ERROR);
        }

        // 从课程中删除
        orgStudentCourseDao.deleteOrgCourseStudent(orgId, fromCourseId, userId, OrgCourseStudentOpStatus.TRANSFER);

        // 删除选座位
        OrgCourseSeat seat = orgCourseSeatDao.getCourseSeat(fromCourseId, userId);
        if (seat != null) {
            orgCourseSeatDao.delById(seat.getId());
        }

        // 从未上完课节删除
        this.deleteStudentFromCourseLesson(orgId, fromCourseId, userId);

        // 删除短信发送记录

        this.orgCourseSmsDao.delCourseSmsRecord(orgId, fromCourseId, UserRole.STUDENT.getRole(),
                Lists.newArrayList(userId));

        // 添加转班标记
        OrgCourseStudentOp orgCourseStudentStatus = new OrgCourseStudentOp();
        orgCourseStudentStatus.setCause(cause);
        orgCourseStudentStatus.setFromCourseId(fromCourseId);
        orgCourseStudentStatus.setToCourseId(toCourseId);
        orgCourseStudentStatus.setStatus(OrgCourseStudentOpStatus.TRANSFER);
        orgCourseStudentStatus.setStudentId(studentId);
        orgCourseStudentStatus.setOrgId(orgId);
        orgCourseStudentOpDao.save(orgCourseStudentStatus);

    }

    /**
     * 学生id和头像url映射
     *
     * @param usrIds
     * @return
     */
    private Map<Long, String> getOrgStudentAvatarUrlMap(Collection<Long> usrIds) {
        Map<Long, String> map = Maps.newHashMap();
        if (CollectionUtils.isEmpty(usrIds)) {
            return map;
        }

        List<Student> students = studentDao.getByUserIds(usrIds, "userId", "avatar");
        // log.debug("students = {}", students);
        Map<Long, Integer> userIdStorageIdMap = Maps.newHashMap();// 学生id到头像id的map

        Set<Integer> avatarIds = Sets.newHashSet();
        for (Student student : students) {
            if (student.getAvatar() != null) {
                avatarIds.add(student.getAvatar());
            }
            userIdStorageIdMap.put(student.getUserId(), student.getAvatar());
        }
        // log.debug("userIdStorageIdMap={}", userIdStorageIdMap);
        Map<Long, String> storageIdUrlMap = Maps.newHashMap();// 头像id到url的map
        // log.debug("avatarIds = {}", avatarIds);
        List<Storage> storages = storageDao.getByIds(avatarIds);
        for (Storage storage : storages) {
            storageIdUrlMap.put(storage.getId(),
                    StorageUtil.constructUrl(storage.getFid(), storage.getMimetype(), storage.getSn()));
        }
        // log.debug("storageIdUrlMap = {}", storageIdUrlMap);

        for (Student student : students) {
            if (student.getAvatar() != null) {
                // log.debug("{} -> {} -> {}", student.getUserId(), userIdStorageIdMap.get(student.getUserId()),
                // storageIdUrlMap.get(userIdStorageIdMap.get(student.getUserId().longValue())));
                map.put(student.getUserId(),
                        storageIdUrlMap.get(userIdStorageIdMap.get(student.getUserId()).longValue()));
            }
        }
        return map;
    }

    @Override
    @Transactional(readOnly = true)
    public List<StudentResponseDto> getOrgCourseStudentList(@NonNull Long orgId, @NonNull Long courseId) {
        List<OrgStudentCourse> courseStudents = this.orgStudentCourseDao.getStudentListByCourseId(orgId, courseId);
        // log.debug("courseStudents = {}", courseStudents);
        Map<Long, OrgStudentCourse> map = Maps.newHashMap();
        Set<Long> userIds = Sets.newHashSet();
        for (OrgStudentCourse orgStudentCourse : courseStudents) {
            if (orgStudentCourse.getStatus().intValue() == 0) {
                userIds.add(orgStudentCourse.getUserId());
                map.put(orgStudentCourse.getUserId(), orgStudentCourse);
            }
        }

        // log.debug("userIds = {}", userIds);
        List<OrgStudent> orgStudents = orgStudentDao.getStudentByUserIds(orgId, userIds);
        Map<Long, String> studentUrlMap = getOrgStudentAvatarUrlMap(userIds);
        // log.debug("studentUrlMap = {}", studentUrlMap);

        boolean isShow = txCascadeCredentialService.isShowMobile(orgId, TianxiaoMContext.getTXCascadeId());
        List<StudentResponseDto> orgStudentCourseResponseDtos = Lists.newArrayList();
        for (OrgStudent orgStudent : orgStudents) {
            StudentResponseDto studentResponseDto = new StudentResponseDto();
            studentResponseDto.setStatus(map.get(orgStudent.getUserId()).getStatus());
            buildStudentResponseDto(studentResponseDto, orgStudent, studentUrlMap);
            if (!isShow) {
                studentResponseDto.setMobile(MaskUtil.maskMobile(studentResponseDto.getMobile()));
            }
            orgStudentCourseResponseDtos.add(studentResponseDto);
        }
        
        // log.debug("orgStudentCourseResponseDtos = {}", orgStudentCourseResponseDtos);
        return orgStudentCourseResponseDtos;
    }
    
    @Override
    @Transactional(readOnly = true)
    public List<StudentResponseDto> getLessonsStudentList(@NonNull Long orgId, @NonNull Long courseId,@NonNull List<Long> lessonIds) {
        List<Long> studentUserIds = orgStudentLessonDao.getUserIdsByLessonIds(orgId, lessonIds);
        
        Set<Long> idSet = new HashSet<Long>(studentUserIds);
        List<OrgStudent> orgStudents = orgStudentDao.getStudentByUserIds(orgId, idSet);

        Map<Long, String> studentUrlMap = getOrgStudentAvatarUrlMap(idSet);

        boolean isShow = txCascadeCredentialService.isShowMobile(orgId, TianxiaoMContext.getTXCascadeId());
        List<StudentResponseDto> orgStudentCourseResponseDtos = Lists.newArrayList();
        for (OrgStudent orgStudent : orgStudents) {
            StudentResponseDto studentResponseDto = new StudentResponseDto();
            buildStudentResponseDto(studentResponseDto, orgStudent, studentUrlMap);
            if (!isShow) {
                studentResponseDto.setMobile(MaskUtil.maskMobile(studentResponseDto.getMobile()));
            }
            orgStudentCourseResponseDtos.add(studentResponseDto);
        }
        return orgStudentCourseResponseDtos;
    }
    

    @Override
    @Transactional(readOnly = true)
    public List<OrgStudentsChooseListDto> getStudentChooseList(@NonNull Long orgId, @NonNull Long courseId, String key,
                                                               PageDto page) {
        // 获取已选本课程的学生的UserIds,没有转班退班
        Set<Long> userIds = Sets.newHashSet();
        if (courseId != null && courseId > 0) {
            List<Long> userIdList = this.orgStudentCourseDao.getStudents(orgId, courseId, 0);
            log.debug("userIdList= {}", userIdList);
            userIds.addAll(userIdList);
        }

        List<OrgStudent> students = this.orgStudentDao.getStudentsNotInUserIds(orgId, userIds, key, DeleteStatus.NORMAL.getValue(), page);
        log.debug("students= {}", students);
        List<OrgStudentsChooseListDto> chooseList = Lists.newArrayList();

        List<Long> studentUserIds = Lists.newArrayList();
        for (OrgStudent stu : students) {
            studentUserIds.add(stu.getUserId().longValue());
        }

        Map<Long, String> studentAvatarUrlMap = getOrgStudentAvatarUrlMap(studentUserIds);

        boolean isMaskMobile = txCascadeCredentialService.isShowMobile(orgId, TianxiaoMContext.getTXCascadeId());

        log.debug("studentAvatarUrlMap= {}", studentAvatarUrlMap);
        for (OrgStudent stu : students) {
            OrgStudentsChooseListDto dto = new OrgStudentsChooseListDto();
            dto.setStudentId(stu.getId());
            dto.setMobile(stu.getMobile());
            if (!isMaskMobile) {
                dto.setMobile(MaskUtil.maskMobile(dto.getMobile()));
            }
            dto.setIsChosen(userIds.contains(stu.getUserId()));
            dto.setName(StudentNameUtil.buildStudentName(stu));
            dto.setAvatarUrl(studentAvatarUrlMap.get(stu.getUserId()));
            dto.setOpenId(stu.getWeixin());
            chooseList.add(dto);
        }

        return chooseList;
    }


    private String getShowMobile(OrgStudent stu) {
        if (StringUtils.isNotBlank(stu.getShowMobile()) && !stu.getShowMobile().contains("*")) {
            return stu.getShowMobile();
        }
        return stu.getMobile();
    }

    /**
     * 把学生从课程的未上课节中删除
     *
     * @param orgId
     * @param courseId
     */
    private void deleteStudentFromCourseLesson(Long orgId, Long courseId, Long userId) {
        List<Long> leftLessonIds = this.orgClassLessonDao.getLeftLessonIds(orgId, courseId);
        log.info("leftLessonIds = {}", leftLessonIds);
        this.orgStudentLessonDao.delStudentFromLesson(orgId, leftLessonIds, userId);
        //签到记录也删除
        Map<String,Object> condition = Maps.newHashMap();
        if(CollectionUtils.isNotEmpty(leftLessonIds)){
            condition.put("lessonId", leftLessonIds);
            condition.put("userId", userId);
            condition.put("orgId", orgId);
            this.orgLessonSignDao.delByCondition(condition);
        }
        
    }

    private void buildStudentResponseDto(StudentResponseDto studentResponseDto, OrgStudent orgStudent,
                                         Map<Long, String> urlMap) {
        studentResponseDto.setStudentId(orgStudent.getId());
        studentResponseDto.setMobile(orgStudent.getMobile());
        studentResponseDto.setWeixin(orgStudent.getWeixin());
        studentResponseDto.setName(StudentNameUtil.buildStudentName(orgStudent));
        studentResponseDto.setAvatarUrl(urlMap.get(orgStudent.getUserId()));
    }

    @Override
    @Transactional(readOnly = true)
    public CourseStudentDetailDto getCourseStudentDetail(@NonNull Long orgId, @NonNull Long courseId,
                                                         @NonNull Long studentId) {

        CourseStudentDetailDto detailDto = new CourseStudentDetailDto();
        OrgCourse orgCourse = this.orgCourseDao.getById(courseId, "name");
        OrgStudent orgStudent = this.orgStudentDao.getById(studentId);
        // 设置课程名称和课程ID
        if (orgCourse != null) {
            detailDto.setCourseName(orgCourse.getName());
            detailDto.setCourseId(courseId);
          //课消规则
            Integer consumeRule = orgCourseConsumeRuleService.getRuleValueByCourseId(orgId, courseId);
            detailDto.setConsumeRule(consumeRule);
            detailDto.setConsumeRuleStr(CourseConsumeRuleEnum.getRuleDescByValue(consumeRule));
        } else {
            log.error("course not exist!");
            throw new BussinessException(CommonErrorCode.PARAM_ERROR);
        }

        boolean isShow = txCascadeCredentialService.isShowMobile(orgId, TianxiaoMContext.getTXCascadeId());

        // 设置学生姓名
        if (orgStudent != null) {
            log.debug("orgStudent = {}", orgStudent);
            detailDto.setStudentId(orgStudent.getId());
            detailDto.setStudentName(StudentNameUtil.buildStudentName(orgStudent));
            detailDto.setMobile(orgStudent.getMobile());
            if (!isShow) {
                detailDto.setMobile(MaskUtil.maskMobile(detailDto.getMobile()));
            }
            Map<Long, String> mapUrl =
                    getOrgStudentAvatarUrlMap(Lists.newArrayList(new Long[]{orgStudent.getUserId()}));
            detailDto.setAvatarUrl(mapUrl.get(orgStudent.getUserId()));
        } else {
            log.error("student not exist!");
            throw new BussinessException(CommonErrorCode.PARAM_ERROR);
        }

        // 设置学生课节信息

        List<OrgLessonSign> orgLessonSign =
                this.orgLessonSignDao.getStudentLessonSign(orgId, courseId, orgStudent.getUserId());
        log.debug("orgLessonSign = {}, orgId = {}, courseId ={},  userId = {}", orgLessonSign, orgId, courseId,
                orgStudent.getUserId());
        List<ClassLessonInfoDto> lessons = Lists.newArrayList();
        for (OrgLessonSign sign : orgLessonSign) {
            ClassLessonInfoDto lessonDto = new ClassLessonInfoDto();
            OrgClassLesson orgClassLesson = orgClassLessonDao.getById(sign.getLessonId());
            if (orgClassLesson != null) {
                lessonDto.setStartTime(orgClassLesson.getStartTime());
                lessonDto.setEndTime(orgClassLesson.getEndTime());
            }
            lessonDto.setIndex(sign.getIndex());
            lessonDto.setLessonId(sign.getLessonId());
            lessonDto.setStatus(sign.getStatus() == null ? 0 : sign.getStatus());
            lessons.add(lessonDto);
        }
        log.debug("lessons ={}", lessons);
        detailDto.setLessons(lessons);
        return detailDto;
    }

    @Override
    @Transactional(readOnly = true)
    public List<OrgStudentsChooseListDto> getLessonStudent(@NonNull Long orgId, @NonNull Long lessonId,
                                                           boolean needSignStatus) {
        List<OrgStudentsChooseListDto> chooseList = Lists.newArrayList();
        // 获取已选本课节的学生的UserIds
        List<Long> userIdList = this.orgStudentLessonDao.getUserIds(lessonId, orgId);
        Set<Long> userIds = Sets.newHashSet(userIdList);

        boolean isShowMobile = txCascadeCredentialService.isShowMobile(orgId, TianxiaoMContext.getTXCascadeId());
        List<OrgStudent> students = orgStudentDao.getStudentByUserIds(orgId, userIds, "mobile", "id", "name", "userId","showMobile");

        Map<Long, String> studentAvatarUrlMap = getOrgStudentAvatarUrlMap(userIdList);

        Map<Long, OrgLessonSign> studentSignMap = Maps.newHashMap();
        if (needSignStatus) {
            List<OrgLessonSign> lessonSigns = orgLessonSignDao.getLessonSignList(orgId, null, lessonId, null,
                    UserRole.STUDENT.getRole(), null, null, null, "userId", "status","updateTime");
            for (OrgLessonSign lessonSign : lessonSigns) {
                studentSignMap.put(lessonSign.getUserId(), lessonSign);
            }
        }

        log.debug("student avatar:{}", studentAvatarUrlMap);
        for (OrgStudent stu : students) {
            OrgStudentsChooseListDto dto = new OrgStudentsChooseListDto();
            dto.setStudentId(stu.getId());
            dto.setMobile(stu.getMobile());
            if (!isShowMobile) {
                dto.setMobile(MaskUtil.maskMobile(dto.getMobile()));
            }
            dto.setIsChosen(userIds.contains(stu.getUserId()));
            dto.setName(StudentNameUtil.buildStudentName(stu));
            dto.setAvatarUrl(studentAvatarUrlMap.get(stu.getUserId()));
            OrgLessonSign lessonSign = studentSignMap.get(stu.getUserId());
            if(lessonSign!=null){
                dto.setStudentSignStatus(SignStatus.getSignStatusByCode(lessonSign.getStatus()));
                dto.setSignInTime(lessonSign.getUpdateTime());
            }else{
                dto.setStudentSignStatus(SignStatus.UNSIGN);
            }
            
            chooseList.add(dto);
        }
        Collections.sort(chooseList, new Comparator<OrgStudentsChooseListDto>() {
            @Override
            public int compare(OrgStudentsChooseListDto o1, OrgStudentsChooseListDto o2) {
                if (o1.getSignStatus() == null) {
                    o1.setStudentSignStatus(SignStatus.UNSIGN);
                }
                if (o2.getSignStatus() == null) {
                    o2.setStudentSignStatus(SignStatus.UNSIGN);
                }
                return o1.getSignStatus().compareTo(o2.getSignStatus());
            }
        });
        return chooseList;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public int delLessonStudent(Long orgId, Long lessonId, Long studentId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(studentId != null && studentId > 0, "studentId is illegal");
        Preconditions.checkArgument(lessonId != null && lessonId > 0, "lessonId is illegal");

        Long userId = orgStudentDao.getUserId(studentId);

        OrgClassLesson lesson = orgClassLessonDao.getById(lessonId, "startTime", "endTime");
        if (lesson == null) {
            log.warn("can not found lesson by lessonId:{}", lessonId);
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "课节ID不正确");
        }
        // TODO 建议后续加上课节时间判断,如果是已经结束课节,不让删除学生

        boolean isSign = orgLessonSignDao.isStudentSignInLesson(orgId, lessonId, userId);
        if (isSign) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "该学生已签到不能删除");
        }
        OrgLessonComment comment =
                orgLessonCommentDao.getLessonCommentDetail(lessonId, userId, UserRole.STUDENT.getRole(), false, "id");

        if (comment != null) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "该学生已经进行了评价,不能删除");
        }
        Map<String, Object> condition = Maps.newHashMap();
        condition.put("orgId", orgId);
        condition.put("lessonId", lessonId);
        condition.put("userId", userId);
        return orgStudentLessonDao.delByCondition(condition);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public int deleteStudentFromLesson(Long orgId, Long lessonId, Collection<Long> userIds) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(lessonId != null && lessonId > 0, "lessonId is illegal");

        OrgClassLesson lesson = orgClassLessonDao.getById(lessonId, "startTime", "endTime");
        if (lesson == null) {
            log.warn("can not found lesson by lessonId:{}", lessonId);
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "课节ID不正确");
        }
        // TODO 建议后续加上课节时间判断,如果是已经结束课节,不让删除学生
        Map<String, Object> condition = Maps.newHashMap();
        condition.put("orgId", orgId);
        condition.put("lessonId", lessonId);
        if (CollectionUtils.isNotEmpty(userIds)) {
            condition.put("userId", userIds);
        }
        return orgStudentLessonDao.delByCondition(condition);
    }
    
    

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void addStudentToLesson(Long orgId, Long courseId, Long lessonId, Collection<Long> studentIds,
                                   boolean isStudentId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        // Preconditions.checkArgument(courseId != null && courseId > 0, "courseId is illegal");
        Preconditions.checkArgument(lessonId != null && lessonId > 0, "lessonId is illegal");
        if (CollectionUtils.isEmpty(studentIds)) {
            log.warn("add student ids is empty");
            return;
        }
        if (isStudentId) {
            Map<Long, Long> userIdMap = orgStudentDao.getStudentIdUserIdMap(studentIds);
            if (MapUtils.isEmpty(userIdMap)) {
                log.warn("add student ids is empty");
                return;
            }
            studentIds = userIdMap.values();
        }

        if (courseId == null) {
            OrgClassLesson lesson = orgClassLessonDao.getById(lessonId, "courseId", "orgId");
            Preconditions.checkArgument(lesson != null && lesson.getOrgId().equals(orgId), "课节ID不正确");
            courseId = lesson.getCourseId();
        }

        List<Long> courseStudents = orgStudentCourseDao.getStudents(orgId, courseId, 0);
        if (!courseStudents.containsAll(studentIds)) {
            log.warn("lesson teacher:{} is not all course teacher ids:{}", studentIds, courseStudents);
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "必须从课程学生中添加课节学生");
        }

        List<Long> lessonIds = Lists.newArrayList(lessonId);
        Map<Long, List<Long>> lessonStudentIds = orgStudentLessonDao.getLessonStudentList(lessonIds);
        List<OrgStudentLesson> delLessonStudent = orgStudentLessonDao.getLessonStudentListDelStatus(lessonIds);
        Map<String,OrgStudentLesson> lesseonStudentMap = Maps.newHashMap();
        for(OrgStudentLesson orgStudentLesson:delLessonStudent){
            String tempKey = orgStudentLesson.getUserId()+"|"+orgStudentLesson.getLessonId();
            lesseonStudentMap.put(tempKey, orgStudentLesson);
        }
        if (lessonStudentIds.containsKey(lessonId)) {
            List<Long> addedStuIds = lessonStudentIds.get(lessonId);
            if (CollectionUtils.isNotEmpty(addedStuIds)) {
                studentIds.removeAll(addedStuIds);
            }
        }
        List<OrgStudentLesson> saveStudentLessons = Lists.newArrayList();
        List<OrgStudentLesson> updateStudentLessons = Lists.newArrayList();
        OrgStudentLesson po = null;
        for (Long studentId : studentIds) {
            po = new OrgStudentLesson();
            po.setCreateTime(new Date());
            po.setLessonId(lessonId);
            po.setOrgId(orgId);
            po.setUserId(studentId);
            String tempKey = studentId+"|"+lessonId;
            if(lesseonStudentMap.containsKey(tempKey)){
                OrgStudentLesson tempStudentLesson = lesseonStudentMap.get(tempKey);
                po.setId(tempStudentLesson.getId());
                po.setDelStatus(0);
                po.setStartStatus(0);
                updateStudentLessons.add(po);
            }else{
                saveStudentLessons.add(po);
            }
            
        }
        log.info("save students :{} to lessonid:{}", saveStudentLessons, lessonId);
        orgStudentLessonDao.saveAll(saveStudentLessons, "createTime", "lessonId", "orgId", "userId");
        for(OrgStudentLesson updateStudentLesson:updateStudentLessons){
            orgStudentLessonDao.update(updateStudentLesson, "createTime", "delStatus", "startStatus");
        }
    }
    
    
    

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void resetLessonStudent(Long orgId, Long courseId, Long lessonId, Collection<Long> studentIds) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(courseId != null && courseId > 0, "courseId is illegal");
        Preconditions.checkArgument(lessonId != null && lessonId > 0, "lessonId is illegal");
        // if (CollectionUtils.isEmpty(studentIds)) {
        // log.warn("add student ids is empty");
        // return;
        // }
        deleteStudentFromLesson(orgId, lessonId, null);
        Map<Long, Long> userIdMap = orgStudentDao.getStudentIdUserIdMap(studentIds);
        if (MapUtils.isEmpty(userIdMap)) {
            log.warn("add student ids is empty");
            return;
        }
        Collection<Long> userIds = userIdMap.values();
        // 清除课节中的学生信息
        // 添加学生到课节
        addStudentToLesson(orgId, courseId, lessonId, userIds, false);
    }

    @Override
    @Transactional(readOnly = true)
    public List<Long> getStudentLessonIds(@NonNull Long orgId, @NonNull Long studentId) {
        Long userId = this.orgStudentDao.getUserId(studentId);
        if (userId == null) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "student not found!");
        }
        return orgStudentLessonDao.getLessonIdsOfStudent(orgId, userId);
    }

    @Override
    @Transactional(readOnly = true)
    public List<Long> getCourseStudentUserIds(Long orgId, Long courseId) {
        return this.orgStudentCourseDao.getStudents(orgId, courseId, 0);
    }

    @Override
    @Transactional(readOnly = true)
    public Map<String, String> getStudentNameAndAvatar(Long orgId, Long studentId) {

        Long userId = this.orgStudentDao.getUserId(studentId);
        Preconditions.checkArgument(userId != null && userId > 0, "student not found!");
        Map<Long, String> urlMap = getOrgStudentAvatarUrlMap(Lists.newArrayList(new Long[]{userId}));
        String avatarUrl = urlMap.get(userId);
        OrgStudent orgStudent = this.orgStudentDao.getStudent(orgId, userId, 0, "name");
        Map<String, String> map = Maps.newHashMap();
        map.put("name", orgStudent != null ? orgStudent.getName() : "");
        map.put("avatarUrl", avatarUrl);
        return map;
    }

    @Override
    @Transactional(readOnly = true)
    public Map<String, Integer> getStudentCourseLessonInfo(Long orgId, Long studentId) {
        Long userId = this.orgStudentDao.getUserId(studentId);
        Preconditions.checkArgument(userId != null && userId > 0, "student not exists!");
        Map<String, Integer> map = Maps.newHashMap();
        
        List<Long> courseIds = this.orgStudentCourseDao.getStudentCourseIds(orgId, userId, null);
        Integer courseCount = 0;
        OrgAccount orgAccount = this.orgAccountDao.getById(orgId, "number");
        if (orgAccount == null) {
            log.error("orgId = {} not exist in orgAccount", orgId);
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "机构帐号错误");
        }
        Long orgNumber = orgAccount.getNumber().longValue();
        List<OrgCourse> orgCourses = this.orgCourseDao.getByIds(courseIds);
        List<Long> courseNoDeleteIds = Lists.newArrayList();
        for (OrgCourse orgCourse : orgCourses) {
            if (orgCourse.getIsDel()==0 && orgCourse.getOrgNumber().equals(orgNumber)) {
                courseCount++;
                courseNoDeleteIds.add(orgCourse.getId());
            }
        }
        map.put("lessonToday", this.orgStudentLessonDao.getStudentLessonCountTodayByCourseId(orgId, userId,courseNoDeleteIds));
        map.put("courseCount", courseCount);
        map.put("commentCount", this.orgLessonCommentDao.getStudentLessonCommentCount(orgId, userId, null));
        return map;
    }
    
    @Override
    public List<CourseStudentSumDto> getCourseStudents(Long orgId, Long courseId, PageDto pageDto) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(courseId != null && courseId > 0, "courseId is illegal");
        
        List<CourseStudentSumDto> result = Lists.newArrayList();
        
        List<OrgStudentCourse> courseStudents = this.orgStudentCourseDao.getStudentListByCourseId(orgId, courseId);
        if (CollectionUtils.isEmpty(courseStudents)) {
            return result;
        }
        
        List<Long> normalStudents = Lists.newArrayList();
        for (OrgStudentCourse orgStudentCourse : courseStudents) {
            if (orgStudentCourse.getStatus().intValue() == 0) {
                normalStudents.add(orgStudentCourse.getUserId());
            }
        }
        if (CollectionUtils.isEmpty(normalStudents)) {
            return result;
        }
        
        List<OrgStudent> orgStudents = this.orgStudentDao.getStudents(orgId, normalStudents, "", DeleteStatus.NORMAL.getValue(), pageDto, "id", "name", "userId");
        if (CollectionUtils.isNotEmpty(orgStudents)) {
            Set<Long> queryStudents = BaseUtils.getPropertiesList(orgStudents, "userId");
            
            // 头像
            Map<Long, String> studentUrlMap = getOrgStudentAvatarUrlMap(queryStudents);
            // 签到统计
            Map<Long, Integer> studentCourseSignMap = this.orgLessonSignDao.getOrgCourseStudentSignCount(queryStudents, courseId);
            // 学生班级总课节
            Map<Long, List<Long>> studentLessonMap = this.orgStudentLessonDao.getStudentCourseLessonMap(orgId, courseId, queryStudents);
            // 剩余课节
            Map<Long, Integer> studentLeftLessonMap = Maps.newHashMap();
            Integer ruleValue = this.orgCourseConsumeRuleService.getRuleValueByCourseId(orgId, courseId);
            if (ruleValue == null || ruleValue == 0) {
                // 默认按时间节点计算课消
                studentLeftLessonMap = this.orgStudentLessonDao.getStudentsLeftLessonMap(orgId, courseId, queryStudents);
            } else {
                // 各种签到状态，用于按到课/请假/缺课算课消时使用
                Map<Long, List<Integer>> studentSignStatusMap = this.orgLessonSignDao.getOrgCourseStudentSignMap(queryStudents, courseId);
                for (Long userId : queryStudents) {
                    int finish = 0;
                    if (studentSignStatusMap.containsKey(userId)) {
                        finish = CourseConsumeRuleEnum.calculateConsumeTimesBySignCodeList(ruleValue, studentSignStatusMap.get(userId));
                    }
                    // 有的学生无排课信息,需要判断
                    if (studentLessonMap.containsKey(userId)) {
                        studentLeftLessonMap.put(userId, studentLessonMap.get(userId).size() - finish);
                    }
                }
            }
            
            for (OrgStudent orgStudent : orgStudents) {
                CourseStudentSumDto dto = new CourseStudentSumDto();
                dto.setStudentId(orgStudent.getId());
                dto.setName(orgStudent.getName());
                dto.setAvatarUrl(studentUrlMap.get(orgStudent.getUserId()));
                
                dto.setSignedLessons(studentCourseSignMap.containsKey(orgStudent.getUserId()) ? studentCourseSignMap.get(orgStudent.getUserId()) : 0);
                dto.setLeftLessons(studentLeftLessonMap.containsKey(orgStudent.getUserId()) ? studentLeftLessonMap.get(orgStudent.getUserId()) : 0);
                dto.setTotalLessons(studentLessonMap.containsKey(orgStudent.getUserId()) ? studentLessonMap.get(orgStudent.getUserId()).size() : 0);
                dto.setFinishLessons(dto.getTotalLessons() - dto.getLeftLessons());
                result.add(dto);
            }
        }
        
        return result;
    }
    
    @Override
    public List<PersonBaseDto> getCourseStudentsBase(Long orgId, Long courseId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(courseId != null && courseId > 0, "courseId is illegal");
        
        List<PersonBaseDto> result = Lists.newArrayList();
        List<OrgStudentCourse> courseStudents = this.orgStudentCourseDao.getStudentListByCourseId(orgId, courseId);
        if (CollectionUtils.isEmpty(courseStudents)) {
            return result;
        }
        
        Set<Long> normalStudents = Sets.newHashSet();
        for (OrgStudentCourse orgStudentCourse : courseStudents) {
            if (orgStudentCourse.getStatus().intValue() == 0) {
                normalStudents.add(orgStudentCourse.getUserId());
            }
        }
        
        if (CollectionUtils.isEmpty(normalStudents)) {
            return result;
        }
        
        List<OrgStudent> orgStudents = this.orgStudentDao.getStudents(orgId, normalStudents, DeleteStatus.NORMAL.getValue(),  null, "id", "name", "userId", "mobile");
        if (CollectionUtils.isNotEmpty(orgStudents)) {
            Set<Long> queryStudents = BaseUtils.getPropertiesList(orgStudents, "userId");
            // 头像
            Map<Long, String> studentUrlMap = getOrgStudentAvatarUrlMap(queryStudents);
            boolean isShowMobile = txCascadeCredentialService.isShowMobile(orgId, TianxiaoMContext.getTXCascadeId());
            
            for (OrgStudent orgStudent : orgStudents) {
                PersonBaseDto dto = new PersonBaseDto();
                dto.setId(orgStudent.getId());
                dto.setName(orgStudent.getName());
                dto.setAvatarUrl(studentUrlMap.get(orgStudent.getUserId()));
                dto.setPinyinName(HanZiPinYinUtils.getLowerCasePinYin(orgStudent.getName()));
                
                if (!isShowMobile) {
                    dto.setMobile(MaskUtil.maskMobile(dto.getMobile()));
                } else {
                    dto.setMobile(orgStudent.getMobile());
                }
                result.add(dto);
            }
        }
        
        return result;
    }
    
    @Override
    public List<PersonBaseDto> getCourseStudentsNotInLesson(Long orgId, Long lessonId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(lessonId != null && lessonId > 0, "lessonId is illegal");
        OrgClassLesson lesson = this.orgClassLessonDao.getById(lessonId, "id", "orgId", "courseId", "delStatus");
        if (lesson == null || lesson.getDelStatus().intValue() == DeleteStatus.DELETED.getValue() || lesson.getOrgId().longValue() != orgId) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "课节不存在或已被删除");
        }
        
        List<PersonBaseDto> result = Lists.newArrayList();
        List<OrgStudentCourse> courseStudents = this.orgStudentCourseDao.getStudentListByCourseId(orgId, lesson.getCourseId());
        if (CollectionUtils.isEmpty(courseStudents)) {
            return result;
        }
        
        Set<Long> normalStudents = Sets.newHashSet();
        for (OrgStudentCourse orgStudentCourse : courseStudents) {
            if (orgStudentCourse.getStatus().intValue() == 0) {
                normalStudents.add(orgStudentCourse.getUserId());
            }
        }
        
        if (CollectionUtils.isEmpty(normalStudents)) {
            return result;
        }
        
        List<Long> lessonStudents = this.orgStudentLessonDao.getUserIdsByLessonIds(orgId, Lists.newArrayList(lessonId));
        // 在班级中，但是不在课节中
        normalStudents.removeAll(lessonStudents);
        
        List<OrgStudent> orgStudents = this.orgStudentDao.getStudents(orgId, normalStudents, DeleteStatus.NORMAL.getValue(),  null, "id", "name", "userId", "mobile");
        if (CollectionUtils.isNotEmpty(orgStudents)) {
            Set<Long> queryStudents = BaseUtils.getPropertiesList(orgStudents, "userId");
            // 头像
            Map<Long, String> studentUrlMap = getOrgStudentAvatarUrlMap(queryStudents);
            for (OrgStudent orgStudent : orgStudents) {
                PersonBaseDto dto = new PersonBaseDto();
                dto.setId(orgStudent.getId());
                dto.setName(orgStudent.getName());
                dto.setAvatarUrl(studentUrlMap.get(orgStudent.getUserId()));
                result.add(dto);
            }
        }
        return result;
    }
}
