
/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */
package com.baijia.tianxiao.sal.course.service.impl;

import java.text.Collator;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.baijia.commons.lang.utils.BaseUtils;
import com.baijia.tianxiao.consants.UserRole;
import com.baijia.tianxiao.constant.AvatarConstants;
import com.baijia.tianxiao.constant.InClassStatus;
import com.baijia.tianxiao.constant.OrgTeacherStatus;
import com.baijia.tianxiao.dal.org.dao.OrgClassLessonDao;
import com.baijia.tianxiao.dal.org.dao.OrgCourseSmsDao;
import com.baijia.tianxiao.dal.org.dao.OrgCourseTeacherDao;
import com.baijia.tianxiao.dal.org.dao.OrgLessonConflictDao;
import com.baijia.tianxiao.dal.org.dao.OrgTeacherBranchDao;
import com.baijia.tianxiao.dal.org.dao.OrgTeacherDao;
import com.baijia.tianxiao.dal.org.dao.OrgTeacherLessonDao;
import com.baijia.tianxiao.dal.org.po.OrgClassLesson;
import com.baijia.tianxiao.dal.org.po.OrgCourseTeacher;
import com.baijia.tianxiao.dal.org.po.OrgLessonConflict;
import com.baijia.tianxiao.dal.org.po.OrgTeacher;
import com.baijia.tianxiao.dal.org.po.OrgTeacherBranch;
import com.baijia.tianxiao.dal.org.po.OrgTeacherLesson;
import com.baijia.tianxiao.dal.storage.dao.StorageDao;
import com.baijia.tianxiao.dal.storage.po.Storage;
import com.baijia.tianxiao.dal.user.dao.TeacherDao;
import com.baijia.tianxiao.dal.user.dao.UserDao;
import com.baijia.tianxiao.dal.user.po.Teacher;
import com.baijia.tianxiao.dal.user.po.User;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.common.api.FilterSoftDelTeacherService;
import com.baijia.tianxiao.sal.course.dto.response.AppTeacherResponseDto;
import com.baijia.tianxiao.sal.course.dto.response.TeacherDto;
import com.baijia.tianxiao.sal.course.dto.response.TeacherRespDto;
import com.baijia.tianxiao.sal.course.dto.response.TeacherResponseDto;
import com.baijia.tianxiao.sal.course.service.CourseTeacherService;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.util.HanZiPinYinUtils;
import com.baijia.tianxiao.util.storage.StorageUtil;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

/**
 * @author cxm
 * @version 1.0
 * @title CourseTeacherServiceImpl
 * @desc TODO
 * @date 2015年12月25日
 */
@Service
@Slf4j
public class CourseTeacherServiceImpl implements CourseTeacherService {

    /**
     *
     */
    public static final String ANONYMOUS_TEACHER_NAME = "匿名老师";

    @Resource
    private TeacherDao teacherDao;

    @Resource
    private OrgCourseTeacherDao orgCourseTeacherDao;

    @Resource
    private OrgTeacherBranchDao orgTeacherBranchDao;

    @Resource
    private StorageDao storageDao;

    @Resource
    private OrgClassLessonDao orgClassLessonDao;

    @Resource
    private OrgTeacherLessonDao orgTeacherLessonDao;

    @Resource
    private OrgTeacherDao orgTeacherDao;

    @Resource
    private OrgCourseSmsDao orgCourseSmsDao;

    @Resource
    private UserDao userDao;

    @Resource
    private OrgLessonConflictDao orgLessonConflictDao;

    @Autowired
    private FilterSoftDelTeacherService filterSoftDelTeacherService;

    @Override
    @Transactional(readOnly = true)
    public List<TeacherResponseDto> listOrgTeacher(String key, Long orgId, Long branchId, Long courseId, PageDto page) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Set<Long> teacherIds = Sets.newHashSet(orgTeacherDao.getTeacherIds(orgId, OrgTeacherStatus.SIGNED.getCode()));
        List<Long> courseTeacherIds = null;

        if (branchId != null) {

            List<OrgTeacherBranch> branchTeachers =
                orgTeacherBranchDao.getBranchTeacher(orgId, branchId, null, teacherIds, "userId");
            if (CollectionUtils.isEmpty(branchTeachers)) {
                return Collections.emptyList();
            }
            teacherIds.clear();
            for (OrgTeacherBranch teacherBranch : branchTeachers) {
                teacherIds.add(teacherBranch.getUserId());
            }
        }
        if (courseId != null && courseId > 0) {
            courseTeacherIds = orgCourseTeacherDao.getTeacherIdsByCourseId(courseId);
            if (CollectionUtils.isNotEmpty(teacherIds) && CollectionUtils.isNotEmpty(courseTeacherIds)) {
                teacherIds.removeAll(courseTeacherIds);
            }
        }
        if (CollectionUtils.isEmpty(teacherIds)) {
            return Lists.newArrayList();
        }
        List<Teacher> teacherList =
            teacherDao.getTeachers(orgId, teacherIds, key, null, null, OrgTeacherStatus.SIGNED.getCode(), page);
        if (CollectionUtils.isNotEmpty(courseTeacherIds)) {
            for (Iterator<Teacher> iterator = teacherList.iterator(); iterator.hasNext();) {
                Teacher teacher = iterator.next();
                if (courseTeacherIds.contains(teacher.getUserId())) {
                    iterator.remove();
                }
            }
        }
        return buildDtoList(teacherList);
    }

    @Override
    @Transactional(readOnly = true)
    public List<TeacherResponseDto> listCourseTeacher(Long orgId, Long courseId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(courseId != null && courseId > 0, "courseId is illegal");
        List<OrgCourseTeacher> courseTeachers = orgCourseTeacherDao.getTeacher(orgId, courseId);
        Set<Long> userIds = Sets.newHashSet();
        for (OrgCourseTeacher courseTeacher : courseTeachers) {
            userIds.add(courseTeacher.getUserId());
        }
        if (CollectionUtils.isNotEmpty(userIds)) {
            List<Teacher> teacherList =
                teacherDao.getTeachers(orgId, userIds, null, null, null, OrgTeacherStatus.SIGNED.getCode(), null);
            filterSoftDelTeacherService.filterTeachers(orgId, teacherList);
            return buildDtoList(teacherList);
        }
        return Collections.emptyList();
    }

    // /**
    // * 针对OrgTeacher里面的老师已经被清除的情况,进行OrgCourseTeacher里面记录的过滤展示 MARK
    // *
    // * @param teacherList
    // * @param userIds
    // */
    // private void checkoutTeachers(Long orgId, List<Teacher> teacherList, Collection<Long> userIds) {
    // if (GenericsUtils.isNullOrEmpty(teacherList) || GenericsUtils.isNullOrEmpty(userIds)) {
    // return;
    // }
    // List<OrgTeacher> teacherByUserIdsAndOrgId = this.orgTeacherDao.getTeacherByUserIdsAndOrgId(orgId, userIds,
    // OrgTeacherStatus.SIGNED.getCode(), true, "userId");
    //
    // List<Long> extractList =
    // com.baijia.commons.lang.utils.collection.CollectionUtils.extractList(teacherByUserIdsAndOrgId,
    // new com.baijia.commons.lang.utils.collection.CollectionUtils.Extracter<Long, OrgTeacher>() {
    // @Override
    // public Long extract(OrgTeacher arg0) {
    // return arg0.getUserId();
    // }
    // });
    // for (Iterator<Teacher> iter = teacherList.iterator(); iter.hasNext();) {
    // Teacher next = iter.next();
    // if (!extractList.contains(next.getUserId())) {
    // iter.remove();
    // }
    // }
    // }

    @Override
    @Transactional(readOnly = true)
    public List<TeacherResponseDto> listCourseTeacher(Long orgId, Collection<Long> courseIds, String key,
        PageDto pageDto) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(CollectionUtils.isNotEmpty(courseIds), "courseIds is empty");
        List<OrgCourseTeacher> courseTeachers = orgCourseTeacherDao.getOrgCourseTeacher(courseIds, pageDto);
        Set<Long> userIds = Sets.newHashSet();
        for (OrgCourseTeacher courseTeacher : courseTeachers) {
            userIds.add(courseTeacher.getUserId());
        }
        if (CollectionUtils.isNotEmpty(userIds)) {
            List<Teacher> teacherList = teacherDao.getTeachers(orgId, userIds, key, null, null, null);
            filterSoftDelTeacherService.filterTeachers(orgId, teacherList);
            return buildDtoList(teacherList);
        }
        return Collections.emptyList();
    }

    @Override
    public List<AppTeacherResponseDto> listOrgTeacherWithCourse(String key, Long orgId, Long courseId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Set<Long> teacherIds = Sets.newHashSet(orgTeacherDao.getTeacherIds(orgId, OrgTeacherStatus.SIGNED.getCode()));

        if (CollectionUtils.isEmpty(teacherIds)) {
            return Lists.newArrayList();
        }
        List<Teacher> teacherList =
            teacherDao.getTeachers(orgId, teacherIds, key, null, null, OrgTeacherStatus.SIGNED.getCode(), null);

        if (CollectionUtils.isEmpty(teacherList)) {
            return Lists.newArrayList();
        }

        List<Long> courseTeacherIds = null;

        if (courseId != null && courseId > 0) {
            courseTeacherIds = orgCourseTeacherDao.getTeacherIdsByCourseId(courseId);
        }

        if (CollectionUtils.isNotEmpty(teacherList)) {
            List<AppTeacherResponseDto> result = Lists.newArrayList();
            Set<Long> avatarIds = Sets.newHashSet();
            for (Teacher teacher : teacherList) {
                if (teacher.getAvatar() != null && teacher.getAvatar() > 0) {
                    avatarIds.add(teacher.getAvatar());
                }
            }
            List<Storage> storageList = storageDao.getByIds(avatarIds);
            Map<Long, String> avatarMap = getAvatarMap(storageList);
            for (Teacher teacher : teacherList) {
                AppTeacherResponseDto dto = new AppTeacherResponseDto();
                if (teacher.getAvatar() != null) {
                    dto.setAvatar(avatarMap.get(teacher.getAvatar()));
                }
                dto.setTeacherId(teacher.getUserId());
                dto.setTeacherName(getTeacherName(teacher));
                dto.setInitial(HanZiPinYinUtils.getInitialStr(dto.getTeacherName()));
                if (CollectionUtils.isNotEmpty(courseTeacherIds) && courseTeacherIds.contains(teacher.getUserId())) {
                    dto.setChosen(InClassStatus.YES.isCode());
                }
                result.add(dto);
            }
            return result;
        }
        return Collections.emptyList();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void addTeacherToCourse(Long orgId, Long courseId, Collection<Long> teacherIds) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(courseId != null && courseId > 0, "courseId is illegal");
        if (CollectionUtils.isEmpty(teacherIds)) {
            log.warn("add teacher ids is empty");
            return;
        }
        List<OrgCourseTeacher> courseTeachers = orgCourseTeacherDao.getTeacher(orgId, courseId);
        for (OrgCourseTeacher t : courseTeachers) {
            teacherIds.remove(t.getUserId());
        }
        List<Teacher> teacherList = teacherDao.getByTeacherIds(teacherIds, "userId", "orgId");
        filterSoftDelTeacherService.filterTeachers(orgId, teacherList);
        List<OrgCourseTeacher> saveCourseTeacher = Lists.newArrayList();
        OrgCourseTeacher po = null;
        for (Teacher teacher : teacherList) {
            if (!orgId.equals(teacher.getOrgId())) {
                log.warn("teacher id:{} is not current org teacher", teacher.getUserId());
                continue;
            }
            po = new OrgCourseTeacher();
            po.setOrgCourseId(courseId);
            po.setUserId(teacher.getUserId());
            po.setCreateTime(new Date());
            po.setUpdateTime(new Date());
            saveCourseTeacher.add(po);
        }
        log.info("save teachers :{} to courseId:{}", saveCourseTeacher, courseId);
        orgCourseTeacherDao.saveAll(saveCourseTeacher, "orgCourseId", "userId", "createTime", "updateTime");
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public int delTeacherFromCourse(Long orgId, Long courseId, Collection<Long> teacherIds) {
        Preconditions.checkArgument(courseId != null && courseId > 0, "courseId is illegal");
        if (CollectionUtils.isEmpty(teacherIds)) {
            log.warn("clear all teacher from courseId:{}", courseId);
            return 0;
        }
        orgCourseTeacherDao.delTeachersFromCourse(courseId, teacherIds);
        orgCourseSmsDao.delCourseSmsRecord(orgId, courseId, UserRole.TEACHER.getRole(), teacherIds);
        List<Long> lessonIds = orgClassLessonDao.getLeftLessonIds(orgId, courseId);
        if (CollectionUtils.isNotEmpty(lessonIds)) {
            Map<String, Object> delCondition = new HashMap<>();
            delCondition.put("lessonId", lessonIds);
            int result = orgTeacherLessonDao.delByCondition(delCondition);
            log.info("del from org teacher lesson by lessonId:{},result:{}", lessonIds, result);
            return result;
        }
        return 0;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public int delTeacherFromCourseForLessonCheck(Long orgId, Long courseId, Collection<Long> teacherIds) {
        Preconditions.checkArgument(courseId != null && courseId > 0, "courseId is illegal");
        if (CollectionUtils.isEmpty(teacherIds)) {
            log.warn("clear all teacher from courseId:{}", courseId);
            return 0;
        }
        Map<Long, OrgClassLesson> mapClassLesson = orgClassLessonDao.getClassLessonMap(orgId, courseId, 0, null);
        Set<Long> allLessonTeacherId = new HashSet<Long>();
        if (mapClassLesson.size() > 0) {
            Map<Long, Long> mapTeacherLesson =
                orgTeacherLessonDao.queryLessonTeacherIdMap(orgId, mapClassLesson.keySet());
            allLessonTeacherId.addAll(mapTeacherLesson.values());
        }
        if (allLessonTeacherId.size() > 0) {
            Map<Long, OrgTeacher> orgTeacherMap = this.orgTeacherDao.getMapByIds(orgId, teacherIds);
            for (Long deleteTeacherId : teacherIds) {
                OrgTeacher orgTeacher = orgTeacherMap.get(deleteTeacherId);
                if (orgTeacher == null) {
                    continue;
                }

                if (orgTeacher.getStatus() == 1 && allLessonTeacherId.contains(deleteTeacherId)) {
                    return 0;
                }
            }
        }
        int result = orgCourseTeacherDao.delTeachersFromCourse(courseId, teacherIds);
        orgCourseSmsDao.delCourseSmsRecord(orgId, courseId, UserRole.TEACHER.getRole(), teacherIds);

        return result;
    }

    @Override
    @Transactional(readOnly = true)
    public List<TeacherResponseDto> getLessonTeacher(@NonNull Long orgId, @NonNull Long lessonId) {
        // 获取已选本课节的老师的UserIds
        List<Long> userIdList = this.orgTeacherLessonDao.getUserIds(lessonId, orgId);
        if (CollectionUtils.isEmpty(userIdList)) {
            return Collections.emptyList();
        }
        List<Teacher> teachers = teacherDao.getTeachers(orgId, userIdList, null, null, null, null);
        OrgClassLesson byId = this.orgClassLessonDao.getById(lessonId);
        if (byId != null && byId.getEndTime().compareTo(new Date()) > 0) {
            filterSoftDelTeacherService.filterTeachers(orgId, teachers);
        }
        return buildDtoList(teachers);
    }

    private List<TeacherResponseDto> buildDtoList(List<Teacher> teacherList) {
        if (CollectionUtils.isNotEmpty(teacherList)) {
            List<TeacherResponseDto> result = Lists.newArrayList();
            Set<Long> avatarIds = Sets.newHashSet();
            for (Teacher teacher : teacherList) {
                if (teacher.getAvatar() != null && teacher.getAvatar() > 0) {
                    avatarIds.add(teacher.getAvatar());
                }
            }
            List<Storage> storageList = storageDao.getByIds(avatarIds);
            Map<Long, String> avatarMap = getAvatarMap(storageList);
            for (Teacher teacher : teacherList) {
                TeacherResponseDto dto = buildDto(teacher, false);
                dto.setAvatar(avatarMap.get(teacher.getAvatar()));
                dto.setSchoolAge(teacher.getSchoolAge());
                dto.setInitial(HanZiPinYinUtils.getInitialStr(dto.getTeacherName()));
                result.add(dto);
            }
            return result;
        }
        return Collections.emptyList();
    }

    private TeacherResponseDto buildDto(Teacher teacher, boolean needAvatar) {
        TeacherResponseDto dto = new TeacherResponseDto();
        if (needAvatar) {
            if (teacher.getAvatar() != null) {
                Storage storage = storageDao.getById(teacher.getAvatar());
                dto.setAvatar(StorageUtil.constructUrl(storage.getFid(), storage.getMimetype(), storage.getSn()));
            }
        }
        dto.setTeacherId(teacher.getUserId());
        User user = userDao.getById(teacher.getUserId());
        dto.setTeacherNumber(user.getNumber());
        dto.setTeacherName(getTeacherName(teacher));
        dto.setMobile(teacher.getMobile());
        return dto;
    }

    private String getTeacherName(Teacher teacher) {
        if (StringUtils.isNoneBlank(teacher.getRealName())) {
            return teacher.getRealName();
        } else if (StringUtils.isNoneBlank(teacher.getNickName())) {
            return teacher.getNickName();
        } else {
            return ANONYMOUS_TEACHER_NAME;
        }
    }

    private Map<Long, String> getAvatarMap(List<Storage> storageList) {
        if (CollectionUtils.isNotEmpty(storageList)) {
            Map<Long, String> result = Maps.newHashMap();
            for (Storage storage : storageList) {
                result.put(storage.getId(),
                    StorageUtil.constructUrl(storage.getFid(), storage.getMimetype(), storage.getSn()));
            }
            return result;
        } else {
            return Collections.emptyMap();
        }
    }

    @Override
    @Transactional(readOnly = true)
    public List<TeacherResponseDto> getTeachers(Collection<Long> teacherIds, Long orgId) {
        List<Teacher> teacherList = teacherDao.getTeachers(orgId, teacherIds, null, null, null, null);
        filterSoftDelTeacherService.filterTeachers(orgId, teacherList);
        return buildDtoList(teacherList);
    }

    @Override
    @Transactional(readOnly = true)
    public TeacherResponseDto getTeacher(Long teacherId, Long orgId) {
        Teacher teacher = teacherDao.getByUserId(teacherId);
        filterSoftDelTeacherService.filterTeachers(orgId, Arrays.asList(teacher));
        return buildDto(teacher, true);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void addTeacherToLesson(Long orgId, Long courseId, Long lessonId, Collection<Long> teacherIds) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(courseId != null && courseId > 0, "courseId is illegal");
        Preconditions.checkArgument(lessonId != null && lessonId > 0, "lessonId is illegal");
        if (CollectionUtils.isEmpty(teacherIds)) {
            log.warn("add teacher ids is empty");
            return;
        }
        if (CollectionUtils.isEmpty(teacherIds)) {
            log.warn("lesson teacher:{} is not all course teacher ids:{}", teacherIds);
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "添加老师不能为空");
        }

        List<OrgTeacherLesson> saveLessonTeachers = Lists.newArrayList();
        OrgTeacherLesson po = null;
        for (Long teacherId : teacherIds) {
            po = new OrgTeacherLesson();
            po.setCreateTime(new Date());
            po.setLessonId(lessonId);
            po.setOrgId(orgId);
            po.setTeacherId(teacherId);
            saveLessonTeachers.add(po);
        }
        log.info("save teachers :{} to lessonId:{}", saveLessonTeachers, lessonId);
        orgTeacherLessonDao.saveAll(saveLessonTeachers, false, "createTime", "lessonId", "orgId", "teacherId");
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public int delTeacherFromLesson(Long orgId, Long lessonId, Collection<Long> teacherIds) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(lessonId != null && lessonId > 0, "lessonId is illegal");

        OrgClassLesson lesson = orgClassLessonDao.getById(lessonId, "startTime", "endTime");
        if (lesson == null) {
            log.warn("can not found lesson by lessonId:{}", lessonId);
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "课节ID不正确");
        }
        // TODO 建议后续加上课节时间判断,如果是已经结束课节,不让删除老师
        Map<String, Object> condition = Maps.newHashMap();
        condition.put("orgId", orgId);
        condition.put("lessonId", lessonId);
        if (CollectionUtils.isNotEmpty(teacherIds)) {
            condition.put("teacherId", teacherIds);
        }
        return orgTeacherLessonDao.delByCondition(condition);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void resetLessonTeacher(Long orgId, Long courseId, Long lessonId, Collection<Long> teacherIds) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(lessonId != null && lessonId > 0, "lessonId is illegal");
        // 清理原来的课节老师信息
        delTeacherFromLesson(orgId, lessonId, null);
        // 添加新老师
        Set<Long> tempTeacherIds = Sets.newHashSet();
        tempTeacherIds.addAll(teacherIds);
        addTeacherToCourse(orgId, courseId, teacherIds);
        addTeacherToLesson(orgId, courseId, lessonId, tempTeacherIds);
    }

    @SuppressWarnings("unchecked")
    @Override
    public List<TeacherRespDto> listPcOrgTeacher(Long orgId, String key, Long courseId, PageDto pageDto) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "机构id非法");
        // 机构的所有老师
        List<Long> orgTeacherIds = orgTeacherDao.getTeacherIds(orgId, OrgTeacherStatus.SIGNED.getCode());
        Set<Long> teacherIds = Sets.newHashSet(orgTeacherIds);
        // 指定课程所有已选的老师
        List<Long> courseTeacherIds = Lists.newArrayList();
        // 剔除已选的老师
        if (courseId != null && courseId > 0) {
            courseTeacherIds = orgCourseTeacherDao.getTeacherIdsByCourseId(courseId);
            if (CollectionUtils.isNotEmpty(teacherIds) && CollectionUtils.isNotEmpty(courseTeacherIds)) {
                teacherIds.removeAll(courseTeacherIds);
            }
        }
        if (CollectionUtils.isEmpty(teacherIds)) {
            return Lists.newArrayList();
        }
        // 可选的老师集合
        List<Teacher> teacherList =
            teacherDao.getTeachers(orgId, teacherIds, key, null, null, OrgTeacherStatus.SIGNED.getCode(), pageDto);
        if (CollectionUtils.isNotEmpty(courseTeacherIds)) {
            for (Iterator<Teacher> iterator = teacherList.iterator(); iterator.hasNext();) {
                Teacher teacher = iterator.next();
                if (courseTeacherIds.contains(teacher.getUserId())) {
                    iterator.remove();
                }
            }
        }
        List<TeacherRespDto> dtos = buildList(teacherList);
        Collections.sort(dtos, new Comparator() {
            @Override
            public int compare(Object object1, Object object2) {
                TeacherRespDto info1 = (TeacherRespDto) object1;
                TeacherRespDto info2 = (TeacherRespDto) object2;
                // 按bean的名字排序
                String name1 = info1.getName();
                if (name1 == null) {
                    name1 = "";
                }
                String name2 = info2.getName();
                if (name2 == null) {
                    name2 = "";
                }
                return Collator.getInstance(Locale.CHINESE).compare(name1, name2);
            }

        });
        return dtos;
    }

    private List<TeacherRespDto> buildList(List<Teacher> teacherList) {
        if (CollectionUtils.isNotEmpty(teacherList)) {
            List<TeacherRespDto> result = Lists.newArrayList();
            Set<Long> avatarIds = Sets.newHashSet();
            for (Teacher teacher : teacherList) {
                if (teacher.getAvatar() != null && teacher.getAvatar() > 0) {
                    avatarIds.add(teacher.getAvatar());
                }
            }
            List<Storage> storageList = storageDao.getByIds(avatarIds);
            Map<Long, String> avatarMap = getAvatarMap(storageList);
            for (Teacher teacher : teacherList) {
                TeacherRespDto dto = new TeacherRespDto();
                dto.setId(teacher.getUserId());
                dto.setName(teacher.getRealName() == null ? "" : teacher.getRealName());
                if (teacher.getAvatar() == null || teacher.getAvatar() == 0
                    || StringUtils.isEmpty(avatarMap.get(teacher.getAvatar()))) {
                    dto.setAvatarUrl(AvatarConstants.STUFF_AVATAR_URL);
                } else {
                    dto.setAvatarUrl(avatarMap.get(teacher.getAvatar()));
                }
                dto.setMobile(teacher.getMobile());
                result.add(dto);
            }
            return result;
        }
        return Collections.emptyList();
    }

    @Override
    public List<TeacherDto> selectTeacher(Long orgId, List<Long> classRoomIds, List<Long> courseIds, Integer type) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "机构id非法");
        // 查询有冲突的课节
        List<Long> confictLessonIds = Lists.newArrayList();
        if (type == 1) {
            List<OrgLessonConflict> confictLessons =
                orgLessonConflictDao.queryConfilctByClassRoomIds(orgId, classRoomIds);
            confictLessonIds = BaseUtils.getPropertyCollections(confictLessons, "lessonId");
        }
        // 查询指定的课节信息
        List<Long> classLessonIds = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(classRoomIds)) {
            List<OrgClassLesson> classLessons =
                orgClassLessonDao.queryByClassRooms(orgId, classRoomIds, courseIds, confictLessonIds);
            classLessonIds = BaseUtils.getPropertyCollections(classLessons, "id");
        }
        // 所有课节的老师
        List<OrgTeacherLesson> teacherLessons = orgTeacherLessonDao.queryByLessonIds(orgId, classLessonIds);
        List<Long> teacherIds = BaseUtils.getPropertyCollections(teacherLessons, "teacherId");
        // 老师信息列表
        List<Teacher> teachers = teacherDao.getByIds(teacherIds);
        this.filterSoftDelTeacherService.filterTeachers(orgId, teachers);
        return TeacherDto.converToDto(teachers);
    }

}
