
/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2017 All Rights Reserved.
 */

package com.baijia.tianxiao.sal.common.impl;

import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.bson.BasicBSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.baijia.tianxiao.beanCopy.BeanInvokeUtils;
import com.baijia.tianxiao.constant.OrgTeacherStatus;
import com.baijia.tianxiao.dal.activity.dao.article.OrgTeacherDocumentMongoDbDao;
import com.baijia.tianxiao.dal.activity.mongo.MongoDBOperator;
import com.baijia.tianxiao.dal.org.dao.OrgLessonCommentDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentLessonDao;
import com.baijia.tianxiao.dal.org.dao.OrgTeacherDao;
import com.baijia.tianxiao.dal.org.dao.OrgTeacherLessonDao;
import com.baijia.tianxiao.dal.org.po.OrgTeacher;
import com.baijia.tianxiao.dal.solr.dto.OrgTeacherDocument;
import com.baijia.tianxiao.dal.user.dao.TeacherDao;
import com.baijia.tianxiao.dal.user.po.Teacher;
import com.baijia.tianxiao.sal.common.api.OrgTeacherForSolrService;
import com.baijia.tianxiao.util.CollectorUtil;
import com.baijia.tianxiao.util.GenericsUtils;
import com.baijia.tianxiao.util.ThreeTuple;
import com.baijia.tianxiao.util.TupleUtil;
import com.baijia.tianxiao.util.TwoTuple;
import com.baijia.tianxiao.util.date.DateUtil;
import com.google.common.collect.Lists;
import com.google.common.base.Function;
import com.google.common.collect.Maps;

import lombok.extern.slf4j.Slf4j;

/**
 * @say little Boy, don't be sad.
 * @name Rezar
 * @time Feb 20, 2017
 * @Desc this guy is too lazy, nothing left.
 */
@Service
@Slf4j
public class OrgTeacherForSolrServiceImpl implements OrgTeacherForSolrService {

    public static final String mongoDbBackupCollection = OrgTeacherDocumentMongoDbDao.COLLECTION;
    public static final String UNIQUE_KEY = "id";

    @Autowired
    private OrgLessonCommentDao orgLessonCommentDao;
    @Autowired
    private TeacherDao teacherDao;
    @Autowired
    private OrgTeacherLessonDao orgTeacherLessonDao;
    @Autowired
    private OrgStudentLessonDao orgStudentLessonDao;
    @Autowired
    private OrgTeacherDao orgTeacherDao;
    @Autowired
    private MongoDBOperator mongoDbOperator;

    @Override
    public Map<Long, OrgTeacherDocument> createOrgTeacherDocuments(Long orgId, Collection<OrgTeacher> orgTeachers,
        Date beginDate, Date endDate, boolean isSaveOrUpdate) {

        if (GenericsUtils.isNullOrEmpty(orgTeachers)) {
            log.info(" empty OrgTeacher  :{} ", orgTeachers);
            return GenericsUtils.emptyMap();
        }

        Map<Long, OrgTeacher> orgTeacherMap = CollectorUtil.collectMap(orgTeachers, new Function<OrgTeacher, Long>() {
            @Override
            public Long apply(OrgTeacher teacher) {
                return teacher.getUserId();
            }
        });

        Set<Long> teacherUserIds = orgTeacherMap.keySet();

        List<Teacher> teachers = null;
        try {
            teachers = teacherDao.getByUserIds(teacherUserIds);
        } catch (Exception e) {
            log.error("error :{} ", e);
        }
        if (GenericsUtils.isNullOrEmpty(teachers)) {
            log.info(" can not find any teacher with teacherUserIds :{} ", teacherUserIds);
            return GenericsUtils.emptyMap();
        }

        // 注意,第一次初始化的时间为当前月份
        String dataMonth = com.baijia.tianxiao.util.date.DateUtil.getCurrentYM(beginDate.getTime());
        Map<Long, OrgTeacherDocument> orgTeacherDocumentMap = Maps.newHashMap();
        // 获取所有的评分
        Map<Long, Long> scoreMap = orgLessonCommentDao.getAveCommentByTeacherIds(teacherUserIds);
        log.info("for teacherUserIds , the scoreMap is :{} ", scoreMap);

        Map<Long, List<ThreeTuple<Long, Integer, Integer>>> teacherLessonIds =
            this.orgTeacherLessonDao.divideTeacherLessonIds(Lists.newArrayList(teacherUserIds), beginDate, endDate);

        log.info("teacherLessonIds are :{} ", teacherLessonIds);

        Map<Long, TwoTuple<Integer, Integer>> classTimeInfos = Maps.newHashMap();
        Map<Long, TwoTuple<List<Long>, List<Long>>> lessonIdMap = Maps.newHashMap();
        for (Map.Entry<Long, List<ThreeTuple<Long, Integer, Integer>>> entry : teacherLessonIds.entrySet()) {
            Long teacherId = entry.getKey();
            List<ThreeTuple<Long, Integer, Integer>> values = entry.getValue();
            if (GenericsUtils.notNullAndEmpty(values)) {
                for (ThreeTuple<Long, Integer, Integer> value : values) {
                    Long lessonId = value.first;
                    Integer finishClassTime = value.second;
                    Integer arrangeClassTime = value.three;
                    if (!classTimeInfos.containsKey(teacherId)) {
                        TwoTuple<Integer, Integer> classTimeTuple = TupleUtil.tuple(finishClassTime, arrangeClassTime);
                        classTimeInfos.put(teacherId, classTimeTuple);
                    } else {
                        TwoTuple<Integer, Integer> twoTuple = classTimeInfos.get(teacherId);
                        Integer finishClassTimeSum = twoTuple.first;
                        Integer arrangeClassTimeSum = twoTuple.second;
                        twoTuple = TupleUtil.tuple(finishClassTime + finishClassTimeSum,
                            arrangeClassTime + arrangeClassTimeSum);
                        classTimeInfos.put(teacherId, twoTuple);
                    }
                    List<Long> finishLessonIds = null;
                    List<Long> arrangeLessonIds = null;
                    if (!lessonIdMap.containsKey(teacherId)) {
                        finishLessonIds = Lists.newArrayList();
                        arrangeLessonIds = Lists.newArrayList();
                        TwoTuple<List<Long>, List<Long>> tuple = TupleUtil.tuple(finishLessonIds, arrangeLessonIds);
                        lessonIdMap.put(teacherId, tuple);
                    } else {
                        TwoTuple<List<Long>, List<Long>> twoTuple = lessonIdMap.get(teacherId);
                        finishLessonIds = twoTuple.first;
                        arrangeLessonIds = twoTuple.second;
                    }
                    arrangeLessonIds.add(lessonId);
                    if (finishClassTime > 0) {
                        finishLessonIds.add(lessonId);
                    }
                }
            }
        }

        log.info("lessonIdMap are:{} ", lessonIdMap);

        // 获取老师的学生数据
        Map<Long, TwoTuple<Integer, Integer>> teacherStudentCountMap = findTeacherStudentCountInfoMap(lessonIdMap);
        log.info("for teacherUserIds , the teacherStudentCountMap is :{} ", teacherStudentCountMap);
        // 获取老师的课时数据
        Map<Long, TwoTuple<Integer, Integer>> teacherClassTimeMap = classTimeInfos;
        // findTeacherClassTimeInfoMap(teacherUserIds, beginDate, endDate);
        log.info("for teacherUserIds , the teacherClassTimeMap is :{} ", teacherClassTimeMap);

        for (Teacher teacher : teachers) {
            Long userId = teacher.getUserId();
            OrgTeacher orgTeacher = orgTeacherMap.get(userId);
            log.info("[OrgTeacherForSolrServiceImpl] orgTeacher is :{} ", orgTeacher);
            OrgTeacherDocument otd = OrgTeacherDocument.newInstance(userId, dataMonth);
            otd.setOrgId(orgTeacher.getOrgId());
            otd.setAvatarStorageId(teacher.getAvatar());
            otd.setCreateTime(teacher.getCreatedAt().getTime());
            otd.setDataMonth(dataMonth);
            otd.setMobile(teacher.getMobile());
            otd.setName(teacher.getRealName());
            Long score = scoreMap.get(userId);
            otd.setScoreNum(score == null ? 0 : score);
            otd.setOrgTeacherId(orgTeacher.getId());
            otd.setUseStatus(orgTeacher.getUseStatus());
            if (orgTeacher.getStatus() == OrgTeacherStatus.TERMINATED.getCode()) {
                otd.setUseStatus(2);
            }
            if (orgTeacher.isInUse()) {
                otd.setPauseTime(null);
            } else {
                otd.setPauseTime(orgTeacher.getUpdateTime() == null ? System.currentTimeMillis()
                    : orgTeacher.getUpdateTime().getTime());
            }

            TwoTuple<Integer, Integer> classTimeInfo = teacherClassTimeMap.get(userId);
            if (classTimeInfo != null) {
                Integer finishedClassTime = classTimeInfo.first; // 已完成的课节时长
                Integer arrangeClassTime = classTimeInfo.second; // 安排的课节时长
                otd.setFinishedClassTime(finishedClassTime);
                otd.setArrangeClassTime(arrangeClassTime);
            }

            TwoTuple<Integer, Integer> studentCountInfo = teacherStudentCountMap.get(userId);
            if (studentCountInfo != null) {
                Integer hasSignupStudentCount = studentCountInfo.first; // 已经签到的学员
                Integer arrangeStudentCount = studentCountInfo.second; // 所有排的学员
                otd.setArrangeStudentCount(arrangeStudentCount);
                otd.setHasSignupStudentCount(hasSignupStudentCount);
            }
            orgTeacherDocumentMap.put(userId, otd);
        }
        if (isSaveOrUpdate) {
            log.info("[OrgTeacherDocument] isSaveOrUpdate is true ,so will batch add document to solr");
            this.addNewDocument(orgTeacherDocumentMap.values());
        }
        return orgTeacherDocumentMap;
    }

    @Override
    public Map<Long, OrgTeacherDocument> createOrgTeacherDocuments(Long orgId, List<Long> teacherUserIds,
        Date beginDate, Date endDate, boolean isSaveOrUpdate) {
        List<OrgTeacher> teacherByUserIdsAndOrgId =
            this.orgTeacherDao.getTeacherByUserIdsAndOrgId(orgId, teacherUserIds, null, null);
        return this.createOrgTeacherDocuments(orgId, teacherByUserIdsAndOrgId, beginDate, endDate, isSaveOrUpdate);
    }

    /**
     * 
     * 查找老师当月所有的排课的学生数以及已授的学生数
     * 
     * @param teacherUserIds
     * @param beginDate
     * @param endDate
     * @return
     */
    private Map<Long, TwoTuple<Integer, Integer>> findTeacherStudentCountInfoMap(
        Map<Long, TwoTuple<List<Long>, List<Long>>> lessonIdMaps) {
        List<Long> finishLessonIds = Lists.newArrayList();
        List<Long> arrangeLessonIds = Lists.newArrayList();
        Map<Long, List<Long>> inversionIdMap = Maps.newHashMap();
        for (Map.Entry<Long, TwoTuple<List<Long>, List<Long>>> entry : lessonIdMaps.entrySet()) {
            Long teacherId = entry.getKey();
            TwoTuple<List<Long>, List<Long>> value = entry.getValue();
            finishLessonIds.addAll(value.first);
            arrangeLessonIds.addAll(value.second);
            for (Long lessonId : value.second) {
                GenericsUtils.addListIfNotExists(inversionIdMap, lessonId, teacherId);
            }
        }
        Map<Long, TwoTuple<Integer, Integer>> studentCountMap =
            this.orgStudentLessonDao.countStudentNumsWithLessonIds(finishLessonIds, arrangeLessonIds);

        Map<Long, TwoTuple<Integer, Integer>> retMap = Maps.newHashMap();
        for (Map.Entry<Long, TwoTuple<Integer, Integer>> entry : studentCountMap.entrySet()) {
            Long lessonId = entry.getKey();
            TwoTuple<Integer, Integer> tupleValue = entry.getValue();
            List<Long> teacherIds = inversionIdMap.get(lessonId);
            if (GenericsUtils.notNullAndEmpty(teacherIds)) {
                for (Long teacherId : teacherIds) {
                    if (!retMap.containsKey(teacherId)) {
                        Integer finishStudentCount = tupleValue.first;
                        Integer totalStudentCount = tupleValue.second;
                        TwoTuple<Integer, Integer> countTuple = TupleUtil.tuple(finishStudentCount, totalStudentCount);
                        retMap.put(teacherId, countTuple);
                    } else {
                        Integer finishStudentCount = retMap.get(teacherId).first;
                        Integer totalStudentCount = retMap.get(teacherId).second;
                        retMap.put(teacherId, TupleUtil.tuple(finishStudentCount + tupleValue.first,
                            totalStudentCount + tupleValue.second));
                    }
                }
            }
        }
        return retMap;
    }

    /**
     * @param teacherUserIds
     * @param beginDate
     * @param endDate
     * @return
     */
    @SuppressWarnings("unused")
    private Map<Long, TwoTuple<Integer, Integer>> findTeacherClassTimeInfoMap(List<Long> teacherUserIds, Date beginDate,
        Date endDate) {
        return orgTeacherLessonDao.getTeacherLessonClassTimeInfoByMonth(teacherUserIds, beginDate, endDate);
    }

    @Override
    public void addNewDocument(Collection<OrgTeacherDocument> otds) {
        if (GenericsUtils.isNullOrEmpty(otds)) {
            return;
        }
        // String collection = SolrConstant.ERP_CLASS_ROOM_COLLECTION;
        // for (OrgTeacherDocument otd : otds) {
        // Map<String, Object> valueByField = createValueByField(otd);
        // try {
        // super.add(collection, valueByField);
        // } catch (SolrServerException | IOException e) {
        // GenericsUtils.logErrorAndInfo(log, e, "can not add document:{} to solr", otd);
        // }
        // }
        log.info("[OrgTeacherDocument] success to add docs into solr ==== :{}", otds);
        // 对数据进行备份处理,避免 solr dump之后系统不可用
        backupByMongoDb(otds);
    }

    /**
     * @param otds
     */
    @SuppressWarnings("rawtypes")
    private void backupByMongoDb(Collection<OrgTeacherDocument> otds) {
        try {
            List<String> ids = Lists.newArrayList();
            for (OrgTeacherDocument otd : otds) {
                ids.add(otd.getId());
            }
            BasicBSONObject searcher = new BasicBSONObject();
            searcher.put("$in", ids);
            List<Map> searchObjs = this.mongoDbOperator.searchObjs(mongoDbBackupCollection,
                new BasicBSONObject("id", searcher), Map.class);
            ids.clear();
            for (Map mapValue : searchObjs) {
                ids.add(mapValue.get("id").toString());
            }
            Collection<OrgTeacherDocument> updateOtds = Lists.newArrayList();
            for (Iterator<OrgTeacherDocument> iter = otds.iterator(); iter.hasNext();) {
                OrgTeacherDocument next = iter.next();
                if (ids.contains(next.getId())) {
                    updateOtds.add(next);
                    iter.remove();
                }
            }
            mongoDbOperator.insertDocuments(mongoDbBackupCollection, otds, UNIQUE_KEY);
            log.info("updateOtds are:{} ", updateOtds);
            this.backUpdateByMongoDB(updateOtds);
        } catch (Exception e) {
            GenericsUtils.logErrorAndInfo(log, e, "can not insert many documents {}", otds);
        }
    }

    /**
     * @param otd
     * @return
     */
    @SuppressWarnings("unused")
    private Map<String, Object> createValueByField(OrgTeacherDocument otd) {
        Map<String, Object> copyToMap = BeanInvokeUtils.copyToMap(otd, null);
        log.info("[OrgTeacherDocument] BeanInvoke copy from otd:{} to map:{} ", otd, copyToMap);
        return copyToMap;
    }

    @Override
    public void updateDocument(Collection<OrgTeacherDocument> otds) {
        if (GenericsUtils.isNullOrEmpty(otds)) {
            return;
        }
        List<OrgTeacherDocument> docs = Lists.newArrayList();
        for (Iterator<OrgTeacherDocument> iter = otds.iterator(); iter.hasNext();) {
            OrgTeacherDocument next = iter.next();
            if (next.getId() == null) {
                docs.add(next);
            }
        }
        if (GenericsUtils.notNullAndEmpty(docs)) {
            log.info("find some docuemnt has not id ,so will attempt to create id for those document :{} ", docs);
            for (OrgTeacherDocument otd_ : docs) {
                if (otd_.getTeacherUserId() != null) {
                    String dataMonth = otd_.getDataMonth();
                    if (GenericsUtils.isNullOrEmpty(dataMonth)) {
                        dataMonth = DateUtil.getCurrentYM();
                    }
                    String createId = OrgTeacherDocument.createId(otd_.getTeacherUserId(), dataMonth);
                    otd_.setId(createId);
                }
            }
        }
        this.addNewDocument(otds);
        // 对数据进行备份处理,避免 solr dump之后系统不可用
        this.backUpdateByMongoDB(otds);
    }

    /**
     * @param otds
     */
    private void backUpdateByMongoDB(Collection<OrgTeacherDocument> otds) {
        Map<String, Object> updateQuery = Maps.newHashMap();
        for (OrgTeacherDocument otd : otds) {
            updateQuery.put("id", otd.getId());
            try {
                this.mongoDbOperator.updateDocument(otd, mongoDbBackupCollection, updateQuery);
            } catch (Exception e) {
                log.info("can not update otd :{} cause by :{} ", otd, e);
            }
            updateQuery.clear();
        }
        log.info("update all orgTeacherDocument over ! ----- ");
    }

    public static void main(String[] args) {
        Date currentTime = new Date();
        Calendar c = Calendar.getInstance();
        c.setTime(currentTime);
        c.add(Calendar.MONTH, 1);
        Date time1 = c.getTime();
        Date time2 = currentTime;
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-01 00:00:00");
        System.out.println(format.format(time1));
        System.out.println(format.format(time2));
    }

}
