
/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2016 All Rights Reserved.
 */

package com.baijia.tianxiao.sal.comment.service.impl;

import com.baijia.tianxiao.consants.UserRole;
import com.baijia.tianxiao.dal.org.dao.AreaDao;
import com.baijia.tianxiao.dal.org.dao.OrgClassLessonDao;
import com.baijia.tianxiao.dal.org.dao.OrgCommentAuditDao;
import com.baijia.tianxiao.dal.org.dao.OrgCourseDao;
import com.baijia.tianxiao.dal.org.dao.OrgCourseTeacherDao;
import com.baijia.tianxiao.dal.org.dao.OrgInfoDao;
import com.baijia.tianxiao.dal.org.dao.OrgLessonCommentDao;
import com.baijia.tianxiao.dal.org.dao.OrgStudentLessonDao;
import com.baijia.tianxiao.dal.org.dao.OrgTeacherLessonDao;
import com.baijia.tianxiao.dal.org.po.Area;
import com.baijia.tianxiao.dal.org.po.CommentAudit;
import com.baijia.tianxiao.dal.org.po.OrgClassLesson;
import com.baijia.tianxiao.dal.org.po.OrgCommentAudit;
import com.baijia.tianxiao.dal.org.po.OrgInfo;
import com.baijia.tianxiao.dal.sync.constant.MsgSyncType;
import com.baijia.tianxiao.dal.sync.dao.TxMsgSyncTimestampDao;
import com.baijia.tianxiao.dal.sync.po.TxMsgSyncTimestamp;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.comment.constant.Biz;
import com.baijia.tianxiao.sal.comment.constant.CommentErrorCode;
import com.baijia.tianxiao.sal.comment.dto.CommentAuditDto;
import com.baijia.tianxiao.sal.comment.dto.OrgCommentAuditDto;
import com.baijia.tianxiao.sal.comment.dto.OrgStudentCommentAuditDto;
import com.baijia.tianxiao.sal.comment.service.LessonCommentAudiService;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.util.ArithUtil;
import com.baijia.tianxiao.util.collection.CollectorUtil;
import com.baijia.tianxiao.util.date.DateUtil;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Resource;

import lombok.extern.slf4j.Slf4j;

/**
 * @author shanyu
 * @version 1.0
 * @title LessonCommentAudiServiceImpl
 * @desc TODO
 * @date 2016年1月3日
 */
@Slf4j
@Service
public class LessonCommentAudiServiceImpl implements LessonCommentAudiService {

    @Resource
    private OrgLessonCommentDao orgLessonCommentDao;

    @Resource
    private OrgClassLessonDao orgClassLessonDao;

    @Resource
    private OrgStudentLessonDao OrgStudentLessonDao;

    @Resource
    private OrgTeacherLessonDao orgTeacherLessonDao;

    @Resource
    private OrgCourseTeacherDao orgCourseTeacherDao;

    @Resource
    private OrgCourseDao orgCourseDao;

    @Resource
    private OrgInfoDao orgInfoDao;

    @Resource
    private OrgCommentAuditDao orgCommentAuditDao;

    @Resource
    private AreaDao areaDao;

    @Resource
    private TxMsgSyncTimestampDao txMsgSyncTimestampDao;

    @Override
    public CommentAuditDto getLessonCommentAudit(Long lessonId, Long orgId) {
        log.info("audit lesson commet");
        CommentAudit commentAudit = this.orgLessonCommentDao.getLessonCommentAudit(lessonId, orgId, false);
        Integer studentCount = this.OrgStudentLessonDao.queryLessonStudentCount(orgId, lessonId);
        return buildCommentInfoDto(commentAudit, studentCount, lessonId);
    }

    private CommentAuditDto buildCommentInfoDto(CommentAudit commentAudit, Integer studentCount, Long id) {
        CommentAuditDto dto = new CommentAuditDto();
        dto.setId(id);
        if (commentAudit != null && studentCount != null && studentCount.intValue() > 0 && commentAudit.getNum() > 0) {
            dto.setCount(commentAudit.getNum());
            if (dto.getCount().intValue() > 0) {
                if (commentAudit.getTotalScore() == null) {
                    dto.setScore(0d);
                } else {
                    dto.setScore(
                            ArithUtil.round(((double) commentAudit.getTotalScore()) / (commentAudit.getNum() * 100), 1));
                }
                dto.setRate(((double) commentAudit.getNum()) / studentCount);
            } else {
                dto.setScore(0d);
                dto.setRate(0d);
            }
        } else {
            dto.setCount(0);
            dto.setRate(0d);
            dto.setScore(0d);
        }
        return dto;
    }

    @Override
    @Transactional(readOnly = true)
    public CommentAuditDto getTeacherCommentAudit(Long teacherId, Long orgId) {
        Date now = new Date();
        log.info("audit teacher comment, teacherId={}, orgId={}, page={}", teacherId, orgId);
        List<Long> lessonIds = this.orgTeacherLessonDao.queryTeacherLessons(orgId, teacherId);
        List<Long> courseIds = this.orgCourseTeacherDao.getTeacherCourseIds(teacherId, null);
        lessonIds = this.orgClassLessonDao.filterLessonListByTime(courseIds, lessonIds, now); // 已经结束的课节
        Integer studentCount = this.OrgStudentLessonDao.queryLessonStudentCount(orgId, lessonIds);
        CommentAudit commentAudit = this.orgLessonCommentDao.getCommentAudit(lessonIds, orgId, false);
        log.debug("teacher comment, commentAudit={}", commentAudit);
        return buildCommentInfoDto(commentAudit, studentCount, teacherId);
    }

    @Override
    @Transactional(readOnly = true)
    public List<CommentAuditDto> getTeacherCourseCommentAudit(Long teacherId, Long orgId, Boolean isSystem,
                                                              Integer userRole, PageDto pageDto) {
        log.info("audit teacher course comment, teacherId={}, orgId={}, page={}", teacherId, orgId, pageDto);
        List<CommentAuditDto> responseDto = Lists.newArrayList();
        List<Long> courseIds = this.orgCourseTeacherDao.getTeacherCourseIds(teacherId, pageDto);
        List<Long> lessonIds = this.orgTeacherLessonDao.queryTeacherLessons(orgId, teacherId);
        lessonIds = this.orgClassLessonDao.filterLessonListByTime(courseIds, lessonIds, new Date());
        log.debug("courseIds={},lessonIds={}", courseIds, lessonIds);
        if (CollectionUtils.isNotEmpty(lessonIds)) {
            List<OrgClassLesson> lessons =
                    this.orgClassLessonDao.queryLessons(orgId, courseIds, null, lessonIds, null, null, null, null);
            List<CommentAudit> commentAudits =
                    this.orgLessonCommentDao.getLessonsCommentAudit(lessonIds, orgId, userRole, isSystem);
            log.debug("audit teacher course, commentAudits={}", commentAudits);
            Map<Long, CommentAudit> commentMap = getCommentMap(commentAudits);
            Map<Long, Integer> studentCountMap = this.OrgStudentLessonDao.queryLessonStudentCountMap(orgId, lessonIds);
            Map<Long, List<OrgClassLesson>> courseLessonMap = getCourseLessonMap(lessons);
            Map<Long, String> courseNameMap = this.orgCourseDao.getCourseNameMap(courseLessonMap.keySet());
            log.debug("studentCountMap={}", studentCountMap);
            log.debug("courseLessonMap={}", courseLessonMap);
            log.debug("courseNameMap={}", courseNameMap);
            for (Long courseId : courseLessonMap.keySet()) {
                CommentAuditDto dto = new CommentAuditDto();
                dto.setId(courseId);
                dto.setName(courseNameMap.get(courseId));
                dto.setTimeoutCount(0);
                dto.setWaitCount(0);
                dto.setCount(0);
                Integer studentCount = getStudentCount(studentCountMap, courseId, courseLessonMap);
                if (studentCount != null && studentCount.intValue() > 0) {
                    getCommentCountAndScore(commentMap, courseId, courseLessonMap, dto, studentCountMap);
                    dto.setRate(((double) dto.getCount()) / studentCount);
                } else {
                    dto.setCount(0);
                    dto.setRate(0d);
                    dto.setScore(0d);
                }
                responseDto.add(dto);
            }
        }
        return responseDto;
    }

    private void getCommentCountAndScore(Map<Long, CommentAudit> commentMap, Long courseId,
                                         Map<Long, List<OrgClassLesson>> courseLessonMap, CommentAuditDto dto, Map<Long, Integer> studentCountMap) {
        Integer score = 0;
        Integer count = 0;
        Date time = DateUtil.getDayDiff(-7);
        List<OrgClassLesson> lessons = courseLessonMap.get(courseId);
        if (CollectionUtils.isNotEmpty(lessons)) {
            for (OrgClassLesson lesson : lessons) {
                CommentAudit commentAudit = commentMap.get(lesson.getId());
                if (commentAudit != null) {
                    count += commentAudit.getNum();
                    int studentCount = 0;
                    if (studentCountMap.get(lesson.getId()) != null) {
                        studentCount = studentCountMap.get(lesson.getId());
                    }
                    if (commentAudit.getTotalScore() != null) {
                        score += commentAudit.getTotalScore();
                    }
                    if (studentCount > 0 && studentCount >= commentAudit.getNum()) {
                        if (lesson.getEndTime().before(time)) {// 失效
                            dto.setWaitCount(dto.getWaitCount() + studentCount - commentAudit.getNum());
                        } else {
                            dto.setTimeoutCount(dto.getTimeoutCount() + studentCount - commentAudit.getNum());
                        }

                    }
                }
            }
        }
        dto.setCount(count);
        if (count.intValue() > 0) {
            dto.setScore(ArithUtil.round(score.doubleValue() / (count * 100), 1));
        } else {
            dto.setScore(0d);
        }
    }

    private Integer getStudentCount(Map<Long, Integer> studentCountMap, Long courseId,
                                    Map<Long, List<OrgClassLesson>> courseLessonMap) {
        Integer count = 0;
        List<OrgClassLesson> lessons = courseLessonMap.get(courseId);
        if (CollectionUtils.isNotEmpty(lessons)) {
            for (OrgClassLesson lesson : lessons) {
                Integer countValue = studentCountMap.get(lesson.getId());
                if (countValue != null) {
                    count += countValue;
                }
            }
        }
        return count;
    }

    private Map<Long, CommentAudit> getCommentMap(List<CommentAudit> commentAudits) {
        return CollectorUtil.collectMap(commentAudits, new Function<CommentAudit, Long>() {
            @Override
            public Long apply(CommentAudit arg0) {
                return arg0.getLessonId();
            }

        });
    }

    private Map<Long, List<OrgClassLesson>> getCourseLessonMap(List<OrgClassLesson> lessons) {
        return CollectorUtil.group(lessons, new Function<OrgClassLesson, Long>() {
            @Override
            public Long apply(OrgClassLesson arg0) {
                return arg0.getCourseId();
            }

        });
    }

    @Override
    @Transactional(readOnly = true)
    public List<CommentAuditDto> getTeacherAuditList(Long orgId, Collection<Long> teacherIds, Integer userRole,
                                                     Boolean isSystem) {
        if (CollectionUtils.isEmpty(teacherIds)) {
            return Lists.newArrayList();
        }
        log.debug("orgId={}, teacherIds={}", orgId, teacherIds);
        List<CommentAuditDto> result = Lists.newArrayList();
        Date endTime = DateUtil.getDayDiff(-7);
        // 未过期
        Map<Long, CommentAudit> commentCountMap =
                this.orgLessonCommentDao.getTeacherCommentCountMap(orgId, teacherIds, userRole, endTime, isSystem);
        Map<Long, Integer> studentCountMap =
                this.OrgStudentLessonDao.getStudentCountOfTeacher(orgId, teacherIds, endTime);
        log.debug("commentCountMap={}", commentCountMap);
        log.debug("studentCountMap={}", studentCountMap);

        // 全部
        Map<Long, CommentAudit> commentCountMapAll =
                this.orgLessonCommentDao.getTeacherCommentCountMap(orgId, teacherIds, userRole, null, isSystem);
        Map<Long, Integer> studentCountMapAll =
                this.OrgStudentLessonDao.getStudentCountOfTeacher(orgId, teacherIds, null);
        log.debug("commentCountMapAll={}", commentCountMapAll);
        log.debug("studentCountMapAll={}", studentCountMapAll);

        for (Long teacherId : teacherIds) {
            CommentAuditDto dto = new CommentAuditDto();
            dto.setId(teacherId);
            int commentCount =
                    commentCountMapAll.get(teacherId) == null ? 0 : commentCountMapAll.get(teacherId).getNum(); // 评价数
            int studentCount = studentCountMapAll.get(teacherId) == null ? 0 : studentCountMapAll.get(teacherId); // 总学生数
            int totalScore =
                    commentCountMapAll.get(teacherId) == null ? 0 : commentCountMapAll.get(teacherId).getTotalScore(); // 总分

            int commentTimeIn = commentCountMap.get(teacherId) == null ? 0 : commentCountMap.get(teacherId).getNum(); // 有效期评价数
            int studentTimeIn = studentCountMap.get(teacherId) == null ? 0 : studentCountMap.get(teacherId); // 有效期学生数

            dto.setId(teacherId);
            if (studentCount > 0) {
                dto.setCount(commentCount);// 评价数
                if (commentCount > 0) {
                    dto.setScore(ArithUtil.round(((double) totalScore / (commentCount * 100)), 1));// 得分
                }
                dto.setWaitCount(studentTimeIn - commentTimeIn);// 待评价
                dto.setTimeoutCount(studentCount - commentCount - dto.getWaitCount());// 已失效
                dto.setRate((double) commentCount / studentCount);// 评价率
                result.add(dto);
            }
        }
        return result;
    }

    @Override
    public OrgCommentAuditDto getOrgCommentAudit(Long orgId, Boolean withSystem) {
        Preconditions.checkArgument(orgId != null, "orgId is null!");
        OrgCommentAuditDto dto = new OrgCommentAuditDto();
        OrgInfo orgInfo = this.orgInfoDao.getOrgInfo(orgId.intValue());
        Area area = this.areaDao.getById(orgInfo.getAreaId());
        dto.setAreaName(area.getName());
        Map<Integer, Integer> commentCountMap = this.orgLessonCommentDao.getCommentCountOfOrg(orgId, withSystem, null);
        Map<Integer, Integer> lowScoreMap =
                this.orgLessonCommentDao.getCommentCountOfOrg(orgId, withSystem, Biz.LOW_SCORE * 100);

        Map<Integer, Integer> commentCountMapWithOutDefault =
                this.orgLessonCommentDao.getCommentCountOfOrg(orgId, false, null);
        Map<Integer, Integer> lowScoreMapWithOutDefault =
                this.orgLessonCommentDao.getCommentCountOfOrg(orgId, false, Biz.LOW_SCORE * 100);

        log.debug("commentCountMap={},lowScoreMap={}", commentCountMap, lowScoreMap);
        log.debug("commentCountMapWithOutDefault={},lowScoreMapWithOutDefault={}", commentCountMapWithOutDefault,
                lowScoreMapWithOutDefault);

        dto.setStudentCommentCount(commentCountMap.get(UserRole.STUDENT.getRole()) == null ? 0
                : commentCountMap.get(UserRole.STUDENT.getRole()));
        dto.setTeacherCommentCount(commentCountMap.get(UserRole.TEACHER.getRole()) == null ? 0
                : commentCountMap.get(UserRole.TEACHER.getRole()));
        Integer studentCount = this.OrgStudentLessonDao.getFinishLessonStudentCount(orgId);
        log.debug("studentCount={}", studentCount);
        if (studentCount > 0) {
            // 不带默认评价
            int studentCommentCount = commentCountMapWithOutDefault.get(UserRole.STUDENT.getRole()) == null ? 0
                    : commentCountMapWithOutDefault.get(UserRole.STUDENT.getRole());
            int teacherCommentCount = commentCountMapWithOutDefault.get(UserRole.TEACHER.getRole()) == null ? 0
                    : commentCountMapWithOutDefault.get(UserRole.TEACHER.getRole());

            int lowScoreStudentCount = lowScoreMapWithOutDefault.get(UserRole.STUDENT.getRole()) == null ? 0
                    : lowScoreMapWithOutDefault.get(UserRole.STUDENT.getRole());

            dto.setStudentCommentRate(ArithUtil.round((double) studentCommentCount / studentCount, 4));
            if (studentCommentCount > 0) {
                dto.setStudentLowRate(ArithUtil.round((double) lowScoreStudentCount / studentCommentCount, 4));
            } else {
                dto.setStudentLowRate(0d);
            }
            dto.setTeacherCommentRate(ArithUtil.round((double) teacherCommentCount / studentCount, 4));
            OrgCommentAudit audit = orgCommentAuditDao.getCommentAuditByOrgId(orgId);
            log.debug("audit={}", audit);
            if (audit != null) {
                dto.setDate(DateUtil.getStrByDate(DateUtil.getDiffDateTime(audit.getUpdateTime(), -1)));
                if (withSystem != null && !withSystem && audit.getCommentCount() > 0) {
                    Integer rank = this.orgCommentAuditDao.getRank(orgId, withSystem);
                    dto.setOrgScore(ArithUtil.round(((double) audit.getScore() / 100), 2));
                    dto.setRank(rank);
                } else if (withSystem == null && audit.getCommentCountAll() > 0) {
                    Integer rank = this.orgCommentAuditDao.getRank(orgId, true);
                    dto.setOrgScore(ArithUtil.round(((double) audit.getScoreAll() / 100), 2));
                    dto.setRank(rank);
                }
            }
        } else {
            dto.setStudentCommentRate(0d);
            dto.setStudentLowRate(0d);
            dto.setTeacherCommentRate(0d);
        }
        if (dto.getRank() == null) {
            throw new BussinessException(CommentErrorCode.COMMENT_RANK_NULL);
        }
        return dto;
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void syncUpdateOrgCommentAudit() {
        TxMsgSyncTimestamp syncInfo =
                this.txMsgSyncTimestampDao.getSyncTimestampByType(MsgSyncType.COMMENT_AUDIT.getSyncType());
        List<OrgCommentAudit> addAudits = Lists.newArrayList(); // 新增统计记录
        List<OrgCommentAudit> updateAudits = Lists.newArrayList(); // 更新统计记录
        Date startTime = null;// 上一次同步时间
        Date endTime = DateUtil.getCurrentDate();// 当天凌晨
        List<OrgCommentAudit> newCommentAuditList = null;
        List<OrgCommentAudit> newCommentAuditListAll = null;
        if (syncInfo == null) {
            syncInfo = new TxMsgSyncTimestamp();
            syncInfo.setSyncType(MsgSyncType.COMMENT_AUDIT.getSyncType());
            syncInfo.setSyncTime(endTime);
            this.txMsgSyncTimestampDao.save(syncInfo, false);
            newCommentAuditList = this.orgLessonCommentDao.getOrgCommentAudit(null, endTime, false);
            newCommentAuditListAll = this.orgLessonCommentDao.getOrgCommentAudit(null, endTime, null);
        } else {
            startTime = syncInfo.getSyncTime();// 上一次同步时间
            syncInfo.setSyncTime(endTime);
            this.txMsgSyncTimestampDao.update(syncInfo, "syncTime");
            newCommentAuditList = this.orgLessonCommentDao.getOrgCommentAudit(startTime, endTime, false);
            newCommentAuditListAll = this.orgLessonCommentDao.getOrgCommentAudit(startTime, endTime, null);
        }
        log.info("new commentAudit={}, startTime={},endTime={}", newCommentAuditList, startTime, endTime);
        log.info("new newCommentAuditListAll={}, startTime={},endTime={}", newCommentAuditListAll, startTime, endTime);
        Map<Long, OrgCommentAudit> newCommentAuditMap =
                CollectorUtil.collectMap(newCommentAuditList, new Function<OrgCommentAudit, Long>() {
                    @Override
                    public Long apply(OrgCommentAudit arg0) {
                        return arg0.getOrgId();
                    }
                });
        if (CollectionUtils.isNotEmpty(newCommentAuditListAll)) {
            Set<Integer> orgIdInts = Sets.newHashSet();
            Set<Long> orgIds = Sets.newHashSet();
            for (OrgCommentAudit audit : newCommentAuditListAll) {
                orgIds.add(audit.getOrgId());
                orgIdInts.add(audit.getOrgId().intValue());
            }
            Map<Integer, Long> areaIdMap = this.orgInfoDao.getAreaMap(orgIdInts);
            Map<Long, OrgCommentAudit> auditMap = this.orgCommentAuditDao.getCommentAuditMapByOrgIds(orgIds);

            for (OrgCommentAudit audit : newCommentAuditListAll) {
                if (auditMap.get(audit.getOrgId()) != null) {// 数据已经存在
                    OrgCommentAudit updateAudit = auditMap.get(audit.getOrgId());
                    updateAudit.setCommentCountAll(updateAudit.getCommentCountAll() + audit.getCommentCountAll());
                    updateAudit.setTotalScoreAll(updateAudit.getTotalScoreAll() + audit.getTotalScoreAll());
                    if (updateAudit.getCommentCountAll() > 0) {
                        updateAudit.setScoreAll(updateAudit.getTotalScoreAll() / updateAudit.getCommentCountAll());
                    } else {
                        updateAudit.setScoreAll(0);
                    }
                    OrgCommentAudit auditWithoutSystem = newCommentAuditMap.get(audit.getOrgId());
                    if (auditWithoutSystem != null) {
                        updateAudit
                                .setCommentCount(updateAudit.getCommentCount() + auditWithoutSystem.getCommentCount());
                        updateAudit.setTotalScore(updateAudit.getTotalScore() + auditWithoutSystem.getTotalScore());
                        if (updateAudit.getCommentCount() > 0) {
                            updateAudit.setScore(updateAudit.getTotalScore() / updateAudit.getCommentCount());
                        } else {
                            updateAudit.setScore(0);
                        }
                    }
                    updateAudits.add(updateAudit);
                } else {
                    audit.setAreaId(areaIdMap.get(audit.getOrgId().intValue()));
                    if (audit.getCommentCountAll() > 0) {
                        audit.setScoreAll(audit.getTotalScoreAll() / audit.getCommentCountAll());
                    } else {
                        audit.setScoreAll(0);
                    }
                    OrgCommentAudit auditWithoutSystem = newCommentAuditMap.get(audit.getOrgId());
                    if (auditWithoutSystem != null) {
                        audit.setCommentCount(auditWithoutSystem.getCommentCount());
                        audit.setTotalScore(auditWithoutSystem.getTotalScore());
                        if (audit.getCommentCount() > 0) {
                            audit.setScore(audit.getTotalScore() / audit.getCommentCount());
                        } else {
                            audit.setScore(0);
                        }
                    } else {
                        audit.setCommentCount(0);
                        audit.setScore(0);
                        audit.setTotalScore(0);
                    }
                    addAudits.add(audit);
                }
            }
            log.info("update commentAudit updateAudits={}", updateAudits);
            log.info("add commentAudit addAudits={}", addAudits);
            // 更新信息
            if (CollectionUtils.isNotEmpty(updateAudits)) {
                this.orgCommentAuditDao.updateAllByOrgId(updateAudits);
            }
            // 添加信息
            if (CollectionUtils.isNotEmpty(addAudits)) {
                this.orgCommentAuditDao.saveAll(addAudits, "orgId", "areaId", "score", "totalScore", "commentCount",
                        "scoreAll", "totalScoreAll", "commentCountAll");
            }
            
            
            this.orgCommentAuditDao.updateAuditTime(orgIds, endTime);

        }
    }

    @Override
    public OrgStudentCommentAuditDto getOrgCommentAudit(Long orgId, Collection<Long> courseIds,
                                                        Collection<Long> teacherIds, Boolean isSystem, Integer userRole) {
        Preconditions.checkArgument(orgId != null, "orgId is null!");
        log.info("audit comments of students with param: courseIds={},teacherIds={}, isSystem={}", courseIds,
                teacherIds, isSystem);
        List<Long> lessonIds = null;
        if (CollectionUtils.isNotEmpty(teacherIds) || CollectionUtils.isNotEmpty(courseIds)) {
            lessonIds = this.orgClassLessonDao.filterLessonList(orgId, courseIds, teacherIds);
        }
        OrgStudentCommentAuditDto response = new OrgStudentCommentAuditDto();
        response.setAll(this.orgLessonCommentDao.getCommentCount(orgId, lessonIds, userRole, null, null, null, isSystem));//所有学员评价数

        response.setWithPic(this.orgLessonCommentDao.getCommentCount(orgId, lessonIds, userRole, null, true, null, isSystem));// 有图评价数

        response.setLowScore(this.orgLessonCommentDao.getCommentCount(orgId, lessonIds, userRole, Biz.LOW_SCORE * 100, null, null, isSystem));// 低分评价数

        response.setHighScore(this.orgLessonCommentDao.getCommentCount(orgId, lessonIds, userRole, null, null,
                Biz.HIGH_SCORE * 100, isSystem));// 高分评价数
        log.info("audit comments of students result ={}", response);
        return response;
    }

    public static void main(String[] args) {
        Date endTime = DateUtil.getCurrentDate();// 当天凌晨
        System.out.println(endTime);
    }

}
