package com.baijia.tianxiao.sal.marketing.commons.utils;

import com.aliyun.mns.client.*;
import com.aliyun.mns.model.*;
import com.baijia.tianxiao.dal.push.constant.MessageSource;
import com.baijia.tianxiao.sal.marketing.commons.exceptions.BusinessException;
import com.baijia.tianxiao.util.properties.PropertiesReader;
import lombok.extern.slf4j.Slf4j;

import java.util.UUID;
import java.util.Vector;

/**
 * Created by liuxp on 16/9/10.
 */
@Slf4j
public class MnsUtils {
    public static String getPrefix() {
        return MessageSource.SIGNUP.getValue() + "$";
    }

    //队列消息的长轮询等待时间，单位是秒
    public static final int POLLING_WAIT_SECONDS = 15;

    //队列消息的最大长度，单位是byte
    private static final long MAX_MSG_SIZE = 2048L;

    private MNSClient client;

    private static MnsUtils mnsUtils;

    private MnsUtils() {
        CloudAccount account = new CloudAccount(
                PropertiesReader.getValue("mns", "mns.accesskeyid"),
                PropertiesReader.getValue("mns", "mns.accesskeysecret"),
                PropertiesReader.getValue("mns", "mns.accountendpoint"));
        client = account.getMNSClient();
    }

    public synchronized static MnsUtils getInstance() {
        if (mnsUtils == null) {
            mnsUtils = new MnsUtils();
        }
        return mnsUtils;
    }

    /**
     * 获取积分的队列
     *
     * @return
     */
    public static CloudQueue getPointsQueue() {
        MnsUtils mnsUtils = MnsUtils.getInstance();
        String subscribeQueueName = PropertiesReader.getValue("mns", "points.msg.queue");
        return mnsUtils.getAndCreateQueue(subscribeQueueName);
    }

    /**
     * 创建队列
     *
     * @param queueName
     * @return
     */
    public CloudQueue getAndCreateQueue(String queueName) {
        CloudQueue queue = client.getQueueRef(queueName);
        if (queue == null) {
            QueueMeta meta = new QueueMeta();
            meta.setQueueName(queueName);
            meta.setPollingWaitSeconds(POLLING_WAIT_SECONDS);
            meta.setMaxMessageSize(MAX_MSG_SIZE);
            queue = client.createQueue(meta);
        }
        return queue;
    }

    /**
     * 创建topic
     *
     * @param topicName
     * @return
     */
    public CloudTopic    getOrCreateTopic(String topicName, String subscribeQueueName) {
        CloudTopic cloudTopic = client.getTopicRef(topicName);
        if (cloudTopic == null) {
            TopicMeta topicMeta = new TopicMeta();
            topicMeta.setTopicName(topicName);

            Vector<String> consumerNameList = new Vector<String>();
            consumerNameList.add(subscribeQueueName);

            QueueMeta queueMetaTemplate = new QueueMeta();
            queueMetaTemplate.setPollingWaitSeconds(POLLING_WAIT_SECONDS);

            CloudPullTopic cloudPullTopic = client.createPullTopic(topicMeta, consumerNameList, true, queueMetaTemplate);
            cloudTopic = cloudPullTopic.getRawTopic();
        }
        return cloudTopic;
    }

    /**
     * 发布消息
     *
     * @param topicName
     * @param messageBody
     * @return
     */
    public TopicMessage publishMessage(String topicName, String subscribeQueueName, String messageBody) {
        CloudTopic cloudTopic = getOrCreateTopic(topicName, subscribeQueueName);
        if (cloudTopic != null) {
            TopicMessage tMessage = new RawTopicMessage();
            tMessage.setBaseMessageBody(messageBody);
            return cloudTopic.publishMessage(tMessage);
        } else {
            throw new BusinessException("mns主题不存在, topicName=" + topicName);
        }
    }

    /**
     * 发送队列消息
     *
     * @param queueName
     * @param messageBody
     */
    public void sendMessage(String queueName, String messageBody) {
        CloudQueue queue = getAndCreateQueue(queueName);
        String msgId = UUID.randomUUID().toString();

        // 发送消息
        Message message = new Message();
        message.setMessageBody(messageBody);
        message.setRequestId(msgId);

        log.info("sendMessage.queueName:{}, messageBody:{}", queueName, messageBody);
        queue.putMessage(message);
    }

    /**
     * 关闭消息队列连接
     */
    public void doAfter() {
        if (client != null) {
            client.close();
        }
    }

    public MNSClient getClient() {
        return client;
    }
}
