package com.baijia.tianxiao.sal.marketing.commons.utils;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.beanutils.ConvertUtils;
import org.apache.commons.beanutils.PropertyUtilsBean;
import org.apache.commons.beanutils.locale.converters.DateLocaleConverter;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.output.ByteArrayOutputStream;
import org.apache.log4j.Logger;
import org.apache.poi.hssf.usermodel.HSSFCell;
import org.apache.poi.hssf.usermodel.HSSFRow;
import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.DataFormat;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.xssf.usermodel.XSSFCell;
import org.apache.poi.xssf.usermodel.XSSFRow;
import org.apache.poi.xssf.usermodel.XSSFSheet;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;

import com.baijia.tianxiao.util.GenericsUtils;

public class ExcelHelper<T> {
    public static final String OFFICE_EXCEL_2003_POSTFIX = "xls";
    public static final String OFFICE_EXCEL_2010_POSTFIX = "xlsx";
    public static final String EMPTY = "";
    public static final String POINT = ".";
    public static final String EXPORT_DATE_FORMAT = "yyyy-MM-dd HH:mm:ss";
    private static final Logger logger = Logger.getLogger(ExcelHelper.class);

    public static PropertyUtilsBean propertyUtilsBean = new PropertyUtilsBean();

    /**
     * 
     * @param request
     * @param response
     * @param data 要导出的数据
     * @param fields 要导出的字段
     * @param fieldValue 表头
     */
    public void exportToExcel(HttpServletRequest request, HttpServletResponse response, List<T> data, String[] fields,
        String[] fieldValue) {
        exportToExcel(request, response, data, fields, fieldValue, null);
    }

    /**
     * 
     * @param request
     * @param response
     * @param data 要导出的数据
     * @param fields 要导出的字段
     * @param fieldValue 表头
     * @param fileName 导出文件名称
     */
    public void exportToExcel(HttpServletRequest request, HttpServletResponse response, List<T> data, String[] fields,
        String[] fieldValue, String fileName) {
        if (fields == null || fieldValue == null) {
            throw new RuntimeException("fields or header can't be null.");
        }

        Workbook wb = new HSSFWorkbook();
        Sheet sheet = wb.createSheet("ExportDataList");

        // Create a row and put some cells in it. Rows are 0 based.
        Row headerTitle = sheet.createRow(0);
        int index = 0;
        for (int i = 0; i < fields.length; i++) {
            Cell cell = headerTitle.createCell(index++ % fields.length);
            cell.setCellValue(fieldValue[i]);
        }

        for (int i = 0; i < data.size(); i++) {
            Row row = sheet.createRow(i + 1);
            T o = data.get(i);

            for (int j = 0; j < fields.length; j++) {
                Cell cell = row.createCell(index++ % fields.length);

                try {
                    cell.setCellValue(BeanUtils.getProperty(o, fields[j]));
                } catch (Exception e) {
                    throw new RuntimeException("export data error", e);
                }
            }
        }

        if (fileName == null || "".equals(fileName)) {
            fileName = System.currentTimeMillis() + ".xls";
        }

        response.setContentType("application/vnd.ms-excel");
        response.setHeader("Content-Disposition", "attachment; filename=" + fileName);
        try {
            wb.write(response.getOutputStream());
        } catch (IOException e) {
            throw new RuntimeException("export data error", e);
        }
    }

    /**
     * { "_id" : "e5c17efb-da27-482b-87b8-fd252af244f1", "popu" : "3", "orgSourceUrl" :
     * "http://bj.houxue.com/xuexiao/249475.html", "orgName" : "北京市西城区华英职业技能培训", "address" : "", "phone" : "",
     * "courseNum" : "0", "orgJpeg" : "http://img.houxue.com/static/school_logo_example.gif", "map" :
     * "http://bj.houxue.com/ditu/249475.html", "teacherCount" : "0", "createDate" : "0000-00-00", "desc" : "" }
     */


    public static File exportToExcelLocalFileFromMap(List<Map<String, String>> source, String file) {

        if (GenericsUtils.isNullOrEmpty(source)) {
            return null;
        }

        System.out.println(source.size());

        Set<String> headSets = source.get(0).keySet();
        String[] fieldValue = new String[headSets.size()];
        headSets.toArray(fieldValue);
        Workbook wb = new HSSFWorkbook();
        Sheet sheet = wb.createSheet("ExportDataList");

        // Create a row and put some cells in it. Rows are 0 based.
        Row headerTitle = sheet.createRow(0);
        int index = 0;
        for (int i = 0; i < fieldValue.length; i++) {
            Cell cell = headerTitle.createCell(index++ % fieldValue.length);
            cell.setCellValue(fieldValue[i]);
        }

        int i = 0;
        for (Map<String, String> data : source) {
            Row row = sheet.createRow(i + 1);
            for (int j = 0; j < fieldValue.length; j++) {
                Cell cell = row.createCell(index++ % fieldValue.length);
                try {
                    String key = fieldValue[j].toString();
                    cell.setCellValue(data.get(key));
                } catch (Exception e) {
                    logger.info("[fieldValue]" + fieldValue[j]);
                    // throw new RuntimeException("export data error", e);
                }
            }
            i++;
        }

        if (file == null || "".equals(file)) {
            file = System.currentTimeMillis() + ".xls";
        }

        System.out.println(file);
        File localFile = new File(file);
        if (!localFile.exists()) {
            try {
                localFile.createNewFile();
            } catch (IOException e) {
                logger.info(e);
                return null;
            }
        }
        OutputStream out = null;
        BufferedOutputStream bos = null;
        try {
            out = new FileOutputStream(localFile);
            bos = new BufferedOutputStream(out);
            wb.write(bos);
            wb = null;
        } catch (Exception e) {
            logger.info("[ writer file failure : ]", e);
        } finally {
            IOUtils.closeQuietly(bos);
            IOUtils.closeQuietly(out);
        }
        return localFile;
    }

    /**
     *
     * @param data 要导出的数据
     * @param fields 要导出的字段
     * @param fieldValue 表头
     * @param file 导出的文件
     */
    public File exportToExcelLocalFile(List<T> data, String[] fields, String[] fieldValue, String file) {
        if (fields == null || fieldValue == null) {
            throw new RuntimeException("fields or header can't be null.");
        }

        Workbook wb = new HSSFWorkbook();
        Sheet sheet = wb.createSheet("ExportDataList");

        // Create a row and put some cells in it. Rows are 0 based.
        Row headerTitle = sheet.createRow(0);
        int index = 0;
        for (int i = 0; i < fields.length; i++) {
            Cell cell = headerTitle.createCell(index++ % fields.length);
            cell.setCellValue(fieldValue[i]);
        }

        for (int i = 0; i < data.size(); i++) {
            Row row = sheet.createRow(i + 1);
            T o = data.get(i);
            for (int j = 0; j < fields.length; j++) {
                Cell cell = row.createCell(index++ % fields.length);
                try {
                    cell.setCellValue(BeanUtils.getProperty(o, fields[j]));
                } catch (Exception e) {
                    throw new RuntimeException("export data error", e);
                }
            }
        }

        if (file == null || "".equals(file)) {
            file = System.currentTimeMillis() + ".xls";
        }

        File localFile = new File(file);
        OutputStream out = null;
        BufferedOutputStream bos = null;
        try {
            out = new FileOutputStream(localFile);
            bos = new BufferedOutputStream(out);
            wb.write(bos);
            wb = null;
        } catch (Exception e) {
            logger.info("[ writer file failure : ]", e);
        } finally {
            IOUtils.closeQuietly(bos);
            IOUtils.closeQuietly(out);
        }
        return localFile;
    }

    public static List<Map<Integer, String>> readExcel(InputStream is, String fileName) throws Exception {
        String postfix = getPostfix(fileName);
        if (OFFICE_EXCEL_2003_POSTFIX.equals(postfix)) {
            return readXls(is);
        } else if (OFFICE_EXCEL_2010_POSTFIX.equals(postfix)) {
            return readXlsx(is);
        } else {
            return null;
        }
    }

    /**
     * Read the Excel 2010
     *
     * @return
     * @throws IOException
     */
    public static List<Map<Integer, String>> readXlsx(InputStream is) throws IOException {
        XSSFWorkbook xssfWorkbook = new XSSFWorkbook(is);
        List<Map<Integer, String>> list = new ArrayList<Map<Integer, String>>();
        // Read the Sheet
        for (int numSheet = 0; numSheet < xssfWorkbook.getNumberOfSheets(); numSheet++) {
            XSSFSheet xssfSheet = xssfWorkbook.getSheetAt(numSheet);
            if (xssfSheet == null) {
                continue;
            }
            // Read the Row
            for (int rowNum = xssfSheet.getFirstRowNum(); rowNum <= xssfSheet.getLastRowNum(); rowNum++) {
                XSSFRow xssfRow = xssfSheet.getRow(rowNum);
                if (xssfRow != null) {
                    Map<Integer, String> map = new HashMap<Integer, String>();
                    int firstCellNum = xssfRow.getFirstCellNum();
                    int lastCellNum = xssfRow.getLastCellNum();

                    for (int cellNum = firstCellNum; cellNum < lastCellNum; cellNum++) {
                        map.put(cellNum, getValue(xssfRow.getCell(cellNum)));
                    }
                    list.add(map);
                }
            }
        }
        return list;
    }

    /**
     * Read the Excel 2003-2007
     *
     * @param path the path of the Excel
     * @return
     * @throws IOException
     */
    public static List<Map<Integer, String>> readXls(InputStream is) throws IOException {
        HSSFWorkbook hssfWorkbook = new HSSFWorkbook(is);
        List<Map<Integer, String>> list = new ArrayList<Map<Integer, String>>();
        // Read the Sheet
        for (int numSheet = 0; numSheet < hssfWorkbook.getNumberOfSheets(); numSheet++) {
            HSSFSheet hssfSheet = hssfWorkbook.getSheetAt(numSheet);
            if (hssfSheet == null) {
                continue;
            }
            // Read the Row
            for (int rowNum = hssfSheet.getFirstRowNum(); rowNum <= hssfSheet.getLastRowNum(); rowNum++) {
                HSSFRow hssfRow = hssfSheet.getRow(rowNum);
                if (hssfRow != null) {
                    Map<Integer, String> map = new HashMap<Integer, String>();
                    int firstCellNum = hssfRow.getFirstCellNum();
                    int lastCellNum = hssfRow.getLastCellNum();
                    for (int cellNum = firstCellNum; cellNum < lastCellNum; cellNum++) {
                        if (null == hssfRow.getCell(firstCellNum)) {
                            break;
                        }
                        map.put(cellNum, getValue(hssfRow.getCell(cellNum)));
                    }
                    list.add(map);
                }
            }
        }
        return list;
    }

    public static String getCellStringValue(Cell cell) {
        NumberFormat nf = NumberFormat.getInstance();
        nf.setGroupingUsed(false);

        if (cell.getCellType() == Cell.CELL_TYPE_BOOLEAN) {
            return String.valueOf(cell.getBooleanCellValue());
        } else if (cell.getCellType() == Cell.CELL_TYPE_NUMERIC) {
            return nf.format(cell.getNumericCellValue());
        } else {
            return cell.getStringCellValue();
        }
    }

    @SuppressWarnings("static-access")
    private static String getValue(XSSFCell xssfRow) {
        DecimalFormat df = new DecimalFormat("0");
        if (xssfRow.getCellType() == xssfRow.CELL_TYPE_BOOLEAN) {
            return String.valueOf(xssfRow.getBooleanCellValue());
        } else if (xssfRow.getCellType() == xssfRow.CELL_TYPE_NUMERIC) {
            return String.valueOf(df.format(xssfRow.getNumericCellValue()));
        } else {
            return String.valueOf(xssfRow.getStringCellValue());
        }
    }

    @SuppressWarnings("static-access")
    private static String getValue(HSSFCell hssfCell) {
        DecimalFormat df = new DecimalFormat("0");
        if (hssfCell.getCellType() == hssfCell.CELL_TYPE_BOOLEAN) {
            return String.valueOf(hssfCell.getBooleanCellValue());
        } else if (hssfCell.getCellType() == hssfCell.CELL_TYPE_NUMERIC) {
            return String.valueOf(df.format(hssfCell.getNumericCellValue()));
        } else if (hssfCell.getCellType() == hssfCell.CELL_TYPE_FORMULA) {
            return String.valueOf(hssfCell.getCellFormula());
        } else {
            return String.valueOf(hssfCell.getStringCellValue());
        }
    }

    /**
     * get postfix of the path
     *
     * @param path
     * @return
     */
    private static String getPostfix(String path) {
        if (path == null || EMPTY.equals(path.trim())) {
            return EMPTY;
        }
        if (path.contains(POINT)) {
            return path.substring(path.lastIndexOf(POINT) + 1, path.length());
        }
        return EMPTY;
    }

    public static <T> void exportToExcel(HttpServletResponse response, List<T> data, String[] fields,
        String[] fieldValue) {
        exportToExcel(response, data, fields, fieldValue, null);
    }

    public static <T> void exportToExcel(HttpServletResponse response, List<T> data, String[] fields,
        String[] fieldValue, String fileName) {

        if (fileName == null || "".equals(fileName)) {
            fileName = System.currentTimeMillis() + ".xlsx";
        }

        response.setContentType("application/vnd.ms-excel");
        try {
            response.setHeader("Content-Disposition", "attachment; filename=" + URLEncoder.encode(fileName, "utf-8"));
        } catch (UnsupportedEncodingException e1) {
            response.setHeader("Content-Disposition", "attachment; filename=" + fileName);
        }

        try {
            exportToExcel(response.getOutputStream(), data, fields, fieldValue);
        } catch (IOException e) {
            throw new RuntimeException("export data error", e);
        }
    }

    public static <T> void exportToExcel(OutputStream os, List<T> data, String[] fields, String[] fieldValue) {
        if (fields == null || fieldValue == null) {
            throw new RuntimeException("fields or header can't be null.");
        }
        Workbook wb = new XSSFWorkbook();
        Sheet sheet = wb.createSheet("ExportDataList");

        // Create a row and put some cells in it. Rows are 0 based.
        Row headerTitle = sheet.createRow(0);
        int index = 0;
        for (int i = 0; i < fields.length; i++) {
            Cell cell = headerTitle.createCell(index++ % fields.length);
            cell.setCellValue(fieldValue[i]);
            sheet.setColumnWidth(cell.getColumnIndex(), fieldValue[i].length() * 8 * 256);
            cell.getCellStyle().setAlignment(CellStyle.ALIGN_CENTER);
        }

        try {
            for (int i = 0; i < data.size(); i++) {
                Row row = sheet.createRow(i + 1);
                T o = data.get(i);

                for (int j = 0; j < fields.length; j++) {
                    try {
                        createCell(sheet, row, index++ % fields.length, propertyUtilsBean.getProperty(o, fields[j]));
                    } catch (Exception e) {
                        e.printStackTrace();
                        throw new RuntimeException("export data error", e);
                    }

                }
            }
            wb.write(os);
            os.flush();
        } catch (IOException e) {
            throw new RuntimeException("export data error", e);
        } finally {
            // IOUtils.closeQuietly(wb);
        }
    }

    public static <T> ZipOutputStream getZipOutputStream(HttpServletResponse response, String fileName)
        throws IOException {

        if (fileName == null || "".equals(fileName)) {
            fileName = System.currentTimeMillis() + ".zip";
        }

        response.setContentType("application/zip");
        response.setHeader("Content-Disposition", "attachment; filename=" + URLEncoder.encode(fileName, "utf-8"));
        ConvertUtils.register(new DateLocaleConverter(Locale.CHINA), Date.class);
        ZipOutputStream out = new ZipOutputStream(response.getOutputStream());
        return out;
    }

    public static <T> void exportToZip(ZipOutputStream out, List<T> data, String[] fields, String[] fieldValue,
        String fileName) {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        try {
            exportToExcel(baos, data, fields, fieldValue);
            out.putNextEntry(new ZipEntry("./tts_export/" + fileName + ".xlsx"));
            out.write(baos.toByteArray());
            out.flush();
            out.closeEntry();
        } catch (IOException e) {
            throw new RuntimeException("export data error", e);
        } finally {
            IOUtils.closeQuietly(baos);
        }
    }

    public static <T> void closeZipOutputStream(ZipOutputStream out) {
        if (out != null) {
            try {
                out.flush();
                out.close();
            } catch (IOException e) {
            }
        }
    }

    private static void createCell(Sheet sheet, Row row, int column, Object value) {
        Cell cell = row.createCell(column);

        if (value != null) {
            DataFormat format = sheet.getWorkbook().createDataFormat();
            CellStyle cellStyle = sheet.getWorkbook().createCellStyle();
            cellStyle.setAlignment(CellStyle.ALIGN_CENTER);
            if (value instanceof Date) {
                sheet.setColumnWidth(cell.getColumnIndex(), 20 * 256);
                cellStyle.setDataFormat(format.getFormat(EXPORT_DATE_FORMAT));
                cell.setCellStyle(cellStyle);
                cell.setCellValue((Date) value);
            } else {
                if (value instanceof Number && (((Number) value).longValue() < Integer.MAX_VALUE)) {
                    cell.setCellType(Cell.CELL_TYPE_NUMERIC);
                    if (value instanceof Double) {
                        cellStyle.setDataFormat(format.getFormat("0.00"));
                        cell.setCellValue(((Number) value).doubleValue());
                    } else {
                        cellStyle.setDataFormat(format.getFormat("0"));
                        cell.setCellValue(((Number) value).longValue());
                    }
                } else {
                    cell.setCellType(Cell.CELL_TYPE_STRING);
                    cell.setCellValue(value.toString());
                    cellStyle.setDataFormat(format.getFormat("@"));
                }
            }
            cell.setCellStyle(cellStyle);
        } else {
            cell.setCellValue("");
        }
    }

    // public static void main(String[] args) {
    // Workbook wb = new HSSFWorkbook();
    // Sheet sheet = wb.createSheet("ExportDataList");
    //
    // // Create a row and put some cells in it. Rows are 0 based.
    // Row headerTitle = sheet.createRow(0);
    // Cell cell = headerTitle.createCell(0);
    // int dx1 = 0;
    // int dy1 = 0;
    // int dx2 = 0;
    // int dy2 = 127;
    // short col = 0;
    // int row = 0;
    // short col2 = 1;
    // int row2 = 1;
    // String value = "123456789012345";
    // int len = 0;
    // try {
    // len = value.getBytes("GBK").length;
    // } catch (UnsupportedEncodingException e1) {
    // // TODO Auto-generated catch block
    // e1.printStackTrace();
    // }
    // System.out.println(len);
    // if (len > 24) {
    // col2 = (short) (col + 3);
    // row2 = row + (len - 1) / 24 + 1;
    // } else {
    // col2 = (short) (col + (len + 2) / 9 + 1);
    // dx2 = ((len + 3) % 9) * 113;
    // }
    // cell.setCellValue(value);
    // Drawing draw = sheet.createDrawingPatriarch();
    //
    // Comment comment = draw.createCellComment(new HSSFClientAnchor(dx1, dy1, dx2, dy2, col, row, col2, row2));
    // comment.setString(new HSSFRichTextString(value));
    // comment.setVisible(false);
    // cell.setCellComment(comment);
    // OutputStream os = null;
    // try {
    // os = new FileOutputStream("/info/" + System.currentTimeMillis() + ".xls");
    // wb.write(os);
    // os.flush();
    // } catch (IOException e) {
    // // TODO Auto-generated catch block
    // e.printStackTrace();
    // } finally {
    // if (os != null) {
    // try {
    // os.close();
    // } catch (IOException e) {
    // // TODO Auto-generated catch block
    // e.printStackTrace();
    // }
    // os = null;
    // }
    // }
    //
    // }
}
