package com.baijia.tianxiao.sal.marketing.draw.service.impl;

import java.lang.reflect.InvocationTargetException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;

import org.apache.commons.beanutils.BeanUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.baijia.tianxiao.dal.activity.constants.TemplateConstant;
import com.baijia.tianxiao.dal.activity.dao.ActivityAccessLogDao;
import com.baijia.tianxiao.dal.activity.dao.TemplateDao;
import com.baijia.tianxiao.dal.activity.dao.TxActivityCommonDao;
import com.baijia.tianxiao.dal.activity.dao.draw.ActivityUserDao;
import com.baijia.tianxiao.dal.activity.dao.draw.DrawActivityDao;
import com.baijia.tianxiao.dal.activity.dao.draw.PrizeInfoDao;
import com.baijia.tianxiao.dal.activity.dao.draw.WinnerDao;
import com.baijia.tianxiao.dal.activity.po.TxActivityCommon;
import com.baijia.tianxiao.dal.activity.po.draw.DrawInfo;
import com.baijia.tianxiao.dal.activity.po.draw.PrizeInfo;
import com.baijia.tianxiao.sal.marketing.commons.constants.Config;
import com.baijia.tianxiao.sal.marketing.commons.constants.TemplateSuffix;
import com.baijia.tianxiao.sal.marketing.commons.enums.ConstantEnums;
import com.baijia.tianxiao.sal.marketing.commons.enums.TemplateTypeCategory;
import com.baijia.tianxiao.sal.marketing.commons.service.RedisService;
import com.baijia.tianxiao.sal.marketing.commons.service.TxActivityCommonService;
import com.baijia.tianxiao.sal.marketing.commons.utils.TwoTuple;
import com.baijia.tianxiao.sal.marketing.draw.dto.DrawActivityBase;
import com.baijia.tianxiao.sal.marketing.draw.dto.DrawActivityDetail;
import com.baijia.tianxiao.sal.marketing.draw.dto.DrawActivityRequest;
import com.baijia.tianxiao.sal.marketing.draw.dto.DrawActivityResponse;
import com.baijia.tianxiao.sal.marketing.draw.dto.DrawCacheDto;
import com.baijia.tianxiao.sal.marketing.draw.service.DrawPrizesService;
import com.baijia.tianxiao.sal.marketing.draw.service.DrawService;
import com.baijia.tianxiao.util.GenericsUtils;

import lombok.extern.slf4j.Slf4j;

/**
 * @author Rezar
 * @createDate :Jan 25, 2016 11:12:23 AM
 * @desc :
 */
@Service
@Slf4j
public class DrawServiceImpl implements DrawService {

    private Logger logger = LoggerFactory.getLogger(getClass());

    @Autowired
    private DrawActivityDao drawActivityDao;
    @Autowired
    private DrawPrizesService drawPrizesService;
    @Autowired
    private PrizeInfoDao prizeInfoDao;
    @Autowired
    private TemplateDao templateDao;
    @Autowired
    private ActivityAccessLogDao accessLogDao;
    @Autowired
    private WinnerDao winnerDao;
    @Autowired
    private ActivityUserDao activityUserDao;
    @Autowired
    private TxActivityCommonDao txActivityCommonDao;
    @Autowired
    private TxActivityCommonService txActivityCommonService;
    @Autowired
    private RedisService redisService;

    @Override
    @Transactional("yunyingTransactionManager")
    public DrawActivityDetail modifyActivity(DrawActivityRequest request) {
        DrawInfo drawInfo = drawActivityDao.getDrawInfoDetail(request.getActivityId(), request.getOrgId());
        if (drawInfo == null) {
            logger.warn("[Draw] Draw is not exist.activityId=" + request.getActivityId());
            return null;
        }
        TwoTuple<DrawInfo, List<PrizeInfo>> buildDetailPo = DrawActivityDetail.buildDetailPo(request);
        DrawInfo di = buildDetailPo.first;
        Long current = new Date().getTime();
        Timestamp currentTimestamp = new Timestamp(current);
        di.setUpdateTime(currentTimestamp);
        this.drawActivityDao.updateDrawActivity(di);
        this.reSetDrawCache(di.getActivityId());

        // 同步tx_activity_conf
        TxActivityCommon conf = txActivityCommonDao.getTxActivityCommon(drawInfo.getOrgId(),
            drawInfo.getTemplateTypeId(), drawInfo.getActivityId());
        txActivityCommonService.saveOrUpdateTxActivityCommon(conf, TemplateTypeCategory.DRAW_TYPE, drawInfo, null);

        List<PrizeInfo> prizeInfos = buildDetailPo.second;
        if (GenericsUtils.notNullAndEmpty(prizeInfos)) {
            Long activityId = request.getActivityId();
            for (PrizeInfo pi : prizeInfos) {
                pi.setActivityId(activityId);
                if (GenericsUtils.isNullOrEmpty(pi.getUrl())) {
                    String defaultImgUrl = ConstantEnums.findProperties(TemplateSuffix
                        .getSuffixKey(TemplateTypeCategory.DRAW_TYPE, drawInfo.getTemplateId(), "DEFAULT_IMAGE"));
                    pi.setUrl(defaultImgUrl);
                }
            }
            this.drawPrizesService.updatePrizes(activityId, prizeInfos);
        }
        templateDao.updateTemplateUseCount(drawInfo.getTemplateId().longValue());

        DrawCacheDto cacheDto = redisService.getDrawActivityBase(request.getActivityId());
        if (cacheDto != null) {
            redisService.setDrawActivityBase(request.getActivityId(),
                getDrawCacheDtoFromDbById(request.getActivityId()));
        }
        return this.getActivityDetail(request);
    }

    @Override
    public DrawActivityDetail getActivityDetail(DrawActivityRequest request) {
        Long activityId = request.getActivityId();
        DrawInfo di = this.drawActivityDao.getDrawInfoDetail(activityId, null, null);
        if (di == null) {
            return null;
        }
        List<PrizeInfo> prizeInfos = this.drawPrizesService.getPrizeInfosByActivityId(activityId);
        DrawActivityDetail detail = DrawActivityDetail.buildDetailDto(di, prizeInfos);
        setStatistical(Arrays.asList(detail));
        DrawActivityResponse dar = new DrawActivityResponse();
        try {
            BeanUtils.copyProperties(dar, detail);
            dar.setPrizes(dar.getPrizeList());
        } catch (IllegalAccessException | InvocationTargetException e) {
            log.error("error : ", e);
        }
        String templateUrl = TemplateConstant.getTemplateUrl(di.getTemplateId());
        templateUrl = templateUrl.replaceAll("templateId=\\d+", "");
        String url = Config.DOMAIN + templateUrl + "activityId=" + activityId + "#!/";
        dar.setUrl(url);
        return dar;
    }

    private void setStatistical(List<? extends DrawActivityBase> baseInfos) {
        if (baseInfos == null) {
            return;
        }
        List<Long> list = new ArrayList<>(baseInfos.size());
        for (DrawActivityBase base : baseInfos) {
            list.add(base.getActivityId());
        }

        Map<Long, Integer> accessMap =
            accessLogDao.selectTotalByActivityId(list, TemplateTypeCategory.DRAW_TYPE.getType());
        Map<Long, Integer> userMap =
            activityUserDao.selectTotalByActivityIdAndCategory(list, TemplateTypeCategory.DRAW_TYPE.getType());
        Map<Long, Integer> winnerMap = winnerDao.selectTotalByActivityId(list);

        for (DrawActivityBase baseInfo : baseInfos) {
            Integer browseCount = accessMap.get(baseInfo.getActivityId());
            Integer userCount = userMap.get(baseInfo.getActivityId());
            Integer winnerCount = winnerMap.get(baseInfo.getActivityId());

            baseInfo.setBrowseCount(browseCount == null ? 0 : browseCount);
            baseInfo.setUserCount(userCount == null ? 0 : userCount);
            baseInfo.setWinnerCount(winnerCount == null ? 0 : winnerCount);
        }
    }

    @Override
    public List<DrawActivityBase> getActivityList(DrawActivityRequest request) {
        List<DrawInfo> listDrawActivity = this.drawActivityDao.listDrawActivity(request.getStatus(),
            request.getLastId(), request.getOrgId(), request.getPageDto());
        if (GenericsUtils.isNullOrEmpty(listDrawActivity)) {
            return Collections.emptyList();
        }
        List<DrawActivityBase> baseInfos = new ArrayList<>(listDrawActivity.size());
        for (DrawInfo di : listDrawActivity) {
            DrawActivityBase baseInfo = new DrawActivityBase().buildDto(di);
            baseInfos.add(baseInfo);
        }
        setStatistical(baseInfos);
        return baseInfos;
    }

    @Override
    @Transactional("yunyingTransactionManager")
    public boolean modifyStatus(DrawActivityRequest request) {
        DrawInfo di = request.buildPo();
        logger.info("[ModifyStatus]" + di.getStatus());
        try {
            di.setUpdateTime(new Timestamp(new Date().getTime()));
            this.drawActivityDao.updateDrawActivity(di);

            // 同步tx_activity_conf
            log.info("[Draw] drawInfo:{}", di);
            TxActivityCommon conf = txActivityCommonDao.getTxActivityCommon(di.getOrgId(),
                TemplateTypeCategory.DRAW_TYPE.getTypeId(), di.getActivityId());
            conf.setStatus(request.getStatus());
            conf.setUpdateTime(new Date());
            txActivityCommonService.saveOrUpdateTxActivityCommon(conf, null, null, null, "status", "updateTime");

            DrawCacheDto cacheDto = redisService.getDrawActivityBase(request.getActivityId());
            if (cacheDto != null) {
                cacheDto.setStatus(request.getStatus());
                logger.info("[ModifyStatus] Cache=" + request.getStatus());
                redisService.setDrawActivityBase(request.getActivityId(), cacheDto);
            }
            return true;
        } catch (Exception ex) {
            logger.error("", ex);
            return false;
        }
    }

    @Override
    public DrawCacheDto getDrawCacheDtoFromDbById(long activityId) {
        DrawInfo drawInfo = drawActivityDao.getDrawInfoDetail(activityId, null);
        List<PrizeInfo> prizeInfos = prizeInfoDao.getPrizeInfosByActivityId(activityId);
        return DrawCacheDto.getInstance(drawInfo, prizeInfos);
    }

    /**
     * @param activityId
     * @return
     */
    public void reSetDrawCache(long activityId) {
        DrawCacheDto drawCacheDto = this.getDrawCacheDtoFromDbById(activityId);
        redisService.setDrawActivityBase(activityId, drawCacheDto);
    }

}
